-- $Id: vcg.adb 12952 2009-04-09 13:13:58Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with Debug;

with
     Cells,
     Fatal,
     File_Utils,
     Graph,
     LexTokenLists,
     Declarations,
     FileSystem,
     SPARK_IO,
     EStrings,
     ErrorHandler,
     ScreenEcho;

use type SPARK_IO.File_Status;
use type File_Utils.FileTypes;

package body VCG
is

   ------------------------------------------------------------------------------

   procedure ProduceVCs (VCGheap                   : in out Cells.Heap_Record;
                         StartNode                 : in     STree.SyntaxNode;
                         SubprogSym                : in     Dictionary.Symbol;
                         Scope                     : in     Dictionary.Scopes;
                         OutputFile                : in     SPARK_IO.File_Type;
                         OutputFileName            : in     EStrings.T;
                         RedType                   : in     CommandLineData.RedTypes;
                         EndPosition               : in     LexTokenManager.TokenPosition;
                         FlowHeap                  : in out Heap.HeapRecord;
                         SemanticErrorInSubprogram : in     Boolean;
                         DataFlowErrorInSubprogram : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     STree.Table;
   --#        in out Declarations.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#           out Graph.Table;
   --#           out StmtStack.S;
   --# derives Declarations.State,
   --#         Dictionary.Dict,
   --#         Graph.Table,
   --#         LexTokenManager.StringTable,
   --#         StmtStack.S,
   --#         VCGheap                     from CommandLineData.Content,
   --#                                          DataFlowErrorInSubprogram,
   --#                                          Declarations.State,
   --#                                          Dictionary.Dict,
   --#                                          FlowHeap,
   --#                                          LexTokenManager.StringTable,
   --#                                          RedType,
   --#                                          Scope,
   --#                                          SemanticErrorInSubprogram,
   --#                                          StartNode,
   --#                                          STree.Table,
   --#                                          SubprogSym,
   --#                                          VCGheap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS           from CommandLineData.Content,
   --#                                          DataFlowErrorInSubprogram,
   --#                                          Declarations.State,
   --#                                          Dictionary.Dict,
   --#                                          EndPosition,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          FlowHeap,
   --#                                          LexTokenManager.StringTable,
   --#                                          OutputFile,
   --#                                          RedType,
   --#                                          Scope,
   --#                                          SemanticErrorInSubprogram,
   --#                                          SPARK_IO.FILE_SYS,
   --#                                          StartNode,
   --#                                          STree.Table,
   --#                                          SubprogSym,
   --#                                          VCGheap &
   --#         FlowHeap                    from *,
   --#                                          CommandLineData.Content,
   --#                                          DataFlowErrorInSubprogram,
   --#                                          Dictionary.Dict,
   --#                                          LexTokenManager.StringTable,
   --#                                          RedType,
   --#                                          Scope,
   --#                                          SemanticErrorInSubprogram,
   --#                                          StartNode,
   --#                                          STree.Table,
   --#                                          SubprogSym,
   --#                                          VCGheap &
   --#         Statistics.TableUsage       from *,
   --#                                          CommandLineData.Content,
   --#                                          DataFlowErrorInSubprogram,
   --#                                          Declarations.State,
   --#                                          Dictionary.Dict,
   --#                                          FlowHeap,
   --#                                          LexTokenManager.StringTable,
   --#                                          RedType,
   --#                                          Scope,
   --#                                          SemanticErrorInSubprogram,
   --#                                          StartNode,
   --#                                          STree.Table,
   --#                                          SubprogSym,
   --#                                          VCGheap &
   --#         null                        from OutputFileName;
      is  separate;

   ------------------------------------------------------------------------------

   procedure FullSymbolName (Item      : in     Dictionary.Symbol;
                             Ancestors :    out LexTokenLists.Lists;
                             List      :    out LexTokenLists.Lists)
   --# global in Dictionary.Dict;
   --# derives Ancestors,
   --#         List      from Dictionary.Dict,
   --#                        Item;
   is
      ReversePrefixList,
      FullSymbolNameList : LexTokenLists.Lists;
      AncestorList       : LexTokenLists.Lists;
      PackSym            : Dictionary.Symbol;
      Scope              : Dictionary.Scopes;

   begin
      ReversePrefixList := LexTokenLists.NullList;
      Scope := Dictionary.GetScope (Item);
      loop
         exit when Dictionary.IsGlobalScope (Scope) or else
           Scope = Dictionary.VisibleScope (Dictionary.GetPredefinedPackageStandard) or else
           Dictionary.IsPredefinedScope (Scope);
         LexTokenLists.Append (ReversePrefixList,
                               Dictionary.GetSimpleName (Dictionary.GetRegion (Scope)));
         Scope := Dictionary.GetEnclosingScope (Scope);
      end loop;

      FullSymbolNameList := LexTokenLists.NullList;

      while ReversePrefixList.Length > 0 loop
         FullSymbolNameList.Length := FullSymbolNameList.Length + 1;
         FullSymbolNameList.Content (FullSymbolNameList.Length) :=
           ReversePrefixList.Content (ReversePrefixList.Length);
         ReversePrefixList.Length := ReversePrefixList.Length - 1;
      end loop;

      LexTokenLists.Append (FullSymbolNameList, Dictionary.GetSimpleName (Item));
      List := FullSymbolNameList;
      AncestorList := LexTokenLists.NullList;
      PackSym := Dictionary.GetLibraryPackage (Dictionary.GetScope (Item));

      if PackSym /= Dictionary.GetPredefinedPackageStandard then
         ReversePrefixList := LexTokenLists.NullList;
         loop
            PackSym := Dictionary.GetPackageParent (PackSym);
            exit when PackSym = Dictionary.NullSymbol;
            LexTokenLists.Append (ReversePrefixList,
                                  Dictionary.GetSimpleName (PackSym));
         end loop;

         while ReversePrefixList.Length > 0 loop
            AncestorList.Length := AncestorList.Length + 1;
            AncestorList.Content (AncestorList.Length) :=
              ReversePrefixList.Content (ReversePrefixList.Length);
            ReversePrefixList.Length := ReversePrefixList.Length - 1;
         end loop;
      end if;
      Ancestors := AncestorList;

   end FullSymbolName;

   --------------------------------------------

   procedure ProduceOutputFiles (SubprogSym       : in     Dictionary.Symbol;
                                 RedType          : in     CommandLineData.RedTypes;
                                 OutputFile       : in out SPARK_IO.File_Type;
                                 DeclarationsFile : in out SPARK_IO.File_Type;
                                 RuleFile         : in out SPARK_IO.File_Type;
                                 OutputFileName   :    out EStrings.T;
                                 OK               :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives DeclarationsFile,
   --#         OutputFile,
   --#         RuleFile          from *,
   --#                                CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.StringTable,
   --#                                RedType,
   --#                                SPARK_IO.FILE_SYS,
   --#                                SubprogSym &
   --#         Ok,
   --#         SPARK_IO.FILE_SYS from CommandLineData.Content,
   --#                                DeclarationsFile,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.StringTable,
   --#                                OutputFile,
   --#                                RedType,
   --#                                RuleFile,
   --#                                SPARK_IO.FILE_SYS,
   --#                                SubprogSym &
   --#         OutputFileName    from CommandLineData.Content,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.StringTable,
   --#                                RedType,
   --#                                SPARK_IO.FILE_SYS,
   --#                                SubprogSym;
   is
      VCSExtension : constant EStrings.T :=
         EStrings.T'(3,
                                         EStrings.Contents'('v', 'c', 'g', others => ' '));
      PFSExtension : constant EStrings.T :=
         EStrings.T'(3,
                                         EStrings.Contents'('p', 'f', 's', others => ' '));
      FDLExtension : constant EStrings.T :=
         EStrings.T'(3,
                                         EStrings.Contents'('f', 'd', 'l', others => ' '));
      DECExtension : constant EStrings.T :=
         EStrings.T'(3,
                                         EStrings.Contents'('d', 'e', 'c', others => ' '));
      RLSExtension : constant EStrings.T :=
         EStrings.T'(3,
                                         EStrings.Contents'('r', 'l', 's', others => ' '));

      LocalOK          : Boolean;

      UnitName         : LexTokenLists.Lists;
      AncestorName     : LexTokenLists.Lists;
      FileName         : EStrings.T;

      --------------------------------------------

      procedure BuildFileNameNest
      --# global in     AncestorName;
      --#        in     CommandLineData.Content;
      --#        in     LexTokenManager.StringTable;
      --#        in     UnitName;
      --#        in out SPARK_IO.FILE_SYS;
      --#           out FileName;
      --#           out LocalOk;
      --# derives FileName          from AncestorName,
      --#                                LexTokenManager.StringTable,
      --#                                UnitName &
      --#         LocalOk,
      --#         SPARK_IO.FILE_SYS from AncestorName,
      --#                                CommandLineData.Content,
      --#                                LexTokenManager.StringTable,
      --#                                SPARK_IO.FILE_SYS,
      --#                                UnitName;
      is
         pos         : LexTokenLists.Lengths;
         Str         : EStrings.T;
         AdjustedDir : EStrings.T;
      begin
         LocalOK := True;
         FileName := EStrings.EmptyString;
         if UnitName.Length /= 1 then -- is not a main program so build nested dirs
            EStrings.AppendExaminerString (FileName, FileSystem.StartOfDirectory);

            pos := 1;
            loop
               exit when pos > AncestorName.Length;
               LexTokenManager.LexStringToString (AncestorName.Content (pos), Str);
               if Str.Length > FileSystem.MaxFileNameLength - 1 then
                  Str.Length := FileSystem.MaxFileNameLength - 1;
               end if;

               -- Note that directories for VCG files are always created
               -- using lower-case names on all platforms.
               EStrings.AppendExaminerStringTruncate (FileName,
                                                      EStrings.LowerCase (Str));
               EStrings.AppendStringTruncate (FileName, "_");
               if LocalOK then
                  -- If the user has asked for an alterative output directory, then start
                  -- there, otherwise start at current working directory
                  AdjustedDir := FileName;
                  CommandLineData.Normalize_FileName_To_Output_Directory (AdjustedDir);

                  FileSystem.IdempotentCreateSubdirectory (AdjustedDir, LocalOK);
               end if;
               EStrings.AppendExaminerStringTruncate (FileName,
                                                      FileSystem.BetweenDirectories);
               pos := pos + 1;
            end loop;

            pos := 1;
            loop
               LexTokenManager.LexStringToString (UnitName.Content (pos), Str);
               if Str.Length > FileSystem.MaxFileNameLength then
                  Str.Length := FileSystem.MaxFileNameLength;
               end if;

               -- Note that directories for VCG files are always created
               -- using lower-case names on all platforms.
               EStrings.AppendExaminerStringTruncate (FileName,
                                                      EStrings.LowerCase (Str));

               -- If the user has asked for an alterative output directory, then start
               -- there, otherwise start at current working directory
               AdjustedDir := FileName;
               CommandLineData.Normalize_FileName_To_Output_Directory (AdjustedDir);

               FileSystem.IdempotentCreateSubdirectory (AdjustedDir, LocalOK);

               exit when pos = UnitName.Length - 1;
               EStrings.AppendExaminerStringTruncate (FileName,
                                                      FileSystem.BetweenDirectories);
               pos := pos + 1;
            end loop;
            EStrings.AppendExaminerStringTruncate (FileName,
                                                   FileSystem.EndOfPath);
         end if;

         LexTokenManager.LexStringToString (UnitName.Content (UnitName.Length),
                                            Str);
         if Str.Length > FileSystem.MaxFileNameLength then
            Str.Length := FileSystem.MaxFileNameLength;
         end if;
         EStrings.AppendExaminerStringTruncate (FileName,
                                                EStrings.LowerCase (Str));
         LocalOK := LocalOK and then
           FileName.Length < EStrings.MaxStringLength - 4;
      end BuildFileNameNest;

      --------------------------------------------

      procedure PutSubprogramName (File     : in SPARK_IO.File_Type;
                                   Sym      : in Dictionary.Symbol;
                                   FileType : in File_Utils.FileTypes)
      --# global in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in     UnitName;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives SPARK_IO.FILE_SYS from *,
      --#                                Dictionary.Dict,
      --#                                File,
      --#                                FileType,
      --#                                LexTokenManager.StringTable,
      --#                                Sym,
      --#                                UnitName;
      is
         pos      : LexTokenLists.Lengths;
         Str,
         HeadLine : EStrings.T;

         PageWidth : constant Natural := 78;

      begin
         if Dictionary.IsFunction (Sym) then
            EStrings.CopyString (HeadLine, "function ");
         elsif Dictionary.IsProcedure (Sym) then
            EStrings.CopyString (HeadLine, "procedure ");
         elsif Dictionary.IsTaskType (Sym) then
            EStrings.CopyString (HeadLine, "task body ");
         else
            EStrings.CopyString (HeadLine, "initialization of ");
         end if;
         pos := 1;
         loop
            LexTokenManager.LexStringToString (UnitName.Content (pos), Str);
            EStrings.AppendExaminerString (HeadLine, Str);
            exit when HeadLine.Length = EStrings.MaxStringLength;
            exit when pos = UnitName.Length;
            EStrings.AppendString (HeadLine, ".");
            pos := pos + 1;
         end loop;
         if (HeadLine.Length + 1) < PageWidth then
            SPARK_IO.Set_Col (File, (PageWidth - HeadLine.Length)/2);
         end if;
         if FileType = File_Utils.DecFile then
            SPARK_IO.Put_Char (File, '{');
            EStrings.PutString (File, HeadLine);
            SPARK_IO.Put_Char (File, '}');
            SPARK_IO.New_Line (File, 1);
         elsif FileType = File_Utils.RuleFile then
            SPARK_IO.Put_String (File, "/*", 0);
            EStrings.PutString (File, HeadLine);
            SPARK_IO.Put_String (File, "*/", 0);
            SPARK_IO.New_Line (File, 1);
         else
            EStrings.PutLine (File, HeadLine);
         end if;
         SPARK_IO.New_Line (File, 2);
      end PutSubprogramName;

      ---------------------------------------------------------

      procedure ProduceOutputFile
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     FileName;
      --#        in     LexTokenManager.StringTable;
      --#        in     RedType;
      --#        in     SubprogSym;
      --#        in     UnitName;
      --#        in out OutputFile;
      --#        in out SPARK_IO.FILE_SYS;
      --#           out LocalOk;
      --#           out OutputFileName;
      --# derives LocalOk,
      --#         OutputFile        from FileName,
      --#                                OutputFile,
      --#                                RedType,
      --#                                SPARK_IO.FILE_SYS &
      --#         OutputFileName    from FileName,
      --#                                RedType &
      --#         SPARK_IO.FILE_SYS from *,
      --#                                CommandLineData.Content,
      --#                                Dictionary.Dict,
      --#                                FileName,
      --#                                LexTokenManager.StringTable,
      --#                                OutputFile,
      --#                                RedType,
      --#                                SubprogSym,
      --#                                UnitName;
      is
         Success : SPARK_IO.File_Status;
      begin
         OutputFileName := FileName;
         if RedType.PFs then
            FileSystem.CheckExtension (OutputFileName,
                                       PFSExtension);
         elsif RedType.VCs then
            FileSystem.CheckExtension (OutputFileName,
                                       VCSExtension);
         end if;

         OutputFileName := FileSystem.CaseOfVCGFiles (OutputFileName);
         SPARK_IO.Create (OutputFile,
                          OutputFileName.Length,
                          OutputFileName.Content,
                          "",
                          Success);

         LocalOK := Success = SPARK_IO.Ok;
         if LocalOK then
            File_Utils.PrintAHeader (OutputFile,
                                 "Semantic Analysis of SPARK Text",
                                 File_Utils.OtherFile);
            PutSubprogramName (OutputFile, SubprogSym, File_Utils.OtherFile);
         end if;
      end ProduceOutputFile;

      ------------------------------------------------------------

      procedure ProduceDeclarationsFile
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     FileName;
      --#        in     LexTokenManager.StringTable;
      --#        in     RedType;
      --#        in     SubprogSym;
      --#        in     UnitName;
      --#        in out DeclarationsFile;
      --#        in out SPARK_IO.FILE_SYS;
      --#           out LocalOk;
      --# derives DeclarationsFile,
      --#         LocalOk           from DeclarationsFile,
      --#                                FileName,
      --#                                RedType,
      --#                                SPARK_IO.FILE_SYS &
      --#         SPARK_IO.FILE_SYS from *,
      --#                                CommandLineData.Content,
      --#                                DeclarationsFile,
      --#                                Dictionary.Dict,
      --#                                FileName,
      --#                                LexTokenManager.StringTable,
      --#                                RedType,
      --#                                SubprogSym,
      --#                                UnitName;
      is
         DeclarationsFileName : EStrings.T;
         Success        : SPARK_IO.File_Status;
      begin
         DeclarationsFileName := FileName;
         if RedType.PFs then
            FileSystem.CheckExtension (DeclarationsFileName,
                                       DECExtension);
         else
            FileSystem.CheckExtension (DeclarationsFileName,
                                       FDLExtension);
         end if;

         -- casing is tgt-dependent
         DeclarationsFileName := FileSystem.CaseOfVCGFiles (DeclarationsFileName);

         SPARK_IO.Create (DeclarationsFile,
                          DeclarationsFileName.Length,
                          DeclarationsFileName.Content,
                          "",
                          Success);
         LocalOK := Success = SPARK_IO.Ok;
         if LocalOK then
            File_Utils.PrintAHeader (DeclarationsFile,
                                 "FDL Declarations",
                                 File_Utils.DecFile);
         end if;
         PutSubprogramName (DeclarationsFile, SubprogSym, File_Utils.DecFile);
      end ProduceDeclarationsFile;

      ------------------------------------------------------------------

      procedure ProduceRuleFile
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     FileName;
      --#        in     LexTokenManager.StringTable;
      --#        in     RedType;
      --#        in     SubprogSym;
      --#        in     UnitName;
      --#        in out RuleFile;
      --#        in out SPARK_IO.FILE_SYS;
      --#           out LocalOk;
      --# derives LocalOk,
      --#         RuleFile          from FileName,
      --#                                RedType,
      --#                                RuleFile,
      --#                                SPARK_IO.FILE_SYS &
      --#         SPARK_IO.FILE_SYS from *,
      --#                                CommandLineData.Content,
      --#                                Dictionary.Dict,
      --#                                FileName,
      --#                                LexTokenManager.StringTable,
      --#                                RedType,
      --#                                RuleFile,
      --#                                SubprogSym,
      --#                                UnitName;
      is
         RuleFileName   : EStrings.T;
         Success        : SPARK_IO.File_Status;

      begin
         RuleFileName := FileName;
         if RedType.VCs then
            FileSystem.CheckExtension (RuleFileName,
                                       RLSExtension);

            -- casing is tgt-dependent
            RuleFileName := FileSystem.CaseOfVCGFiles (RuleFileName);

            SPARK_IO.Create (RuleFile,
                             RuleFileName.Length,
                             RuleFileName.Content,
                             "",
                             Success);
            LocalOK := Success = SPARK_IO.Ok;

            if LocalOK then
               File_Utils.PrintAHeader (RuleFile,
                                        "Proof Rule Declarations",
                                        File_Utils.RuleFile);
            end if;
            PutSubprogramName (RuleFile, SubprogSym, File_Utils.RuleFile);
         else -- no rule file required so set to Standard_Output to prevent constraint error
            LocalOK := True;
            RuleFile := SPARK_IO.Standard_Output;
         end if;
      end ProduceRuleFile;

      ------------------------------------------------------------------

   begin -- ProduceOutputFiles
      FullSymbolName (SubprogSym,
                      AncestorName,
                      UnitName);

      BuildFileNameNest;

      -- FileName now contains the basename of the required output file(s)
      -- without an extension.  It is also a relative to the current working
      -- directory - for example for subprogram P.Q, we end up with FileName
      -- being "p/q"
      --
      -- If the user has requested an alternative output directory, then we adjust
      -- FileName now
      CommandLineData.Normalize_FileName_To_Output_Directory (FileName);

      OutputFileName := EStrings.EmptyString;

      if LocalOK then
         ProduceOutputFile;
         if LocalOK then
            ProduceDeclarationsFile;
            if LocalOK then
               ProduceRuleFile;
            end if;
         end if;
      end if;
      OK := LocalOK;
      --# accept Flow, 601, DeclarationsFile, OutputFile, "ignore data coupling between files thro' SPARK_IO" &
      --#        Flow, 601, RuleFile, OutputFile, "ignore data coupling between files thro' SPARK_IO" &
      --#        Flow, 601, RuleFile, DeclarationsFile, "ignore data coupling between files thro' SPARK_IO";
   end ProduceOutputFiles;

   ---------------------------------------------------------------------------

   procedure GenerateVCsLocal (StartNode                 : in     STree.SyntaxNode;
                               Scope                     : in     Dictionary.Scopes;
                               RedType                   : in     CommandLineData.RedTypes;
                               EndPosition               : in     LexTokenManager.TokenPosition;
                               FlowHeap                  : in out Heap.HeapRecord;
                               SemanticErrorInSubprogram : in     Boolean;
                               DataFlowErrorInSubprogram : in     Boolean)
   --# global in     CommandLineData.Content;
   --#        in     STree.Table;
   --#        in out Declarations.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out Graph.Table;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out StmtStack.S;
   --# derives Declarations.State,
   --#         Dictionary.Dict,
   --#         FlowHeap,
   --#         Graph.Table,
   --#         LexTokenManager.StringTable,
   --#         Statistics.TableUsage,
   --#         StmtStack.S                 from *,
   --#                                          CommandLineData.Content,
   --#                                          DataFlowErrorInSubprogram,
   --#                                          Dictionary.Dict,
   --#                                          FlowHeap,
   --#                                          LexTokenManager.StringTable,
   --#                                          RedType,
   --#                                          Scope,
   --#                                          SemanticErrorInSubprogram,
   --#                                          SPARK_IO.FILE_SYS,
   --#                                          StartNode,
   --#                                          STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS           from CommandLineData.Content,
   --#                                          DataFlowErrorInSubprogram,
   --#                                          Dictionary.Dict,
   --#                                          EndPosition,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          FlowHeap,
   --#                                          LexTokenManager.StringTable,
   --#                                          RedType,
   --#                                          Scope,
   --#                                          SemanticErrorInSubprogram,
   --#                                          SPARK_IO.FILE_SYS,
   --#                                          StartNode,
   --#                                          STree.Table;
   is
      SubprogSym        : Dictionary.Symbol;
      OutputFile        : SPARK_IO.File_Type;
      OutputFileName    : EStrings.T;
      DeclarationsFile  : SPARK_IO.File_Type;
      RuleFile          : SPARK_IO.File_Type;
      OK                : Boolean;
      Success           : SPARK_IO.File_Status;
      VCGheap              : Cells.Heap_Record;
   begin
      OutputFile       := SPARK_IO.Standard_Output;
      DeclarationsFile := SPARK_IO.Standard_Output;
      RuleFile         := SPARK_IO.Standard_Output;

      if RedType /= CommandLineData.NoVCG then
         SubprogSym := Dictionary.GetRegion (Scope);
         ProduceOutputFiles (SubprogSym,
                             RedType,
                             OutputFile,
                             DeclarationsFile,
                             RuleFile,
                             OutputFileName,
                             OK);
         if OK then
            Cells.Initialize (VCGheap);
            Declarations.StartProcessing (VCGheap);
            ProduceVCs (VCGheap,
                        StartNode,
                        SubprogSym,
                        Scope,
                        OutputFile,
                        OutputFileName,
                        RedType,
                        EndPosition,
                        FlowHeap,
                        SemanticErrorInSubprogram,
                        DataFlowErrorInSubprogram);
            Declarations.OutputDeclarations (VCGheap,
                                             DeclarationsFile,
                                             RuleFile,
                                             Scope,
                                             RedType.VCs,
                                             EndPosition);
            Cells.Report_Usage (VCGheap);
         else
            -- Unable to create output files
            ErrorHandler.SemanticWarning (406,
                                          EndPosition,
                                          LexTokenManager.NullString);
         end if;
         --# accept Flow, 10, Success, "Expected ineffective assignment to Success" &
         --#        Flow, 10, OutputFile, "Expected ineffective assignment to OutputFile" &
         --#        Flow, 10, DeclarationsFile, "Expected ineffective assignment to DeclarationsFile" &
         --#        Flow, 10, RuleFile, "Expected ineffective assignment to RuleFile";
         SPARK_IO.Close (OutputFile, Success);                -- ignore ineff assig
         SPARK_IO.Close (DeclarationsFile, Success);          -- ignore ineff assig
         if RedType.VCs then
            SPARK_IO.Close (RuleFile, Success);   -- ignore ineff assig
         end if;
         --# end accept;
      end if;
      --# accept Flow, 33, Success, "Expected Success to be neither referenced nor exported";
   exception
      --# hide GenerateVCsLocal;
      when Fatal.StaticLimit =>
         -- Here owing to a VCG Heap/table exhausted.
         -- We need to close open files, making sure they are at least
         -- syntactically legal for the Simplifier.
         -- We insert an explicitly False VC here, so it is sure
         -- to be undischarged and picked up by POGS

         ErrorHandler.SemanticWarning (409,
                                       EndPosition,
                                       LexTokenManager.NullString);


         if SPARK_IO.Is_Open (OutputFile) then
            declare
               HeadLine         : EStrings.T;
               Str              : EStrings.T;
               UnitName         : LexTokenLists.Lists;
               AncestorName     : LexTokenLists.Lists;
            begin
               FullSymbolName (SubprogSym,
                               AncestorName,
                               UnitName);

               -- The header of the VC has to have the correct name and prefix
               -- for POGS, so we have to deduce this here from SubprogSym
               if Dictionary.IsFunction (SubprogSym) then
                  EStrings.CopyString (HeadLine, "function_");
               elsif Dictionary.IsTaskType (SubprogSym) then
                  EStrings.CopyString (HeadLine, "task_type_");
               else -- must be a procedure
                  EStrings.CopyString (HeadLine, "procedure_");
               end if;

               LexTokenManager.LexStringToString (UnitName.Content (UnitName.Length), Str);
               EStrings.AppendExaminerString (HeadLine, EStrings.LowerCase (Str));
               EStrings.AppendString (HeadLine, "_1.");

               SPARK_IO.New_Line (OutputFile, 1);
               SPARK_IO.Put_Line (OutputFile,
                                  "/* False VC generated due to VCG heap exhausted */", 0);
               SPARK_IO.New_Line (OutputFile, 2);
               SPARK_IO.Put_Line (OutputFile,
                                  "For path(s) from start to finish:", 0);
               SPARK_IO.New_Line (OutputFile, 1);
               EStrings.PutLine (OutputFile, HeadLine);
               SPARK_IO.Put_Line (OutputFile, "H1:    true .", 0);
               SPARK_IO.Put_Line (OutputFile, "        ->", 0);
               SPARK_IO.Put_Line (OutputFile, "C1:    false .", 0);
               SPARK_IO.New_Line (OutputFile, 2);
               SPARK_IO.Close (OutputFile, Success);
            end;
         end if;

         if RedType.VCs and then SPARK_IO.Is_Open (RuleFile) then
            SPARK_IO.Close (RuleFile, Success);
         end if;

         if SPARK_IO.Is_Open (DeclarationsFile) then
            -- Make sure the FDL file is termianted properly before closing it
            Declarations.PrintDeclarationTail (DeclarationsFile);
            SPARK_IO.Close (DeclarationsFile, Success);
         end if;

         -- We DONT'T re-raise here - there may be other subprograms
         -- requiring VC Generation in the enclosing unit, so we
         -- carry on.

      when others =>
         -- Any other exception reaching here.
         -- We need to close open files, then re-raise
         if SPARK_IO.Is_Open (OutputFile) then
            SPARK_IO.Close (OutputFile, Success);
         end if;

         if RedType.VCs and then SPARK_IO.Is_Open (RuleFile) then
            SPARK_IO.Close (RuleFile, Success);
         end if;

         if SPARK_IO.Is_Open (DeclarationsFile) then
            -- Make sure the FDL file is termianted properly before closing it
            Declarations.PrintDeclarationTail (DeclarationsFile);
            SPARK_IO.Close (DeclarationsFile, Success);
         end if;

         raise;
   end GenerateVCsLocal;


   procedure GenerateVCs (StartNode                 : in     STree.SyntaxNode;
                          Scope                     : in     Dictionary.Scopes;
                          RedType                   : in     CommandLineData.RedTypes;
                          EndPosition               : in     LexTokenManager.TokenPosition;
                          FlowHeap                  : in out Heap.HeapRecord;
                          SemanticErrorInSubprogram : in     Boolean;
                          DataFlowErrorInSubprogram : in     Boolean)
   is
      ErrorsInSubprogramOrItsSgnature : Boolean;
   begin
      if RedType /= CommandLineData.NoVCG then

         Invoked := True;

         ErrorsInSubprogramOrItsSgnature := SemanticErrorInSubprogram or
           (RedType.VCs and then
              not Dictionary.SubprogramSignatureIsWellformed (Dictionary.IsAbstract,
                                                              Dictionary.GetRegion (Scope)));

         if ErrorsInSubprogramOrItsSgnature then
            ErrorHandler.SemanticWarning (408,
                                          EndPosition,
                                          LexTokenManager.NullString);
         end if;

         GenerateVCsLocal (StartNode,
                           Scope,
                           RedType,
                           EndPosition,
                           FlowHeap,
                           ErrorsInSubprogramOrItsSgnature,
                           DataFlowErrorInSubprogram);
      end if;
   end GenerateVCs;

end VCG;
