-- $Id: stree.adb 12351 2009-02-02 15:03:51Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with SystemErrors,
     Statistics;
package body STree
is

   type NodeValue is
      record
         NodeType    : SPSymbols.SPSymbol;
         Position    : LexTokenManager.TokenPosition;
         Parent,
         Next,
         Child       : SyntaxNode;
         Ref         : ExaminerConstants.RefType;
      end record;

   type SyntaxTreeContents is array (SyntaxNode) of NodeValue;

   type TableStructure is
      record
         FreeList       : SyntaxNode;
         TopUsed        : SyntaxNode;
         CurrSyntaxNode : SyntaxNode;
         Contents       : SyntaxTreeContents;
      end record;

   Table : TableStructure;

   ------------------------------------------------------------------------

   procedure RetrieveCurrentRoot (Root : out SyntaxNode)
   is
   begin
      Root := Table.CurrSyntaxNode;
      Table.CurrSyntaxNode := NullNode;
   end RetrieveCurrentRoot;

   ------------------------------------------------------------------------

   function NodeToRef (Node : SyntaxNode) return ExaminerConstants.RefType
   is
   begin
      return ExaminerConstants.RefType (Node);
   end NodeToRef;

   ------------------------------------------------------------------------

   function RefToNode (Ref  : ExaminerConstants.RefType) return SyntaxNode
   is
   begin
      return SyntaxNode (Ref);
   end RefToNode;

   ------------------------------------------------------------------------

   procedure AllocateNode (Node : out SyntaxNode)
   --# global in out Table;
   --# derives Node,
   --#         Table from Table;
   is
      NewNode : SyntaxNode;
   begin
      if Table.FreeList /= NullNode then
         NewNode := Table.FreeList;
         -- Child field used as free list pointer.
         Table.FreeList :=
            Table.Contents (Table.FreeList).Child;
      elsif Table.TopUsed /= SyntaxTreeContents'Last then
         Table.TopUsed := Table.TopUsed + 1;
         NewNode := Table.TopUsed;
      else
         NewNode := NullNode;
      end if;

      if NewNode = NullNode then
         SystemErrors.FatalError (SystemErrors.SyntaxTreeOverflow, "");
      else
         Table.Contents (NewNode).Position :=
            LexTokenManager.TokenPosition'(0, 0);
         Table.Contents (NewNode).Parent := NullNode;
         Table.Contents (NewNode).Next := NullNode;
         Table.Contents (NewNode).Child := NullNode;
         Table.Contents (NewNode).Ref := 0;
      end if;

      Node := NewNode;
   end AllocateNode;

   ------------------------------------------------------------------------

   procedure NewProduction (Production : in SPSymbols.SPNonTerminal;
                            Node       : out SyntaxNode)
   is
      CurrNode  : SyntaxNode;
   begin
      AllocateNode (CurrNode);
      Table.Contents (CurrNode).NodeType := Production;
      Table.CurrSyntaxNode := CurrNode;
      Node := CurrNode;
   end NewProduction;

   ------------------------------------------------------------------------

   procedure NewTerminal (Terminal    : in SPSymbols.SPTerminal;
                          TerminalVal : in LexTokenManager.LexValue;
                          Node        : out SyntaxNode)
   is
      CurrNode  : SyntaxNode;
   begin
      AllocateNode (CurrNode);
      Table.Contents (CurrNode).NodeType := Terminal;
      Table.Contents (CurrNode).Position := TerminalVal.Position;
      Table.Contents (CurrNode).Ref :=
         ExaminerConstants.RefType (LexTokenManager.LexStringRef (TerminalVal.TokenStr));
      Node := CurrNode;
   end NewTerminal;

   ------------------------------------------------------------------------

   procedure AddDerivative (Child_Node : in SyntaxNode)
   is
      CurrNode  : SyntaxNode;
      LastChild : SyntaxNode;
      ThePosition : LexTokenManager.TokenPosition;
   begin
      CurrNode := Table.CurrSyntaxNode;
      LastChild := Table.Contents (CurrNode).Child;
      if LastChild = NullNode then
         Table.Contents (CurrNode).Child := Child_Node;
         ThePosition := Table.Contents (Child_Node).Position;
         Table.Contents (CurrNode).Position := ThePosition;
      else
         loop
            exit when Table.Contents (LastChild).Next = NullNode;
            LastChild := Table.Contents (LastChild).Next;
         end loop;
         if Table.Contents (CurrNode).Position =
            LexTokenManager.TokenPosition'(0, 0) then
            ThePosition := Table.Contents (Child_Node).Position;
            Table.Contents (CurrNode).Position := ThePosition;
         end if;
         Table.Contents (LastChild).Next := Child_Node;
      end if;
      Table.Contents (Child_Node).Parent := CurrNode;
   end AddDerivative;

   ------------------------------------------------------------------------

   procedure AddChildNode (Node         : in SyntaxNode;
                           ChildNode    : in SyntaxNode;
                           LinkToParent : in Boolean)
   is
      LastChild : SyntaxNode;
   begin
      LastChild := Table.Contents (Node).Child;
      if LastChild = NullNode then
         Table.Contents (Node).Child := ChildNode;
      else
         loop
            exit when Table.Contents (LastChild).Next = NullNode;
            LastChild := Table.Contents (LastChild).Next;
         end loop;
         Table.Contents (LastChild).Next := ChildNode;
      end if;
      if LinkToParent then
         Table.Contents (ChildNode).Parent := Node;
      end if;
   end AddChildNode;

   ------------------------------------------------------------------------

   function Child_Node (CurrentNode : SyntaxNode) return SyntaxNode
   is
   begin
      return Table.Contents (CurrentNode).Child;
   end Child_Node;

   ------------------------------------------------------------------------

   function Next_Sibling (CurrentNode : SyntaxNode) return SyntaxNode
   is
   begin
      return Table.Contents (CurrentNode).Next;
   end Next_Sibling;

   ------------------------------------------------------------------------

   function ParentNode (CurrentNode : SyntaxNode) return SyntaxNode
   is
   begin
      return Table.Contents (CurrentNode).Parent;
   end ParentNode;

   ------------------------------------------------------------------------

   function SyntaxNodeType (Node : SyntaxNode) return SPSymbols.SPSymbol
   is
   begin
      return Table.Contents (Node).NodeType;
   end SyntaxNodeType;

   ------------------------------------------------------------------------

   function NodePosition (Node : SyntaxNode) return LexTokenManager.TokenPosition
   is
      Position   : LexTokenManager.TokenPosition;
   begin
      if Node = NullNode then
         Position := LexTokenManager.TokenPosition'(0, 0);
      else
         Position := Table.Contents (Node).Position;
      end if;
      return Position;
   end NodePosition;

   ------------------------------------------------------------------------

   function NodeLexString (Node : SyntaxNode) return LexTokenManager.LexString
   is
      LexStr     : LexTokenManager.LexString;
   begin
      if Node /= NullNode and then
         (Table.Contents (Node).NodeType in
          SPSymbols.SPTerminal)
      then
         LexStr := LexTokenManager.ConvertLexStringRef
            (Natural (Table.Contents (Node).Ref));
      else
         LexStr := LexTokenManager.NullString;
      end if;
      return LexStr;
   end NodeLexString;

   ------------------------------------------------------------------------

   procedure CollectNode (Node : in SyntaxNode)
   --# global in out Table;
   --# derives Table from *,
   --#                    Node;
   is
      TheNodeValue : NodeValue;
   begin
      TheNodeValue := Table.Contents (NullNode);
      Table.Contents (Node) := TheNodeValue;
      Table.Contents (Node).Child := Table.FreeList;
      Table.FreeList := Node;
   end CollectNode;

   ----------------------------------------------------------------------

   procedure DeleteSyntaxTree (Root          : in SyntaxNode;
                               KeepConstants : in Boolean)
   is
      CurrNode,
      TheNextNode,
      TempNode : SyntaxNode;
   begin
      CurrNode := Root;
      loop
         TheNextNode := Child_Node (CurrNode);

         -- Subprogram constraints are needed by
         -- the VCG later, so we don't return such nodes to the free list.
         -- Similarly, constant declarations are needed by the Declarations
         -- package for proof rule generation.  Furthermore, we also need
         -- to protect generic_actual_part to preserve expressions used
         -- to initialize generic formal objects
         while TheNextNode /= NullNode and then
           ((SyntaxNodeType (TheNextNode) = SPSymbols.procedure_constraint) or
              (SyntaxNodeType (TheNextNode) = SPSymbols.function_constraint) or
              (KeepConstants and SyntaxNodeType (TheNextNode) = SPSymbols.constant_declaration) or
              (SyntaxNodeType (TheNextNode) = SPSymbols.generic_actual_part))
         loop
            Table.Contents (TheNextNode).Parent := NullNode;
            TempNode := Next_Sibling (TheNextNode);
            Table.Contents (TheNextNode).Next := NullNode;
            TheNextNode := TempNode;
         end loop;

         if TheNextNode = NullNode then
            loop
               TheNextNode := Next_Sibling (CurrNode);
               while TheNextNode /= NullNode and then
                 ((SyntaxNodeType (TheNextNode) = SPSymbols.procedure_constraint) or
                    (SyntaxNodeType (TheNextNode) = SPSymbols.function_constraint) or
                    (KeepConstants and SyntaxNodeType (TheNextNode) = SPSymbols.constant_declaration) or
                    (SyntaxNodeType (TheNextNode) = SPSymbols.generic_actual_part))
               loop
                  Table.Contents (TheNextNode).Parent := NullNode;
                  TempNode := Next_Sibling (TheNextNode);
                  Table.Contents (TheNextNode).Next := NullNode;
                  TheNextNode := TempNode;
               end loop;
               if TheNextNode /= NullNode then
                  CollectNode (CurrNode);
                  exit;
               end if;
               TheNextNode := ParentNode (CurrNode);
               CollectNode (CurrNode);
               exit when TheNextNode = NullNode;
               CurrNode := TheNextNode;
            end loop;
         end if;
         exit when TheNextNode = NullNode;
         CurrNode := TheNextNode;
      end loop;
   end DeleteSyntaxTree;

   -----------------------------------------------------------------------

   function TraverseAcross (RootNode : SyntaxNode;
                            CurrNode : SyntaxNode) return SyntaxNode
   --# global in Table;
   -- Traverses across the tree in a pre-order fashion.
   is
      TheNextNode : SyntaxNode;
   begin
      TheNextNode := CurrNode;
      while TheNextNode /= NullNode loop
         if TheNextNode = RootNode then
            TheNextNode := NullNode;
            exit;
         end if;
         if Next_Sibling (TheNextNode) /= NullNode then
            TheNextNode := Next_Sibling (TheNextNode);
            exit;
         end if;
         TheNextNode := ParentNode (TheNextNode);
      end loop;
      return TheNextNode;
   end TraverseAcross;

   -----------------------------------------------------------------------

   function Traverse (RootNode : SyntaxNode;
                      CurrNode : SyntaxNode) return SyntaxNode
   --# global in Table;
   -- Traverses the tree in a pre-order fashion.
   is
      TheNextNode : SyntaxNode;
   begin
      if Child_Node (CurrNode) /= NullNode then
         -- Depth first
         TheNextNode := Child_Node (CurrNode);
      else
         -- breadth next
         TheNextNode := TraverseAcross (RootNode, CurrNode);
      end if;
      return TheNextNode;
   end Traverse;

   -----------------------------------------------------------------------

   function LastChildOf (StartNode : SyntaxNode)
                        return SyntaxNode
   is
      LastValidNodeFound,
      TheNextNode : SyntaxNode;

   begin
      LastValidNodeFound := StartNode;
      TheNextNode := StartNode;
      while TheNextNode /= NullNode loop
         LastValidNodeFound := TheNextNode;
         TheNextNode := Child_Node (TheNextNode);
      end loop;

      return LastValidNodeFound;

   end LastChildOf;

   --------------------------------------------------------------

   function LastSiblingOf (StartNode : SyntaxNode)
                          return SyntaxNode
   is
      LastValidNodeFound,
      TheNextNode : SyntaxNode;

   begin
      LastValidNodeFound := StartNode;
      TheNextNode := StartNode;
      while TheNextNode /= NullNode loop
         LastValidNodeFound := TheNextNode;
         TheNextNode := Next_Sibling (TheNextNode);
      end loop;

      return LastValidNodeFound;

   end LastSiblingOf;

   --------------------------------------------------------------------

   procedure AddNodeSymbol (Node : in     SyntaxNode;
                            Sym  : in     Dictionary.Symbol)
   is
   begin
      Table.Contents (Node).Ref := Dictionary.SymbolRef (Sym);
   end AddNodeSymbol;

   --------------------------------------------------------------------

   function NodeSymbol (Node : SyntaxNode) return Dictionary.Symbol
   is
      Sym : Dictionary.Symbol;
   begin
      Sym := Dictionary.ConvertSymbolRef (Table.Contents (Node).Ref);
      return Sym;
   end NodeSymbol;

   -------------------------------------------------------

   procedure ReportUsage
   is
   begin
      -- TopUsed is peak usage, because freelist is used up before allocating
      -- a new cell at the top of the table
      Statistics.SetTableUsage (Statistics.SyntaxTree,
                                Integer (Table.TopUsed));
   end ReportUsage;

   -------------------------------------------------------

   function NextNodeType (It : Iterator) return Iterator
   --# global in Table;
   is
      Node   : SyntaxNode;
      NextIt : Iterator;
   begin
      Node := It.Current;
      while Node /= NullNode loop
         --# accept Flow, 41, "Expected stable expression";
         if It.SearchDirection = Down then -- expect stable expression
            if SyntaxNodeType (Node) = It.SearchNodeType then
               -- This node was returned in the search so ignore all nodes
               -- below it.
               Node := TraverseAcross (It.Root, Node);
            else
               -- Get the next pre-order node
               Node := Traverse (RootNode => It.Root,
                                 CurrNode => Node);
            end if;
         else
            Node := ParentNode (Node);
         end if;
         --# end accept;
         exit when SyntaxNodeType (Node) = It.SearchNodeType;
      end loop;

      if Node = NullNode then
         NextIt := NullIterator;
      else
         NextIt :=
           Iterator'(TheSearchKind   => It.TheSearchKind,
                     SearchNodeType  => It.SearchNodeType,
                     SearchDirection => It.SearchDirection,
                     Current         => Node,
                     Root            => It.Root);
      end if;
      return NextIt;
   end NextNodeType;

   -------------------------------------------------------

   function IsBranchNode (Node : SyntaxNode) return Boolean
   --# global in Table;
   is
   begin
      return Node /= NullNode and then
         -- has more than one child
        Child_Node (Node) /= NullNode and then
        Next_Sibling (Child_Node (Node)) /= NullNode;
   end IsBranchNode;

   -------------------------------------------------------

   function NextBranch (It : Iterator) return Iterator
   --# global in Table;
   is
      Node   : SyntaxNode;
      NextIt : Iterator;
   begin
      Node := It.Current;
      while Node /= NullNode loop
         --# accept Flow, 41, "Expected stable expression";
         if It.SearchDirection = Down then -- expect stable expression
            Node := Traverse (RootNode => It.Root,
                              CurrNode => Node);
         else
            Node := ParentNode (Node);
         end if;
         --# end accept;
         exit when IsBranchNode (Node);
      end loop;

      if Node = NullNode then
         NextIt := NullIterator;
      else
         NextIt :=
           Iterator'(TheSearchKind   => It.TheSearchKind,
                     SearchNodeType  => It.SearchNodeType,
                     SearchDirection => It.SearchDirection,
                     Current         => Node,
                     Root            => It.Root);
      end if;
      return NextIt;
   end NextBranch;

   -------------------------------------------------------

   function IsFormalParameterNode (Node     : SyntaxNode;
                                   RootNode : SyntaxNode) return Boolean
   --# global in Table;
   is
      MyNode : SyntaxNode;
      Result : Boolean;
   begin
      if SyntaxNodeType (Node) /= SPSymbols.identifier or else
        SyntaxNodeType (ParentNode (Node)) /= SPSymbols.simple_name then
         Result := False;
      else
         Result := True;
         MyNode := Node;
         while MyNode /= RootNode  loop
            if SyntaxNodeType (MyNode) = SPSymbols.expression then
               Result := False;
               exit;
            end if;
            MyNode := ParentNode (MyNode);
         end loop;
      end if;
      return Result;
   end IsFormalParameterNode;

   -------------------------------------------------------

   function NextFormalParameter (It : Iterator) return Iterator
   --# global in Table;
   is
      Node   : SyntaxNode;
      NextIt : Iterator;
   begin
      Node := It.Current;
      while Node /= NullNode loop
         Node := Traverse (RootNode => It.Root,
                           CurrNode => Node);
         exit when IsFormalParameterNode (Node, It.Root);
      end loop;

      if Node = NullNode then
         NextIt := NullIterator;
      else
         NextIt :=
           Iterator'(TheSearchKind   => It.TheSearchKind,
                     SearchNodeType  => It.SearchNodeType,
                     SearchDirection => It.SearchDirection,
                     Current         => Node,
                     Root            => It.Root);
      end if;
      return NextIt;
   end NextFormalParameter;

   -------------------------------------------------------

   function NextNode (It : Iterator) return Iterator
   is
      NextIt : Iterator;
   begin
      case It.TheSearchKind is
         when Undefined =>
            NextIt := NullIterator;
            SystemErrors.FatalError (SysErr => SystemErrors.SyntaxTreeWalkError,
                                     Msg => "in STree.NextNode");
         when NodeTypeSearch =>
            NextIt := NextNodeType (It);
         when BranchSearch =>
            NextIt := NextBranch (It);
         when FormalParameterSearch =>
            NextIt := NextFormalParameter (It);
      end case;
      return NextIt;
   end NextNode;

   -------------------------------------------------------

   function FindFirstNode (NodeKind    : SPSymbols.SPSymbol;
                           FromRoot    : SyntaxNode;
                           InDirection : TraverseDirection) return Iterator
   is
      It : Iterator;
   begin
      It := Iterator'(TheSearchKind   => NodeTypeSearch,
                      SearchNodeType  => NodeKind,
                      SearchDirection => InDirection,
                      Current         => FromRoot,
                      Root            => FromRoot);

      if SyntaxNodeType (FromRoot) /= It.SearchNodeType then
         It := NextNode (It);
      end if;
      return It;
   end FindFirstNode;

   -------------------------------------------------------

   function FindFirstBranchNode (FromRoot    : SyntaxNode;
                                 InDirection : TraverseDirection) return Iterator
   is
      It : Iterator;
   begin
      It := Iterator'(TheSearchKind   => BranchSearch,
                      SearchNodeType  => SPSymbols.SPEND,
                      SearchDirection => InDirection,
                      Current         => FromRoot,
                      Root            => FromRoot);

      if not IsBranchNode (FromRoot) then
         It := NextNode (It);
      end if;
      return It;
   end FindFirstBranchNode;

   -------------------------------------------------------

   function FindFirstFormalParameterNode (FromRoot : SyntaxNode) return Iterator
   is
      It : Iterator;
   begin
      if SyntaxNodeType (FromRoot) /= SPSymbols.named_argument_association then
         It := NullIterator;
         SystemErrors.FatalError (SysErr => SystemErrors.SyntaxTreeWalkError,
                                  Msg => "in STree.FindFormalParameterNode");
      else
         It := FindFirstNode (NodeKind    => SPSymbols.identifier,
                              FromRoot    => FromRoot,
                              InDirection => Down);
      end if;

      return Iterator'(TheSearchKind   => FormalParameterSearch,
                       SearchNodeType  => SPSymbols.SPEND,
                       SearchDirection => Down,
                       Current         => It.Current,
                       Root            => FromRoot);
   end FindFirstFormalParameterNode;

   -------------------------------------------------------

   function GetNode (It : Iterator) return SyntaxNode
   is
   begin
      return It.Current;
   end GetNode;

   -------------------------------------------------------

   function IsNull (It : Iterator) return Boolean
   is
   begin
      return It = NullIterator;
   end IsNull;

   -------------------------------------------------------

   function FindLastItemInDependencyRelation (Node : SyntaxNode) return SyntaxNode
      is separate;

   -------------------------------------------------------

   function FindLastActualParameterNode (FromRoot : SyntaxNode) return SyntaxNode
   is
      LastFormal : SyntaxNode;
      LastActual : SyntaxNode;
      It         : Iterator;
   begin
      LastFormal := NullNode;
      if SyntaxNodeType (FromRoot) /= SPSymbols.named_argument_association then
         SystemErrors.FatalError (SysErr => SystemErrors.SyntaxTreeWalkError,
                                  Msg => "in STree.FindLastActualParameterNode");
      else
         -- Get the first formal
         It := FindFirstFormalParameterNode (FromRoot => FromRoot);
         -- find the last formal
         while not IsNull (It) loop
            LastFormal := GetNode (It);
            It := NextNode (It);
         end loop;
      end if;

      if LastFormal = NullNode then
         LastActual := NullNode;
      else
         -- work out the last actual
         -- simple_name => expression
         LastActual := Next_Sibling
           (GetNode
              (FindFirstNode (NodeKind    => SPSymbols.simple_name,
                              FromRoot    => LastFormal,
                              InDirection => Up)));
      end if;
      return LastActual;
   end FindLastActualParameterNode;

   -------------------------------------------------------

   function ExpressionFromPositionalArgumentAssociation (Node : SyntaxNode) return SyntaxNode
   is
      Result : SyntaxNode;
   begin
      SystemErrors.RTAssert (SyntaxNodeType (Node) = SPSymbols.positional_argument_association,
                             SystemErrors.PreconditionFailure,
                             "in function ExpressionFromPositionalArgumentAssociation");
      Result := Child_Node (Node);
      if SyntaxNodeType (Result) /= SPSymbols.expression and then
        SyntaxNodeType (Result) /= SPSymbols.annotation_expression then
         Result := Next_Sibling (Result);
      end if;
      SystemErrors.RTAssert (SyntaxNodeType (Result) = SPSymbols.expression or
                               SyntaxNodeType (Result) = SPSymbols.annotation_expression,
                             SystemErrors.PostConditionFailure,
                             "in function ExpressionFromPositionalArgumentAssociation");
      return Result;
   end ExpressionFromPositionalArgumentAssociation;

   -------------------------------------------------------

   function ExpressionFromNamedArgumentAssociation (Node : SyntaxNode) return SyntaxNode
   is
      Result : SyntaxNode;
   begin
      SystemErrors.RTAssert (SyntaxNodeType (Node) = SPSymbols.named_argument_association,
                             SystemErrors.PreconditionFailure,
                             "in function ExpressionFromNamedArgumentAssociation");
      Result := Child_Node (Node);
      if SyntaxNodeType (Result) /= SPSymbols.simple_name and then
        SyntaxNodeType (Result) /= SPSymbols.annotation_simple_name then
         Result := Next_Sibling (Result);
      end if;
      -- skip over parameter name to get expression
      Result := Next_Sibling (Result);
      SystemErrors.RTAssert (SyntaxNodeType (Result) = SPSymbols.expression or
                               SyntaxNodeType (Result) = SPSymbols.annotation_expression,
                             SystemErrors.PostConditionFailure,
                             "in function ExpressionFromNamedArgumentAssociation");
      return Result;
   end ExpressionFromNamedArgumentAssociation;

   -------------------------------------------------------

   function LoopParameterSpecFromEndOfLoop (Node : SyntaxNode) return SyntaxNode
   is
      Result : SyntaxNode;
   begin
      SystemErrors.RTAssert (SyntaxNodeType (Node) = SPSymbols.end_of_loop,
                             SystemErrors.PreconditionFailure,
                             "in function LoopParameterSpecFromEndOfLoop");

      Result := Child_Node (ParentNode (Node));   -- simple_name or loop_statement_opt
      if SyntaxNodeType (Result) = SPSymbols.simple_name then
         Result := Next_Sibling (Result);       -- loop_statement_opt
      end if;
      Result := Child_Node (Child_Node (Result));

      SystemErrors.RTAssert (Result = NullNode or else
                               SyntaxNodeType (Result) = SPSymbols.loop_parameter_specification or else
                               SyntaxNodeType (Result) = SPSymbols.condition,
                             SystemErrors.PostConditionFailure,
                             "in function LoopParameterSpecFromEndOfLoop");
      return Result;
   end LoopParameterSpecFromEndOfLoop;

   -------------------------------------------------------

   function IdentifierHasTildeSuffix (Node : SyntaxNode) return Boolean
   is
   begin
      SystemErrors.RTAssert (SyntaxNodeType (Node) = SPSymbols.identifier,
                             SystemErrors.PreconditionFailure,
                             "in function IdentifierHasTildeSuffix");
      return SyntaxNodeType (Next_Sibling (Node)) = SPSymbols.tilde;
   end IdentifierHasTildeSuffix;

   -------------------------------------------------------

   function IdentifierHasPercentSuffix (Node : SyntaxNode) return Boolean
   is
   begin
      SystemErrors.RTAssert (SyntaxNodeType (Node) = SPSymbols.identifier,
                             SystemErrors.PreconditionFailure,
                             "in function IdentifierHasPercentSuffix");
      return SyntaxNodeType (Next_Sibling (Node)) = SPSymbols.percent;
   end IdentifierHasPercentSuffix;

   -------------------------------------------------------

begin
   Table.CurrSyntaxNode := NullNode;
   --# accept Flow, 23, Table.Contents, "Init partial but effective";
   Table.Contents (NullNode) := -- Expect flow error on 1st write to array
      NodeValue'(SPSymbols.SPEND,
                 LexTokenManager.TokenPosition'(0, 0),
                 NullNode,
                 NullNode,
                 NullNode,
                 0);
   --# end accept;
   Table.FreeList := NullNode;
   Table.TopUsed  := NullNode;
   --# accept Flow, 602, Table, Table.Contents, "Init partial but effective";
end STree; --  Init. is partial but effective.  Expect 1 warning.
