-- $Id: spparser.adb 13071 2009-04-21 14:16:24Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with SPSymbols,
     SPProductions,
     STree,
     SPParserActions,
     SPParserGoto,
     SPExpectedSymbols,
     SPRelations,
     SparkLex,
     LexTokenManager,
     ExaminerConstants,
     ErrorHandler,
     SystemErrors;

use type STree.SyntaxNode;
use type SPParserActions.SPParseAct;
use type SPParserActions.SPActionKind;
use type SPProductions.SPState;
use type SPSymbols.SPSymbol;

package body SPParser is

   procedure Put_Symbol (File : in SPARK_IO.File_Type;
                         Item : in SPSymbols.SPSymbol)
   --# global in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                File,
   --#                                Item;
   is
      --# hide Put_Symbol; -- hidden since uses 'Image
   begin
      if SPARK_IO.Valid_File (File) then
         SPARK_IO.Put_String (File, SPSymbols.SPSymbol'Image (Item), 0);
      end if;
   end Put_Symbol;


   procedure SPParse (ProgText       : in SPARK_IO.File_Type;
                      MaxStackSize   : out Natural;
                      FileEnd        : out Boolean)
   is
      --# inherit ErrorHandler,
      --#         ExaminerConstants,
      --#         SPProductions,
      --#         SPSymbols,
      --#         STree,
      --#         SystemErrors;
      package SPStackManager
      --# own SPStack    : SPStackStruct;
      --#     SPStackPtr : SPStackPtrVal;
      --# initializes SPStack,
      --#             SPStackPtr;
      is
         subtype SPStackPtrVal is Natural range 0 .. ExaminerConstants.SPStackSize;

         type SPStackEntry is
            record
               StateNo : SPProductions.SPState;
               SPSym   : SPSymbols.SPSymbol;
               Node    : STree.SyntaxNode;
            end record;

         SPStackPtr : SPStackPtrVal;

         procedure SPPush (St   : in SPProductions.SPState;
                          Sym  : in SPSymbols.SPSymbol;
                          Node : in STree.SyntaxNode);
         --# global in out SPStack;
         --#        in out SPStackPtr;
         --# derives SPStack    from *,
         --#                         Node,
         --#                         SPStackPtr,
         --#                         ST,
         --#                         Sym &
         --#         SPStackPtr from *;

         procedure SPTop (Top : out SPStackEntry);
         --# global in SPStack;
         --#        in SPStackPtr;
         --# derives Top from SPStack,
         --#                  SPStackPtr;

         procedure SPPop (Top : out SPStackEntry; PopOff : in SPProductions.SPRight);
         --# global in     SPStack;
         --#        in out SPStackPtr;
         --# derives SPStackPtr from *,
         --#                         PopOff &
         --#         Top        from PopOff,
         --#                         SPStack,
         --#                         SPStackPtr;

         procedure SPLook (StackEntry : out SPStackEntry; Pos : in SPStackPtrVal);
         --# global in SPStack;
         --#        in SPStackPtr;
         --# derives StackEntry from Pos,
         --#                         SPStack,
         --#                         SPStackPtr;

         procedure SPRemove (NoStates : in SPStackPtrVal);
         --# global in out SPStackPtr;
         --# derives SPStackPtr from *,
         --#                         NoStates;

      end SPStackManager;


      --# inherit ErrorHandler,
      --#         LexTokenManager,
      --#         SPARK_IO,
      --#         SPParserActions,
      --#         SPParserGoto,
      --#         SPProductions,
      --#         SPStackManager,
      --#         SPSymbols,
      --#         STree;
      package SPActions

      is

         procedure ShiftAction (State     : in SPProductions.ValidStates;
                               Sym       : in SPSymbols.SPTerminal;
                               LexVal    : in LexTokenManager.LexValue;
                               PuncToken : in Boolean);
         --# global in out SPStackManager.SPStack;
         --#        in out SPStackManager.SPStackPtr;
         --#        in out STree.Table;
         --# derives SPStackManager.SPStack    from *,
         --#                                        PuncToken,
         --#                                        SPStackManager.SPStackPtr,
         --#                                        State,
         --#                                        STree.Table,
         --#                                        Sym &
         --#         SPStackManager.SPStackPtr from * &
         --#         STree.Table               from *,
         --#                                        LexVal,
         --#                                        PuncToken,
         --#                                        Sym;

         procedure ReduceAction (ReduceSymbol : in SPSymbols.SPNonTerminal;
                                ReduceBy     : in SPProductions.SPRight);
         --# global in out SPStackManager.SPStack;
         --#        in out SPStackManager.SPStackPtr;
         --#        in out STree.Table;
         --# derives SPStackManager.SPStack,
         --#         STree.Table               from ReduceBy,
         --#                                        ReduceSymbol,
         --#                                        SPStackManager.SPStack,
         --#                                        SPStackManager.SPStackPtr,
         --#                                        STree.Table &
         --#         SPStackManager.SPStackPtr from *,
         --#                                        ReduceBy;
      end SPActions;



      SPCurrentSym : SPSymbols.SPSymbol;
      SPCurrState  : SPProductions.SPState;
      SPHaltCalled : Boolean;
      LexVal       : LexTokenManager.LexValue;
      PuncToken    : Boolean;

      NoOfTerminals, NoOfNonTerminals : SPExpectedSymbols.SPEssSymRange;
      TerminalList, NonTerminalList : SPExpectedSymbols.SPExpSymList;

      SPStackTop   : SPStackManager.SPStackEntry;

      SPAct        : SPParserActions.SPParseAct;


      --# inherit CommandLineData,
      --#         Dictionary,
      --#         ErrorHandler,
      --#         ExaminerConstants,
      --#         LexTokenManager,
      --#         SPActions,
      --#         SparkLex,
      --#         SPARK_IO,
      --#         SPParserActions,
      --#         SPParserGoto,
      --#         SPProductions,
      --#         SPRelations,
      --#         SPStackManager,
      --#         SPSymbols,
      --#         STree;
      package SPErrorRecovery
      is

         procedure SPRecover (ProgText      : in SPARK_IO.File_Type;
                              CurrentToken  : in SPSymbols.SPSymbol;
                              CurrentLexVal : in LexTokenManager.LexValue;
                              PuncToken     : in Boolean;
                              Halt          : out Boolean);
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out LexTokenManager.StringTable;
         --#        in out SparkLex.CurrLine;
         --#        in out SPARK_IO.File_Sys;
         --#        in out SPStackManager.SPStack;
         --#        in out SPStackManager.SPStackPtr;
         --#        in out STree.Table;
         --# derives ErrorHandler.ErrorContext,
         --#         LexTokenManager.StringTable,
         --#         SparkLex.CurrLine,
         --#         SPARK_IO.File_Sys           from CommandLineData.Content,
         --#                                          CurrentLexVal,
         --#                                          CurrentToken,
         --#                                          Dictionary.Dict,
         --#                                          ErrorHandler.ErrorContext,
         --#                                          LexTokenManager.StringTable,
         --#                                          ProgText,
         --#                                          SparkLex.CurrLine,
         --#                                          SPARK_IO.File_Sys,
         --#                                          SPStackManager.SPStack,
         --#                                          SPStackManager.SPStackPtr,
         --#                                          STree.Table &
         --#         Halt,
         --#         SPStackManager.SPStack,
         --#         SPStackManager.SPStackPtr,
         --#         STree.Table                 from CommandLineData.Content,
         --#                                          CurrentLexVal,
         --#                                          CurrentToken,
         --#                                          Dictionary.Dict,
         --#                                          ErrorHandler.ErrorContext,
         --#                                          LexTokenManager.StringTable,
         --#                                          ProgText,
         --#                                          PuncToken,
         --#                                          SparkLex.CurrLine,
         --#                                          SPARK_IO.File_Sys,
         --#                                          SPStackManager.SPStack,
         --#                                          SPStackManager.SPStackPtr,
         --#                                          STree.Table;

      end SPErrorRecovery;


      package body SPStackManager is
         subtype SPStackIndex is Positive range 1 .. ExaminerConstants.SPStackSize;
         type SPStackStruct is array (SPStackIndex) of SPStackEntry;

         SPStack : SPStackStruct;

         procedure SPPush (St   : in SPProductions.SPState;
                          Sym  : in SPSymbols.SPSymbol;
                          Node : in STree.SyntaxNode)
         is
         begin
            if SPStackPtr < SPStackIndex'Last then
               SPStackPtr := SPStackPtr + 1;
               SPStack (SPStackPtr) := SPStackEntry'(St, Sym, Node);
            else
               SystemErrors.FatalError (SystemErrors.ParseStackOverflow, "");
            end if;
         end SPPush;

         procedure SPTop (Top : out SPStackEntry)
         is
         begin
            Top := SPStack (SPStackPtr);
         end SPTop;

         procedure SPPop (Top : out SPStackEntry; PopOff : in SPProductions.SPRight)
         is
         begin
            if SPStackPtr > Natural (PopOff) then
               SPStackPtr := SPStackPtr - Natural (PopOff);
               Top := SPStack (SPStackPtr);
            else
               SPStackPtr := 0;
               Top := SPStackEntry'(SPProductions.NoState, SPSymbols.SPDEFAULT,
                                    STree.NullNode);
            end if;
         end SPPop;

         procedure SPLook (StackEntry : out SPStackEntry; Pos : in SPStackPtrVal)
         is
         begin
            if SPStackPtr > Pos then
               StackEntry := SPStack (SPStackPtr - Pos);
            else
               StackEntry := SPStackEntry'(SPProductions.NoState,
                                           SPSymbols.SPDEFAULT,
                                           STree.NullNode);
            end if;
         end SPLook;

         procedure SPRemove (NoStates : in SPStackPtrVal)
         is
         begin
            if SPStackPtr > NoStates then
               SPStackPtr := SPStackPtr - NoStates;
            else
               SPStackPtr := 0;
            end if;
         end SPRemove;

      begin
         SPStackPtr := 0;
         --# accept F, 31, SPStack,
         --#    "Only the stack pointer needs to be initialized" &
         --#        F, 32, SPStack,
         --#    "Only the stack pointer needs to be initialized";
      end SPStackManager; -- flow error SPStack undefined expected

      package body SPActions
      is

         procedure ShiftAction (State     : in SPProductions.ValidStates;
                               Sym       : in SPSymbols.SPTerminal;
                               LexVal    : in LexTokenManager.LexValue;
                               PuncToken : in Boolean)
         is
            Node : STree.SyntaxNode;
         begin
            if not PuncToken then
               STree.NewTerminal (Sym, LexVal, Node);
            else
               Node := STree.NullNode;
            end if;
            SPStackManager.SPPush (State, Sym, Node);
         end ShiftAction;

         procedure ReduceAction (ReduceSymbol : in SPSymbols.SPNonTerminal;
                                ReduceBy     : in SPProductions.SPRight)
         is
            Node        : STree.SyntaxNode;
            SPElement      : SPStackManager.SPStackEntry;


            SPCurrState : SPProductions.SPState;
            StackPointer : SPStackManager.SPStackPtrVal;
         begin
            STree.NewProduction (ReduceSymbol, Node);
            StackPointer := SPStackManager.SPStackPtrVal (ReduceBy);
            loop
               exit when StackPointer = 0;
               StackPointer := StackPointer - 1;
               SPStackManager.SPLook (SPElement, StackPointer);
               if SPElement.Node /= STree.NullNode then
                  STree.AddDerivative (SPElement.Node);
               end if;
            end loop;
            SPStackManager.SPPop (SPElement, ReduceBy);
            SPCurrState := SPParserGoto.SPGoto (SPElement.StateNo, ReduceSymbol);
            SPStackManager.SPPush (SPCurrState, ReduceSymbol, Node);
         end ReduceAction;
      end SPActions;


      package body SPErrorRecovery is

         procedure SPRecover (ProgText      : in SPARK_IO.File_Type;
                              CurrentToken  : in SPSymbols.SPSymbol;
                              CurrentLexVal : in LexTokenManager.LexValue;
                              PuncToken     : in Boolean;
                              Halt          : out Boolean)
         is

            type BufIndex is range 0 .. ExaminerConstants.SPErrLookahead;
            --# assert BufIndex'Base is Short_Short_Integer; -- for GNAT

            type TokenBuffer is array (BufIndex) of SPSymbols.SPTerminal;
            type LexBuffer is array (BufIndex) of LexTokenManager.LexValue;
            type TypeBuffer is array (BufIndex) of Boolean;


            type LocalRecoverySuccess is (NoSuccess, WrongToken, MissingToken, ExtraToken);

            TokenList      : TokenBuffer;
            LexValList     : LexBuffer;
            TokenTypes     : TypeBuffer;
            HigherEntry,
            LowerEntry     : SPStackManager.SPStackEntry;
            StackDepth,
            Pos            : SPStackManager.SPStackPtrVal;
            Node           : STree.SyntaxNode;
            LocalSuccess   : LocalRecoverySuccess;
            Success,
            Stop,
            Done           : Boolean;

            SymListSize    : Natural;
            SymList        : ErrorHandler.ErrSymList;
            ReplacementSym,
            RecoverySym    : SPSymbols.SPSymbol;
            CurrState      : SPProductions.SPState;
            Index,
            LastBufIndex   : BufIndex;

            SPElement      : SPStackManager.SPStackEntry;
            SPAct          : SPParserActions.SPParseAct;

            LexToken       : SPSymbols.SPTerminal;
            LexTokenValue  : LexTokenManager.LexValue;
            LexTokenType   : Boolean;

            procedure CheckFollowingTokens (Tokens         : in TokenBuffer;
                                            StartIndex,
                                               LastIndex      : in BufIndex;
                                            StackPos       : in SPStackManager.SPStackPtrVal;
                                            NextState      : in SPProductions.SPState;
                                            RecoveryOK     : out Boolean)
            --# global in SPStackManager.SPStack;
            --#        in SPStackManager.SPStackPtr;
            --# derives RecoveryOK from LastIndex,
            --#                         NextState,
            --#                         SPStackManager.SPStack,
            --#                         SPStackManager.SPStackPtr,
            --#                         StackPos,
            --#                         StartIndex,
            --#                         Tokens;
            is
               type RecoveryStackIndex is range 0 .. ExaminerConstants.SPErrLookahead * 2 + 1;
               --# assert RecoveryStackIndex'Base is Short_Short_Integer; -- for GNAT

               -- times two to allow for reduction by 0
               type RecoveryStack is array (RecoveryStackIndex) of SPProductions.SPState;

               LocalStack     : RecoveryStack;
               LocalStackPtr  : RecoveryStackIndex;
               ParseStackptr  : SPStackManager.SPStackPtrVal;
               CurrState      : SPProductions.SPState;
               SPAct          : SPParserActions.SPParseAct;
               SPElement      : SPStackManager.SPStackEntry;
               Index          : BufIndex;
               Done           : Boolean;
            begin
               -- This code could do with refactoring to remove the conditional
               -- flow errors and to render it free from RTE - TJJ.

               --# accept F, 23, LocalStack,
               --#    "The stack pointer is all that is need to determine the extent of the stack" &
               --#        F, 501, LocalStack,
               --#    "The stack pointer is all that is need to determine the extent of the stack" &
               --#        F, 504, LocalStack,
               --#    "The stack pointer is all that is need to determine the extent of the stack" &
               --#        F, 602, RecoveryOK, LocalStack,
               --#    "The stack pointer is all that is need to determine the extent of the stack";


               -- check further tokens so that we do not recover too soon
               RecoveryOK := True;
               ParseStackptr := StackPos;
               if NextState /= SPProductions.NoState then
                  LocalStack (1) := NextState;
                  LocalStackPtr := 1;
               else
                  LocalStackPtr := 0;
               end if;
               if LocalStackPtr > 0 then
                  CurrState := LocalStack (LocalStackPtr);
               else
                  SPStackManager.SPLook (SPElement, ParseStackptr);
                  CurrState := SPElement.StateNo;
               end if;
               Index := StartIndex;
               Done := False;
               loop
                  --SPARK_IO.Put_String (SPARK_IO.STANDARD_OUTPUT, "state ",0);
                  --SPARK_IO.Put_Integer (SPARK_IO.STANDARD_OUTPUT, Integer (CurrState), 5, 10);
                  SPAct := SPParserActions.SPA (CurrState, Tokens (Index));
                  --SPARK_IO.Put_String (SPARK_IO.STANDARD_OUTPUT, "trying ",0);
                  --Put_Symbol (SPARK_IO.STANDARD_OUTPUT, Tokens (Index));
                  case SPAct.Act is
                     when SPParserActions.Shift =>
                        --SPARK_IO.Put_LINE (SPARK_IO.STANDARD_OUTPUT, " - shift",0);
                        CurrState := SPAct.State;
                        LocalStackPtr := LocalStackPtr + 1;
                        LocalStack (LocalStackPtr) := CurrState;
                        if Index < LastIndex then
                           Index := Index + 1;
                        else
                           Done := True;
                        end if;
                     when SPParserActions.Reduce =>
                        --SPARK_IO.Put_LINE (SPARK_IO.STANDARD_OUTPUT, " - reduce",0);
                        if Integer (LocalStackPtr) > Integer (SPAct.RedBy) then
                           LocalStackPtr := LocalStackPtr - RecoveryStackIndex (SPAct.RedBy);
                           CurrState := LocalStack (LocalStackPtr);
                        else
                           ParseStackptr := ParseStackptr +
                              SPStackManager.SPStackPtrVal'(Integer (SPAct.RedBy) -
                                                            Integer (LocalStackPtr));
                           LocalStackPtr := 0;
                           SPStackManager.SPLook (SPElement, ParseStackptr);
                           CurrState := SPElement.StateNo;
                        end if;
                        CurrState := SPParserGoto.SPGoto (CurrState, SPAct.Symbol);
                        LocalStackPtr := LocalStackPtr + 1;
                        LocalStack (LocalStackPtr) := CurrState;
                     when SPParserActions.Accpt =>
                        --SPARK_IO.Put_LINE (SPARK_IO.STANDARD_OUTPUT, " - accept",0);
                        if Tokens (Index) = SPSymbols.SPEND then
                           Done := True;
                        else
                           LocalStackPtr := 1;
                           LocalStack (1) := 1;  -- First state
                           CurrState := 1;
                        end if;
                     when SPParserActions.Error =>
                        --SPARK_IO.Put_LINE (SPARK_IO.STANDARD_OUTPUT, " - error",0);
                        RecoveryOK := False;
                        Done := True;
                  end case;
                  exit when Done;
               end loop;
            end CheckFollowingTokens;

            procedure FindLocalError (StackTop     : in SPStackManager.SPStackEntry;
                                     TokenList    : in out TokenBuffer;
                                     RecoveryPosn : out BufIndex;
                                     Success      : out LocalRecoverySuccess)
            --# global in SPStackManager.SPStack;
            --#        in SPStackManager.SPStackPtr;
            --# derives RecoveryPosn,
            --#         Success,
            --#         TokenList    from SPStackManager.SPStack,
            --#                           SPStackManager.SPStackPtr,
            --#                           StackTop,
            --#                           TokenList;
            is
               RecoveryToken,
               RecoverySymbol   : SPSymbols.SPSymbol;
               RecoveryPossible,
               RecoveryOK       : Boolean;
               RecoveryAct      : SPParserActions.SPParseAct;
               Status           : LocalRecoverySuccess;
               Index            : SPParserActions.ActionIndex;
               FirstToken       : SPSymbols.SPTerminal;
            begin
               FirstToken := TokenList (1);
               Index := SPParserActions.FirstActionIndex;
               Status := NoSuccess;
               RecoveryPossible := True;
               RecoveryToken := SPSymbols.SPDEFAULT;
               -- the initialization of this variable is not strictly
               -- necessary but it avoids conditional data-flow errors.
               loop
                  SPParserActions.ScanActionTable (StackTop.StateNo, Index, RecoveryAct, RecoverySymbol);
                  exit when RecoveryAct = SPParserActions.ErrorAction;

                  --Put_Symbol (SPARK_IO.STANDARD_OUTPUT, RecoverySymbol);
                  if (RecoverySymbol /= SPSymbols.SPDEFAULT) and (RecoverySymbol /= SPSymbols.SPEND) then
                     -- check for invalid extra token
                     if RecoverySymbol = TokenList (2) then
                        --SPARK_IO.Put_LINE (SPARK_IO.STANDARD_OUTPUT, "Trying local error - extra token",0);
                        CheckFollowingTokens (TokenList, 2,
                                             BufIndex (ExaminerConstants.SPLocalErrLookahead),
                                             0, SPProductions.NoState, RecoveryOK);
                        if RecoveryOK then
                           if Status = NoSuccess then
                              Status := ExtraToken;
                           else
                              RecoveryPossible := False;
                           end if;
                        end if;
                     end if;
                     -- check for missing token
                     if RecoveryPossible then
                        -- SPARK_IO.Put_LINE (SPARK_IO.STANDARD_OUTPUT, "Trying local error - missing token",0);
                        TokenList (0) := RecoverySymbol;
                        CheckFollowingTokens (TokenList, 0,
                                             BufIndex (ExaminerConstants.SPLocalErrLookahead),
                                             0, SPProductions.NoState, RecoveryOK);
                        if RecoveryOK then
                           if Status = NoSuccess then
                              Status := MissingToken;
                              RecoveryToken := RecoverySymbol;
                           else
                              RecoveryPossible := False;
                           end if;
                        end if;
                     end if;
                     -- wrongly spelt token
                     if RecoveryPossible then
                        if SparkLex.SimilarTokens (RecoverySymbol, TokenList (1)) then
                           TokenList (1) := RecoverySymbol;
                           -- SPARK_IO.Put_LINE (SPARK_IO.STANDARD_OUTPUT, "Trying local error - wrongly spelt token",0);
                           CheckFollowingTokens (TokenList, 1,
                                                BufIndex (ExaminerConstants.SPLocalErrLookahead),
                                                0, SPProductions.NoState, RecoveryOK);
                           if RecoveryOK then
                              if Status = NoSuccess then
                                 Status := WrongToken;
                                 RecoveryToken := RecoverySymbol;
                              else
                                 RecoveryPossible := False;
                              end if;
                           end if;
                           TokenList (1) := FirstToken;
                        end if;
                     end if;
                  end if;
                  exit when not RecoveryPossible;
               end loop;
               if RecoveryPossible then
                  if Status = MissingToken then
                     TokenList (0) := RecoveryToken; -- flow err from non-exec path
                     RecoveryPosn := 0;
                  elsif Status = WrongToken then
                     TokenList (1) := RecoveryToken; -- flow err from non-exec path
                     RecoveryPosn := 1;
                  else
                     RecoveryPosn := 2;
                  end if;
                  Success := Status;
               else
                  Success := NoSuccess;
                  RecoveryPosn := 0;
               end if;
            end FindLocalError;

            procedure FindErrorPhrase (HigherEntry : in out SPStackManager.SPStackEntry;
                                      LowerEntry  : in SPStackManager.SPStackEntry;
                                      StackPos    : in SPStackManager.SPStackPtrVal;
                                      TokenList   : in TokenBuffer;
                                      Success     : out Boolean)
            --# global in SPStackManager.SPStack;
            --#        in SPStackManager.SPStackPtr;
            --# derives HigherEntry,
            --#         Success     from HigherEntry,
            --#                          LowerEntry,
            --#                          SPStackManager.SPStack,
            --#                          SPStackManager.SPStackPtr,
            --#                          StackPos,
            --#                          TokenList;
            is
               RecoveryOK,
               RecoveryFound,
               RecoveryPossible : Boolean;
               RecoverySymbol   : SPSymbols.SPNonTerminal;
               RecoveryEntry    : SPStackManager.SPStackEntry;
               RecoveryState,
               NextState        : SPProductions.SPState;
               Index            : SPParserGoto.GotoIndex;
               SPAct            : SPParserActions.SPParseAct;
            begin
               RecoveryEntry := SPStackManager.SPStackEntry'(SPProductions.NoState, SPSymbols.SPDEFAULT,
                                                             STree.NullNode);
               Index := SPParserGoto.FirstGotoIndex;
               RecoveryPossible := True;
               RecoveryFound := False;
               loop
                  SPParserGoto.ScanGotoTable (LowerEntry.StateNo, Index, RecoveryState, RecoverySymbol);
                  exit when RecoveryState = SPProductions.NoState;
                  SPAct := SPParserActions.SPA (RecoveryState, TokenList (1));
                  if SPAct.Act = SPParserActions.Shift or SPAct.Act = SPParserActions.Accpt then
                     if HigherEntry.StateNo = SPProductions.NoState or else
                        RecoverySymbol = HigherEntry.SPSym or else
                        SPRelations.SPLeftCorner (RecoverySymbol, HigherEntry.SPSym) then
                        CheckFollowingTokens (TokenList, 1, BufIndex'Last, StackPos,
                                             RecoveryState, RecoveryOK);
                        if RecoveryOK then
                           --# accept F, 20, NextState, "NextState is guarded by RecoveryFound";
                           if not RecoveryFound then
                              NextState := SPAct.State;
                              RecoveryFound := True;
                              RecoveryEntry := SPStackManager.SPStackEntry'(RecoveryState,
                                                                            RecoverySymbol,
                                                                            STree.NullNode);
                           elsif SPAct.State /= NextState then -- expected flow error
                              RecoveryPossible := False;
                           end if;
                        end if;
                     end if;
                  end if;
                  exit when not RecoveryPossible;
               end loop;
               if RecoveryPossible and RecoveryFound then
                  Success := True;
                  HigherEntry := RecoveryEntry;
               else
                  Success := False;
               end if;
               --# accept F, 602, HigherEntry, NextState,
               --#    "NextState is guarded by RecoveryFound" &
               --#        F, 602, Success, NextState,
               --#    "NextState is guarded by RecoveryFound";
            end FindErrorPhrase;

         begin  --SPRecover
            Stop := False;
            --# accept f, 23, TokenList, "Whole array is initialized." &
            --#        f, 23, LexValList, "Whole array is initialized." &
            --#        f, 23, TokenTypes, "Whole array is initialized.";
            TokenList (1) := CurrentToken;
            LexValList (1) := CurrentLexVal;
            TokenTypes (1) := PuncToken;
            LexValList (0) :=
               LexTokenManager.LexValue'(LexTokenManager.TokenPosition'(0, 0),
                                         LexTokenManager.NullString);
            TokenTypes (0) := False;
            for Ix in BufIndex range 2  .. BufIndex (ExaminerConstants.SPLocalErrLookahead) loop
               SparkLex.ExaminerLex (ProgText, LexToken, LexTokenValue, LexTokenType);
               TokenList (Ix)  := LexToken;
               LexValList (Ix) := LexTokenValue;
               TokenTypes (Ix) := LexTokenType;
            end loop;
            --# end accept;
            Success := False;
            SPStackManager.SPTop (HigherEntry);
            -- try local error recovery
            FindLocalError (HigherEntry, TokenList, Index, LocalSuccess);
            --# accept F, 23, SymList,
            --#    "Access to SymList Elements is guarded by SymListSize";
            if LocalSuccess /= NoSuccess then
               -- produce recovery message
               if LocalSuccess = WrongToken or LocalSuccess = ExtraToken then
                  SymListSize := 1;
                  SymList (1) := CurrentToken;
               else
                  SymListSize := 0;
               end if;
               if LocalSuccess = WrongToken or LocalSuccess = MissingToken then
                  ReplacementSym := TokenList (Index);
               else
                  ReplacementSym := SPSymbols.SPDEFAULT;
               end if;
               --# accept F, 504, SymList,
               --#    "Access to SymList is guarded by SymListSize";
               ErrorHandler.SyntaxRecovery (LexValList (1), ReplacementSym, SPSymbols.SPDEFAULT, SymListSize, SymList);
               --# end accept;
               -- if LocalSuccess = MissingToken then
               -- SPARK_IO.Put_LINE (SPARK_IO.STANDARD_OUTPUT, "Local error - missing token",0);
               -- elsif LocalSuccess = ExtraToken then
               -- SPARK_IO.Put_LINE (SPARK_IO.STANDARD_OUTPUT, "Local error - extra token",0);
               -- else
               -- SPARK_IO.Put_LINE (SPARK_IO.STANDARD_OUTPUT, "Local error - wrongly spelt token",0);
               -- end if;
               Success := True;
               LastBufIndex := BufIndex (ExaminerConstants.SPLocalErrLookahead);
            else
               for Ix in BufIndex range BufIndex (ExaminerConstants.SPLocalErrLookahead) + 1  .. BufIndex'Last loop
                  SparkLex.ExaminerLex (ProgText, LexToken, LexTokenValue, LexTokenType);
                  TokenList (Ix)  := LexToken;
                  LexValList (Ix) := LexTokenValue;
                  TokenTypes (Ix) := LexTokenType;
               end loop;
               RecoverySym := TokenList (1);
               loop
                  SPStackManager.SPTop (LowerEntry);
                  HigherEntry := SPStackManager.SPStackEntry'(SPProductions.NoState,
                                                              SPSymbols.SPDEFAULT,
                                                              STree.NullNode);
                  StackDepth := 0;
                  while not Success and StackDepth < SPStackManager.SPStackPtr loop
                     FindErrorPhrase (HigherEntry, LowerEntry, StackDepth, TokenList, Success);
                     if Success then
                        -- produce recovery message
                        -- Put_LINE (STANDARD_OUTPUT, "Non-local error");
                        Pos := StackDepth; SymListSize := 0;
                        while Pos > 0 and SymListSize < Natural (ErrorHandler.ErrSymRange'Last) loop
                           Pos := Pos - 1;
                           SPStackManager.SPLook (SPElement,
                                                 Pos);
                           SymListSize := SymListSize + 1;
                           SymList (ErrorHandler.ErrSymRange (SymListSize)) :=
                              SPElement.SPSym;
                        end loop;
                        --# accept F, 41, "Pos is updated in the outer loop.";
                        if Pos /= 0 then  -- expected flow error
                           SymListSize := StackDepth;
                           SPStackManager.SPTop (SPElement);
                        --# accept F, 504, SymList,
                        --#    "Update of element of SymList - Accesses guarded by SymListSize";
                           SymList (ErrorHandler.ErrSymRange'Last) :=
                             SPElement.SPSym;
                        --# end accept;
                        end if;
                        --# end accept;
                        --# accept F, 504, SymList,
                        --#    "Access to SymList is guarded by SymListSize";
                        ErrorHandler.SyntaxRecovery (LexValList (1),
                                                     HigherEntry.SPSym, RecoverySym, SymListSize, SymList);
                        --# end accept;
                        -- patch stack
                        SPStackManager.SPRemove (StackDepth);
                        STree.NewProduction (HigherEntry.SPSym, Node);
                        SPStackManager.SPPush (HigherEntry.StateNo, HigherEntry.SPSym, Node);
                        Index := 1;
                        LastBufIndex := BufIndex'Last;
                        exit;
                     end if;
                     HigherEntry := LowerEntry;
                     StackDepth := StackDepth + 1;
                     SPStackManager.SPLook (LowerEntry, StackDepth);
                  end loop;
                  exit when Success or TokenList (1) = SPSymbols.SPEND;
                  for Ix in BufIndex range 1 .. BufIndex'Last - 1 loop
                     TokenList (Ix) := TokenList (Ix + 1);
                     LexValList (Ix) := LexValList (Ix + 1);
                     TokenTypes (Ix) := TokenTypes (Ix + 1);
                  end loop;

                  SparkLex.ExaminerLex (ProgText, LexToken, LexTokenValue, LexTokenType);
                  TokenList (BufIndex'Last)  := LexToken;
                  LexValList (BufIndex'Last) := LexTokenValue;
                  TokenTypes (BufIndex'Last) := LexTokenType;
                  RecoverySym := SPSymbols.SPDEFAULT;

               end loop;
            end if;
            -- perform action on following tokens
            if Success then
               SPStackManager.SPTop (HigherEntry);
               CurrState := HigherEntry.StateNo;
               Done := False;
               loop
                  SPAct := SPParserActions.SPA (CurrState, TokenList (Index));
                  case SPAct.Act is
                     when SPParserActions.Shift =>
                        SPActions.ShiftAction (SPAct.State, TokenList (Index),
                                              LexValList (Index), TokenTypes (Index));
                        CurrState := SPAct.State;
                        --# accept F, 501, LastBufIndex, "Access guarded by Success.";
                        if Index < LastBufIndex then -- flow error expected
                           Index := Index + 1;
                        else
                           Done := True;
                        end if;
                        --# end accept;
                     when SPParserActions.Reduce =>
                        SPActions.ReduceAction (SPAct.Symbol, SPAct.RedBy);
                        SPStackManager.SPTop (HigherEntry);
                        CurrState := HigherEntry.StateNo;
                     when SPParserActions.Accpt =>
                        Stop := True;
                        Done := True;
                     when others => -- doesn't arise
                        Done := True;
                  end case;
                  exit when Done;
               end loop;
            end if;
            Halt := Stop;
            --# accept F, 602, SPARK_IO.File_Sys, TokenList, "Accessed elements are defined." &
            --#        F, 602, SPARK_IO.File_Sys, LexValList, "Accessed elements are defined." &
            --#        F, 602, SPARK_IO.File_Sys, SymList, "Accessed elements are defined." &
            --#        F, 602, LexTokenManager.StringTable, TokenList, "Accessed elements are defined." &
            --#        F, 602, LexTokenManager.StringTable, LexValList, "Accessed elements are defined." &
            --#        F, 602, LexTokenManager.StringTable, SymList, "Accessed elements are defined." &
            --#        F, 602, ErrorHandler.ErrorContext, TokenList, "Accessed elements are defined." &
            --#        F, 602, ErrorHandler.ErrorContext, LexValList, "Accessed elements are defined." &
            --#        F, 602, ErrorHandler.ErrorContext, SymList, "Accessed elements are defined." &
            --#        F, 602, SparkLex.CurrLine, TokenList, "Accessed elements are defined." &
            --#        F, 602, SparkLex.CurrLine, LexValList, "Accessed elements are defined." &
            --#        F, 602, SparkLex.CurrLine, SymList, "Accessed elements are defined." &
            --#        F, 602, STree.Table, TokenList, "Accessed elements are defined." &
            --#        F, 602, STree.Table, LexValList, "Accessed elements are defined." &
            --#        F, 602, STree.Table, SymList, "Accessed elements are defined." &
            --#        F, 602, STree.Table, Tokentypes, "Accessed elements are defined." &
            --#        F, 602, STree.Table, LastBufIndex, "Accesses guarded by Success" &
            --#        F, 602, SPStackManager.SPStack, TokenList, "Accessed elements are defined." &
            --#        F, 602, SPStackManager.SPStack, LexValList, "Accessed elements are defined." &
            --#        F, 602, SPStackManager.SPStack, SymList, "Accessed elements are defined." &
            --#        F, 602, SPStackManager.SPStack, Tokentypes, "Accessed elements are defined." &
            --#        F, 602, SPStackManager.SPStack, LastBufIndex, "Accesses guarded by Success" &
            --#        F, 602, SPStackManager.SPStackPtr, TokenList, "Accessed elements are defined." &
            --#        F, 602, SPStackManager.SPStackPtr, LexValList, "Accessed elements are defined." &
            --#        F, 602, SPStackManager.SPStackPtr, SymList, "Accessed elements are defined." &
            --#        F, 602, SPStackManager.SPStackPtr, Tokentypes, "Accessed elements are defined." &
            --#        F, 602, SPStackManager.SPStackPtr, LastBufIndex, "Accesses guarded by Success" &
            --#        F, 602, Halt, TokenList, "Accessed elements are defined." &
            --#        F, 602, Halt, LexValList, "Accessed elements are defined." &
            --#        F, 602, Halt, SymList, "Accessed elements are defined." &
            --#        F, 602, Halt, Tokentypes, "Accessed elements are defined." &
            --#        F, 602, Halt, LastBufIndex, "Accesses guarded by Success";
         end SPRecover;
      end SPErrorRecovery;


      --  Unused procedure, but leave here for debugging
      procedure SPPrintAction (OutputFile   : in SPARK_IO.File_Type;
                               SPAct        : in SPParserActions.SPParseAct;
                               SPCurrState  : in SPProductions.SPState;
                               SPCurrentSym : in SPSymbols.SPSymbol)
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                OutputFile,
      --#                                SPAct,
      --#                                SPCurrentSym,
      --#                                SPCurrState;
      is
      begin
         SPARK_IO.Put_String (OutputFile, " STATE: ", 0);
         SPARK_IO.Put_Integer (OutputFile, Integer (SPCurrState), 5, 10);
         SPARK_IO.Put_String (OutputFile, " SYMBOL: ", 0);
         Put_Symbol (OutputFile, SPCurrentSym);
         SPARK_IO.New_Line (OutputFile, 1);
         case SPAct.Act is
            when SPParserActions.Shift =>
               SPARK_IO.Put_String (OutputFile, " ACTION : SHIFT ", 0);
               SPARK_IO.Put_Integer (OutputFile, Integer (SPAct.State), 5, 10);
               SPARK_IO.New_Line (OutputFile, 1);
            when SPParserActions.Reduce =>
               SPARK_IO.Put_String (OutputFile, " ACTION : REDUCE ", 0);
               SPARK_IO.Put_String (OutputFile, " SYMBOL : ", 0);
               Put_Symbol (OutputFile, SPAct.Symbol);
               SPARK_IO.Put_String (OutputFile, "  REDUCE BY : ", 0);
               SPARK_IO.Put_Integer (OutputFile, Integer (SPAct.RedBy), 5, 10);
               SPARK_IO.Put_String (OutputFile, "  PROD NO : ", 0);
               SPARK_IO.Put_Integer (OutputFile, Integer (SPAct.ProdNo), 5, 10);
               SPARK_IO.New_Line (OutputFile, 1);
            when SPParserActions.Accpt =>
               SPARK_IO.Put_String (OutputFile, " ACTION : ACCEPT", 0);
               SPARK_IO.New_Line (OutputFile, 1);
            when SPParserActions.Error =>
               SPARK_IO.Put_String (OutputFile, " ACTION : ERROR", 0);
               SPARK_IO.New_Line (OutputFile, 1);
         end case;
      end SPPrintAction;
      pragma Unreferenced (SPPrintAction);

      --  Unused procedure, but leave here for debugging
      procedure SPPrintStack (OutputFile : in SPARK_IO.File_Type)
      --# global in     SPStackManager.SPStack;
      --#        in     SPStackManager.SPStackPtr;
      --#        in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                OutputFile,
      --#                                SPStackManager.SPStack,
      --#                                SPStackManager.SPStackPtr;
      is
         SPElement : SPStackManager.SPStackEntry;
      begin
         SPARK_IO.Put_Line (OutputFile, " STACK VALUES :", 0);
         for Ix in reverse SPStackManager.SPStackPtrVal
            range 0 .. SPStackManager.SPStackPtr loop
            SPStackManager.SPLook (SPElement, Ix);
            SPARK_IO.Put_Integer (OutputFile, Integer (SPElement.StateNo), 5, 10);
            SPARK_IO.Put_String (OutputFile, " SYMBOL: ", 0);
            Put_Symbol (OutputFile, SPElement.SPSym);
            SPARK_IO.New_Line (OutputFile, 1);
         end loop;
      end SPPrintStack;
      pragma Unreferenced (SPPrintStack);

      -- Declarations to write SPPrintStack to a named file
      -- OutputFile : SPARK_IO.File_Type;
      -- Status : SPARK_IO.File_Status;

   begin  --SPParse
      -- SPARK_IO.CREATE (OutputFile, "spark.out", "", Status);
      SPCurrState := 1;
      SparkLex.ExaminerLex (ProgText, SPCurrentSym, LexVal, PuncToken);
      SPStackManager.SPPush (1, SPSymbols.SPACCEPT, STree.NullNode);
      SPHaltCalled := False;
      while not SPHaltCalled loop
         SPAct := SPParserActions.SPA (SPCurrState, SPCurrentSym);
         -- SPPrintAction (OutputFile, SPAct, SPCurrState, SPCurrentSym); -- to write to named dump file
         -- SPPrintAction (SPARK_IO.Standard_Output, SPAct, SPCurrState, SPCurrentSym); -- to dump to screen
         case SPAct.Act is
            when SPParserActions.Shift =>
               SPCurrState := SPAct.State;
               SPActions.ShiftAction (SPCurrState, SPCurrentSym, LexVal, PuncToken);
               SparkLex.ExaminerLex (ProgText, SPCurrentSym, LexVal, PuncToken);
            when SPParserActions.Reduce =>
               SPActions.ReduceAction (SPAct.Symbol, SPAct.RedBy);
               SPStackManager.SPTop (SPStackTop);
               SPCurrState := SPStackTop.StateNo;
            when SPParserActions.Accpt =>
               SPHaltCalled := True;
            when SPParserActions.Error =>
               -- SPPrintStack (OutputFile);
               SPStackManager.SPTop (SPStackTop);
               SPExpectedSymbols.GetExpectedSymbols (SPStackTop.StateNo, NoOfTerminals, TerminalList,
                                                    NoOfNonTerminals, NonTerminalList);
               ErrorHandler.SyntaxError (LexVal, SPCurrentSym, SPStackTop.SPSym,
                                        NoOfTerminals, NoOfNonTerminals,
                                        TerminalList, NonTerminalList);
               SPErrorRecovery.SPRecover (ProgText, SPCurrentSym, LexVal, PuncToken, SPHaltCalled);
               if not SPHaltCalled then
                  SPStackManager.SPTop (SPStackTop);
                  SPCurrState := SPStackTop.StateNo;
                  SparkLex.ExaminerLex (ProgText, SPCurrentSym, LexVal, PuncToken);
               end if;
               -- SPPrintStack (OutputFile);
         end case;
      end loop;
      MaxStackSize := 0;
      case SPCurrentSym is
         when SPSymbols.SPEND =>
            FileEnd := True;
         when others =>
            FileEnd := False;
      end case;
   end SPParse;
end SPParser;
