-- $Id: sparklex.adb 11889 2008-12-12 15:49:12Z rod chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with ErrorHandler,
  Ada.Characters.Handling,
  CommandLineData,
  SystemErrors;

package body SparkLex
--# own CurrLine is LineManager.CurrLine;
is

   EndOfText       : constant Character := Ada.Characters.Latin_1.ETX;
   EndOfLine       : constant Character := Ada.Characters.Latin_1.CR;
   EndOfAnnotation : constant Character := Ada.Characters.Latin_1.EM;
   LineTooLong     : constant Character := Ada.Characters.Latin_1.ESC;

   --# inherit Ada.Characters.Latin_1,
   --#         CommandLineData,
   --#         Dictionary,
   --#         ErrorHandler,
   --#         EStrings,
   --#         LexTokenManager,
   --#         SparkLex,
   --#         SPARK_IO,
   --#         SystemErrors;
   package LineManager
   --# own CurrLine : ProgramLine;
   is
      --++++ Type ProgramLine and own variable CurrLine are made visible to ++++--
      --++++ avoid excessive copying of the line contents by other modules. ++++--
      ----------------------------------------------------------------------------

      type ProgramLine is record
         Context       : SparkLex.ProgramContext;
         AnnoContext   : SparkLex.AnnotationContext;
         LineNo        : LexTokenManager.LineNumbers;
         LineLength,
         LastTokenPos,
         CurrPos,
         LookaheadPos  : EStrings.Lengths;
         Conts         : EStrings.Line;
      end record;

      CurrLine      : ProgramLine; -- By SPARK Rules, read only outside LineManager.
      ------------------------------------------------------------------------------

      procedure ClearLine;
      --# global out CurrLine;
      --# derives CurrLine from ;

      procedure CopyOutLine (Line : out ProgramLine);
      --# global in CurrLine;
      --# derives Line from CurrLine;

      procedure CopyInLine (Line : in ProgramLine);
      --# global out CurrLine;
      --# derives CurrLine from Line;

      procedure RecordCurrPos;
      --# global in out CurrLine;
      --# derives CurrLine from *;

      procedure ResetCurrPos;
      --# global in out CurrLine;
      --# derives CurrLine from *;

      procedure InspectChar (Ch : out Character);
      --# global in CurrLine;
      --# derives Ch from CurrLine;

      procedure AcceptChar;
      --# global in out CurrLine;
      --# derives CurrLine from *;

      procedure LookaheadChar (Ch : out Character);
      --# global in out CurrLine;
      --# derives Ch,
      --#         CurrLine from CurrLine;

      procedure AcceptLookahead;
      --# global in out CurrLine;
      --# derives CurrLine from *;

      procedure RejectLookahead;
      --# global in out CurrLine;
      --# derives CurrLine from *;

      procedure NextSigChar (ProgText : in SPARK_IO.File_Type);
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in out CurrLine;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LexTokenManager.StringTable;
      --#        in out SPARK_IO.File_Sys;
      --# derives CurrLine,
      --#         ErrorHandler.ErrorContext,
      --#         LexTokenManager.StringTable,
      --#         SPARK_IO.File_Sys           from CommandLineData.Content,
      --#                                          CurrLine,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          LexTokenManager.StringTable,
      --#                                          ProgText,
      --#                                          SPARK_IO.File_Sys;

      procedure SetContext (NewContext : in SparkLex.ProgramContext);
      --# global in out CurrLine;
      --# derives CurrLine from *,
      --#                       NewContext;

      procedure SetAnnoContext (NewContext : in SparkLex.AnnotationContext);
      --# global in out CurrLine;
      --# derives CurrLine from *,
      --#                       NewContext;
      pragma Unreferenced (SetAnnoContext); -- not used at present

   end LineManager;

   NoOfRW      : constant Natural  := 117;
   MinRWLength : constant Positive := 1;
   MaxRWLength : constant Positive := 20;

   type AnnoType is (StartAnno, ProofAnno, HideAnno, OtherAnno, NoAnno);

   subtype RWLength is Positive range 1 .. MaxRWLength;
   subtype RWIndex  is Natural  range 0 .. NoOfRW - 1;

   subtype ResWord is String (RWLength);
   type RWPair is record
      Word  : ResWord;
      Token : SPSymbols.SPTerminal;
   end record;
   type RWList is array (RWIndex) of RWPair;

   RW : constant RWList := RWList'
      (RWPair'(ResWord'("abort               "), SPSymbols.RWabort),
       RWPair'(ResWord'("abs                 "), SPSymbols.RWabs),
       RWPair'(ResWord'("abstract            "), SPSymbols.RWabstract),
       RWPair'(ResWord'("accept              "), SPSymbols.RWaccept),
       RWPair'(ResWord'("access              "), SPSymbols.RWaccess),
       RWPair'(ResWord'("aliased             "), SPSymbols.RWaliased),
       RWPair'(ResWord'("all                 "), SPSymbols.RWall),
       RWPair'(ResWord'("and                 "), SPSymbols.RWand),
       RWPair'(ResWord'("are_interchangeable "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("array               "), SPSymbols.RWarray),
       RWPair'(ResWord'("as                  "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("assert              "), SPSymbols.RWassert),
       RWPair'(ResWord'("assume              "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("at                  "), SPSymbols.RWat),

       RWPair'(ResWord'("begin               "), SPSymbols.RWbegin),
       RWPair'(ResWord'("body                "), SPSymbols.RWbody),

       RWPair'(ResWord'("case                "), SPSymbols.RWcase),
       RWPair'(ResWord'("check               "), SPSymbols.RWcheck),
       RWPair'(ResWord'("const               "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("constant            "), SPSymbols.RWconstant),

       RWPair'(ResWord'("declare             "), SPSymbols.RWdeclare),
       RWPair'(ResWord'("delay               "), SPSymbols.RWdelay),
       RWPair'(ResWord'("delta               "), SPSymbols.RWdelta),
       RWPair'(ResWord'("derives             "), SPSymbols.RWderives),
       RWPair'(ResWord'("digits              "), SPSymbols.RWdigits),
       RWPair'(ResWord'("div                 "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("do                  "), SPSymbols.RWdo),

       RWPair'(ResWord'("element             "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("else                "), SPSymbols.RWelse),
       RWPair'(ResWord'("elsif               "), SPSymbols.RWelsif),
       RWPair'(ResWord'("end                 "), SPSymbols.RWend),
       RWPair'(ResWord'("entry               "), SPSymbols.RWentry),
       RWPair'(ResWord'("exception           "), SPSymbols.RWexception),
       RWPair'(ResWord'("exit                "), SPSymbols.RWexit),

       RWPair'(ResWord'("finish              "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("first               "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("for                 "), SPSymbols.RWfor),
       RWPair'(ResWord'("for_all             "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("for_some            "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("from                "), SPSymbols.RWfrom),
       RWPair'(ResWord'("function            "), SPSymbols.RWfunction),

       RWPair'(ResWord'("generic             "), SPSymbols.RWgeneric),
       RWPair'(ResWord'("global              "), SPSymbols.RWglobal),
       RWPair'(ResWord'("goal                "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("goto                "), SPSymbols.RWgoto),

       RWPair'(ResWord'("hide                "), SPSymbols.RWhide),

       RWPair'(ResWord'("if                  "), SPSymbols.RWif),
       RWPair'(ResWord'("in                  "), SPSymbols.RWin),
       RWPair'(ResWord'("inherit             "), SPSymbols.RWinherit),
       RWPair'(ResWord'("initializes         "), SPSymbols.RWinitializes),
       RWPair'(ResWord'("is                  "), SPSymbols.RWis),

       RWPair'(ResWord'("last                "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("limited             "), SPSymbols.RWlimited),
       RWPair'(ResWord'("loop                "), SPSymbols.RWloop),

       RWPair'(ResWord'("main_program        "), SPSymbols.RWmain_program),
       RWPair'(ResWord'("may_be_deduced      "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("may_be_deduced_from "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("may_be_replaced_by  "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("mod                 "), SPSymbols.RWmod),

       RWPair'(ResWord'("new                 "), SPSymbols.RWnew),
       RWPair'(ResWord'("nonfirst            "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("nonlast             "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("not                 "), SPSymbols.RWnot),
       RWPair'(ResWord'("not_in              "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("null                "), SPSymbols.RWnull),

       RWPair'(ResWord'("odd                 "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("of                  "), SPSymbols.RWof),
       RWPair'(ResWord'("or                  "), SPSymbols.RWor),
       RWPair'(ResWord'("others              "), SPSymbols.RWothers),
       RWPair'(ResWord'("out                 "), SPSymbols.RWout),
       RWPair'(ResWord'("own                 "), SPSymbols.RWown),

       RWPair'(ResWord'("package             "), SPSymbols.RWpackage),
       RWPair'(ResWord'("pending             "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("post                "), SPSymbols.RWpost),
       RWPair'(ResWord'("pragma              "), SPSymbols.RWpragma),
       RWPair'(ResWord'("pre                 "), SPSymbols.RWpre),
       RWPair'(ResWord'("pred                "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("private             "), SPSymbols.RWprivate),
       RWPair'(ResWord'("procedure           "), SPSymbols.RWprocedure),
       RWPair'(ResWord'("proof               "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("protected           "), SPSymbols.RWprotected),

       RWPair'(ResWord'("raise               "), SPSymbols.RWraise),
       RWPair'(ResWord'("range               "), SPSymbols.RWrange),
       RWPair'(ResWord'("real                "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("record              "), SPSymbols.RWrecord),
       RWPair'(ResWord'("rem                 "), SPSymbols.RWrem),
       RWPair'(ResWord'("renames             "), SPSymbols.RWrenames),
       RWPair'(ResWord'("requeue             "), SPSymbols.RWrequeue),
       RWPair'(ResWord'("requires            "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("return              "), SPSymbols.RWreturn),
       RWPair'(ResWord'("reverse             "), SPSymbols.RWreverse),

       RWPair'(ResWord'("save                "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("select              "), SPSymbols.RWselect),
       RWPair'(ResWord'("separate            "), SPSymbols.RWseparate),
       RWPair'(ResWord'("sequence            "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("set                 "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("some                "), SPSymbols.RWsome),
       RWPair'(ResWord'("sqr                 "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("start               "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("strict_subset_of    "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("subset_of           "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("subtype             "), SPSymbols.RWsubtype),
       RWPair'(ResWord'("succ                "), SPSymbols.predefined_FDL_identifier),

       RWPair'(ResWord'("tagged              "), SPSymbols.RWtagged),
       RWPair'(ResWord'("task                "), SPSymbols.RWtask),
       RWPair'(ResWord'("terminate           "), SPSymbols.RWterminate),
       RWPair'(ResWord'("then                "), SPSymbols.RWthen),
       RWPair'(ResWord'("type                "), SPSymbols.RWtype),

       RWPair'(ResWord'("until               "), SPSymbols.RWuntil),
       RWPair'(ResWord'("update              "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("use                 "), SPSymbols.RWuse),

       RWPair'(ResWord'("var                 "), SPSymbols.predefined_FDL_identifier),

       RWPair'(ResWord'("when                "), SPSymbols.RWwhen),
       RWPair'(ResWord'("where               "), SPSymbols.predefined_FDL_identifier),
       RWPair'(ResWord'("while               "), SPSymbols.RWwhile),
       RWPair'(ResWord'("with                "), SPSymbols.RWwith),

       RWPair'(ResWord'("xor                 "), SPSymbols.RWxor));

   NoOfFDL_RW      : constant Natural  := 35;
   subtype FDL_RWIndex  is Natural  range 0 .. NoOfFDL_RW - 1;
   type FDL_RWList is array (FDL_RWIndex) of ResWord;

   FDL_RW : constant FDL_RWList := FDL_RWList'
      (ResWord'("are_interchangeable "),
       ResWord'("as                  "),
       ResWord'("assume              "),

       ResWord'("const               "),

       ResWord'("div                 "),

       ResWord'("element             "),

       ResWord'("finish              "),
       ResWord'("first               "),
       ResWord'("for_all             "),
       ResWord'("for_some            "),

       ResWord'("goal                "),
       ResWord'("last                "),

       ResWord'("may_be_deduced      "),
       ResWord'("may_be_deduced_from "),
       ResWord'("may_be_replaced_by  "),

       ResWord'("nonfirst            "),
       ResWord'("nonlast             "),
       ResWord'("not_in              "),

       ResWord'("odd                 "),

       ResWord'("pending             "),
       ResWord'("pred                "),
       ResWord'("proof               "),

       ResWord'("real                "),
       ResWord'("requires            "),

       ResWord'("save                "),
       ResWord'("sequence            "),
       ResWord'("set                 "),
       ResWord'("sqr                 "),
       ResWord'("start               "),
       ResWord'("strict_subset_of    "),
       ResWord'("subset_of           "),
       ResWord'("succ                "),

       ResWord'("update              "),

       ResWord'("var                 "),

       ResWord'("where               "));

   --  Ada2005 defines 3 new reserved words, which we need to be able
   --  to check for to warn for future upward incompatibility in SPARK83
   --  and/or SPARK95 code.
   RWOverriding   : constant ResWord := ResWord'("overriding          ");
   RWSynchronized : constant ResWord := ResWord'("synchronized        ");
   RWInterface    : constant ResWord := ResWord'("interface           ");

   subtype Offset is Integer range 0 .. 3;
   type PossiblePrefixes is (FieldPrefix, UpdatePrefix);
   type PrefixStrings    is array (Offset) of Character;
   type PrefixTables     is array (PossiblePrefixes) of PrefixStrings;
   PrefixTable : constant PrefixTables :=
      PrefixTables'(FieldPrefix  => PrefixStrings'('f', 'l', 'd', '_'),
                    UpdatePrefix => PrefixStrings'('u', 'p', 'f', '_'));

   package body LineManager is separate;

   --++++  The following functions are defined as per Ada LRM Chapter 2. ++++--
   -------------------------------------------------------------------------------

   -- Note a "Special Character" here to the SPARK Lexer is NOT the
   -- same as Ada95 LRM's Ada.Characters.Handling.Is_Special
   function SpecialCharacter (Ch : Character) return Boolean is
      Result : Boolean;
   begin
      case Ch is
         when '"' | '#' | '&' | ''' | '(' | ')' | '*' | '+' | ',' | '-' | '.' |
            '/' | ':' | ';' | '<' | '=' | '>' | '_' | '|' =>
            Result := True;
         when others =>
            Result := False;
      end case;
      return Result;
   end SpecialCharacter;

   function FormatEffector (Ch : Character) return Boolean is
      Result : Boolean;
   begin
      case Ch is
         when Ada.Characters.Latin_1.HT | Ada.Characters.Latin_1.VT |
              Ada.Characters.Latin_1.CR | Ada.Characters.Latin_1.LF |
              Ada.Characters.Latin_1.FF =>
            Result := True;
         when others   =>
            Result := False;
      end case;
      return Result;
   end FormatEffector;

   function OtherSpecialCharacter (Ch : Character) return Boolean is
      Result : Boolean;
   begin
      case Ch is
         when '!' | '$' | '%' | '?' | '@' | '[' |
              '\' | ']' | '^' | '`' | '{' | '}' | '~' =>
            Result := True;
         when others =>
            Result := False;
      end case;
      return Result;
   end OtherSpecialCharacter;

   -- Note a "Basic Character" here to the SPARK Lexer is NOT the
   -- same as Ada95 LRM's Ada.Characters.Handling.Is_Basic
   function BasicGraphicCharacter (Ch : Character) return Boolean is
   begin
      return Ada.Characters.Handling.Is_Upper (Ch) or else
             Ada.Characters.Handling.Is_Digit (Ch) or else
             SpecialCharacter (Ch) or else
             Ch = ' ';
   end BasicGraphicCharacter;

   -- Note a "Graphic Character" here to the SPARK Lexer is NOT the
   -- same as Ada95 LRM's Ada.Characters.Handling.Is_Graphic
   function GraphicCharacter (Ch : Character) return Boolean is
   begin
      return BasicGraphicCharacter (Ch) or else
             Ada.Characters.Handling.Is_Lower (Ch) or else
             OtherSpecialCharacter (Ch);
   end GraphicCharacter;

   function Separator (Ch : Character) return Boolean is
      Result : Boolean;
   begin
      case Ch is
         when ' ' |
              Ada.Characters.Latin_1.HT | Ada.Characters.Latin_1.VT |
              Ada.Characters.Latin_1.CR | Ada.Characters.Latin_1.LF |
              Ada.Characters.Latin_1.FF =>
            Result := True;
         when others   =>
            Result := False;
      end case;
      return Result;
   end Separator;

   function SimpleDelimiter (Ch : Character) return Boolean is
      Result : Boolean;
   begin
      case Ch is
         when '&' | ''' | '(' | ')' | '*' | '+' | ',' | '-' | '.' |
              '/' | ':' | ';' | '<' | '=' | '>' | '|' | '[' | ']' |
              '@' | '~' | '%' =>
            Result := True;
         when others =>
            Result := False;
      end case;
      return Result;
   end SimpleDelimiter;

   function LetterOrDigit (Ch : Character) return Boolean is
   begin
      return Ada.Characters.Handling.Is_Letter (Ch) or else
             Ada.Characters.Handling.Is_Digit (Ch);
   end LetterOrDigit;

   function ExtendedDigit (Ch : Character) return Boolean is
   begin
      -- We consider an extended digit to be a decimal digit or
      -- any letter, so...
      return LetterOrDigit (Ch);
   end ExtendedDigit;

-------------------------------------------------------------------------------

   procedure ClearLineContext
   --# global out LineManager.CurrLine;
   --# derives LineManager.CurrLine from ;
   is
   begin
      LineManager.ClearLine;
   end ClearLineContext;

   procedure StoreLineContext (FileLine : out LineContext)
   --# global in LineManager.CurrLine;
   --# derives FileLine from LineManager.CurrLine;
   is
      Line : LineManager.ProgramLine;
   begin
      LineManager.CopyOutLine (Line);
      FileLine.Context := Line.Context;
      -- field-by-field update of record will give flow error here
      FileLine.AnnoContext := Line.AnnoContext;
      FileLine.LineNo := Line.LineNo;
      FileLine.LineLength := Line.LineLength;
      FileLine.Conts := Line.Conts;
      FileLine.LastTokenPos := Line.LastTokenPos;
      FileLine.CurrPos := Line.CurrPos;
      FileLine.LookaheadPos := Line.LookaheadPos;
   end StoreLineContext;

   procedure RestoreLineContext (FileLine : in LineContext)
   --# global out LineManager.CurrLine;
   --# derives LineManager.CurrLine from FileLine;
   is
      Line : LineManager.ProgramLine;
   begin
      Line.Context := FileLine.Context;
      -- field-by-field update of record will give flow error here
      Line.AnnoContext := FileLine.AnnoContext;
      Line.LineNo := FileLine.LineNo;
      Line.Conts := FileLine.Conts;
      Line.LineLength := FileLine.LineLength;
      Line.LastTokenPos := FileLine.LastTokenPos;
      Line.CurrPos := FileLine.CurrPos;
      Line.LookaheadPos := FileLine.LookaheadPos;
      LineManager.CopyInLine (Line);
   end RestoreLineContext;

   ---------------------------------------------------------------------------

   procedure CheckReserved (StartPos, EndPos : in     EStrings.Lengths;
                            Token            :    out SPSymbols.SPTerminal)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     LineManager.CurrLine;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EndPos,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        LineManager.CurrLine,
   --#                                        SPARK_IO.File_Sys,
   --#                                        StartPos &
   --#         Token                     from CommandLineData.Content,
   --#                                        EndPos,
   --#                                        LineManager.CurrLine,
   --#                                        StartPos;
   is
      type CmpRes is (CEQ, CLT, CGT);
      Leng   : Natural;
      Ix     : RWIndex;
      IL, IU : Natural;
      RWIx   : ResWord;
      Result : CmpRes;
      LToken : SPSymbols.SPTerminal;

      -----------------------------------------

      procedure CompRW (RWord            : in     ResWord;
                        StartPos, EndPos : in     EStrings.Lengths;
                        Result           :    out CmpRes)
      --# global in LineManager.CurrLine;
      --# derives Result from EndPos,
      --#                     LineManager.CurrLine,
      --#                     RWord,
      --#                     StartPos;
      is
         Ch1, Ch2   : Character;
         LX         : EStrings.Lengths;
         RWX        : RWIndex;
         CompResult : CmpRes;
      begin
         RWX := 1; LX := StartPos;

         loop
            Ch1 := RWord (RWX);

            -- Reserved words in lower case.
            Ch2 := Ada.Characters.Handling.To_Lower (LineManager.CurrLine.Conts (LX));

            if Ch1 < Ch2 then
               CompResult := CLT;
            elsif Ch1 > Ch2 then
               CompResult := CGT;
            elsif LX = EndPos then
               if RWX = MaxRWLength or else RWord (RWX + 1) = ' 'then
                  CompResult := CEQ;
               else
                  CompResult := CGT;
               end if;
            else
               CompResult := CEQ;
            end if;
            exit when CompResult /= CEQ or else LX = EndPos;
            LX := LX + 1; RWX := RWX + 1;
         end loop;
         Result := CompResult;
      end CompRW;

      function CheckFLDorUPF return SPSymbols.SPTerminal
      --# global in Leng;
      --#        in LineManager.CurrLine;
      --#        in StartPos;
      is
         PrefixSort : PossiblePrefixes;
         Result     : SPSymbols.SPTerminal;

         ----------------------------------------------

         procedure CheckRestOfPrefix
         --# global in     LineManager.CurrLine;
         --#        in     PrefixSort;
         --#        in     StartPos;
         --#           out Result;
         --# derives Result from LineManager.CurrLine,
         --#                     PrefixSort,
         --#                     StartPos;
         is
            Ptr : Offset;
         begin
            Ptr := 1;
            loop
               if Ada.Characters.Handling.To_Lower
                  (LineManager.CurrLine.Conts (StartPos + Ptr)) /=
                  PrefixTable (PrefixSort) (Ptr) then
                  Result := SPSymbols.identifier;
                  exit;
               end if;

               if Ptr = Offset'Last then
                  Result := SPSymbols.predefined_FDL_identifier;
                  exit;
               end if;

               Ptr := Ptr + 1;
            end loop;
         end CheckRestOfPrefix;

         ----------------------------------------------

      begin  --CheckFLDorUPF
         Result := SPSymbols.identifier;
         if Leng >= 5 then -- minimum length a valid fld_ or upf_ could be
            if Ada.Characters.Handling.To_Lower
               (LineManager.CurrLine.Conts (StartPos)) = 'f' then

               PrefixSort := FieldPrefix;
               CheckRestOfPrefix;
            elsif Ada.Characters.Handling.To_Lower
               (LineManager.CurrLine.Conts (StartPos)) = 'u' then

               PrefixSort := UpdatePrefix;
               CheckRestOfPrefix;
            else
               Result := SPSymbols.identifier;
            end if;
         end if;
         return Result;
      end CheckFLDorUPF;

      -----------------------------------------------------------

      function ConvertFDL (Token : SPSymbols.SPTerminal)
                          return SPSymbols.SPTerminal
      --# global in CommandLineData.Content;
      is
         Result : SPSymbols.SPTerminal;
      begin
         if Token = SPSymbols.predefined_FDL_identifier and then
            not CommandLineData.Content.FDLreserved then
            Result := SPSymbols.identifier;
         else
            Result := Token;
         end if;
         return Result;
      end ConvertFDL;

      -----------------------------------------------------------

      function ConvertReserved (Token : SPSymbols.SPTerminal)
                               return SPSymbols.SPTerminal
      --# global in CommandLineData.Content;
      is
         Result : SPSymbols.SPTerminal;
      begin
         if Token = SPSymbols.RWsome and then
            not CommandLineData.Content.FDLreserved then
            Result := SPSymbols.identifier;
         else
            Result := Token;
         end if;
         return Result;
      end ConvertReserved;

      -----------------------------------------------------------

      function Convert95Reserved (Token : SPSymbols.SPTerminal)
                                return SPSymbols.SPTerminal
      --# global in CommandLineData.Content;
      is
         Result : SPSymbols.SPTerminal;
      begin
         if (Token = SPSymbols.RWaliased or else
             Token = SPSymbols.RWprotected or else
             Token = SPSymbols.RWrequeue or else
             Token = SPSymbols.RWtagged or else
             Token = SPSymbols.RWuntil) and then
            CommandLineData.IsSpark83 then
            Result := SPSymbols.identifier;
         elsif
            Token = SPSymbols.RWabstract and then
            CommandLineData.IsSpark83 and then
            not CommandLineData.Content.FDLreserved then
            Result := SPSymbols.identifier;
         else
            Result := Token;
         end if;
         return Result;
      end Convert95Reserved;

      procedure CheckAda2005Reserved (Token : SPSymbols.SPTerminal)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     EndPos;
      --#        in     LexTokenManager.StringTable;
      --#        in     LineManager.CurrLine;
      --#        in     StartPos;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.File_Sys;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.File_Sys         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        EndPos,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.StringTable,
      --#                                        LineManager.CurrLine,
      --#                                        SPARK_IO.File_Sys,
      --#                                        StartPos,
      --#                                        Token;
      is
         LResult1 : CmpRes;
         LResult2 : CmpRes;
         LResult3 : CmpRes;
      begin
         -- Only compare and complain for identifiers appearing
         -- in code, not annotations, AND if we're in SPARK95 mode.
         -- This avoids warning about "pragma Interface" everywhere
         -- in SPARK83 code.
         if Token = SPSymbols.identifier and then
           LineManager.CurrLine.Context = InAda and then
           CommandLineData.IsSpark95 then

            CompRW (RWInterface,    StartPos, EndPos, LResult1);
            CompRW (RWSynchronized, StartPos, EndPos, LResult2);
            CompRW (RWOverriding,   StartPos, EndPos, LResult3);

            if LResult1 = CEQ or LResult2 = CEQ or LResult3 = CEQ then
               ErrorHandler.SemanticWarning
                 (7,
                  LexTokenManager.TokenPosition'(LineManager.CurrLine.LineNo,
                                                 StartPos),
                  LexTokenManager.NullString);
            end if;
         end if;

      end CheckAda2005Reserved;

      -----------------------------------------------------------

   begin -- CheckReserved
      Leng := (EndPos - StartPos) + 1;
      if Leng >= MinRWLength and Leng <= MaxRWLength then
         IL := RWIndex'First;
         IU := RWIndex'Last;

         loop
            CompRW (RW (IL).Word, StartPos, EndPos, Result);
            if Result = CGT then
               exit;
            end if;
            CompRW (RW (IU).Word, StartPos, EndPos, Result);
            if Result = CLT then
               exit;
            end if;

            Ix := (IL + IU) / 2;
            RWIx := RW (Ix).Word;
            CompRW (RWIx, StartPos, EndPos, Result);
            case Result is
               when CEQ => null;
               when CGT => IU := Ix - 1;
               when CLT => IL := Ix + 1;
            end case;
            exit when Result = CEQ or else IL > IU;
         end loop;

         if Result = CEQ then
            --# accept F, 501, Ix, "Ix always defined on this path";
            LToken := RW (Ix).Token;
            --# end accept;
         else
            LToken :=  CheckFLDorUPF;
         end if;
      else
         LToken := CheckFLDorUPF;
      end if;

      LToken := Convert95Reserved (ConvertReserved (ConvertFDL (LToken)));

      CheckAda2005Reserved (LToken);

      Token := LToken;
      --# accept F, 602, SPARK_IO.File_Sys,         Ix, "Ix always defined here" &
      --#        F, 602, ErrorHandler.ErrorContext, Ix, "Ix always defined here" &
      --#        F, 602, Token,                     Ix, "Ix always defined here";
   end CheckReserved;


   function CheckFDL_RW (ExStr : EStrings.T) return Boolean
   is
      type CmpRes is (CEQ, CLT, CGT);
      IsFDL_RW : Boolean := True;
      Ix       : FDL_RWIndex;
      FDL_RWIx : ResWord;
      IL, IU   : Natural;
      Result   : CmpRes;

      procedure CompFDL_RW (RWord  : in     ResWord;
                            ExStr  : in     EStrings.T;
                            Result :    out CmpRes)
      --# derives Result from ExStr,
      --#                     RWord;
      is
         Ch1, Ch2   : Character;
         LX         : EStrings.Lengths;
         RWX        : RWIndex;
         CompResult : CmpRes;
      begin
         RWX := 1; LX := 1;

         loop
            Ch1 := RWord (RWX);

            -- Reserved words in lower case.
            Ch2 := Ada.Characters.Handling.To_Lower (ExStr.Content (LX));

            if Ch1 < Ch2 then
               CompResult := CLT;
            elsif Ch1 > Ch2 then
               CompResult := CGT;
            elsif LX = ExStr.Length then
               if RWX = MaxRWLength or else RWord (RWX + 1) = ' ' then
                  CompResult := CEQ;
               else
                  CompResult := CGT;
               end if;
            else
               CompResult := CEQ;
            end if;
            exit when CompResult /= CEQ or else LX = ExStr.Length;
            LX := LX + 1; RWX := RWX + 1;
         end loop;
         Result := CompResult;
      end CompFDL_RW;

   begin
      if ExStr.Length > MaxRWLength or ExStr.Length < MinRWLength then
         IsFDL_RW := False;
      else
         IL := FDL_RWIndex'First;
         IU := FDL_RWIndex'Last;
         loop
            CompFDL_RW (FDL_RW (IL), ExStr, Result);
            if Result = CGT then
               exit;
            end if;
            CompFDL_RW (FDL_RW (IU), ExStr, Result);
            if Result = CLT then
               exit;
            end if;

            Ix := (IL + IU) / 2;
            FDL_RWIx := FDL_RW (Ix);
            CompFDL_RW (FDL_RWIx, ExStr, Result);
            case Result is
               when CEQ => null;
               when CGT => IU := Ix - 1;
               when CLT => IL := Ix + 1;
            end case;
            exit when Result = CEQ or else IL > IU;
         end loop;

         if Result /= CEQ then
            IsFDL_RW := False;
         end if;
      end if;

      return IsFDL_RW;
   end CheckFDL_RW;

   ---------------------------------------------------------------------------

   --  Returns the type of the annotation starting at the lookahead position in
   --  the line buffer.
   procedure CheckAnnoType (UnfinishedAnno : in     Boolean;
                            AnnoKind       :    out AnnoType)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     LineManager.CurrLine;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives AnnoKind                  from CommandLineData.Content,
   --#                                        LineManager.CurrLine,
   --#                                        UnfinishedAnno &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        LineManager.CurrLine,
   --#                                        SPARK_IO.File_Sys;
   is
      type ElementType is (EndOfLineOrText, NonCharacter, RWOrIdent);

      Unused,
      StartPosn,
      EndPosn     : EStrings.Lengths;
      AnnoToken   : SPSymbols.SPTerminal;
      NextElement : ElementType;

      procedure CheckNextElement (StartPosn   : in     EStrings.Lengths;
                                  EndPosn     :    out EStrings.Lengths;
                                  NextElement :    out ElementType;
                                  Symbol      :    out SPSymbols.SPTerminal)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in     LineManager.CurrLine;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.File_Sys;
      --# derives EndPosn,
      --#         NextElement               from LineManager.CurrLine,
      --#                                        StartPosn &
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.File_Sys         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.StringTable,
      --#                                        LineManager.CurrLine,
      --#                                        SPARK_IO.File_Sys,
      --#                                        StartPosn &
      --#         Symbol                    from CommandLineData.Content,
      --#                                        LineManager.CurrLine,
      --#                                        StartPosn;
      is
         Ch             : Character;
         LocalEndPosn,
         LocalStartPosn : EStrings.Lengths;
      begin
         LocalStartPosn := StartPosn;
         loop
            Ch := LineManager.CurrLine.Conts (LocalStartPosn);
            exit when Ch = EndOfLine or
              Ch = EndOfText or
              not Separator (Ch);
            LocalStartPosn := LocalStartPosn + 1;
         end loop;
         LocalEndPosn := LocalStartPosn;
         if Ch = EndOfLine or Ch = EndOfText then
            NextElement := EndOfLineOrText;
            Symbol      := SPSymbols.illegal_id; -- not used
         elsif not Ada.Characters.Handling.Is_Letter (Ch) then
            NextElement := NonCharacter;
            Symbol      := SPSymbols.illegal_id;  -- not used
         else
            NextElement := RWOrIdent;

            -- Scan the next identifier, but then see if it's a reserved word.
            while LetterOrDigit (LineManager.CurrLine.Conts (LocalEndPosn)) or
              LineManager.CurrLine.Conts (LocalEndPosn) = '_' loop

               LocalEndPosn := LocalEndPosn + 1;
            end loop;

            LocalEndPosn := LocalEndPosn - 1;
            CheckReserved (LocalStartPosn, LocalEndPosn, Symbol);
         end if;
         EndPosn := LocalEndPosn;

      end CheckNextElement;


   begin
      StartPosn := LineManager.CurrLine.LookaheadPos + 1;
      CheckNextElement (StartPosn, EndPosn, NextElement, AnnoToken);
      if NextElement = EndOfLineOrText then
         AnnoKind := NoAnno;
      elsif NextElement = NonCharacter then
         AnnoKind := OtherAnno;
      else
         case AnnoToken is
            when SPSymbols.RWmain_program | SPSymbols.RWinherit |
               SPSymbols.RWown | SPSymbols.RWinitializes |
               SPSymbols.RWglobal | SPSymbols.RWderives | SPSymbols.RWdeclare |
               SPSymbols.RWpre | SPSymbols.RWpost =>
               AnnoKind := StartAnno;

            when SPSymbols.RWreturn =>
               if (UnfinishedAnno) then
                  -- Still in an annotation so it's not the start of a new one
                  AnnoKind := OtherAnno;
               else
                  -- New annotation
                  AnnoKind := StartAnno;
               end if;
            when SPSymbols.RWassert | SPSymbols.RWcheck |
               SPSymbols.RWtype | SPSymbols.RWsubtype |
              SPSymbols.RWfunction | SPSymbols.RWaccept |
              SPSymbols.RWend =>

               AnnoKind := ProofAnno;

            when SPSymbols.RWfor =>
               -- do a second look ahead to check for "some" or "all"
               StartPosn := EndPosn + 1;
               --# accept F, 10, Unused, "Unused not referenced here";
               CheckNextElement (StartPosn, Unused, NextElement, AnnoToken);
               --# end accept;
               if NextElement = RWOrIdent and then
                  (AnnoToken = SPSymbols.RWsome or AnnoToken = SPSymbols.RWall) then
                  AnnoKind := OtherAnno;
               else
                  AnnoKind := ProofAnno;
               end if;
            when SPSymbols.RWhide =>
               AnnoKind := HideAnno;
            when others =>
               -- When a proof constant declaration occurs
               -- interpreting --# as proof context is
               -- handled by HyphIntro.
               AnnoKind := OtherAnno;
         end case;
      end if;
      --# accept F, 33, Unused, "Unused not referenced here";
   end CheckAnnoType;

   -- Main implementation of the lexical analyser, common to both
   -- the Examiner and SPARKFormat.
   procedure Lex (ProgText    : in     SPARK_IO.File_Type;
                  AllowDollar : in     Boolean;
                  Token       :    out SPSymbols.SPTerminal;
                  LexVal      :    out LexTokenManager.LexValue;
                  PunctToken  :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out LineManager.CurrLine;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.ErrorContext,
   --#         LexTokenManager.StringTable,
   --#         LexVal,
   --#         LineManager.CurrLine,
   --#         PunctToken,
   --#         SPARK_IO.File_Sys,
   --#         Token                       from AllowDollar,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          LexTokenManager.StringTable,
   --#                                          LineManager.CurrLine,
   --#                                          ProgText,
   --#                                          SPARK_IO.File_Sys;
   is separate;

   procedure ExaminerLex (ProgText   : in     SPARK_IO.File_Type;
                          Token      :    out SPSymbols.SPTerminal;
                          LexVal     :    out LexTokenManager.LexValue;
                          PunctToken :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out LineManager.CurrLine;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.ErrorContext,
   --#         LexTokenManager.StringTable,
   --#         LexVal,
   --#         LineManager.CurrLine,
   --#         PunctToken,
   --#         SPARK_IO.File_Sys,
   --#         Token                       from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          LexTokenManager.StringTable,
   --#                                          LineManager.CurrLine,
   --#                                          ProgText,
   --#                                          SPARK_IO.File_Sys;
   is
   begin
      Lex (ProgText, False, Token, LexVal, PunctToken);
   end ExaminerLex;

   procedure SPARKFormatLex (ProgText   : in     SPARK_IO.File_Type;
                             Token      :    out SPSymbols.SPTerminal;
                             LexVal     :    out LexTokenManager.LexValue;
                             PunctToken :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out LineManager.CurrLine;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.ErrorContext,
   --#         LexTokenManager.StringTable,
   --#         LexVal,
   --#         LineManager.CurrLine,
   --#         PunctToken,
   --#         SPARK_IO.File_Sys,
   --#         Token                       from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          LexTokenManager.StringTable,
   --#                                          LineManager.CurrLine,
   --#                                          ProgText,
   --#                                          SPARK_IO.File_Sys;
   is
   begin
      -- For SPARKFormat, we allow identifiers to begin with a '$'
      -- character so that SPARKFormat doesn't mess up annotations
      -- containing GNATPREP symbols
      Lex (ProgText, True, Token, LexVal, PunctToken);
   end SPARKFormatLex;


   function SimilarTokens (Token1, Token2 : SPSymbols.SPTerminal) return Boolean
   is
      type TokenType is (Delimiter, ReservedWord, Id, Number, Chars, OtherToken);

      TokenType1,
      TokenType2 : TokenType;
      Result     : Boolean;

      function TypeOfToken (Token : SPSymbols.SPTerminal) return TokenType
      is
         TokenKind : TokenType;
      begin
         case Token is
            when SPSymbols.ampersand | SPSymbols.apostrophe | SPSymbols.left_paren |
               SPSymbols.right_paren | SPSymbols.multiply |
               SPSymbols.plus | SPSymbols.comma | SPSymbols.minus |
               SPSymbols.point | SPSymbols.divide | SPSymbols.colon |
               SPSymbols.semicolon |
               SPSymbols.less_than | SPSymbols.equals | SPSymbols.greater_than |
               SPSymbols.vertical_bar | SPSymbols.arrow |
               SPSymbols.double_dot | SPSymbols.double_star | SPSymbols.becomes |
               SPSymbols.not_equal |
               SPSymbols.greater_or_equal | SPSymbols.less_or_equal |
               SPSymbols.left_label_paren | SPSymbols.right_label_paren |
               SPSymbols.box | SPSymbols.implies | SPSymbols.is_equivalent_to |
               SPSymbols.tilde | SPSymbols.square_open |
               SPSymbols.square_close |
               SPSymbols.percent
               => TokenKind := Delimiter;
            when SPSymbols.integer_number | SPSymbols.real_number |
               SPSymbols.based_integer | SPSymbols.based_real
               => TokenKind := Number;
            when SPSymbols.character_literal | SPSymbols.string_literal
               => TokenKind := Chars;
            when SPSymbols.identifier
               => TokenKind := Id;
            when SPSymbols.RWabort | SPSymbols.RWabs | SPSymbols.RWabstract | SPSymbols.RWaccept |
               SPSymbols.RWaccess | SPSymbols.RWaliased | SPSymbols.RWall | SPSymbols.RWand |
               SPSymbols.RWandthen | SPSymbols.RWany |
               SPSymbols.RWarray |
               SPSymbols.RWassert |
               SPSymbols.RWat | SPSymbols.RWbegin | SPSymbols.RWbody |
               SPSymbols.RWcase | SPSymbols.RWcheck |
               SPSymbols.RWconstant | SPSymbols.RWdeclare | SPSymbols.RWdelay |
               SPSymbols.RWdelta | SPSymbols.RWderives | SPSymbols.RWdigits |
               SPSymbols.RWdo |
               SPSymbols.RWelse | SPSymbols.RWelsif |
               SPSymbols.RWend | SPSymbols.RWentry | SPSymbols.RWexception |
               SPSymbols.RWexit |
               SPSymbols.RWfor | SPSymbols.RWforall | SPSymbols.RWforsome |
               SPSymbols.RWfrom | SPSymbols.RWfunction | SPSymbols.RWgeneric |
               SPSymbols.RWglobal | SPSymbols.RWgoto |
               SPSymbols.RWhide | SPSymbols.RWif |
               SPSymbols.RWin | SPSymbols.RWinherit | SPSymbols.RWinitializes |
               SPSymbols.RWis |
               SPSymbols.RWlimited | SPSymbols.RWloop |
               SPSymbols.RWmain_program | SPSymbols.RWmod | SPSymbols.RWnew |
               SPSymbols.RWnot |
               SPSymbols.RWnotin | SPSymbols.RWnull |
               SPSymbols.RWof |
               SPSymbols.RWor |
               SPSymbols.RWorelse | SPSymbols.RWothers |
               SPSymbols.RWout | SPSymbols.RWown | SPSymbols.RWpackage |
               SPSymbols.RWpost | SPSymbols.RWpragma |
               SPSymbols.RWpre | SPSymbols.RWprivate | SPSymbols.RWprotected |
               SPSymbols.RWprocedure | SPSymbols.RWraise |
               SPSymbols.RWrange | SPSymbols.RWrecord | SPSymbols.RWrequeue | SPSymbols.RWrem |
               SPSymbols.RWrenames | SPSymbols.RWreturn |
               SPSymbols.RWreverse |
               SPSymbols.RWselect | SPSymbols.RWseparate |
               SPSymbols.RWsome | SPSymbols.RWsubtype | SPSymbols.RWtagged |
               SPSymbols.RWtask | SPSymbols.RWterminate | SPSymbols.RWthen |
               SPSymbols.RWtype | SPSymbols.RWuntil | SPSymbols.RWuse |
               SPSymbols.RWwhen |
               SPSymbols.RWwhile | SPSymbols.RWwith |
               SPSymbols.RWxor
               => TokenKind := ReservedWord;
            when others
               => TokenKind := OtherToken;
         end case;
         return TokenKind;
      end TypeOfToken;

   begin
      TokenType1 := TypeOfToken (Token1);
      TokenType2 := TypeOfToken (Token2);
      case TokenType1 is
         when Delimiter =>
            case TokenType2 is
               when Delimiter =>
                  Result := True;
               when others =>
                  Result := False;
            end case;
         when Number =>
            case TokenType2 is
               when Number =>
                  Result := True;
               when others =>
                  Result := False;
            end case;
         when Chars =>
            case TokenType2 is
               when Chars =>
                  Result := True;
               when others =>
                  Result := False;
            end case;
         when Id =>
            case TokenType2 is
               when Id =>
                  Result := True;
               when ReservedWord =>
                  Result := True;
               when others =>
                  Result := False;
            end case;
         when ReservedWord =>
            case TokenType2 is
               when Id =>
                  Result := True;
               when ReservedWord =>
                  Result := True;
               when others =>
                  Result := False;
            end case;
         when others =>
            Result := False;
      end case;
      return Result;
   end SimilarTokens;

end SparkLex;
