-- $Id: sparklex-lex.adb 13063 2009-04-21 12:05:08Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with LexTokenLists;

separate (SparkLex)
procedure Lex (ProgText    : in     SPARK_IO.File_Type;
               AllowDollar : in     Boolean;
               Token       :    out SPSymbols.SPTerminal;
               LexVal      :    out LexTokenManager.LexValue;
               PunctToken  :    out Boolean)
is
   StartLine : LexTokenManager.LineNumbers;
   StartPosn,
   EndPosn : EStrings.Lengths;
   LexStr : LexTokenManager.LexString;
   NextToken : SPSymbols.SPTerminal;
   Ch : Character;
   ErrVal : LexTokenManager.LexValue;
   HiddenPart,
   EndHideFound : Boolean;
   HideDesignator,
   EndDesignator : LexTokenLists.Lists;

   procedure GetIdent (Token : out SPSymbols.SPTerminal)
   --# global in out LineManager.CurrLine;
   --# derives LineManager.CurrLine,
   --#         Token                from LineManager.CurrLine;
      is separate;

   procedure GetNumber (Token : out SPSymbols.SPTerminal)
   --# global in out LineManager.CurrLine;
   --# derives LineManager.CurrLine,
   --#         Token                from LineManager.CurrLine;
      is separate;

   procedure HyphIntro (Token : out SPSymbols.SPTerminal)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LineManager.CurrLine;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        LineManager.CurrLine,
   --#                                        SPARK_IO.File_Sys &
   --#         LineManager.CurrLine,
   --#         Token                     from CommandLineData.Content,
   --#                                        LineManager.CurrLine;
      is separate;

   procedure GetString (Token : out SPSymbols.SPTerminal)
   --# global in out LineManager.CurrLine;
   --# derives LineManager.CurrLine,
   --#         Token                from LineManager.CurrLine;
      is separate;

   procedure ApostIntro (Token : out SPSymbols.SPTerminal)
   --# global in out LineManager.CurrLine;
   --# derives LineManager.CurrLine,
   --#         Token                from LineManager.CurrLine;
      is separate;

   procedure LTIntro (Token : out SPSymbols.SPTerminal)
   --# global in out LineManager.CurrLine;
   --# derives LineManager.CurrLine,
   --#         Token                from LineManager.CurrLine;
      is separate;

   procedure NextLex (Token : out SPSymbols.SPTerminal)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LineManager.CurrLine;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        LineManager.CurrLine,
   --#                                        SPARK_IO.File_Sys &
   --#         LineManager.CurrLine,
   --#         Token                     from CommandLineData.Content,
   --#                                        LineManager.CurrLine;
   --
   -- The Ada Lexis allows the class of a token to be determined by the
   -- first character in the text string representing it. Given the first
   -- character of the string representing the token is at the current position
   -- in the line buffer, LineManager.CurrLine, NextLex determines the class of the token and
   -- calls a procedure to recognise that class of token.
   -- On exit Token is set to a value representing the token.
      is separate;

   -- Modified list of reserved words which are punctuation tokens
   -- this is a knock on effect from change in syntax of proof contexts
   function IsPunctToken (Token : SPSymbols.SPTerminal) return Boolean
   is
      Result : Boolean;
   begin
      case Token is
         when SPSymbols.left_paren | SPSymbols.right_paren | SPSymbols.comma |
           SPSymbols.colon | SPSymbols.semicolon | SPSymbols.becomes |
           SPSymbols.double_dot | SPSymbols.point | SPSymbols.apostrophe |
           SPSymbols.vertical_bar | SPSymbols.arrow |
           SPSymbols.annotation_start | SPSymbols.annotation_end |
           SPSymbols.proof_context |
           SPSymbols.RWabort | SPSymbols.RWaccess |
           SPSymbols.RWall |
           SPSymbols.RWarray | SPSymbols.RWassert |
           SPSymbols.RWat | SPSymbols.RWbegin |
           SPSymbols.RWbody | SPSymbols.RWcase |
           SPSymbols.RWcheck | SPSymbols.RWconstant |
           SPSymbols.RWdeclare   | SPSymbols.RWdelta |
           SPSymbols.RWderives | SPSymbols.RWdigits | SPSymbols.RWdo |
           SPSymbols.RWelse |
           SPSymbols.RWelsif | SPSymbols.RWend | SPSymbols.RWentry |
           SPSymbols.RWexception |
           SPSymbols.RWfor | SPSymbols.RWfrom |
           SPSymbols.RWfunction | SPSymbols.RWgeneric | SPSymbols.RWglobal |
           SPSymbols.RWgoto | SPSymbols.RWhide |
           SPSymbols.RWif | SPSymbols.RWin |
           SPSymbols.RWinherit | SPSymbols.RWinitializes |
           SPSymbols.RWis |
           SPSymbols.RWlimited | SPSymbols.RWloop |
           SPSymbols.RWnew |
           SPSymbols.RWnotin |
           SPSymbols.RWof |
           SPSymbols.RWothers | SPSymbols.RWout |
           SPSymbols.RWown | SPSymbols.RWpackage |
           SPSymbols.RWpost | SPSymbols.RWpragma | SPSymbols.RWpre |
           SPSymbols.RWprivate | SPSymbols.RWprocedure |
           SPSymbols.RWraise |
           SPSymbols.RWrange | SPSymbols.RWrecord | SPSymbols.RWrenames |
           SPSymbols.RWreturn | SPSymbols.RWreverse |
           SPSymbols.RWselect | SPSymbols.RWseparate |
           SPSymbols.RWsubtype |
           SPSymbols.RWtask | SPSymbols.RWterminate | SPSymbols.RWthen |
           SPSymbols.RWtype | SPSymbols.RWuse |
           SPSymbols.RWwhen |
           SPSymbols.RWwhile | SPSymbols.RWwith |
            -- SPARK95 reserved words
           SPSymbols.RWabstract | SPSymbols.RWrequeue | SPSymbols.RWprotected |
           SPSymbols.RWuntil

            => Result := True;
         when others
            => Result := False;
      end case;
      return Result;
   end IsPunctToken;


   procedure CheckFollowingToken
      (PossibleFollower : in SPSymbols.SPSymbol;
       TransformedToken : in SPSymbols.SPSymbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     ProgText;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out LineManager.CurrLine;
   --#        in out NextToken;
   --#        in out SPARK_IO.File_Sys;
   --#           out LexStr;
   --# derives ErrorHandler.ErrorContext,
   --#         LexTokenManager.StringTable,
   --#         SPARK_IO.File_Sys           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          LexTokenManager.StringTable,
   --#                                          LineManager.CurrLine,
   --#                                          ProgText,
   --#                                          SPARK_IO.File_Sys &
   --#         LexStr                      from  &
   --#         LineManager.CurrLine        from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          LexTokenManager.StringTable,
   --#                                          PossibleFollower,
   --#                                          ProgText,
   --#                                          SPARK_IO.File_Sys &
   --#         NextToken                   from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          LexTokenManager.StringTable,
   --#                                          LineManager.CurrLine,
   --#                                          PossibleFollower,
   --#                                          ProgText,
   --#                                          SPARK_IO.File_Sys,
   --#                                          TransformedToken;
   is
      FollowingToken : SPSymbols.SPTerminal;
   begin
      LineManager.NextSigChar (ProgText);
      LineManager.RecordCurrPos;
      GetIdent (FollowingToken);
      if FollowingToken = SPSymbols.identifier then
         CheckReserved
            (LineManager.CurrLine.LastTokenPos,
             LineManager.CurrLine.CurrPos - 1,
             FollowingToken);
         if FollowingToken = PossibleFollower then
            NextToken := TransformedToken;
         else
            LineManager.ResetCurrPos;
         end if;
      else
         LineManager.ResetCurrPos;
      end if;
      LexStr := LexTokenManager.NullString;
   end CheckFollowingToken;

   ---------------------

   procedure InsertStringLiteral
      (LineIn                  : in     EStrings.Line;
       StartPos,
       EndPos                  : in     EStrings.Positions;
       LexStr                  :    out LexTokenManager.LexString)
   --# global in out LexTokenManager.StringTable;
   --# derives LexStr,
   --#         LexTokenManager.StringTable from EndPos,
   --#                                          LexTokenManager.StringTable,
   --#                                          LineIn,
   --#                                          StartPos;

   is
      ModifiedLine : EStrings.Line;
      LocalPtr     : EStrings.Positions;
      Ch           : Character;
      SkipNext     : Boolean := False;
   begin
      -- copy leading quote
      --# accept F, 23, ModifiedLine, "Array update anomaly";
      ModifiedLine (1) := LineIn (StartPos);
      --# end accept;

      -- copy character up to closing quote eliminating doubled quotes
      LocalPtr := 2;
      for I in EStrings.Positions range StartPos + 1 .. EndPos - 1 loop
         if SkipNext then
            SkipNext := False;
         else
            Ch := LineIn (I);
            ModifiedLine (LocalPtr) := Ch;
            LocalPtr := LocalPtr + 1;
            SkipNext := Ch = '"';
         end if;
      end loop;

      -- copy closing quote
      ModifiedLine (LocalPtr) := LineIn (EndPos);

      -- put in string table
      LexTokenManager.InsertCaseSensitiveLexString (ModifiedLine,
                                                    1,
                                                    LocalPtr,
                                                    LexStr);
      --# accept F, 602, LexTokenManager.StringTable, ModifiedLine, "Array update anomaly" &
      --#        F, 602, LexStr,                      ModifiedLine, "Array update anomaly";
   end InsertStringLiteral;

   ---------------------

begin -- Lex

   --# accept W, 169, LineManager.CurrLine.AnnoContext, "Direct updates here OK";

   HiddenPart := False;
   HideDesignator := LexTokenLists.NullList;
   loop
      --# assert True; -- for RTC generation
      LineManager.NextSigChar (ProgText);
      StartLine := LineManager.CurrLine.LineNo;
      StartPosn := LineManager.CurrLine.CurrPos;
      NextLex (NextToken);
      EndPosn := LineManager.CurrLine.CurrPos - 1;

      --# assert True; -- for RTC generation

      case NextToken is
         when SPSymbols.identifier =>
            CheckReserved (StartPosn, EndPosn, NextToken);
            if NextToken = SPSymbols.identifier then

               if LineManager.CurrLine.Conts (StartPosn) = '$' then
                  --# accept F, 41, "Stable expression here OK";
                  if AllowDollar then
                     null;
                  else
                     -- Identifier starting with a $, which is illegal in
                     -- this case.  Issue a lexical error and then treat
                     -- as an identifier starting at the next character.
                     ErrorHandler.LexError ("Illegal token",
                                            "Token ignored",
                                            LexTokenManager.LexValue'
                                              (LexTokenManager.TokenPosition'
                                                 (StartLine, StartPosn),
                                               LexTokenManager.NullString));
                     StartPosn := StartPosn + 1;
                  end if;
                  --# end accept;
               end if;

               LexTokenManager.InsertLexString
                  (LineManager.CurrLine.Conts, StartPosn, EndPosn, LexStr);
               if (LineManager.CurrLine.Context = InAnnotation) then
                  LineManager.CurrLine.AnnoContext := MidAnnotation;
               end if;

            elsif NextToken = SPSymbols.RWand then
               CheckFollowingToken (SPSymbols.RWthen, SPSymbols.RWandthen);

            elsif NextToken = SPSymbols.RWor then
               CheckFollowingToken (SPSymbols.RWelse, SPSymbols.RWorelse);

            elsif NextToken = SPSymbols.RWnot then
               CheckFollowingToken (SPSymbols.RWin, SPSymbols.RWnotin);

            elsif NextToken = SPSymbols.RWfor then
               CheckFollowingToken (SPSymbols.RWall, SPSymbols.RWforall);

               if NextToken = SPSymbols.RWfor then
                  CheckFollowingToken (SPSymbols.RWsome, SPSymbols.RWforsome);
               end if;
            else
               LexStr := LexTokenManager.NullString;
            end if;

         when SPSymbols.string_literal  =>
            InsertStringLiteral (LineManager.CurrLine.Conts,
                                 StartPosn,
                                 EndPosn,
                                 LexStr);

         when SPSymbols.character_literal |
            SPSymbols.integer_number | SPSymbols.real_number |
            SPSymbols.based_integer =>

            LexTokenManager.InsertCaseSensitiveLexString
               (LineManager.CurrLine.Conts,
                StartPosn,
                EndPosn,
                LexStr);

         when SPSymbols.annotation_start | SPSymbols.proof_context =>
            if LineManager.CurrLine.Context = InAnnotation then
               NextToken := SPSymbols.annotation_continuation;
            else
               LineManager.SetContext (InAnnotation);
            end if;

            if (LineManager.CurrLine.AnnoContext = EndedAnnotation) then
               LineManager.CurrLine.AnnoContext := StartAnnotation;
            end if;

            LexStr := LexTokenManager.NullString;

         when SPSymbols.annotation_end =>
            LineManager.SetContext (InAda);
            LexStr := LexTokenManager.NullString;

         when SPSymbols.hide_directive =>
            -- skip over hide
            LineManager.NextSigChar (ProgText);

            --# accept F, 10, NextToken, "Skipping so ineffective assignment";
            NextLex (NextToken);
            --# end accept;

            -- now get designator
            LineManager.NextSigChar (ProgText);
            StartLine := LineManager.CurrLine.LineNo;
            StartPosn := LineManager.CurrLine.CurrPos;
            NextLex (NextToken);
            EndPosn := LineManager.CurrLine.CurrPos - 1;
            if NextToken = SPSymbols.identifier then
               CheckReserved (StartPosn, EndPosn, NextToken);
            end if;

            --# assert True; -- for RTC generation

            if NextToken /= SPSymbols.identifier then
               -- ???? Report Error ???? --
               ErrVal := LexTokenManager.LexValue'
                  (LexTokenManager.TokenPosition'(StartLine, StartPosn),
                   LexTokenManager.NullString);
               ErrorHandler.LexError ("Hide designator missing",
                                      "Hide directive ignored",
                                      ErrVal);
               LineManager.ResetCurrPos;
               NextToken := SPSymbols.comment;
            else
               HiddenPart := True;
               HideDesignator := LexTokenLists.NullList;
               LexTokenManager.InsertLexString
                  (LineManager.CurrLine.Conts, StartPosn, EndPosn, LexStr);
               LexTokenLists.Append (HideDesignator, LexStr);

               loop
                  --# assert True; -- for RTC generation
                  LineManager.NextSigChar (ProgText);
                  LineManager.InspectChar (Ch);
                  exit when Ch /= '.';
                  LineManager.LookaheadChar (Ch);
                  LineManager.RejectLookahead;
                  exit when Ch = '.';   -- stop if ..
                  LineManager.AcceptChar; -- absorb dot
                  LineManager.NextSigChar (ProgText);
                  StartLine := LineManager.CurrLine.LineNo;
                  StartPosn := LineManager.CurrLine.CurrPos;
                  NextLex (NextToken);
                  EndPosn := LineManager.CurrLine.CurrPos - 1;
                  if NextToken = SPSymbols.identifier then
                     CheckReserved (StartPosn, EndPosn, NextToken);
                  end if;
                  if NextToken /= SPSymbols.identifier then
                     -- ???? Report Error ???? --
                     ErrVal := LexTokenManager.LexValue'
                        (LexTokenManager.TokenPosition'(StartLine,
                                                        StartPosn),
                         LexTokenManager.NullString);
                     ErrorHandler.LexError ("Hide designator incomplete",
                                            "Hide directive ignored",
                                            ErrVal);
                     LineManager.ResetCurrPos;
                     NextToken := SPSymbols.comment;
                     HiddenPart := False;
                     exit;
                  end if;
                  LexTokenManager.InsertLexString
                     (LineManager.CurrLine.Conts, StartPosn, EndPosn, LexStr);
                  LexTokenLists.Append (HideDesignator, LexStr);
               end loop;
            end if;
            LexStr := LexTokenManager.NullString;

         when SPSymbols.apostrophe =>
            LineManager.NextSigChar (ProgText);  -- Check for attribute designator.
            LineManager.InspectChar (Ch);
            if Ada.Characters.Handling.Is_Letter (Ch) then
               StartLine := LineManager.CurrLine.LineNo;
               StartPosn := LineManager.CurrLine.CurrPos;
               GetIdent (NextToken);
               EndPosn := LineManager.CurrLine.CurrPos - 1;
               LexTokenManager.InsertLexString (LineManager.CurrLine.Conts,
                                                StartPosn,
                                                EndPosn,
                                                LexStr);
               if NextToken = SPSymbols.identifier then
                  NextToken := SPSymbols.attribute_ident;
               else
                  -- ???? Report Error ???? --
                  ErrVal := LexTokenManager.LexValue'
                     (LexTokenManager.TokenPosition'(StartLine, StartPosn),
                      LexStr);
                  ErrorHandler.LexError ("Attribute identifier expected",
                                         "Attribute identifier assumed",
                                         ErrVal);
                  NextToken := SPSymbols.attribute_ident; -- Error recovery.
               end if;
            else
               LexStr := LexTokenManager.NullString;
            end if;

         when SPSymbols.illegal_id =>
            NextToken := SPSymbols.identifier;   -- Error recovery.
            LexTokenManager.InsertLexString (LineManager.CurrLine.Conts,
                                             StartPosn,
                                             EndPosn,
                                             LexStr);
            -- ???? Report Error ???? --
            ErrVal := LexTokenManager.LexValue'
               (LexTokenManager.TokenPosition'(StartLine, StartPosn),
                LexStr);
            ErrorHandler.LexError ("Illegal identifier",
                                   "Identifier assumed",
                                   ErrVal);

         when SPSymbols.illegal_number =>
            NextToken := SPSymbols.integer_number;
            LexTokenManager.InsertLexString (LineManager.CurrLine.Conts,
                                             StartPosn,
                                             EndPosn,
                                             LexStr);
            -- ???? Report Error ???? --
            ErrVal := LexTokenManager.LexValue'
               (LexTokenManager.TokenPosition'(StartLine, StartPosn),
                LexStr);
            if LineManager.CurrLine.Conts (StartPosn) = '0' then
               ErrorHandler.LexError ("Illegal number - possible mis-use of '0' instead of 'O' as first letter of identifier",
                                      "Number assumed",
                                      ErrVal);
            else
               ErrorHandler.LexError ("Illegal number",
                                      "Number assumed",
                                      ErrVal);
            end if;



         when SPSymbols.based_real =>
            LexTokenManager.InsertLexString (LineManager.CurrLine.Conts,
                                             StartPosn,
                                             EndPosn,
                                             LexStr);
            -- ???? Report Error ???? --
            ErrVal := LexTokenManager.LexValue'
               (LexTokenManager.TokenPosition'(StartLine, StartPosn),
                LexStr);
            ErrorHandler.LexError ("Based real numbers are not allowed",
                                   "Number assumed",
                                   ErrVal);

         when SPSymbols.unterminated_string =>
            NextToken := SPSymbols.string_literal;  -- Error recovery.
            LexTokenManager.InsertLexString (LineManager.CurrLine.Conts,
                                             StartPosn,
                                             EndPosn,
                                             LexStr);
            -- ???? Report Error ???? --
            ErrVal := LexTokenManager.LexValue'
               (LexTokenManager.TokenPosition'(StartLine, StartPosn),
                LexStr);
            ErrorHandler.LexError ("Unterminated string",
                                   "String assumed",
                                   ErrVal);

         when SPSymbols.illegal_comment =>
            LexStr := LexTokenManager.NullString;
            NextToken := SPSymbols.comment;  -- Error recovery.

            -- ???? Report Error ???? --
            ErrVal := LexTokenManager.LexValue'
               (LexTokenManager.TokenPosition'(StartLine, StartPosn),
                LexTokenManager.NullString);
            ErrorHandler.LexError ("Illegal character in comment",
                                   "Illegal character ignored",
                                   ErrVal);

         when SPSymbols.illegal_token =>
            LexStr := LexTokenManager.NullString;
            -- ???? Report Error ???? --
            ErrVal := LexTokenManager.LexValue'
               (LexTokenManager.TokenPosition'(StartLine, StartPosn),
                LexTokenManager.NullString);
            ErrorHandler.LexError ("Illegal token",
                                   "Token ignored",
                                   ErrVal);

         when SPSymbols.LONGLINE =>
            LexStr := LexTokenManager.NullString;
            -- ???? Give Error Message ???? --
            ErrVal := LexTokenManager.LexValue'
               (LexTokenManager.TokenPosition'(StartLine, EndPosn),
                LexTokenManager.NullString);
            ErrorHandler.LexError ("Line too long",
                                   "Ignoring excess characters",
                                   ErrVal);

         when SPSymbols.semicolon =>
            LexStr := LexTokenManager.NullString;
            if (LineManager.CurrLine.Context = InAnnotation) then
               LineManager.CurrLine.AnnoContext := EndedAnnotation;
            end if;

         when others =>
            LexStr := LexTokenManager.NullString;
      end case;

      --# assert True; -- for RTC generation

      exit when HiddenPart or
              (NextToken /= SPSymbols.comment and
               NextToken /= SPSymbols.annotation_continuation and
               NextToken /= SPSymbols.LONGLINE and
               NextToken /= SPSymbols.illegal_token);
   end loop;

   --# assert True; -- for RTC generation

   if HiddenPart then
      EndHideFound := False;

      loop
         --# assert True; -- for RTC generation
         LineManager.NextSigChar (ProgText);
         StartLine := LineManager.CurrLine.LineNo;
         StartPosn := LineManager.CurrLine.CurrPos;
         NextLex (NextToken);
         EndPosn := LineManager.CurrLine.CurrPos - 1;

         if NextToken = SPSymbols.SPEND then
            EndHideFound := True;
         elsif NextToken = SPSymbols.identifier then
            CheckReserved (StartPosn, EndPosn, NextToken);
         end if;
         if NextToken = SPSymbols.RWend then
            LineManager.NextSigChar (ProgText);
            StartLine := LineManager.CurrLine.LineNo;
            StartPosn := LineManager.CurrLine.CurrPos;
            NextLex (NextToken);
            EndPosn := LineManager.CurrLine.CurrPos - 1;

            if NextToken = SPSymbols.identifier then
               EndDesignator := LexTokenLists.NullList;
               LexTokenManager.InsertLexString
                  (LineManager.CurrLine.Conts, StartPosn, EndPosn, LexStr);
               LexTokenLists.Append (EndDesignator, LexStr);

               -- Process remainder of dotted name (if any)
               loop
                  --# assert True; -- for RTC generation
                  LineManager.NextSigChar (ProgText);
                  LineManager.InspectChar (Ch);
                  exit when Ch /= '.';
                  LineManager.LookaheadChar (Ch);
                  LineManager.RejectLookahead;
                  exit when Ch = '.';   -- stop if '..'
                  LineManager.AcceptChar; -- absorb dot
                  LineManager.NextSigChar (ProgText);
                  StartPosn := LineManager.CurrLine.CurrPos;
                  NextLex (NextToken); -- get expected identifier
                  EndPosn := LineManager.CurrLine.CurrPos - 1;
                  exit when NextToken /= SPSymbols.identifier;
                  LexTokenManager.InsertLexString
                     (LineManager.CurrLine.Conts, StartPosn, EndPosn, LexStr);
                  LexTokenLists.Append (EndDesignator, LexStr);
               end loop;

               if NextToken = SPSymbols.identifier then
                  EndHideFound := LexTokenLists.EqUnit (HideDesignator,
                                                        EndDesignator);
               end if;
            end if;
         end if;

         --# assert True; -- for RTC generation

         if EndHideFound then
            HiddenPart := False;
            NextToken := SPSymbols.hide_directive;
            LexStr := LexTokenManager.NullString;
         end if;
         exit when not HiddenPart;
      end loop;
   end if;

   --# assert True; -- for RTC generation

   Token := NextToken;
   LexVal := LexTokenManager.LexValue'
      (LexTokenManager.TokenPosition'(StartLine, StartPosn),
       LexStr);
   PunctToken := IsPunctToken (NextToken);

end Lex;
