-- $Id: sparklex-lex-getnumber.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (SparkLex.Lex)
procedure GetNumber (Token : out SPSymbols.SPTerminal)
   --
   -- Given that the character at the current position in the line buffer is a
   -- digit GetNum recognises the maximal length subsequence from the line buffer
   -- defined by one of the following regular expressions :-
   --    integer_number ::= integer [exponent]
   --    real_number    ::= integer.integer [exponent]
   --    based_integer  ::= integer#based_integer# [exponent]
   --    based_real     ::= integer#based_integer.based_integer# [exponent]
   --    illegal_number ::= numeric_literal non_number_separator {non_number_separator}
   -- where
   --    non_number_separator is any chahracter which cannot legally separate
   --    a numeric_literal from the next token in the Ada lexis.
   -- On exit Token is set according to which of  the expressions is recognised.
   -- The current position of the line buffer is the character immediately
   -- following the recognised subsequence.
is
   Ch    : Character;
   Legal : Boolean;
   TokenSoFar : SPSymbols.SPTerminal;

   function NumSep (Ch : Character) return Boolean is
      Result : Boolean;
   begin
      Result := Ch = ' ' or else Ch = EndOfLine or else
         Ch = EndOfText or else
         Ch = LineTooLong or else
         SimpleDelimiter (Ch) or else FormatEffector (Ch);
      return Result;
   end NumSep;

   procedure GetInteger (Legal : out Boolean)
   --# global in out LineManager.CurrLine;
   --# derives Legal,
   --#         LineManager.CurrLine from LineManager.CurrLine;
   --
   -- This procedure recognises the maximal length sequence from the
   -- line buffer satisfying one of the two following regular expressions
   --    integer     ::= digit {[underline] digit}
   --    illegal_int ::= integer underline
   --
   -- If an integer is recognised, then Legal is set to TRUE otherwise False.
   --
   is
      type NumState is (UnderlineOrDigit, DigitOnly, IllegalNumState);
      State : NumState;
      Ch    : Character;
   begin
      LineManager.AcceptChar;  -- First digit already recognised.
      State := UnderlineOrDigit;
      loop
         LineManager.InspectChar (Ch);
         case State is
            when UnderlineOrDigit =>
               if Ch = '_' then
                  State := DigitOnly;
               end if;
            when DigitOnly =>
               if Ada.Characters.Handling.Is_Digit (Ch) then
                  State := UnderlineOrDigit;
               else
                  State := IllegalNumState;
               end if;
            when IllegalNumState =>
               null;
         end case;
         exit when (not Ada.Characters.Handling.Is_Digit (Ch) and Ch /= '_') or
            State = IllegalNumState;
         LineManager.AcceptChar;
      end loop;

      Legal := (State = UnderlineOrDigit);
   end GetInteger;

   procedure GetBasedInteger (Legal : out Boolean)
   --# global in out LineManager.CurrLine;
   --# derives Legal,
   --#         LineManager.CurrLine from LineManager.CurrLine;
   --
   -- This procedure recognises the maximal length sequence from the
   -- line buffer satisfying on of the the two following regular expression
   --    based_integer ::= extended_digit {[underline] extended_digit}
   --    illegal_based ::= based_integer underline
   --
   -- If a based_integer is recognised, then Legal is set to TRUE otherwise False.
   --
   is
      type NumState is (UnderlineOrExtDigit, ExtDigitOnly, IllegalNumState);
      State : NumState;
      Ch    : Character;
   begin
      LineManager.AcceptChar;  -- First extended digit already recognised.
      State := UnderlineOrExtDigit;
      loop
         LineManager.InspectChar (Ch);
         case State is
            when UnderlineOrExtDigit =>
               if Ch = '_' then
                  State := ExtDigitOnly;
               end if;
            when ExtDigitOnly =>
               if ExtendedDigit (Ch) then
                  State := UnderlineOrExtDigit;
               else
                  State := IllegalNumState;
               end if;
            when IllegalNumState =>
               null;
         end case;
         exit when (not ExtendedDigit (Ch) and Ch /= '_') or State = IllegalNumState;
         LineManager.AcceptChar;
      end loop;

      Legal := (State = UnderlineOrExtDigit);
   end GetBasedInteger;

begin
   GetInteger (Legal);

   if Legal then
      TokenSoFar := SPSymbols.integer_number;
   else
      TokenSoFar := SPSymbols.illegal_number;
   end if;
   --# assert TokenSoFar in SPSymbols.SPTerminal;

   LineManager.InspectChar (Ch);
   if Ch = '#' then
      LineManager.AcceptChar;
      LineManager.InspectChar (Ch);
      if ExtendedDigit (Ch) then
         GetBasedInteger (Legal);
         if Legal and TokenSoFar /= SPSymbols.illegal_number then
            TokenSoFar := SPSymbols.based_integer;
         else
            TokenSoFar := SPSymbols.illegal_number;
         end if;
      end if;
   end if;
   --# assert Ch in character and TokenSoFar in SPSymbols.SPTerminal;


   LineManager.InspectChar (Ch);
   if Ch = '.' then
      LineManager.LookaheadChar (Ch); -- Check for '..' symbol.
      LineManager.RejectLookahead;   -- Current and lookahead position at first '.' char.
      if TokenSoFar = SPSymbols.integer_number then
         if Ada.Characters.Handling.Is_Digit (Ch) then
            LineManager.AcceptChar;      -- Accept decimal point, '.', char.
            GetInteger (Legal);
            if Legal then
               TokenSoFar := SPSymbols.real_number;
            else
               TokenSoFar := SPSymbols.illegal_number;
            end if;
         end if;
      elsif TokenSoFar = SPSymbols.based_integer then
         if ExtendedDigit (Ch) then
            LineManager.AcceptChar;      -- Accept decimal point, '.', char.
            GetBasedInteger (Legal);
            if Legal then
               TokenSoFar := SPSymbols.based_real;
            else
               TokenSoFar := SPSymbols.illegal_number;
            end if;
         end if;
      end if;
   end if;
   --# assert Ch in character and TokenSoFar in SPSymbols.SPTerminal;

   if TokenSoFar = SPSymbols.based_integer or TokenSoFar = SPSymbols.based_real then
      LineManager.InspectChar (Ch);
      if Ch = '#' then
         LineManager.AcceptChar;
      else
         TokenSoFar := SPSymbols.illegal_number;
      end if;
   end if;
   --# assert Ch in character and TokenSoFar in SPSymbols.SPTerminal;

   LineManager.InspectChar (Ch);
   if Ch = 'E' or Ch = 'e' then
      LineManager.AcceptChar;
      LineManager.InspectChar (Ch);
      if Ch = '+' then
         LineManager.AcceptChar;
         LineManager.InspectChar (Ch);
      elsif Ch = '-' then
         LineManager.AcceptChar;
         LineManager.InspectChar (Ch);
         if TokenSoFar /= SPSymbols.real_number and
            TokenSoFar /= SPSymbols.based_real then
            TokenSoFar := SPSymbols.illegal_number;
         end if;
      end if;
      if Ada.Characters.Handling.Is_Digit (Ch) then
         GetInteger (Legal);
         if not Legal then
            TokenSoFar := SPSymbols.illegal_number;
         end if;
      else
         TokenSoFar := SPSymbols.illegal_number;
      end if;
   end if;
   --# assert Ch in character and TokenSoFar in SPSymbols.SPTerminal;

   LineManager.InspectChar (Ch);
   if NumSep (Ch) then
      Token := TokenSoFar;
   else
      while not NumSep (Ch) loop
         LineManager.AcceptChar;
         LineManager.InspectChar (Ch);
      end loop;
      Token := SPSymbols.illegal_number;
   end if;

end GetNumber;
