-- $Id: sparkhtml.adb 13063 2009-04-21 12:05:08Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with SPARK_IO,
     Ada.Characters.Handling,
     FileSystem,
     ScreenEcho,
     EStrings,
     ELStrings,
     CommandLineData,
     LongStringUtilities;

use type SPARK_IO.File_Status;
use type FileSystem.TypFileSpecStatus;
use type ELStrings.T;

package body SparkHTML is

   ErrorReferenceFileName : constant ELStrings.T :=
      ELStrings.T'(ELStrings.Lengths (10),
                                              ELStrings.Contents'('e', 'r', 'r', 'o', 'r', 's', '.', 'h',
                                              't', 'm', others => ' '));

   -- These states are used in the parsing of the report file.
   type ReportFileStates is (ReportJustStarted,
                             ReportBannerStarted,
                             ReportBannerFinished,
                             ReportDateFound,
                             ReportOptionsFound,
                             ReportIndexFileFound,
                             ReportWarningFileFound,
                             ReportTargetCompilerDataFound,
                             ReportTargetConfigFileFound,
                             ReportSourceExtensionFound,
                             ReportListingExtensionFound,
                             ReportDictionaryFound,
                             ReportReportFileFound,
                             ReportHTMLFlagFound,
                             ReportStatisticsFlagFound,
                             ReportFDLIdentifiersFound,
                             ReportFlowAnalysisFound,
                             ReportLanguageOptionFound,
                             ReportAnnotationCharacterFound,
                             ReportSelectedFilesStarted,
                             ReportIndexFilesStarted,
                             ReportMetaFilesStarted,
                             ReportWarningListStarted,
                             ReportSourceListStarted,
                             ReportMissingFilesStarted,
                             ReportSourceFileStarted,
                             ReportListingFileNameFound,
                             ReportUnitNameFound,
                             ReportNoUnitsFound,
                             ReportUnitTypeFound,
                             ReportAnalysedMessageFound,
                             ReportStartOfErrors,
                             ReportEndOfErrors,
                             ReportJustificationsSummaryFound,
                             ReportSummarizedWarningsFound,
                             ReportLineHeaderFound,
                             ReportErrorSourceLineFound,
                             ReportErrorSourcePointerFound,
                             ReportErrorMessageFound,
                             ReportBlankAfterErrorFound,
                             ReportTargetConfigListStarted,
                             ReportTargetErrorLine,
                             ReportTargetErrorNextLine,
                             ReportEndOfReportFile);

   type ListingFileStates is (ListingJustStarted,
                              ListingBannerStarted,
                              ListingBannerFinished,
                              ListingDateFound,
                              ListingLineHeadingFound,
                              ListingSourceLineFound,
                              ListingErrorSourcePointerFound,
                              ListingErrorMessageFound,
                              ListingEndOfListingFile);

   GenerateHTML : Boolean := True;
   -- Set to false if fatal HTML error occurs to prevent further HTML generation.

   HTMLWorkDir  : ELStrings.T;
   SparkWorkDir : ELStrings.T;

   ---------------------------------------------------------------------------

   -- This function returns true if and only if C is a character representing
   -- a digit.

   function Digit (C : Character) return Boolean
   is

   begin
      return C in '0' .. '9';
   end Digit;

   ---------------------------------------------------------------------------

   -- This function prepends the given string with the name of the selected
   -- direcotry into which HTML is being generated.  The appropriate
   -- target-dependant directory separator is also added.
   --
   -- No error checking is performed.  If the string overflows it is
   -- truncated by the AppendString routines.

   function HTMLFileName (FileName : ELStrings.T)
                         return ELStrings.T
   --# global in CommandLineData.Content;
   is

      ReturnFileName : ELStrings.T;

   begin

      ReturnFileName := ELStrings.EmptyString;
      ELStrings.AppendExaminerString (ReturnFileName, FileSystem.StartOfDirectory);
      ELStrings.AppendExaminerString (ReturnFileName, CommandLineData.Content.HTMLDirectory);
      ELStrings.AppendExaminerString (ReturnFileName, FileSystem.EndOfPath);
      ELStrings.AppendExaminerLongString (ReturnFileName, FileName);

      return ReturnFileName;

   end HTMLFileName;


   ---------------------------------------------------------------------------


   -- This function returns InString with all occurences of FromChar changed
   -- to ToChar.
   --
   -- No error checking is necessary.  The for loop ensures that bounds aren't
   -- exceeded.

   function Translate (InString : ELStrings.T;
                       FromChar : Character;
                       ToChar   : Character) return ELStrings.T

   is

      OutContent : ELStrings.Contents;

   begin

      OutContent := ELStrings.EmptyString.Content;

      for i in ELStrings.Positions range ELStrings.Positions'First .. InString.Length loop

         if InString.Content (i) = FromChar then
            OutContent (i) := ToChar;
         else
            OutContent (i) := InString.Content (i);
         end if;
      end loop;

      return ELStrings.T'(Content => OutContent,
                                                     Length  => InString.Length);
   end Translate;


   ---------------------------------------------------------------------------

   -- This function returns the index of the first character on a line that isn't
   -- a space.
   --
   -- No error checking is necessary.  The for loop ensures that bounds aren't
   -- exceeded.

   function FirstChar (TheString : ELStrings.T) return ELStrings.Positions
   is

      Pos : ELStrings.Positions;

   begin

      Pos := ELStrings.Positions'First;

      for I in ELStrings.Positions range ELStrings.Positions'First .. TheString.Length loop
         if TheString.Content (I) /= ' ' then
            Pos := I;
            exit;
         end if;
      end loop;

      return Pos;

   end FirstChar;


   ---------------------------------------------------------------------------

   -- This procedure appends the given character to the given ExaminerLongString.
   -- A similar procedure exists in the ExaminerLongStrings package but it has an
   -- overflow parameter.  This is often not necessary for my uses and so this
   -- simpler version avoids the need for local variables and ineffective
   -- assignments.
   --
   -- If the maximum ExaminerLongString length is exceeded then nothing happens.

   procedure AppendCharWithoutCheck (EStr : in out ELStrings.T;
                                     Char : in     Character)
      --# derives EStr from *,
      --#                   Char;

   is

   begin

      if EStr.Length < ELStrings.MaxStringLength then
         EStr.Length := EStr.Length + 1;
         EStr.Content (EStr.Length) := Char;
      end if;

   end AppendCharWithoutCheck;

   ---------------------------------------------------------------------------

   -- This function returns the given character if the character can be used in
   -- an HTML name and '_' otherwise.
   -- Characters the are allowed in HTML names are letters ([A-Za-z]), digits ([0-9]
   -- hyphens ("-"), underscores ("_"), colons (":") and periods (".").
   --
   -- No error checking necessary.

   function HTMLNameChar (C : Character) return Character is

      OutChar : Character;

   begin

      if C in 'A' .. 'Z' or
         C in 'a' .. 'z' or
         C in '0' .. '9' or
         C = '-' or
         C = '_' or
         C = ':' or
         C = '.'
      then
         OutChar := C;
      else
         OutChar := '_';
      end if;

      return OutChar;

   end HTMLNameChar;


   ---------------------------------------------------------------------------

   -- Applies HTMLNameChar to each character in the string.
   --
   -- No error checking is necessary.  The for loop ensures that bounds aren't
   -- exceeded.

   function HTMLName (EStr : ELStrings.T) return ELStrings.T
   is

      OutString : ELStrings.T;

   begin

      OutString := ELStrings.EmptyString;

      for I in ELStrings.Positions range ELStrings.Positions'First .. EStr.Length loop

         AppendCharWithoutCheck (OutString, HTMLNameChar (EStr.Content (I)));

      end loop;

      return OutString;

   end HTMLName;


   ---------------------------------------------------------------------------

   -- This function removes all occurrences of special HTML characters and replaces
   -- them with the HTML codes required to display them.
   -- The conversions done are:
   --
   --    >   becomes   &gt;
   --    <   becomes   &lt;
   --    &   becomes   &amp;
   --    "   becomes   &quot;
   --
   -- The for loop ensures that bounds of the input string aren't exceeded.  If
   -- string overflow occurs then the output string is truncated by the Append
   -- routines.

   function ConvertSpecialHTMLChars (Line : ELStrings.T)
                                    return ELStrings.T

   is

      OutString : ELStrings.T;

   begin

      OutString := ELStrings.EmptyString;

      for I in ELStrings.Positions range ELStrings.Positions'First .. Line.Length loop

         case Line.Content (I) is

            when '<' =>
               ELStrings.AppendString (OutString, "&lt;");
            when '>' =>
               ELStrings.AppendString (OutString, "&gt;");
            when '&' =>
               ELStrings.AppendString (OutString, "&amp;");
            when '"' =>
               ELStrings.AppendString (OutString, "&quot;");
            when others =>
               AppendCharWithoutCheck (OutString, Line.Content (I));
         end case;

      end loop;

      return OutString;

   end ConvertSpecialHTMLChars;

   ---------------------------------------------------------------------------

   -- This function encloses the line given in HTML tags to make it bold type.
   --
   -- If the line given is too long for the tags to be added then it is
   -- returned unchanged.

   function HTMLEmbolden (Line : ELStrings.T)
                         return ELStrings.T
   is
      OutString : ELStrings.T;
   begin
      if Line.Length < ELStrings.MaxStringLength - 7 then
         ELStrings.CopyString (OutString, "<B>");
         ELStrings.AppendExaminerLongString (OutString, Line);
         ELStrings.AppendString (OutString, "</B>");
      else
         OutString := Line;
      end if;

      return OutString;
   end HTMLEmbolden;

   ---------------------------------------------------------------------------

   -- This function does a target-specific append of a subdirectory name.
   -- The function works with or without trailing separators.
   --
   -- The path returned will always have a trailing separator.

   function AppendDirectoryString (Path : ELStrings.T;
                                   Dir  : EStrings.T)
                                  return ELStrings.T
   is

      PathOut : ELStrings.T;
      DirIn   : ELStrings.T;

   begin

      DirIn := ELStrings.ToExaminerLongString (Dir);

      PathOut := Path;

      -- Add trailing separator if necessary.
      if PathOut.Content (PathOut.Length) /= FileSystem.BetweenDirectories.Content (FileSystem.BetweenDirectories.Length) then
         ELStrings.AppendExaminerString (PathOut, FileSystem.BetweenDirectories);
      end if;

      ELStrings.AppendExaminerLongString (PathOut, DirIn);
      ELStrings.AppendExaminerString (PathOut, FileSystem.EndOfPath);

      return PathOut;

   end AppendDirectoryString;

   ---------------------------------------------------------------------------
   -- This procedure takes two paths and returns the URI of the first path
   -- relative to the second path using the global value of the SparkWorkDir as
   -- the current directory.
   --
   -- The directories passed must have trailing separators.  The directory returned has
   -- a trailing separator.  A separator is defined as being the first character
   -- of the ExaminerLongString returned by FileSystem.BetweenDirectories.
   --
   -- For use on Windows, it is not possible to give a URI in all instances if
   -- the directories exist on two different drives.  For example, if the
   -- paths given are "C:\foo\" and "D:\bar\".
   --
   -- We could simply return the absolute path but if we then want to view
   -- the URI on a Unix system we will find that the path does not exist.
   --
   -- I therefore have included a "Success" parameter which is set to false if
   -- the paths are on 2 different drives.
   --
   procedure HTMLRelativeDir (DirIn, RelativeToIn : in     ELStrings.T;
                              DirRelative         :    out ELStrings.T;
                              SuccessOut          :    out Boolean)
      --# global in SparkWorkDir;
      --# derives DirRelative,
      --#         SuccessOut  from DirIn,
      --#                          RelativeToIn,
      --#                          SparkWorkDir;
   is

      Dir, RelativeTo       : ELStrings.T;
      OutString             : ELStrings.T;
      RemainingDir          : ELStrings.T;
      RemainingRelativeTo   : ELStrings.T;
      PoppedDirectory       : ELStrings.T;

      WorkingDevice         : ELStrings.T;
      DirDevice             : ELStrings.T;
      RelativeToDevice      : ELStrings.T;

      I                     : ELStrings.Positions;

      DirectoryFound        : Boolean;
      WorkingDeviceFound    : Boolean;
      DirDeviceFound        : Boolean;
      RelativeToDeviceFound : Boolean;

      Success               : Boolean;

      -- This function returns the first character of the ExaminerLongString.
      -- The result is undefined if the EmptyString is given.

      function GetFirst (S : String) return Character
      is
      begin
         return S (S'First);
      end GetFirst;

      -- This function chops the first character from the ExaminerLongString.
      -- If the EmptyString is given then the EmptyString is returned.

      function GetRest (S : ELStrings.T) return ELStrings.T
      is
         Result : ELStrings.T;
      begin
         if S.Length <= 1 then
            Result := ELStrings.EmptyString;
         else
            Result := LongStringUtilities.Section (S, ELStrings.Positions'First + 1, S.Length - 1);
         end if;

         return Result;
      end GetRest;

      -- Does a case insensitive comparison of the first characters of
      -- the given strings.  Returns true if the first character of each string
      -- is the same.  Returns false otherwise.  Additionally, the result is
      -- undefined if one of the strings is the empty string.

      function SameFirst (Path1, Path2 : String) return Boolean
      is
      begin
         return Ada.Characters.Handling.To_Lower (GetFirst (Path1)) =
            Ada.Characters.Handling.To_Lower (GetFirst (Path2));
      end SameFirst;

      -- This function looks for a device name and returns it.
      --
      -- If no device name is present then the EmptyString is returned.
      --
      -- A device name is ususally the name of a Windows drive (of the form "X:")
      function GetDevicePrefix (Path : in ELStrings.T)
                               return ELStrings.T
      is
         ColonFound        : Boolean;
         Pos               : ELStrings.Positions;
         Device            : ELStrings.T;
      begin

         if Path.Length = 0 then

            Device := ELStrings.EmptyString;

         else

            ColonFound  := False;
            Pos         := Path.Length;

            for I in ELStrings.Positions range ELStrings.Positions'First .. Path.Length loop

               -- Have we found a colon?
               if Path.Content (I) = ':' then
                  ColonFound := True;
                  Pos        := I;
                  exit;
               end if;

               -- If we find a directory separator or StartOfDirectory then
               -- exit the loop with ColonFound = False.
               if Path.Content (I) = GetFirst (FileSystem.StartOfDirectory.Content) or
                  Path.Content (I) = GetFirst (FileSystem.BetweenDirectories.Content) then
                  exit;  -- ColonFound is already False;
               end if;

            end loop;

            if ColonFound then
               Device := LongStringUtilities.Section (Path, ELStrings.Positions'First, Pos);
            else
               -- Leave Path as it is.  Set Device to empty string.
               Device := ELStrings.EmptyString;
            end if;

         end if;

         return Device;

      end GetDevicePrefix;

      -- This procedure does a GetDevicePrefix and removes the device name from
      -- the path given.
      procedure ChopDevicePrefix (Path   : in out ELStrings.T;
                                  Device :    out ELStrings.T;
                                  Found  :    out Boolean)
         --# derives Device,
         --#         Found,
         --#         Path   from Path;
      is

         DeviceName  : ELStrings.T;

      begin

         DeviceName := GetDevicePrefix (Path);

         -- GetDevicePrefix returns EmptyString if no device name was found.
         if DeviceName.Length > 0 then

            Path   := LongStringUtilities.Section (Path, DeviceName.Length + 1, Path.Length - DeviceName.Length);
            Device := DeviceName;
            Found  := True;

         else

            Device := ELStrings.EmptyString;
            Found  := False;

         end if;

      end ChopDevicePrefix;


      -- This function takes a pathname (must NOT be preceded by an device name) parameter.
      -- If the pathname is absolute (begins with a BetweenDirectories character) then it is
      -- returned unchanged.  Otherwise it is returned with SparkWorkDir prepended to it.
      -- The returned path will not have a device name.
      function MakeAbsolute (Path : ELStrings.T) return ELStrings.T
         --# global in SparkWorkDir;
      is

         OutString   : ELStrings.T;
         Device      : ELStrings.T;

      begin

         if SameFirst (Path.Content, FileSystem.BetweenDirectories.Content) then
            OutString := Path;
         else
            -- Directory is relative to current - append CurrentDir
            OutString := SparkWorkDir;
            ELStrings.AppendExaminerLongString (OutString, Path);
         end if;

         -- Remove device name if it exists.
         Device := GetDevicePrefix (OutString);

         -- Length of device name is 0 if no device was found.
         if Device.Length > 0 then
            OutString := LongStringUtilities.Section (OutString,
                                                      Device.Length + 1,
                                                      OutString.Length - Device.Length);
         end if;

         return OutString;

      end MakeAbsolute;

      -- This procedure returns and removes the first subdirectory name from
      -- the given path.  The given path must not have an device name or a
      -- leading separator.  The popped subdirectory name will also not have a
      -- leading separator. The remaining directory name has no leading
      -- separator.
      --
      -- e.g. if the input path is "foo/bar/baz/" then the path returned
      -- will be "bar/baz/" and the subdirectory returned will be "foo".
      --
      -- If the URL flag is set then the separator '/' is used rather than
      -- a target specific separator.
      --
      -- DirFound is true if and only if a directory separator was found in the
      -- path.
      --
      -- If the path has length 0 or a directory separator is not found then
      --       - Path is returned unchanged
      --       - Dir is ELStrings.EmptyString
      --       - Dirfound is false
      --
      procedure PopSubDir (Path     : in out ELStrings.T;
                           URL      : in     Boolean;
                           Dir      :    out ELStrings.T;
                           DirFound :    out Boolean)
         --# derives Dir,
         --#         Dirfound,
         --#         Path     from Path,
         --#                       URL;
      is

         BetweenSeparator : ELStrings.T;
         EndSeparator     : ELStrings.T;
         SeparatorPos     : ELStrings.Positions;
         SeparatorFound   : Boolean;

      begin

         if Path.Length = 0 then

            Dir := ELStrings.EmptyString;
            DirFound := False;

         else

            if URL then

               ELStrings.CopyString (BetweenSeparator, "/");
               EndSeparator := BetweenSeparator;

            else

               BetweenSeparator := ELStrings.ToExaminerLongString (FileSystem.BetweenDirectories);
               EndSeparator     := ELStrings.ToExaminerLongString (FileSystem.EndOfPath);

            end if;

            LongStringUtilities.FindExaminerSubString (Path, BetweenSeparator,
                                                       SeparatorFound, SeparatorPos);

            if not SeparatorFound then
               -- Maybe last directory?
               LongStringUtilities.FindExaminerSubString (Path, EndSeparator,
                                                          SeparatorFound, SeparatorPos);
            end if;

            if SeparatorFound then

               Dir := LongStringUtilities.Section (Path,
                                                   ELStrings.Positions'First,
                                                   SeparatorPos - 1);
               Path := LongStringUtilities.Section (Path, SeparatorPos + 1,
                                                    Path.Length - SeparatorPos);
               DirFound := True;

            else

               Dir := ELStrings.EmptyString;
               DirFound := False;

            end if;

         end if;

      end PopSubDir;

      -- This function removes the last directory in the URL given.
      -- The input path should have a trailing separator and the output
      -- path will also have a trailing separator.
      --
      -- If no separator is found then the string is returned unchanged.
      --
      -- e.g. RemoveLastDirectory ("foo/bar/baz/") = "foo/bar/"
      --
      -- This routine is used when removing "../" from pathnames.
      --
      -- NOTE: This only works with URLs - the directory separator must be '/'
      --

      function RemoveLastDirectory (Path : ELStrings.T)
                                   return ELStrings.T
      is

         OutString : ELStrings.T;
         Pos       : ELStrings.Lengths;

      begin

         OutString := Path;
         Pos       := Path.Length;

         -- Remember not to include the trailing BetweenDirectories character in the search.
         for I in reverse ELStrings.Positions range ELStrings.Positions'First .. Path.Length - 1 loop
            if Path.Content (I) = '/' then
               Pos := I;
               exit;
            end if;
         end loop;

         -- Pos will still equal Path.Length if separator was not found.
         if Pos /= Path.Length then  -- separator was found
            OutString := LongStringUtilities.Section (Path, ELStrings.Positions'First, Pos);
         end if;

         return OutString;

      end RemoveLastDirectory;

      -- This function resolves a URL containing references to the previous
      -- directory "../" and the current directory "./".  The path given must have
      -- a trailing '/' character.
      -- The function works with or without a leading directory separator (this is
      -- copied if it exists.
      --
      -- NOTE: This function only works with URL's.  The directory separator
      --       used is always '/' and current and parent directories are
      --       "." and ".." respectively.

      function RemoveDots (InPath : ELStrings.T)
                          return ELStrings.T
      is

         SubDirFound : Boolean;
         Path        : ELStrings.T;
         OutPath     : ELStrings.T;
         NextDir     : ELStrings.T;

      begin

         Path    := InPath;
         OutPath := ELStrings.EmptyString;

         if Path.Content (ELStrings.Positions'First) = '/' then
            -- Copy the leading separator.
            Path := GetRest (Path);
            AppendCharWithoutCheck (OutPath, '/');
         end if;

         loop

            PopSubDir (Path, True, NextDir, SubDirFound);

            exit when not SubDirFound;

            if ELStrings.Eq1String (NextDir, ".") then
               null; -- do nothing
            elsif ELStrings.Eq1String (NextDir, "..") then
               OutPath := RemoveLastDirectory (OutPath);
            else
               ELStrings.AppendExaminerLongString (OutPath, NextDir);
               AppendCharWithoutCheck (OutPath, '/');
            end if;

         end loop;

         return OutPath;

      end RemoveDots;

      -- This function converts a directory name into a URL.
      function ConvertToURL (Path : ELStrings.T) return ELStrings.T
      is
      begin
         -- On Windows, pathnames might contain '\' which need
         -- to be transformed unto '/' to be a URL.  On other
         -- platforms, this is a no-op.
         return Translate (Path, '\', '/');
      end ConvertToURL;

   begin  -- HTMLRelativeDir

      -- Initialise variables
      Success    := True;
      OutString  := ELStrings.EmptyString;

      -- Copy input parameters to local variables.
      Dir := DirIn;
      RelativeTo := RelativeToIn;

      -- Get device names.
      -- Device names are removed from the input directories.
      -- SparkWorkDir is not to be modified.

      WorkingDevice := GetDevicePrefix  (SparkWorkDir);
      WorkingDeviceFound := WorkingDevice.Length > 0;

      ChopDevicePrefix (Dir, DirDevice, DirDeviceFound);
      ChopDevicePrefix (RelativeTo, RelativeToDevice, RelativeToDeviceFound);

      -- We cannot create links if files are on different NT drives or different
      -- VAX devices.
      if DirDeviceFound then
         --Dir contains a device name.
         if RelativeToDeviceFound then
            -- RelativeTo also contains a device name.  Are they the same?
            -- Fail if different devices.
            if not ELStrings.EqString (DirDevice, RelativeToDevice) then
               Success := False;  -- Files are on different drives.
            end if;
         else
            -- Dir contains a device name and RelativeTo is on the current drive.
            -- Is the current device equal to the device that Dir is on?
            if (not WorkingDeviceFound) or else (not ELStrings.EqString (DirDevice, WorkingDevice)) then
               Success := False;
            end if;
         end if;
      else
         -- Dir does not contain a drive specification - does RelativeTo?
         if RelativeToDeviceFound then
            -- RelativeTo contains a device name and Dir is on the current drive.
            -- Is the current device equal to the device that RelativeTo is on?
            if (not WorkingDeviceFound) or else (not ELStrings.EqString (RelativeToDevice, WorkingDevice)) then
               Success := False;
            end if;
         end if;
      end if;

      -- Do nothing (return empty string) if directories are identical, or if
      -- the previous checks failed.
      if (not ELStrings.EqString (Dir, RelativeTo)) and Success then

         -- Make directories absolute
         Dir        := MakeAbsolute (Dir);
         RelativeTo := MakeAbsolute (RelativeTo);

         -- Convert directories to URL's.
         Dir        := ConvertToURL (Dir);
         RelativeTo := ConvertToURL (RelativeTo);

         -- Remove "./" and "../" and make case-insensitive where required.
         Dir := ELStrings.ToExaminerLongString
           (ELStrings.ToExaminerString (RemoveDots (Dir)));
         RelativeTo := ELStrings.ToExaminerLongString
           (ELStrings.ToExaminerString (RemoveDots (RelativeTo)));

         -- Initialize counter.
         I := ELStrings.Positions'First;

         -- Skip common prefix.  We want I to point to the character that begins
         -- the name of the first different subdirectory.  e.g. if we have
         -- /foo/bar/ and /foo/baz/ we want to point to the 'b'
         loop
            exit when I > Dir.Length or
               I > RelativeTo.Length or
               Dir.Content (I) /= RelativeTo.Content (I);
            I := I + 1;
         end loop;

         if I < Dir.Length and I < RelativeTo.Length then
            -- Back up to previous directory separator (in case we're comparing, e.g.
            -- \foo\bar and \foo\baz, in which case the common prefix is c:\foo\ba)
            while Dir.Content (I) /= '/'
               and I > ELStrings.Positions'First loop
               I := I - 1;
            end loop;

            -- Now we want to point to just past the separator so
            I := I + 1;
         end if;

         -- Now remove the common ancestor directories.
         if I > Dir.Length then
            RemainingDir := ELStrings.EmptyString;
         else
            RemainingDir         := LongStringUtilities.Section (Dir, I, (Dir.Length - I) + 1);
         end if;

         if I > RelativeTo.Length then
            RemainingRelativeTo := ELStrings.EmptyString;
         else
            RemainingRelativeTo  := LongStringUtilities.Section (RelativeTo, I, (RelativeTo.Length - I) + 1);
         end if;

         -- For each subdirectory remaining in RelativeTo we add a "../" to OutString
         loop
            --# accept Flow, 10, PoppedDirectory, "Expected ineffective assignment to PoppedDirectory";
            PopSubDir (RemainingRelativeTo,
                       True,
                       PoppedDirectory,
                       DirectoryFound);  -- Flow error expected - PoppedDir not used.
            --# end accept;

            exit when not DirectoryFound;

            ELStrings.AppendString (OutString, "../");

         end loop;

         -- For each subdirectory remaining in Dir we add "<subdir>/"
         loop

            PopSubDir (RemainingDir, True, PoppedDirectory, DirectoryFound);
            exit when not DirectoryFound;

            ELStrings.AppendExaminerLongString (OutString, PoppedDirectory);
            ELStrings.AppendString (OutString, "/");

         end loop;

      end if;

      DirRelative := OutString;
      SuccessOut  := Success;

   end HTMLRelativeDir;  -- Flow error expected.  SparkWorkDir is NOT updated.

   ---------------------------------------------------------------------------

   -- This procedure uses HTMLRelativeDir to give the URI of a file relative
   -- to a given directory and the global SparkWorkDir as the current directory.
   --
   -- The

   procedure HTMLRelativeFile (File, RelativeTo  : in     ELStrings.T;
                               FileRelative      :    out ELStrings.T;
                               SuccessOut        :    out Boolean)
      --# global in SparkWorkDir;
      --# derives FileRelative,
      --#         SuccessOut   from File,
      --#                           RelativeTo,
      --#                           SparkWorkDir;
   is

      DirectoryName   : ELStrings.T;
      RelativeDirName : ELStrings.T;
      FileName        : ELStrings.T;
      OutString       : ELStrings.T;
      SeparatorPos    : ELStrings.Positions;
      Success         : Boolean;

   begin

      SeparatorPos := File.Length;

      -- Get the filename and extension.  Filename might have a path
      -- or a drive name as a prefix.
      for I in reverse ELStrings.Positions range ELStrings.Positions'First .. File.Length loop
         if File.Content (I) = FileSystem.EndOfPath.Content (FileSystem.EndOfPath.Length)
            or File.Content (I) = ':' then
            SeparatorPos := I;
            exit;
         end if;
      end loop;

      -- Separate the file and directory name.
      if SeparatorPos = File.Length then -- no pathname given, just filename.
         DirectoryName := ELStrings.EmptyString;
         FileName      := File;
      else
         DirectoryName := LongStringUtilities.Section (File,
                                                       ELStrings.Positions'First,
                                                       SeparatorPos);
         FileName      := LongStringUtilities.Section (File,
                                                       SeparatorPos + 1,
                                                       File.Length - SeparatorPos);
      end if;

      -- Interpret the directory name as a URL relative to RelativeTo
      HTMLRelativeDir (DirectoryName, RelativeTo, RelativeDirName, Success);

      -- If the relative directory operation was successful then we use the
      -- new directory name.  Otherwise we simply return the filename given.
      if Success then
         OutString := RelativeDirName;
         ELStrings.AppendExaminerLongString (OutString, FileName);
      else
         OutString := File;
      end if;

      FileRelative := OutString;
      SuccessOut   := Success;

   end HTMLRelativeFile;

   ---------------------------------------------------------------------------

   -- This procedure is global as it is used by both the report and listing file
   -- translators.
   --
   -- This procedure processes an error message line.  Some errors get wrapped over
   -- more than one line and this procedure should only be used in generating
   -- HTML for the first line of any error message.
   --
   -- It assumes that the line is formatted as follows:
   --
   --   <flash> (nnnn)  <error_type> : <xxx> : <error_message>
   --
   -- The actual pattern matched is slightly different but we only expect lines
   -- of this type (or the special cases described below) and lines of this type
   -- will be parsed correctly.
   --
   -- The <error_type> and the error code <xxx> are used to generate an HTML
   -- link of the form <A HREF="error_file#error_link">, where error_file is
   -- the (relative) location of the error messages file and error_link is the
   -- name of an anchor in the file, typically made from the first word of the
   -- error type (converted to lower case) and the error number, separated by a hyphen.
   -- Note that there are some special cases where the name of the anchor is not
   -- constructed in this way.
   --
   -- The HTML link is placed before the start of the line (the closing tag is
   -- placed at the end of the line).  The link is also returned in the ErrorLink
   -- out parameter so that it can be saved for use in any continuations
   -- of this message line.
   --
   -- For example, if the Line parameter is the ExaminerLongString:
   --
   --    *** (419)   Semantic Error      : 1: Message
   --
   -- The line output will be:
   --
   -- <A HREF="error_file#semantic-1">  *** (419)   Semantic Error      : 1: Message</A>
   --
   -- and the ErrorLink parameter will be:
   --
   -- <A HREF="error_file#semantic-1">
   --
   -- If no tag is added then the ErrorLink parameter is set to EmptyString.
   --
   -- The special cases (i.e. those which don't fit the above format, mainly
   -- because they don't have error numbers) are handled by the procedure
   -- HandleSpecialCases as described below.
   procedure ProcessFirstErrorMessageLine (Line       : in out ELStrings.T;
                                           Lookahead  : in     ELStrings.T;
                                           ErrorLink  :    out ELStrings.T)
      --# global in out SPARK_IO.FILE_SYS;
      --# derives ErrorLink,
      --#         Line              from Line,
      --#                                Lookahead &
      --#         SPARK_IO.FILE_SYS from *,
      --#                                Line,
      --#                                Lookahead;
   is

      SavedLine   : ELStrings.T;
      ErrorType   : ELStrings.T;
      ErrorNo     : ELStrings.T;
      OutString   : ELStrings.T;
      Char        : Character;
      Char2       : Character;
      Char3       : Character;
      Error       : Boolean;
      SpecialCase : Boolean;

      -- The function FlashCharacter is true if and only if the input is
      -- a character used by the Examiner in a "flash" at the start of
      -- an error message.

      function FlashCharacter (Char : Character) return Boolean
      is
      begin
         return Char = '?' or else
            Char = '-' or else
            Char = '+' or else
            Char = '*' or else
            Char = '!';
      end FlashCharacter;
      pragma Inline (FlashCharacter);

      -- This procedure handles the special cases of error messages that are not
      -- of the form described in the commentary for ProcessFirstErrorMessageLine.
      -- These special cases are mainly errors that don't have error numbers
      -- associated with them or that have no reference in the error reference
      -- file.
      --
      -- The "special case" errors are as follows:
      --
      --   *** Syntax Error : ";" expected.
      --   *** Syntax Error : No APRAGMA can be start with reserved word "IS".
      --   *** Syntax Error : No complete PROCEDURE_SPECIFICATION can be followed by ANNOTATION_START here.
      --   *** Syntax Error : No complete PROCEDURE_SPECIFICATION can be followed by reserved word "IS" here.
      --   *** Syntax Error : reserved word "INHERIT" expected.
      --   Any other syntax errors (these won't have links)
      --   Warning : No semantic checks carried out, text may not be legal SPARK.
      --   Any lexical errors (these won't have links)
      --
      --   The line output is the line input but with an HTML link to the correct
      --   place in the error reference file (or no link if no such reference
      --   exists).
      --
      --   The error_link parameter is set to the HTML tag used to generate the link
      --   or, if not link is generated, the empty string.
      --
      --   The SpecialCase parameter is set to true if and only if a special case
      --   was found.  If this parameter is set then the caller should not try
      --   to process the error message line in the usual way.
      --
      --   The procedure uses a fairly naive SubString search mechanism that could
      --   potentially incorrectly match a string and flag it as a SpecialCase
      --   but this is unlikely because when this procedure is called we know that
      --   we have a line containing an error message and we know what the error
      --   messages are.
      procedure HandleSpecialCases (Line        : in out ELStrings.T;
                                    Lookahead   : in     ELStrings.T;
                                    ErrorLink   :    out ELStrings.T;
                                    SpecialCase :    out Boolean)
         --# derives ErrorLink,
         --#         Line        from Line,
         --#                          Lookahead &
         --#         SpecialCase from Line;

      is

         AddTag           : Boolean;
         SyntaxErrorFound : Boolean;
         WarningFound     : Boolean;
         LexicalFound     : Boolean;
         ErrorLinkName    : ELStrings.T;
         ErrorLinkString  : ELStrings.T;
         OutString        : ELStrings.T;

      begin

         AddTag        := False;
         SpecialCase   := False;
         ErrorLink     := ELStrings.EmptyString;
         ErrorLinkName := ELStrings.EmptyString;

         -- Check for the various error types that have special cases.

         SyntaxErrorFound := LongStringUtilities.IsSubString (EStr         => Line,
                                                              SearchString => "Syntax Error");
         WarningFound     := LongStringUtilities.IsSubString (EStr         => Line,
                                                              SearchString => "Warning");
         LexicalFound     := LongStringUtilities.IsSubString (EStr         => Line,
                                                              SearchString => "Lexical Error");

         if SyntaxErrorFound then

            -- HTML directives:
            --! <NameFormat> <"syntax-"><Name>
            --! <ErrorFormat> <"*** Syntax Error : "><Error>

            -- HTML output
            --! <Name> semicolon-expected
            --! <Error> &quot;;&quot; expected.
            --! If this is reported at the end of the input file it may well
            --!  be caused by the misspelling of an identifier in a hide directive.
            --!  The parser then skips all the following text looking for the
            --!  misspelled identifier but finds the end of file first where it
            --!  reports a syntax error.

            --! <Name> no-apragma
            --! <Error> No APRAGMA can be start with reserved word &quot;IS&quot;
            --! This can occur when a stub for an embedded subprogram is wrongly
            --! terminated by a semicolon.

            --! <Name> procedure-spec-annotation-start
            --! <Error> No complete PROCEDURE_SPECIFICATION can be followed by ANNOTATION_START here.
            --! This can occur when the reserved word </i>body<i> has been
            --! omitted from the declaration of a package body. This error
            --! will occur at the annotation placed between the
            --! specification and the reserved word </i>is<i> of the first
            --! subprogram.

            --! <Name> procedure-spec-is
            --! <Error> No complete PROCEDURE_SPECIFICATION can be followed by reserved word &quot;IS&quot; here.
            --! This can occur when the reserved word </i>body<i> has been omitted
            --! from the declaration of a package body. This error will occur at the
            --! reserved word </i>is<i> which introduces the body of the first subprogram.

            --! <Name> inherit-expected
            --! <Error> reserved word &quot;INHERIT&quot; expected.
            --! This occurs where the annotation on a subprogram body is placed after
            --! the reserved word </i>is<i> instead of before it.

            --! <Name> simple-expression-rbracket
            --! <Error> No complete SIMPLE_EXPRESSION can be followed by &quot;)&quot; here.
            --! This can occur in an aggregate expression when there is a mixure of
            --! named and positional association being used.

            --! <Name> simple-expression-comma
            --! <Error> No complete SIMPLE_EXPRESSION can be followed by &quot;,&quot; here.
            --! This can occur in an aggregate expression when there is a mixure of
            --! named and positional association being used.

            -- All syntax errors are special cases.
            SpecialCase := True;

            if LongStringUtilities.IsSubString (EStr         => Line,
                                                SearchString => ";&quot; expected") then
               ELStrings.CopyString (ErrorLinkName, "syntax-semicolon-expected");
               AddTag := True;
            end if;

            if LongStringUtilities.IsSubString (EStr         => Line,
                                                SearchString => "No APRAGMA") then
               ELStrings.CopyString (ErrorLinkName, "syntax-no-apragma");
               AddTag := True;
            end if;

            if LongStringUtilities.IsSubString
               (EStr         => Line,
                SearchString => "No complete PROCEDURE_SPECIFICATION") then

               if LongStringUtilities.IsSubString
                  (EStr         => Lookahead,
                   SearchString => "_START here.") then

                  ELStrings.CopyString (ErrorLinkName, "syntax-procedure-spec-annotation-start");
                  AddTag := True;

               elsif LongStringUtilities.IsSubString
                  (EStr         => Lookahead,
                   SearchString => "&quot;IS&quot; here.") then

                  ELStrings.CopyString (ErrorLinkName, "syntax-procedure-spec-is");
                  AddTag := True;

               end if;

            end if;

            if LongStringUtilities.IsSubString
               (EStr         => Line,
                SearchString => "No complete SIMPLE_EXPRESSION") then

               if LongStringUtilities.IsSubString
                 (EStr         => Lookahead,
                  SearchString => "&quot;)&quot; here.") then

                  ELStrings.CopyString (ErrorLinkName, "syntax-simple-expression-rbracket");
                  AddTag := True;

               elsif LongStringUtilities.IsSubString
                 (EStr         => Lookahead,
                  SearchString => "&quot;,&quot; here.") then

                  ELStrings.CopyString (ErrorLinkName, "syntax-simple-expression-comma");
                  AddTag := True;

               end if;

            end if;


            if LongStringUtilities.IsSubString (EStr         => Line,
                                                SearchString => "reserved word &quot;INHERIT&quot; expected") then
               ELStrings.CopyString (ErrorLinkName, "syntax-inherit-expected");
               AddTag := True;
            end if;

         elsif WarningFound then

            -- Not all warnings are special cases - only set SpecialCase to True if
            -- a special case is detected.

            if LongStringUtilities.IsSubString (EStr         => Line,
                                                SearchString => "No semantic checks carried out,") then
               ELStrings.CopyString (ErrorLinkName, "warning-no-semantic-checks");
               AddTag := True;
               SpecialCase := True;
            end if;

         elsif LexicalFound then

            -- All lexical errors are special cases.
            SpecialCase := True;

            -- Lexical errors are not included in the file of error explanations
            -- so no processing is done.

         end if;

         if AddTag then
            ELStrings.CopyString (ErrorLinkString, "<A TARGET=""rbottom"" HREF=""");
            ELStrings.AppendExaminerLongString (ErrorLinkString, ErrorReferenceFileName);
            ELStrings.AppendString (ErrorLinkString, "#");
            ELStrings.AppendExaminerLongString (ErrorLinkString, ErrorLinkName);
            ELStrings.AppendString (ErrorLinkString, """>");
            ErrorLink := ErrorLinkString;

            OutString := ErrorLinkString;
            ELStrings.AppendExaminerLongString (OutString, Line);
            ELStrings.AppendString (OutString, "</A>");
            Line := OutString;
         end if;

      end HandleSpecialCases;

   begin  -- ProcessFirstErrorMessageLine

      HandleSpecialCases (Line, Lookahead, ErrorLink, SpecialCase);

      if not SpecialCase then

         Error := False;

         ErrorType := ELStrings.EmptyString;
         ErrorNo   := ELStrings.EmptyString;

         SavedLine := Line;

         -- Get characters until flash is found (either ???, ---, +++, ***, !!!)
         loop
            ELStrings.PopChar (Line, Char);
            exit when FlashCharacter (Char) or Line = ELStrings.EmptyString;
         end loop;

         -- Look for two more flash characters
         ELStrings.PopChar (Line, Char2);
         ELStrings.PopChar (Line, Char3);

         if Char2 /= Char or Char3 /= Char then
            Error := True;
         end if;

         -- Look for a space and a '('
         ELStrings.PopChar (Line, Char2);
         ELStrings.PopChar (Line, Char3);

         if Char2 /= ' ' or Char3 /= '(' then
            Error := True;
         end if;

         -- Skip line number (up to next ')')
         loop
            ELStrings.PopChar (Line, Char);
            exit when Char = ')' or Line = ELStrings.EmptyString;
         end loop;

         -- Skip whitespace
         loop
            ELStrings.PopChar (Line, Char);
            exit when Char /= ' ' or Line = ELStrings.EmptyString;
         end loop;

         -- Char is first character of ErrorType
         AppendCharWithoutCheck (ErrorType, Char);

         -- Get rest of ErrorType (up to next ' ')
         loop
            ELStrings.PopChar (Line, Char);
            exit when Char = ' ' or Line = ELStrings.EmptyString;
            AppendCharWithoutCheck (ErrorType, Char);
         end loop;

         -- Skip up to colon
         loop
            ELStrings.PopChar (Line, Char);
            exit when Char = ':' or Line = ELStrings.EmptyString;
         end loop;

         -- Skip whitespace
         loop
            ELStrings.PopChar (Line, Char);
            exit when Char /= ' ' or Line = ELStrings.EmptyString;
         end loop;

         -- Char is first character of ErrorNo
         AppendCharWithoutCheck (ErrorNo, Char);

         -- Get rest of ErrorNo (up to next ':')
         loop
            ELStrings.PopChar (Line, Char);
            exit when Char = ':' or Line = ELStrings.EmptyString;
            AppendCharWithoutCheck (ErrorNo, Char);
         end loop;

         -- If an error occurred report this and show which line it occurred on.
         if Error or Line = ELStrings.EmptyString then
            ScreenEcho.Put_Line ("An error occurred while parsing the following error message line.");
            ELStrings.PutLine (SPARK_IO.Standard_Output, SavedLine);
         end if;

         ErrorType := ELStrings.LowerCase (ErrorType);

         -- Generate the output string.
         ELStrings.CopyString (OutString, "<A HREF=""");
         ELStrings.AppendExaminerLongString (OutString, ErrorReferenceFileName);
         ELStrings.AppendString (OutString, "#");
         ELStrings.AppendExaminerLongString (OutString, ErrorType);
         ELStrings.AppendString (OutString, "-");
         ELStrings.AppendExaminerLongString (OutString, ErrorNo);
         ELStrings.AppendString (OutString, """ TARGET=""rbottom"">");

         ErrorLink := OutString;

         ELStrings.AppendExaminerLongString (OutString, SavedLine);
         ELStrings.AppendString (OutString, "</A>");

         Line := OutString;

      end if;

      Line := HTMLEmbolden (Line);

   end ProcessFirstErrorMessageLine;

   -- This procedure processes subsequent error message lines.  These contain
   -- word-wrapped portions of the error message, but no flash, error number
   -- and so on.  We therefore need to use the tag stored in SavedErrorLink
   -- as the link for this line.
   --
   -- The line is assumed to have some spaces at the front of the it.  Placing the
   -- tag before these turns the spaces into links which looks really silly, so
   -- the tag is placed immediately before the first non-space character.
   --
   -- If SavedErrorLink is EmptyString then no tag is added.

   procedure ProcessNextErrorMessageLine (Line : in out ELStrings.T;
                                          Link : in     ELStrings.T)
      --# derives Line from *,
      --#                   Link;
   is

      OutString : ELStrings.T;
      Char      : Character;

   begin

      if not ELStrings.EqString (Line, ELStrings.EmptyString) then
         OutString := ELStrings.EmptyString;

         -- Copy up to first non-space character
         loop
            ELStrings.PopChar (Line, Char);
            exit when Char /= ' ';
            AppendCharWithoutCheck (OutString, Char);
         end loop;

         ELStrings.AppendExaminerLongString (OutString, Link);

         AppendCharWithoutCheck (OutString, Char);
         ELStrings.AppendExaminerLongString (OutString, Line);
         ELStrings.AppendString (OutString, "</A>");

         Line := OutString;

      end if;

      Line := HTMLEmbolden (Line);

   end ProcessNextErrorMessageLine;

   --
   --  -------------
   --  InitSparkHTML
   --  -------------
   --
   --  This procedure initialises the HTML output of the examiner.
   --
   --  It is assumed that the HTML output flag has already been tested.  The call to
   --  InitSparkHTML should look something like:
   --
   --    if CommandLineData.Content.HTML then
   --       SparkHTML.InitSparkHTML;
   --    end if;
   --
   --  The procedure does exactly the following:
   --
   --     - creates a subdirectory of the current directory if one
   --       doesn't already exist.  The directory name is given by
   --       CommandLineData.Content.HTMLDirectory
   --     - prints a message to the screen informing the user that HTML output
   --       has been requested;
   --     - creates a top-level frames file in the HTML directory as described in
   --       the design;
   --     - creates an empty HTML file to fill the bottom frame (this is so that
   --       Netscape Navigator behaves correctly);
   --     - sets GenerateHTML to false if anything went wrong, to prevent further
   --       HTML generation.
   --
   --  Error trapping:
   --
   --  If anything unusual happens then the procedure will echo a message to the
   --  screen saying what went wrong and, in the case of errors which should stop
   --  HTML generation, GenerateHTML will be set to "false".
   --
   --  Failed HTML initialisation should not affect the analysis.  If intialisation
   --  fails, the GenerateHTML flag will stop calls to other SparkHTML routines
   --  from doing anything.  It is therefore safe to make those calls.

   procedure InitSparkHTML is

      NewDirOK    : Boolean;
      FrameFileOK : Boolean;
      DirName     : EStrings.T;

      -- This procedure writes HTML content to the base frame file.
      -- The procedure does not open or close the frame file, it assumes
      -- that a file has been opened.
      -- The FrameFile parameter must be the handle of an open file.
      -- The ReportFileName parameter is the location of the file to
      -- go in the top frame, relative to the frame file.
      -- The BlankfileName parameter is the location of the file to go
      -- in the top frame, relative to the frame file.
      --
      -- Error trapping:
      --
      -- All error trapping is performed within SPARK_IO.  The SPARK_IO
      -- procedures called do not return status information and so error
      -- trapping is not possible here.

      procedure WriteFrameFile (FrameFile      : in SPARK_IO.File_Type;
                                ReportFileName : in ELStrings.T;
                                BlankFileName  : in ELStrings.T)
         --# global in     CommandLineData.Content;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives SPARK_IO.FILE_SYS from *,
         --#                                BlankFileName,
         --#                                CommandLineData.Content,
         --#                                FrameFile,
         --#                                ReportFileName;
      is
         TempExaminerLongString : ELStrings.T;
      begin
         SPARK_IO.Put_Line (FrameFile, "<!DOCTYPE HTML PUBLIC ""-//W3C//DTD HTML 4.0 Frameset//EN""", 57);
         SPARK_IO.Put_Line (FrameFile, "            ""http://www.w3.org/TR/REC-html40/frameset.dtd"">", 59);
         SPARK_IO.Put_Line (FrameFile, "<html>", 6);
         SPARK_IO.Put_Line (FrameFile, "  <head>", 8);
         SPARK_IO.Put_Line (FrameFile, "    <title>SPARK HTML Output</title>", 36);
         SPARK_IO.Put_Line (FrameFile, "  </head>", 9);
         SPARK_IO.Put_Line (FrameFile, "  <frameset rows=""60%,*"">", 25);

         ELStrings.CopyString (TempExaminerLongString,
                                         "    <frame name=""rtop"" scrolling=""auto"" src=""");
         ELStrings.AppendExaminerLongString (TempExaminerLongString, ReportFileName);
         ELStrings.AppendString (TempExaminerLongString, """>");

         if CommandLineData.Content.PlainOutput then
            TempExaminerLongString := ELStrings.LowerCase (TempExaminerLongString);
         end if;
         SPARK_IO.Put_Line (FrameFile,  TempExaminerLongString.Content, TempExaminerLongString.Length);

         ELStrings.CopyString (TempExaminerLongString,
                                         "    <frame name=""rbottom"" scrolling=""auto"" src=""");
         ELStrings.AppendExaminerLongString (TempExaminerLongString, BlankFileName);
         ELStrings.AppendString (TempExaminerLongString, """>");

         SPARK_IO.Put_Line (FrameFile,  TempExaminerLongString.Content, TempExaminerLongString.Length);

         SPARK_IO.Put_Line (FrameFile, "  </frameset>", 13);
         SPARK_IO.Put_Line (FrameFile, "</html>", 7);
      end WriteFrameFile;

      -- This procedure does the following:
      --
      --    - creates a file called "spark.htm" in the HTML subdirectory
      --      of the current directory (the HTML directory is assumed to
      --      already exist),
      --    - derives the report file name from that held in
      --      CommandLineData.Content for inclusion in the frame file,
      --    - creates a blank file to fill the bottom frame,
      --    - calls WriteFrameFile to write the content of the base frame
      --      file,
      --    - closes the frame file and blank file,
      --    - returns a status value of True if no errors occurred,
      --    - returns a status value of False and displays a message to
      --      the user if an error occurred.
      --
      -- Error Trapping:
      --
      -- Where calls to procedures in SPARK_IO return status parameters the
      -- parameter is checked.  If the status is not OK then a message is
      -- echoed to the screen to inform the user.  The message does not
      -- display the type of error (but could in future) but explains what
      -- was happening when the error occurred.
      --
      -- This procedure returns a status value itself.  If the status value
      -- is False then it is suggested that no further HTML generation should
      -- take place.  The caller should test the returned status and set the
      -- GenerateHTML flag appropriately.

      procedure CreateFrameFile (Success : out Boolean)
         --# global in     CommandLineData.Content;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives SPARK_IO.FILE_SYS,
         --#         Success           from CommandLineData.Content,
         --#                                SPARK_IO.FILE_SYS;
      is
         FrameFile          : SPARK_IO.File_Type;
         BlankFile          : SPARK_IO.File_Type;
         FrameFileCreatedOK : SPARK_IO.File_Status;
         FrameFileClosedOK  : SPARK_IO.File_Status;
         BlankFileCreatedOK : SPARK_IO.File_Status;
         BlankFileClosedOK  : SPARK_IO.File_Status;

         HTMLReportFileName : ELStrings.T;

         FrameFileName      : ELStrings.T;
         BlankFileName      : ELStrings.T;
         FullFrameFileName  : ELStrings.T;
         FullBlankFileName  : ELStrings.T;

      begin
         -- Initialise variables.
         Success      := True;  -- Set to false when fatal error occurs.
         FrameFile    := SPARK_IO.Null_File;
         BlankFile    := SPARK_IO.Null_File;

         ELStrings.CopyString (FrameFileName, "spark.htm");
         ELStrings.CopyString (BlankFileName, "blank.htm");

         -- These files both reside in the HTML directory so their
         -- names need to have the HTML directory name prepended.
         FullFrameFileName := FileSystem.CaseOfFilesForCreateL (HTMLFileName (FrameFileName));
         FullBlankFileName := FileSystem.CaseOfFilesForCreateL (HTMLFileName (BlankFileName));

         -- Get the name of the HTML report file for inclusion in the HTML.
         HTMLReportFileName := ELStrings.ToExaminerLongString (
                                  FileSystem.JustFile (CommandLineData.Content.ReportFileName, True));
         HTMLReportFileName := Translate (HTMLReportFileName, '.', '_');
         ELStrings.AppendString (HTMLReportFileName, ".htm");

         -- Create the frame file.
         SPARK_IO.Create (File          => FrameFile,
                          Name_Length   => FullFrameFileName.Length,
                          Name_Of_File  => FullFrameFileName.Content,
                          Form_Of_File  => "",
                          Status        => FrameFileCreatedOK);

         if FrameFileCreatedOK /= SPARK_IO.Ok then

            ScreenEcho.Put_Line ("An error occurred while attemping to create the HTML frame file.");
            Success := False;

         else

            SPARK_IO.Create (File          => BlankFile,
                             Name_Length   => FullBlankFileName.Length,
                             Name_Of_File  => FullBlankFileName.Content,
                             Form_Of_File  => "",
                             Status        => BlankFileCreatedOK);

            if BlankFileCreatedOK /= SPARK_IO.Ok then

               ScreenEcho.Put_Line ("An error occurred while attempting to create the blank HTML file.");
               -- There is not an else here as not being able to write the blank file is
               -- not fatal.  Success remains true and we continue as we still have to
               -- close the FrameFile.

            end if;

            WriteFrameFile (FrameFile, HTMLReportFileName, BlankFileName);

            --# accept Flow, 10, FrameFile, "Expected ineffective assignment to FrameFile";
            SPARK_IO.Close (File   => FrameFile,
                            Status => FrameFileClosedOK);
            --# end accept;

            if FrameFileClosedOK /= SPARK_IO.Ok then
               ScreenEcho.Put_Line ("An error occurred while attempting to close the HTML frame file.");
               Success := False;
            end if;
            -- There is not an else here as we need to try and close the other file.

            --# accept Flow, 10, BlankFile, "Expected ineffective assignment to BlankFile";
            SPARK_IO.Close (File   => BlankFile,
                            Status => BlankFileClosedOK);
            --# end accept;

            if BlankFileClosedOK /= SPARK_IO.Ok then
               ScreenEcho.Put_Line ("An error occurred while attempting to close the blank HTML file.");
               -- Not closing the blank file is non-fatal, Success remains true.
            end if;

         end if;

      end CreateFrameFile;

      -- Returns the name of the HTML subdirectory required for passing
      -- to FileSystem.IdempotentCreateSubdir.

      function HTMLSubDirName return EStrings.T
      --# global in CommandLineData.Content;
      is

         DirName : EStrings.T;

      begin

         DirName := FileSystem.StartOfDirectory;

         EStrings.AppendExaminerString (DirName, CommandLineData.Content.HTMLDirectory);

         -- IdempotentCreateSubdir expects:
         --   on Unix/NT:  just the name of the directory

         return DirName;

      end HTMLSubDirName;

      procedure CopyErrorsFile
      --# global in     CommandLineData.Content;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives SPARK_IO.FILE_SYS from *,
      --#                                CommandLineData.Content;
      is

         SourceFileName        : EStrings.T;
         SourceFullFileName    : EStrings.T;
         SourceFileStatus      : FileSystem.TypFileSpecStatus;
         SourceFile            : SPARK_IO.File_Type;
         SourceFileOpenOK      : SPARK_IO.File_Status;
         SourceFileClosedOK    : SPARK_IO.File_Status;

         DestFileName          : ELStrings.T;
         DestFile              : SPARK_IO.File_Type;
         DestFileCreatedOK     : SPARK_IO.File_Status;
         DestFileClosedOK      : SPARK_IO.File_Status;

         CopyBuffer            : ELStrings.T;

      begin

         SourceFile := SPARK_IO.Null_File;
         DestFile   := SPARK_IO.Null_File;

         -- Start by trying to open output file.
         ELStrings.CopyString (DestFileName, "errors.htm");
         DestFileName := FileSystem.CaseOfFilesForCreateL (HTMLFileName (DestFileName));

         SPARK_IO.Create (File          => DestFile,
                          Name_Length   => DestFileName.Length,
                          Name_Of_File  => DestFileName.Content,
                          Form_Of_File  => "",
                          Status        => DestFileCreatedOK);

         if DestFileCreatedOK /= SPARK_IO.Ok then
            ScreenEcho.Put_Line ("An error occurred while trying to create the HTML errors file.");
         else
            -- Now locate the input file.

            SourceFileName := FileSystem.ExaminerLibDirectory;

            EStrings.AppendExaminerString (SourceFileName, FileSystem.EndOfPath);

            EStrings.AppendString (SourceFileName, "errors.htm");
            FileSystem.FindFullFileName (SourceFileName, SourceFileStatus, SourceFullFileName);
            if SourceFileStatus /= FileSystem.FileFound then
               -- Output simple message to destination file.
               SPARK_IO.Put_Line (DestFile, "Sorry, could not locate errors.htm.", 35);
               --# accept Flow, 10, DestFileClosedOK, "Expected ineffective assignment to DestFileClosedOK";
               SPARK_IO.Close (DestFile, DestFileClosedOK);    -- Unused variable DestFileClosedOK
                                                               --Ignore error on close.
               --# end accept;
            else

               -- Open the file and copy it.
               SPARK_IO.Open (File         => SourceFile,
                              Mode_Of_File => SPARK_IO.In_File,
                              Name_Length  => SourceFullFileName.Length,
                              Name_Of_File => SourceFullFileName.Content,
                              Form_Of_File => "",
                              Status       => SourceFileOpenOK);

               if SourceFileOpenOK /= SPARK_IO.Ok then
                  ScreenEcho.Put_Line ("An error occurred while opening the HTML errors file");
               else
                  while not SPARK_IO.End_Of_File (SourceFile) loop
                     ELStrings.GetLine (File => SourceFile,
                                                  EStr => CopyBuffer);
                     ELStrings.PutLine (File => DestFile,
                                                  EStr => CopyBuffer);
                  end loop;
               end if;

               --# accept Flow, 10, SourceFile, "Expected ineffective assignment to SourceFile" &
               --#        Flow, 10, SourceFileClosedOK, "Expected ineffective assignment to SourceFileClosedOK";
               SPARK_IO.Close (SourceFile, SourceFileClosedOK); -- 2 ineffective assignments OK
               --# end accept;

            end if;
         end if;
         --# accept Flow, 10, DestFile, "Expected ineffective assignment to DestFile" &
         --#        Flow, 10, DestFileClosedOK, "Expected ineffective assignment to DestFileClosedOK";
         SPARK_IO.Close (DestFile, DestFileClosedOK);  -- 2 ineffective assignments OK
         --# end accept;

         --# accept Flow, 33, SourceFileClosedOK, "Expected SourceFileClosedOK to be neither referenced nor exported" &
         --#        Flow, 33, DestFileClosedOK, "Expected DestFileClosedOK to be neither referenced nor exported";
      end CopyErrorsFile;  -- 2 Flow errors OK


   begin -- InitSparkHTML

      -- Initialise working directories - ensure trailing slashes for use in
      -- relative filename procedures.
      SparkWorkDir := ELStrings.ToExaminerLongString (FileSystem.WorkingDirectory);

      -- Ensure trailing separator.
      if SparkWorkDir.Content (SparkWorkDir.Length) /= FileSystem.EndOfPath.Content (FileSystem.EndOfPath.Length) then
         ELStrings.AppendExaminerString (SparkWorkDir, FileSystem.EndOfPath);
      end if;

      HTMLWorkDir := ELStrings.ToExaminerLongString (FileSystem.WorkingDirectory);
      HTMLWorkDir := AppendDirectoryString (HTMLWorkDir, CommandLineData.Content.HTMLDirectory);

      -- Get name for creating subdirectory.
      DirName  := HTMLSubDirName;

      -- Create subdirectory.
      FileSystem.IdempotentCreateSubdirectory (Path => DirName,
                                               Ok   => NewDirOK);
      CopyErrorsFile;
      CreateFrameFile (FrameFileOK);
      if not (NewDirOK and FrameFileOK) then
         ScreenEcho.Put_Line ("Error ocurred while initialising HTML generation.");
         ScreenEcho.Put_Line ("No further HTML generation will occur.");
         GenerateHTML := False;
         -- Suppress further HTML generation.
      end if;
   end InitSparkHTML;  -- flow errors expected due to false coupling through SPARK_IO and
   -- FileStatus not being used.

   ---------------------------------------------------------------------------

   --
   --  -------------
   --  GenReportHTML
   --  -------------
   --
   --  This procedure generates the HTMLified report file from the plain text
   --  report file.
   --
   --  It is assumed that the HTML output flag has already been tested.  The call to
   --  GenReportHTML should look something like:
   --
   --    if CommandLineData.Content.HTML then
   --       SparkHTML.GenReportHTML;
   --    end if;
   --
   --  The checking of the GenerateHTML flag is done internally.
   --
   --  If the GenerateHTML flag is false this procedure does nothing, otherwise,
   --  it does exactly the following:
   --
   --     - creates an HTML file (named <rep_file>.htm where <rep_file> is the name
   --       of the report file with all '.' characters changed to '_' characters;
   --     - processes each line of the plain text report file using the
   --       ProcessReportLine procedure (the functionality of this is described
   --       at the declaration of ProcessReportLine);
   --     - writes the processed lines to the HTML report file;
   --     - sets GenerateHTML to false if anything goes wrong, to prevent further
   --       HTML generation.
   --
   --  Error trapping:
   --
   --  The HTML report file is essentially a guide to the HTMLified listings and
   --  so if something causes the HTML report file generation to fail we should
   --  suppress all following HTML generation by settign GenerateHTML to "false".
   --
   --  If anything unusual happens then the procedure will echo a message to the
   --  screen saying what went wrong and, in the case of errors which should stop
   --  HTML generation, GenerateHTML will be set to "false".
   --
   procedure GenReportHTML is

      HTMLReportFile          : SPARK_IO.File_Type;
      HTMLReportFileName      : ELStrings.T;
      HTMLReportFileCreatedOK : SPARK_IO.File_Status;
      HTMLReportFileClosedOK  : SPARK_IO.File_Status;

      PlainReportFile         : SPARK_IO.File_Type;
      PlainReportFileName     : EStrings.T;
      PlainReportFileOpenOK   : SPARK_IO.File_Status;
      PlainReportFileClosedOK : SPARK_IO.File_Status;

      LineBuffer              : ELStrings.T;
      LookaheadBuffer         : ELStrings.T;
      SavedErrorLink          : ELStrings.T;
      SavedListingFile        : ELStrings.T;

      ReportFileState         : ReportFileStates := ReportJustStarted;

      HTMLMessage             : EStrings.T;

      -- Sub Programs
      -- ------------

      -- This sub-procedure writes HTML content to the start of the report file.
      -- It assumes that the file handle HTMLReportFile is the open HTML
      -- report file.  This procedure will not open or close the file.
      --
      -- The HTML written specifies the title of the page and some formatting
      -- tags.  The formatting is <PRE> (pre-processed text) which displays
      -- text exactly as given, and <TT> which sets the typewriter-text font;
      -- I use this because it usually results in a fixed-width font being
      -- used.
      --
      -- When writing the end of the report file these tags need to be closed,
      -- as do the <BODY> and <HTML> tags.  This should be done by calling
      -- the WriteHTMLReportFooter procedure.
      --
      -- Error trapping:
      --
      -- All error trapping is performed within SPARK_IO.
      -- The SPARK_IO routines called do not return status parameters and so
      -- no error trapping can be done here.

      procedure WriteHTMLReportHeader
         --# global in     HTMLReportFile;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives SPARK_IO.FILE_SYS from *,
         --#                                HTMLReportFile;

      is

      begin

         SPARK_IO.Put_Line (HTMLReportFile,
                            "<!DOCTYPE HTML PUBLIC ""-//W3C//DTD HTML 4.0 Transitional//EN""", 61);
         SPARK_IO.Put_Line (HTMLReportFile,
                            "            ""http://www.w3.org/TR/REC-html40/loose.dtd"">", 56);
         SPARK_IO.Put_Line (HTMLReportFile,
                            "<html>", 6);
         SPARK_IO.Put_Line (HTMLReportFile,
                            "  <head>", 8);
         SPARK_IO.Put_Line (HTMLReportFile,
                            "    <title>SPARK Examiner HTML Report File</title>", 50);
         SPARK_IO.Put_Line (HTMLReportFile,
                            "  </head>", 9);
         SPARK_IO.Put_Line (HTMLReportFile,
                            "  <body>", 8);
         SPARK_IO.Put_Line (HTMLReportFile,
                            "    <pre>", 9);
         SPARK_IO.Put_Line (HTMLReportFile,
                            "      <tt>", 10);

      end WriteHTMLReportHeader;


      -- This subprocedure writes HTML content to the end of the report file.
      -- It assumes that the file handle HTMLReportFile is the open HTML
      -- report file.  This procedure will not open or close the file.
      --
      -- The HTML written closes all the formatting tags that were opened
      -- by the call to WriteHTMLReportHeader.
      --
      -- Error trapping:
      --
      -- All error trapping is performed within SPARK_IO.
      -- The SPARK_IO routines called do not return status parameters and so
      -- no error trapping can be done here.

      procedure WriteHTMLReportFooter
         --# global in     HTMLReportFile;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives SPARK_IO.FILE_SYS from *,
         --#                                HTMLReportFile;

      is

      begin

         SPARK_IO.Put_Line (HTMLReportFile, "      </tt>", 11);
         SPARK_IO.Put_Line (HTMLReportFile, "    </pre>", 10);
         SPARK_IO.Put_Line (HTMLReportFile, "  </body>", 9);
         SPARK_IO.Put_Line (HTMLReportFile, "</html>", 7);

      end WriteHTMLReportFooter;

      -- This procedure is used to convert a line of the report file into HTML.
      -- It is effectively a parser for the report file.
      --
      -- The procedure design is based on a state machine.  The global variable
      -- ReportFileState records our current location within the report file (in
      -- terms of what information has already been read).
      --
      -- Given a line, we can determine the line's meaning from our current state
      -- and the contents of the line (all blank lines are ignored).  For example,
      -- if we are reading the warning list and we discover a line starting with
      -- the string "Source Filename(s) used were:" then we know that we are now
      -- processing the source file list.
      --
      -- We can use this method to parse almost all the information in the source
      -- file.  Those bits that we don't parse are usually optional (such as the
      -- flag "rtc" in the options list) and require no translation to HTML anyway.
      --
      -- Once the procedure understands what a line represents it updates the
      -- ReportFileState and processes the line by calling an appropriate
      -- subroutine.
      --
      -- The procedure contains a debugging feature which reports each line that it
      -- finds to the screen along with a message for each line that it recognises.
      -- This only happens if the -debug switch is given on the commandline.

      procedure ProcessReportLine (Line      : in out ELStrings.T;
                                   Lookahead : in     ELStrings.T)
         --# global in     CommandLineData.Content;
         --#        in     HTMLWorkDir;
         --#        in     SparkWorkDir;
         --#        in out ReportFileState;
         --#        in out SavedErrorLink;
         --#        in out SavedListingFile;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives Line              from *,
         --#                                CommandLineData.Content,
         --#                                HTMLWorkDir,
         --#                                Lookahead,
         --#                                ReportFileState,
         --#                                SavedErrorLink,
         --#                                SavedListingFile,
         --#                                SparkWorkDir &
         --#         ReportFileState   from *,
         --#                                Line &
         --#         SavedErrorLink    from *,
         --#                                Line,
         --#                                Lookahead,
         --#                                ReportFileState &
         --#         SavedListingFile  from *,
         --#                                CommandLineData.Content,
         --#                                Line,
         --#                                ReportFileState &
         --#         SPARK_IO.FILE_SYS from *,
         --#                                CommandLineData.Content,
         --#                                Line,
         --#                                Lookahead,
         --#                                ReportFileState;

      is

         StartPos      : Integer;
         CompareString : ELStrings.T;
         DebugMessage  : ELStrings.T;

         -- This function takes a filename and some text as input and returns the text
         -- enclosed in HTML tags that form a link to the file relative to the
         -- specified "RelativeTo" directory.
         --
         -- A filename beginning with an @ symbol is interpreted as a metafile and
         -- the @ is removed from the location placed in the tag.
         --
         -- Error checking is assumed to occur within the string handling routines.
         -- If the HTMLRelativeFile procedure can not generate the URI then
         -- the link is not added and the file name returned, followed by a
         -- message saying "[Link unavailable]".

         function CreateFileLinkTagWithText (FileNameIn          : ELStrings.T;
                                             RelativeTo          : ELStrings.T;
                                             Text                : ELStrings.T)
                                            return ELStrings.T
            --# global in SparkWorkDir;
         is

            OutString            : ELStrings.T;
            FileName             : ELStrings.T;
            Location             : ELStrings.T;
            Success              : Boolean;

         begin

            FileName := FileNameIn;

            -- The OutString is the complete string that will be output including
            -- the tags and the original filename.
            OutString := ELStrings.EmptyString;

            -- Chop off the leading @ (if there is one).
            if FileName.Content (ELStrings.Positions'First) = '@' then
               FileName := LongStringUtilities.Section (FileName,
                                                        ELStrings.Positions'First + 1,
                                                        FileName.Length - 1);
            end if;

            -- Find the relative URI.
            HTMLRelativeFile (FileName, RelativeTo, Location, Success);

            if not Success then
               OutString := FileNameIn;
               ELStrings.AppendString (OutString, " [Link unavailable] ");
            else
               -- Create the tag.
               ELStrings.AppendString (OutString, "<A HREF=""");

               ELStrings.AppendExaminerLongString (OutString, Location);
               ELStrings.AppendString (OutString,
                                                 """  TYPE=""x-text/spark"" TARGET=""rbottom"" >");
               ELStrings.AppendExaminerLongString (OutString, Text);
               ELStrings.AppendString (OutString, "</A>");
            end if;

            return OutString;

         end CreateFileLinkTagWithText;

         -- This function does the same as CreateFileLinkTagWithText but the text
         -- used is the filename given.

         function CreateFileLinkTag (FileNameIn          : ELStrings.T;
                                     RelativeTo          : ELStrings.T)
                                    return ELStrings.T
            --# global in SparkWorkDir;
         is

         begin

            return CreateFileLinkTagWithText (FileNameIn => FileNameIn,
                                              RelativeTo => RelativeTo,
                                              Text       => FileNameIn);
         end CreateFileLinkTag;


         -- This file turns a line from the options section of the report file
         -- that specifies a file location and adds HTML tags to create a link
         -- to the file.
         --
         -- An "Option Line" is defined to be a line from the options section
         -- of the report file.  This line is of the form:
         --
         --    <option_name> = <file_location>
         --
         -- The procedure simply copies the line up to the '=' character
         -- and assumes that whatever follows the '=' is the filename.

         function CreateOptionFileLink (Line : ELStrings.T)
                                       return ELStrings.T
            --# global in HTMLWorkDir;
            --#        in SparkWorkDir;
         is

            TempLine      : ELStrings.T;
            I             : ELStrings.Positions;
            FileName      : ELStrings.T;
            FileLink      : ELStrings.T;

         begin

            TempLine      := ELStrings.EmptyString;
            FileName      := ELStrings.EmptyString;
            I             := ELStrings.Positions'First;

            -- Copy up to the '='
            loop
               AppendCharWithoutCheck (TempLine, Line.Content (I));
               exit when Line.Content (I) = '=';
               I := I + 1;
            end loop;

            -- Point to the first character of the filename.
            I := I + 1;

            -- Get the name of the index file by copying up to the end of line
            while I <= Line.Length loop
               AppendCharWithoutCheck (FileName, Line.Content (I));
               I := I + 1;
            end loop;

            -- Add HTML tag
            FileLink := CreateFileLinkTag (FileName, HTMLWorkDir);

            ELStrings.AppendExaminerLongString (TempLine, FileLink);

            return TempLine;

         end CreateOptionFileLink;

         -- This function takes a line containing a number of spaces followed by
         -- a file name and puts HTML link tags in to form a link to the file.
         -- If the file starts with an @ symbol it is interpreted as a metafile
         -- and the @ is removed.
         --
         -- This function is used to generate HTML for lines which just contain
         -- whitespace and filenames, such as the selected files list, index files
         -- list, and metafiles list.

         function CreateFileLink (Line : ELStrings.T)
                                 return ELStrings.T
            --# global in HTMLWorkDir;
            --#        in SparkWorkDir;
         is

            TempLine      : ELStrings.T;
            I             : ELStrings.Positions;
            FileName      : ELStrings.T;
            FileLink      : ELStrings.T;

         begin

            TempLine      := ELStrings.EmptyString;
            FileName      := ELStrings.EmptyString;
            I             := ELStrings.Positions'First;

            -- Copy whitespace up to the first character.
            loop
               exit when Line.Content (I) /= ' ';
               AppendCharWithoutCheck (TempLine, Line.Content (I));
               I := I + 1;
            end loop;

            -- Get the name of the file by copying up to the end of line
            while I <= Line.Length loop
               AppendCharWithoutCheck (FileName, Line.Content (I));
               I := I + 1;
            end loop;

            -- Add HTML tag
            FileLink := CreateFileLinkTag (FileName, HTMLWorkDir);

            ELStrings.AppendExaminerLongString (TempLine, FileLink);

            return TempLine;

         end CreateFileLink;

         -- This function is used to process a line from the source file list.
         -- The line is assumed to be some space characters followed by a
         -- file name.  The function creates the file link as described in the
         -- commentary for CreateFileLink, and appends to it a link to the
         -- analysis section for that file.
         --
         -- The link created is a link of the form:
         --
         --    <A HREF="#html_name"> ...  </A>
         --
         -- where html_name is the result of applying the HTMLName function to
         -- the filename.
         --
         -- The HTMLName function must also be used to create the anchor that this
         -- link refers to.

         function ProcessSourceListLine (Line : ELStrings.T)
                                        return ELStrings.T
            --# global in HTMLWorkDir;
            --#        in SparkWorkDir;
         is

            OutString    : ELStrings.T;
            FileName     : ELStrings.T;
            I            : ELStrings.Positions;

         begin

            I := FirstChar (Line);
            FileName := LongStringUtilities.Section (Line, I, (Line.Length - I) + 1);

            OutString := CreateFileLink (Line);

            ELStrings.AppendString (OutString, " [<A HREF=""#");
            ELStrings.AppendExaminerLongString (OutString, HTMLName (FileName));
            ELStrings.AppendString (OutString, """>View analysis</A>]");

            return OutString;

         end ProcessSourceListLine;

         -- This function processes a line from the main section of the report file
         -- of the form:
         --
         --    some_text: file_name
         --
         --  to produce a line of the form:
         --
         --    some_text: <A NAME="html_name">file_link</A>
         --
         --  where:
         --
         --    html_name is the result of applying the function HTMLName to file_name
         --              to generate the anchor tag referenced from the source file
         --              list (see commentary for ProcessSourceListLine)
         --    file_link is the result of applying the function CreateFileLinkTag to
         --              file_name to produce a link to the file
         --
         --  This function should be used to produce HTML for the lines in the body
         --  of the report file that beign "Source Filename: "

         function ProcessSourceFileNameLine (Line : ELStrings.T)
                                            return ELStrings.T
            --# global in HTMLWorkDir;
            --#        in SparkWorkDir;
         is

            OutString    : ELStrings.T;
            I            : ELStrings.Positions;
            FileName     : ELStrings.T;
            Anchor       : ELStrings.T;

         begin

            OutString := ELStrings.EmptyString;
            I         := ELStrings.Positions'First;

            -- Copy up to and including ':'
            loop
               AppendCharWithoutCheck (OutString, Line.Content (I));
               exit when Line.Content (I) = ':';
               I := I + 1;
            end loop;


            -- Point to next character
            I := I + 1;

            -- Copy spaces up to first char of file name
            loop
               exit when Line.Content (I) /= ' ';
               AppendCharWithoutCheck (OutString, Line.Content (I));
               I := I + 1;
            end loop;

            -- Extract the filename
            FileName := LongStringUtilities.Section (Line, I, (Line.Length - I) + 1);

            -- Create the anchor tag
            ELStrings.CopyString (Anchor, "<A NAME=""");
            ELStrings.AppendExaminerLongString (Anchor, HTMLName (FileName));
            ELStrings.AppendString (Anchor, """></A>");

            -- Append the anchor tag to the output line
            ELStrings.AppendExaminerLongString (OutString, Anchor);

            -- Create the link to the source file and append that to the output line
            ELStrings.AppendExaminerLongString (OutString,
                                                          CreateFileLinkTag (FileName,
                                                                             HTMLWorkDir));


            return OutString;

         end ProcessSourceFileNameLine;

         -- This procedure processes a line from the main section of the report file
         -- of the form:
         --
         --    some_text: file_name
         --
         --  to produce a line of the form:
         --
         --    some_text: file_link html_link
         --
         --  where:
         --
         --    file_link is the result of applying the function CreateFileLinkTag to
         --              file_name to produce a link to the file
         --
         --    html_link is the text "[View HTML]" enclosed in tags that make it a
         --              link to the HTML version of the above file.  The HTML
         --              version is assumed to be the name (without directory)
         --              of the specified listing file, with all '.'s changed
         --              to '_'s, with a ".htm" extension and residing in the HTML
         --              subdirectory of the current directory; e.g. H:\foo\bar.lst
         --              becomes HTML/bar_lst.htm
         --
         --  It also updates the value of SavedListingFile to be a reference to the
         --  HTML version of the listing for use in processing the source error lines.
         --
         --  This function should be used to produce HTML for the lines in the body
         --  of the report file that beign "Listing Filename: "

         procedure ProcessListingFileNameLine (Line : in out ELStrings.T)
         --# global in     CommandLineData.Content;
         --#        in     HTMLWorkDir;
         --#        in     SparkWorkDir;
         --#           out SavedListingFile;
         --# derives Line             from *,
         --#                               CommandLineData.Content,
         --#                               HTMLWorkDir,
         --#                               SparkWorkDir &
         --#         SavedListingFile from CommandLineData.Content,
         --#                               Line;
         is

            OutString : ELStrings.T;
            I         : ELStrings.Positions;

            FileName  : ELStrings.T;
            HTMLFileLink : ELStrings.T;

            HTMLLinkText : ELStrings.T;

         begin

            OutString := ELStrings.EmptyString;
            I := ELStrings.Positions'First;

            ELStrings.CopyString (HTMLLinkText, "[View HTML]");

            -- Copy up to and including ':'
            loop
               AppendCharWithoutCheck (OutString, Line.Content (I));
               exit when Line.Content (I) = ':';
               I := I + 1;
            end loop;

            -- Point to next character
            I := I + 1;

            -- Copy spaces up to first char of file name
            loop
               exit when Line.Content (I) /= ' ';
               AppendCharWithoutCheck (OutString, Line.Content (I));
               I := I + 1;
            end loop;

            -- Extract the filename
            FileName := LongStringUtilities.Section (Line, I, (Line.Length - I) + 1);

            -- Append link to plain text listing.
            ELStrings.AppendExaminerLongString (OutString,
                                                          CreateFileLinkTag (FileNameIn => FileName,
                                                                             RelativeTo => HTMLWorkDir));

            -- And add a space
            AppendCharWithoutCheck (OutString, ' ');

            -- Create URL for HTML listing file in SavedListingFile
            SavedListingFile := ELStrings.ToExaminerLongString (CommandLineData.Content.HTMLDirectory);
            ELStrings.AppendString (SavedListingFile, "/");
            ELStrings.AppendExaminerLongString (
               SavedListingFile,
               Translate (ELStrings.ToExaminerLongString (
                             FileSystem.JustFile (Fn  => ELStrings.ToExaminerString (FileName),
                                                  Ext => True)),
                          '.',
                          '_'));

            ELStrings.AppendString (SavedListingFile, ".htm");

            -- Create HTML tags.
            HTMLFileLink := CreateFileLinkTagWithText (FileNameIn => SavedListingFile,
                                                       RelativeTo => HTMLWorkDir,
                                                       Text       => HTMLLinkText);

            ELStrings.AppendExaminerLongString (OutString, HTMLFileLink);

            Line := OutString;

         end ProcessListingFileNameLine;

         -- *** Comment

         procedure ProcessErrorSourceLine (Line : in out ELStrings.T)
            --# global in HTMLWorkDir;
            --#        in SavedListingFile;
            --#        in SparkWorkDir;
            --# derives Line from *,
            --#                   HTMLWorkDir,
            --#                   SavedListingFile,
            --#                   SparkWorkDir;
         is

            OutString       : ELStrings.T;
            I               : ELStrings.Positions;
            LineNo          : ELStrings.T;

            RelativeListing : ELStrings.T;
            LinkSuccess     : Boolean;
            Link            : ELStrings.T;

         begin

            if SavedListingFile.Length /= 0 then  -- there is a listing file

               OutString := ELStrings.EmptyString;
               I         := ELStrings.Positions'First;
               LineNo    := ELStrings.EmptyString;

               -- Copy up to the first non-space.
               loop
                  exit when Line.Content (I) /= ' ';
                  AppendCharWithoutCheck (OutString, Line.Content (I));
                  I := I + 1;
               end loop;

               -- Copy digits to LineNo
               while Digit (Line.Content (I)) loop
                  AppendCharWithoutCheck (LineNo, Line.Content (I));
                  I := I + 1;
               end loop;

               -- Create relative location of SavedListingFile
               HTMLRelativeFile (SavedListingFile,
                                 HTMLWorkDir,
                                 RelativeListing,
                                 LinkSuccess);

               if LinkSuccess then
                  -- Create link to listing based on saved listing file.
                  ELStrings.CopyString (Link, "<A TARGET=""rbottom"" HREF=""");
                  ELStrings.AppendExaminerLongString (Link, RelativeListing);
                  ELStrings.AppendString (Link, "#line");
                  ELStrings.AppendExaminerLongString (Link, LineNo);
                  ELStrings.AppendString (Link, """>");
                  ELStrings.AppendExaminerLongString (Link, LineNo);
                  ELStrings.AppendString (Link, "</A>");

               else
                  -- The link is just the number.
                  Link := LineNo;
               end if;

               -- Append link to OutString
               ELStrings.AppendExaminerLongString (OutString, Link);

               -- Append rest of line
               ELStrings.AppendExaminerLongString (OutString,
                                                             LongStringUtilities.Section (Line,
                                                                                          I,
                                                                                          (Line.Length - I) + 1));

               Line := OutString;

            end if;

         end ProcessErrorSourceLine;

      begin  -- ProcessReportLine
         if Line.Length > 0 then

            StartPos := FirstChar (Line);
            DebugMessage := ELStrings.EmptyString;

            if CommandLineData.Content.Debug.HTML then
               ELStrings.PutLine (SPARK_IO.Standard_Output, Line);  -- Line used for debugging.
            end if;

            case ReportFileState is
               when ReportJustStarted =>
                  if ELStrings.Eq1String (LongStringUtilities.Section (Line, StartPos, 5),
                                                    "*****") then
                     ReportFileState := ReportBannerStarted;
                     ELStrings.CopyString (DebugMessage,
                                                     "ReportBannerStarted");
                  end if;
               when ReportBannerStarted =>
                  if ELStrings.Eq1String (LongStringUtilities.Section (Line, StartPos, 5),
                                                    "*****") then
                     ReportFileState := ReportBannerFinished;
                     ELStrings.CopyString (DebugMessage, "ReportBannerFinished");
                  end if;
               when ReportBannerFinished =>
                  if ELStrings.Eq1String (LongStringUtilities.Section (Line, StartPos, 7),
                                                    "DATE : ") then
                     ReportFileState := ReportDateFound;
                     ELStrings.CopyString (DebugMessage, "ReportDateFound");
                  end if;
               when ReportDateFound =>
                  if ELStrings.Eq1String (LongStringUtilities.Section (Line, StartPos, 8),
                                                    "Options:") then
                     ReportFileState := ReportOptionsFound;
                     ELStrings.CopyString (DebugMessage, "ReportOptionsFound");
                  end if;
               when ReportOptionsFound =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 11);
                  if ELStrings.Eq1String (CompareString,  "index_file=") then
                     ReportFileState := ReportIndexFileFound;
                     Line := CreateOptionFileLink (Line);
                     ELStrings.CopyString (DebugMessage, "ReportIndexFileFound");
                  elsif ELStrings.Eq1String (CompareString,  "noindex_fil") then
                     ReportFileState := ReportIndexFileFound;
                     ELStrings.CopyString (DebugMessage, "ReportIndexFileFound");
                  end if;
               when ReportIndexFileFound =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 13);
                  if ELStrings.Eq1String (CompareString,  "warning_file=") then
                     ReportFileState := ReportWarningFileFound;
                     Line := CreateOptionFileLink (Line);
                     ELStrings.CopyString (DebugMessage, "ReportWarningFileFound");
                  elsif ELStrings.Eq1String (CompareString,  "nowarning_fil") then
                     ReportFileState := ReportWarningFileFound;
                     ELStrings.CopyString (DebugMessage, "ReportWarningFileFound");
                  end if;
               when ReportWarningFileFound =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 21);
                  if ELStrings.Eq1String (CompareString,  "target_compiler_data=") then
                     ReportFileState := ReportTargetCompilerDataFound;
                     Line := CreateOptionFileLink (Line);
                     ELStrings.CopyString (DebugMessage, "ReportTargetCompilerDataFound");
                  elsif ELStrings.Eq1String (CompareString,  "notarget_compiler_dat") then
                     ReportFileState := ReportTargetCompilerDataFound;
                     ELStrings.CopyString (DebugMessage, "ReportTargetCompilerDataFound");
                  end if;
               when ReportTargetCompilerDataFound =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 12);
                  if ELStrings.Eq1String (CompareString,  "config_file=") then
                     ReportFileState := ReportTargetConfigFileFound;
                     Line := CreateOptionFileLink (Line);
                     ELStrings.CopyString (DebugMessage, "ReportTargetConfigFileFound");
                  elsif ELStrings.Eq1String (CompareString,  "noconfig_fil") then
                     ReportFileState := ReportTargetConfigFileFound;
                     ELStrings.CopyString (DebugMessage, "ReportTargetConfigFileFound");
                  end if;
               when ReportTargetConfigFileFound =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 17);
                  if ELStrings.Eq1String (CompareString,  "source_extension=") then
                     ReportFileState := ReportSourceExtensionFound;
                     ELStrings.CopyString (DebugMessage, "ReportSourceExtensionFound");
                  end if;
               when ReportSourceExtensionFound =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 18);
                  if ELStrings.Eq1String (CompareString,  "listing_extension=") then
                     ReportFileState := ReportListingExtensionFound;
                     ELStrings.CopyString (DebugMessage, "ReportListingExtensionFound");
                  end if;
               when ReportListingExtensionFound =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 12);
                  if ELStrings.Eq1String (CompareString,  "dictionary_f") then
                     ReportFileState := ReportDictionaryFound;
                     Line := CreateOptionFileLink (Line);
                     ELStrings.CopyString (DebugMessage, "ReportDictionaryFound");
                  elsif ELStrings.Eq1String (CompareString,  "nodictionary") then
                     ReportFileState := ReportDictionaryFound;
                     ELStrings.CopyString (DebugMessage, "ReportDictionaryFound");
                  end if;
               when ReportDictionaryFound =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 12);
                  if ELStrings.Eq1String (CompareString,  "report_file=") then
                     ReportFileState := ReportReportFileFound;
                     Line := CreateOptionFileLink (Line);
                     ELStrings.CopyString (DebugMessage, "ReportReportFileFound");
                  end if;
                  -- NOTE: checking for "noreport_file" is a stupid thing to do!
               when ReportReportFileFound =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 4);
                  if ELStrings.Eq1String (CompareString,  "html") then
                     -- NOTE: checking for nohtml is an equally stupid thing to do!
                     ReportFileState := ReportHTMLFlagFound;
                     ELStrings.CopyString (DebugMessage, "ReportHTMLFlagFound");
                  end if;
               when ReportHTMLFlagFound =>
                  -- The RTC and VCG options appear here but as they are optional
                  -- we shall ignore them.
                  CompareString := LongStringUtilities.Section (Line, StartPos, 10);
                  if ELStrings.Eq1String (CompareString,  "statistics") or
                     ELStrings.Eq1String (CompareString,  "nostatisti") then
                     ReportFileState := ReportStatisticsFlagFound;
                     ELStrings.CopyString (DebugMessage, "ReportStatisticsFlagFound");
                  end if;
               when ReportStatisticsFlagFound =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 15);
                  if ELStrings.Eq1String (CompareString,  "fdl_identifiers") or
                     ELStrings.Eq1String (CompareString,  "nofdl_identifie") then
                     ReportFileState := ReportFDLIdentifiersFound;
                     ELStrings.CopyString (DebugMessage, "ReportFDLIdentifiersFound");
                  end if;
               when ReportFDLIdentifiersFound =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 14);
                  if ELStrings.Eq1String (CompareString,  "flow_analysis=") then
                     ReportFileState := ReportFlowAnalysisFound;
                     ELStrings.CopyString (DebugMessage, "ReportFlowAnalysisFound");
                  end if;
               when ReportFlowAnalysisFound =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 5);
                  if ELStrings.Eq1String (CompareString,  "ada83") or
                     ELStrings.Eq1String (CompareString,  "ada95") then
                     ReportFileState := ReportLanguageOptionFound;
                     ELStrings.CopyString (DebugMessage, "ReportLanguageOptionFound");
                  end if;
               when ReportLanguageOptionFound =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 21);
                  if ELStrings.Eq1String (CompareString,  "annotation_character=") then
                     ReportFileState := ReportAnnotationCharacterFound;
                     ELStrings.CopyString (DebugMessage, "ReportAnnotationCharacterFound");
                  end if;
               when ReportAnnotationCharacterFound =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 15);
                  if ELStrings.Eq1String (CompareString,  "Selected files:") then
                     ReportFileState := ReportSelectedFilesStarted;
                     ELStrings.CopyString (DebugMessage, "ReportSelectedFilesStarted");
                  end if;
               when ReportSelectedFilesStarted =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 24);
                  if ELStrings.Eq1String (CompareString,  "Index Filename(s) used w") or
                     ELStrings.Eq1String (CompareString,  "No Index files were used") then
                     ReportFileState := ReportIndexFilesStarted;
                     ELStrings.CopyString (DebugMessage, "ReportIndexFilesStarted");
                  else
                     Line := CreateFileLink (Line);
                  end if;
               when ReportIndexFilesStarted =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 18);
                  if ELStrings.Eq1String (CompareString,  "Meta File(s) used ") or
                     ELStrings.Eq1String (CompareString,  "No Meta Files used") then
                     ReportFileState := ReportMetaFilesStarted;
                     ELStrings.CopyString (DebugMessage, "ReportMetaFilesStarted");
                  else
                     Line := CreateFileLink (Line);
                  end if;
               when ReportMetaFilesStarted =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 22);
                  if ELStrings.Eq1String (CompareString,  "Full warning reporting") or
                     ELStrings.Eq1String (CompareString,  "Summary warning report") then
                     ReportFileState := ReportWarningListStarted;
                     ELStrings.CopyString (DebugMessage, "ReportWarningListStarted");
                  else
                     Line := CreateFileLink (Line);
                  end if;
               when ReportWarningListStarted =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 26);
                  if ELStrings.Eq1String (CompareString, "Target configuration file:") then
                     ReportFileState := ReportTargetConfigListStarted;
                     ELStrings.CopyString (DebugMessage, "ReportTargetConfigListStarted");
                  elsif ELStrings.Eq1String (CompareString, "Source Filename(s) used we") then
                     ReportFileState := ReportSourceListStarted;
                     ELStrings.CopyString (DebugMessage, "ReportSourceListStarted");
                  end if;
               when ReportTargetConfigListStarted =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 4);
                  if ELStrings.Eq1String (CompareString, "Line") then
                     ReportFileState := ReportTargetErrorLine;
                     ELStrings.CopyString (DebugMessage, "ReportTargetErrorLine");
                  end if;
               when ReportTargetErrorLine =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 3);
                  if ELStrings.Eq1String (CompareString,  "!!!") or else
                    ELStrings.Eq1String (CompareString,  "***") or else
                    ELStrings.Eq1String (CompareString,  "---") or else
                    ELStrings.Eq1String (CompareString,  "???") then
                     ProcessFirstErrorMessageLine (Line, Lookahead, SavedErrorLink);
                     ReportFileState := ReportTargetErrorNextLine;
                     ELStrings.CopyString (DebugMessage, "ReportTargetErrorNextLine");
                  else
                     CompareString := LongStringUtilities.Section (Line, StartPos, 29);
                     if ELStrings.Eq1String (CompareString,  "Source Filename(s) used were:") then
                        ReportFileState := ReportSourceListStarted;
                        ELStrings.CopyString (DebugMessage, "ReportSourceListStarted");
                     end if;
                  end if;
               when ReportTargetErrorNextLine =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 29);
                  if StartPos > 10 then -- Extra lines of the error message will be indented by 11 characters
                     ProcessNextErrorMessageLine (Line, SavedErrorLink);
                  elsif ELStrings.Eq1String (CompareString,  "Source Filename(s) used were:") then
                     ReportFileState := ReportSourceListStarted;
                     ELStrings.CopyString (DebugMessage, "ReportSourceListStarted");
                  else
                     CompareString := LongStringUtilities.Section (Line, StartPos, 3);
                     if ELStrings.Eq1String (CompareString,  "!!!") or else
                       ELStrings.Eq1String (CompareString,  "***") or else
                       ELStrings.Eq1String (CompareString,  "---") or else
                       ELStrings.Eq1String (CompareString,  "???") then
                        ProcessFirstErrorMessageLine (Line, Lookahead, SavedErrorLink);
                        ReportFileState := ReportTargetErrorNextLine;
                        ELStrings.CopyString (DebugMessage, "ReportTargetErrorNextLine");
                     else
                        ReportFileState := ReportTargetErrorLine;
                        ELStrings.CopyString (DebugMessage, "ReportTargetErrorLine");
                     end if;
                  end if;
               when ReportSourceListStarted =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 16);
                  if ELStrings.Eq1String (CompareString,  "Source Filename:") then
                     Line := ProcessSourceFileNameLine (Line);
                     ReportFileState := ReportSourceFileStarted;
                     ELStrings.CopyString (DebugMessage, "ReportSourceFileStarted");
                  elsif ELStrings.Eq1String (CompareString,  "The following we") then
                     ReportFileState := ReportMissingFilesStarted;
                     ELStrings.CopyString (DebugMessage, "ReportMissingFilesStarted");
                  else
                     Line := ProcessSourceListLine (Line);
                  end if;
               when ReportMissingFilesStarted =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 16);
                  if ELStrings.Eq1String (CompareString,  "Source Filename:") then
                     Line := ProcessSourceFileNameLine (Line);
                     ReportFileState := ReportSourceFileStarted;
                     ELStrings.CopyString (DebugMessage, "ReportSourceFileStarted");
                  end if;
               when ReportSourceFileStarted =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 15);
                  if ELStrings.Eq1String (CompareString,  "Listing Filenam") then
                     ProcessListingFileNameLine (Line);
                     ReportFileState := ReportListingFileNameFound;
                     ELStrings.CopyString (DebugMessage, "ReportListingFileNameFound");
                  elsif ELStrings.Eq1String (CompareString,  "No Listing File") then
                     SavedListingFile := ELStrings.EmptyString;
                     ReportFileState := ReportListingFileNameFound;
                     ELStrings.CopyString (DebugMessage, "ReportListingFileNameFound");
                  elsif ELStrings.Eq1String (CompareString,   "***     no unit") then
                     ReportFileState := ReportNoUnitsFound;
                     ELStrings.CopyString (DebugMessage, "ReportNoUnitsFound");
                  end if;
               when ReportListingFileNameFound =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 10);
                  if ELStrings.Eq1String (CompareString,  "Unit name:") then
                     ReportFileState := ReportUnitNameFound;
                     ELStrings.CopyString (DebugMessage, "ReportUnitNameFound");
                  elsif ELStrings.Eq1String (CompareString,  "***     no") then
                     ReportFileState := ReportNoUnitsFound;
                     ELStrings.CopyString (DebugMessage, "ReportNoUnitsFound");
                  end if;
               when ReportUnitNameFound =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 10);
                  if ELStrings.Eq1String (CompareString,  "Unit type:") then
                     ReportFileState := ReportUnitTypeFound;
                     ELStrings.CopyString (DebugMessage, "ReportUnitTypeFound");
                  end if;
               when ReportUnitTypeFound =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 22);
                  if ELStrings.Eq1String (CompareString,  "Unit has been analysed") or
                     ELStrings.Eq1String (CompareString,  "Unit has been parsed o") then
                     ReportFileState := ReportAnalysedMessageFound;
                     ELStrings.CopyString (DebugMessage, "ReportAnalysedMessageFound");
                  end if;
               when ReportAnalysedMessageFound | ReportNoUnitsFound =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 10);
                  if ELStrings.Eq1String (CompareString,  "No errors ") or
                     ELStrings.Eq1String (CompareString,  "No summari") then
                     ReportFileState := ReportEndOfErrors;
                     ELStrings.CopyString (DebugMessage, "ReportEndOfErrors");
                  elsif ELStrings.Eq1String (CompareString,  "Unit name:") then
                     ReportFileState := ReportUnitNameFound;
                     ELStrings.CopyString (DebugMessage, "ReportUnitNameFound");
                  elsif ELStrings.Eq1String (CompareString, "***     Un") then
                     ReportFileState := ReportEndOfErrors;
                     ELStrings.CopyString (DebugMessage, "ReportEndOfErrors");
                  else
                     CompareString := LongStringUtilities.Section (Line, Line.Length - 9, 10);
                     if ELStrings.Eq1String (CompareString, "warning(s)") then
                        ReportFileState := ReportStartOfErrors;
                        ELStrings.CopyString (DebugMessage, "ReportStartOfErrors");
                     end if;
                  end if;
               when ReportStartOfErrors =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 4);
                  if ELStrings.Eq1String (CompareString, "Line") then
                     ReportFileState := ReportLineHeaderFound;
                     ELStrings.CopyString (DebugMessage, "ReportLineHeaderFound");
                  end if;
               when ReportLineHeaderFound =>
                  ProcessErrorSourceLine (Line);
                  ReportFileState := ReportErrorSourceLineFound;
                  ELStrings.CopyString (DebugMessage, "ReportErrorSourceLineFound");
               when ReportErrorSourceLineFound =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 1);
                  if ELStrings.Eq1String (CompareString,  "^") then
                     -- Process Error Pointer Line (do nothing?)
                     ReportFileState := ReportErrorSourcePointerFound;
                     ELStrings.CopyString (DebugMessage, "ReportErrorSourcePointerFound");
                  else  -- Some errors don't have pointers
                     CompareString := LongStringUtilities.Section (Line, StartPos, 3);
                     if ELStrings.Eq1String (CompareString,  "!!!") or else
                        ELStrings.Eq1String (CompareString,  "***") or else
                        ELStrings.Eq1String (CompareString,  "---") or else
                        ELStrings.Eq1String (CompareString,  "???") then

                        ProcessFirstErrorMessageLine (Line, Lookahead, SavedErrorLink);

                        ReportFileState := ReportErrorMessageFound;
                        ELStrings.CopyString (DebugMessage, "ReportErrorMessageFound");
                     end if;
                  end if;
               when ReportErrorSourcePointerFound =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 3);
                  if ELStrings.Eq1String (CompareString,  "!!!") or else
                     ELStrings.Eq1String (CompareString,  "***") or else
                     ELStrings.Eq1String (CompareString,  "---") or else
                     ELStrings.Eq1String (CompareString,  "???") then

                     ProcessFirstErrorMessageLine (Line, Lookahead, SavedErrorLink);
                     ReportFileState := ReportErrorMessageFound;
                     ELStrings.CopyString (DebugMessage, "ReportErrorMessageFound");
                  end if;
               when ReportErrorMessageFound | ReportEndOfErrors =>
                  if StartPos > 10 then -- Extra lines of the error message will be indented by 11 characters
                     ProcessNextErrorMessageLine (Line, SavedErrorLink);
                     ELStrings.CopyString (DebugMessage, "ReportNextErrorMessageLineFound");
                  else
                     CompareString := LongStringUtilities.Section (Line, StartPos, 16);
                     if ELStrings.Eq1String (CompareString,  "Source Filename:") then
                        Line := ProcessSourceFileNameLine (Line);
                        ReportFileState := ReportSourceFileStarted;
                        ELStrings.CopyString (DebugMessage, "ReportSourceFileStarted");
                     elsif ELStrings.Eq1String (CompareString,  "No summarized wa") then
                        ReportFileState := ReportEndOfErrors;
                        ELStrings.CopyString (DebugMessage, "ReportEndOfErrorsFound");
                     elsif ELStrings.Eq1String (CompareString,  "--End of file---") then
                        ReportFileState := ReportEndOfReportFile;
                        ELStrings.CopyString (DebugMessage, "ReportEndOfReportFile");

                     elsif ELStrings.Eq1String (CompareString,  "Expected message") then
                        ReportFileState := ReportJustificationsSummaryFound;
                        ELStrings.CopyString (DebugMessage, "ReportJustificationsSummaryFound");

                     else -- next error source line found, next error found or
                        -- summarized warnings
                        CompareString := LongStringUtilities.Section (Line, StartPos, 3);
                        if ELStrings.Eq1String (CompareString,  "!!!") or else
                           ELStrings.Eq1String (CompareString,  "***") or else
                           ELStrings.Eq1String (CompareString,  "---") or else
                           ELStrings.Eq1String (CompareString,  "???") then

                           ProcessFirstErrorMessageLine (Line, Lookahead, SavedErrorLink);
                           ReportFileState := ReportErrorMessageFound;
                           ELStrings.CopyString (DebugMessage, "ReportErrorMessageFound");
                        else  -- error source line or summarized warnings
                           if Line.Length > 9 then
                              CompareString := LongStringUtilities.Section (Line, Line.Length - 9, 10);
                              if ELStrings.Eq1String (CompareString, "omprising:") then
                                 ReportFileState := ReportSummarizedWarningsFound;
                              else
                                 ReportFileState := ReportErrorSourceLineFound;
                              end if;
                           else
                              ReportFileState := ReportErrorSourceLineFound;
                           end if;
                           if ReportFileState = ReportSummarizedWarningsFound then
                              ELStrings.CopyString (DebugMessage, "ReportSummarizedWarningsFound");
                           else
                              ProcessErrorSourceLine (Line);
                              ELStrings.CopyString (DebugMessage, "ReportNextErrorSourceLineFound");
                           end if;
                        end if;
                     end if;
                  end if;
               when ReportJustificationsSummaryFound =>
                     CompareString := LongStringUtilities.Section (Line, StartPos, 16);
                     if ELStrings.Eq1String (CompareString,  "Source Filename:") then
                        Line := ProcessSourceFileNameLine (Line);
                        ReportFileState := ReportSourceFileStarted;
                        ELStrings.CopyString (DebugMessage, "ReportSourceFileStarted");
                     elsif ELStrings.Eq1String (CompareString,  "No summarized wa") then
                        ReportFileState := ReportEndOfErrors;
                        ELStrings.CopyString (DebugMessage, "ReportEndOfErrorsFound");
                     elsif ELStrings.Eq1String (CompareString,  "--End of file---") then
                        ReportFileState := ReportEndOfReportFile;
                        ELStrings.CopyString (DebugMessage, "ReportEndOfReportFile");
                     else
                        -- Here, we could process the justification table line to include
                        -- HTML links the relevant source files, lines of code, and error
                        -- messages.  TBD!
                        -- Both "Brief" and "Full" justifications mode need to be dealt with here.
                        null;
                     end if;

               when ReportSummarizedWarningsFound =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 7);
                  if ELStrings.Eq1String (CompareString, "(*Note:") then
                     ReportFileState := ReportEndOfErrors;
                     ELStrings.CopyString (DebugMessage, "ReportEndOfErrorsFound");
                  elsif ELStrings.Eq1String (CompareString, "Source ") then
                     Line := ProcessSourceFileNameLine (Line);
                     ReportFileState := ReportSourceFileStarted;
                     ELStrings.CopyString (DebugMessage, "ReportSourceFileStarted");
                  end if;
               when ReportBlankAfterErrorFound |
                    ReportEndOfReportFile =>
                  ScreenEcho.Put_Line ("An error occurred during HTML report file generation: Invalid ReportFileState");
            end case;

            if CommandLineData.Content.Debug.HTML then
               ELStrings.PutLine (SPARK_IO.Standard_Output, DebugMessage);  -- Line used for debugging.
            end if;

         end if;
      end ProcessReportLine;

   begin  -- GenReportHTML

      -- Do nothing if previous HTML generation failed in some way.
      if GenerateHTML then

         -- Inform user that HTML generation is taking place.
         EStrings.CopyString (HTMLMessage, "Generating report file HTML");
         if CommandLineData.Content.Echo and not CommandLineData.Content.Brief then
            ScreenEcho.Echo (HTMLMessage);
         end if;

         -- Initialise SavedErrorLink and SavedListingFile.
         SavedErrorLink   := ELStrings.EmptyString;
         SavedListingFile := ELStrings.EmptyString;

         -- Initialise file handles.
         HTMLReportFile  := SPARK_IO.Null_File;
         PlainReportFile := SPARK_IO.Null_File;

         -- Generate filename of the form HTML/<report_filename>.htm
         PlainReportFileName := FileSystem.JustFile (CommandLineData.Content.ReportFileName, True);
         HTMLReportFileName := Translate (ELStrings.ToExaminerLongString (PlainReportFileName), '.', '_');
         ELStrings.AppendString (HTMLReportFileName, ".htm");
         HTMLReportFileName := FileSystem.CaseOfFilesForCreateL (HTMLFileName (HTMLReportFileName));

         -- Create HTML report file.
         SPARK_IO.Create (File          => HTMLReportFile,
                          Name_Length   => HTMLReportFileName.Length,
                          Name_Of_File  => HTMLReportFileName.Content,
                          Form_Of_File  => "",
                          Status        => HTMLReportFileCreatedOK);

         -- Check for errors.  Stop HTML generation if create failed.
         if HTMLReportFileCreatedOK /= SPARK_IO.Ok then

            ScreenEcho.Put_Line ("An error occurred while creating the HTML report file.");
            ScreenEcho.Put_Line ("No further HTML generation will occur.");
            GenerateHTML := False;

         else  -- file created successfully.

            -- Open report file for input
            CommandLineData.Normalize_FileName_To_Output_Directory (PlainReportFileName);

            -- Open report file for input
            SPARK_IO.Open (File         => PlainReportFile,
                           Mode_Of_File => SPARK_IO.In_File,
                           Name_Length  => PlainReportFileName.Length,
                           Name_Of_File => PlainReportFileName.Content,
                           Form_Of_File => "",
                           Status       => PlainReportFileOpenOK);

            -- Check for errors.  Stop HTML generation if open failed.
            if PlainReportFileOpenOK /= SPARK_IO.Ok then

               ScreenEcho.Put_Line ("An error occurred while opening the report file for HTML generation.");
               ScreenEcho.Put_Line ("No further HTML generation will occur.");
               GenerateHTML := False;

            else  -- file opened successfully.

               WriteHTMLReportHeader;

               -- Fill LineBuffer and LookaheadBuffer
               if not SPARK_IO.End_Of_File (PlainReportFile) then
                  ELStrings.GetLine (File => PlainReportFile,
                                               EStr => LineBuffer);
                  LineBuffer := ConvertSpecialHTMLChars (LineBuffer);
                  if not SPARK_IO.End_Of_File (PlainReportFile) then
                     ELStrings.GetLine (File => PlainReportFile,
                                                  EStr => LookaheadBuffer);
                     LookaheadBuffer := ConvertSpecialHTMLChars (LookaheadBuffer);

                     -- Process first line
                     ProcessReportLine (LineBuffer, LookaheadBuffer);

                     -- Write line and process rest of file.
                     loop
                        ELStrings.PutLine (File => HTMLReportFile,
                                                     EStr => LineBuffer);

                        if SPARK_IO.End_Of_File (PlainReportFile) then

                           -- Process and output the lookahead buffer.
                           --# accept Flow, 10, SavedErrorLink, "Expected ineffective assignment to SavedErrorLink" &
                           --#        Flow, 10, SavedListingFile, "Expected ineffective assignment to SavedListingFile" &
                           --#        Flow, 10, ReportFileState, "Expected ineffective assignment to ReportFileState";
                           ProcessReportLine (LookaheadBuffer,
                                              ELStrings.EmptyString);  -- Flow errors expected
                           --# end accept;
                           -- this is the last call and so the saved values will not be used.
                           ELStrings.PutLine (File => HTMLReportFile,
                                                        EStr => LookaheadBuffer);
                           exit;
                        end if;

                        LineBuffer := LookaheadBuffer;

                        ELStrings.GetLine (File => PlainReportFile,
                                                     EStr => LookaheadBuffer);
                        LookaheadBuffer := ConvertSpecialHTMLChars (LookaheadBuffer);

                        ProcessReportLine (LineBuffer, LookaheadBuffer);

                     end loop;

                  end if;

               end if;

               WriteHTMLReportFooter;

               -- Close input report file.
               --# accept Flow, 10, PlainReportFile, "Expected ineffective assignment to PlainReportFile";
               SPARK_IO.Close (File   => PlainReportFile,  -- Flow error expected
                               Status => PlainReportFileClosedOK);
               --# end accept;


               -- Check for errors.  Stop HTML generation if close failed.
               if PlainReportFileClosedOK /= SPARK_IO.Ok then
                  ScreenEcho.Put_Line ("An error occurred while closing the report file after HTML generation.");
                  ScreenEcho.Put_Line ("No further HTML generation will occur.");
                  GenerateHTML := False;
               end if;
               --  We don't use an else here as we need to try and close the HTML file too.

               -- Close HTML output file.
               --# accept Flow, 10, HTMLReportFile, "Expected ineffective assignment to HTMLReportFile";
               SPARK_IO.Close (File   => HTMLReportFile,  -- Flow error expected
                               Status => HTMLReportFileClosedOK);
               --# end accept;

               -- Check for errors.  Stop HTML generation if close failed.
               if HTMLReportFileClosedOK /= SPARK_IO.Ok then
                  ScreenEcho.Put_Line ("An error occurred while closing the HTML report file after HTML generation.");
                  ScreenEcho.Put_Line ("No further HTML generation will occur.");
                  GenerateHTML := False;
               end if;

            end if;

         end if;

      end if;
      --# accept Flow, 601, GenerateHTML, HTMLWorkDir, "False coupling in SPARK_IO" &
      --#        Flow, 601, GenerateHTML, SparkWorkDir, "False coupling in SPARK_IO";
   end GenReportHTML;  -- Flow errors expected due to false coupling in SPARK_IO.

   -----------------------------------------------------------------------------
   --
   --  -------------
   --  GenListingHTML
   --  -------------
   --
   --  This procedure generates the HTMLified listing file from the plain text
   --  listing file.
   --
   --  It is assumed that the HTML output flag has already been tested.  The call to
   --  GenListingHTML should look something like:
   --
   --    if CommandLineData.Content.HTML then
   --       SparkHTML.GenListingHTML;
   --    end if;
   --
   --  The checking of the GenerateHTML flag is done internally.
   --
   --  If the GenerateHTML flag is false this procedure does nothing, otherwise,
   --  it does exactly the following:
   --
   --     - creates an HTML file (named <lis_file>.htm where <lis_file> is the name
   --       of the listing file associated with the file descriptor passed as a
   --       parameter with all '.' characters changed to '_' characters;
   --     - processes each line of the plain text listing file using the
   --       ProcessListingLine procedure (the functionality of this is described
   --       at the declaration of ProcessListingLine);
   --     - writes the processed lines to the HTML listing file;
   --
   --  Error trapping:
   --
   --  Incorrect generation of a listing files should not affect further HTML generation.
   --  In fact, it is better that we try and generate HTML for as many listing files as
   --  possible.  So if HTML generation fails in this procedure the GenerateHTML flag is
   --  not set to false.
   --
   procedure GenListingHTML (FileDescriptor : in ContextManager.FileDescriptors)
   is

      Message : EStrings.T;

      SavedErrorLink : ELStrings.T;

      HTMLListingFile  : SPARK_IO.File_Type;
      PlainListingFile : SPARK_IO.File_Type;

      HTMLListingFileName   : ELStrings.T;
      PlainListingFileName  : EStrings.T;
      EchoedListingFileName : EStrings.T;

      HTMLListingFileCreatedOK : SPARK_IO.File_Status;
      PlainListingFileOpenOK   : SPARK_IO.File_Status;

      HTMLListingFileClosedOK  : SPARK_IO.File_Status;
      PlainListingFileClosedOK : SPARK_IO.File_Status;

      LineBuffer      : ELStrings.T;
      LookaheadBuffer : ELStrings.T;

      ListingFileState : ListingFileStates := ListingJustStarted;

      --
      -- Subprograms
      --
      -- This sub-procedure writes HTML content to the start of the listing file.
      -- It assumes that the file handle HTMLListingFile is the open HTML
      -- listing file.  This procedure will not open or close the file.
      --
      -- The HTML written specifies the title of the page (using the filename
      -- specified as a parameter) and some formatting tags.  The formatting
      -- is <PRE> (pre-processed text) which displays text exactly as given,
      -- and <TT> which sets the typewriter-text font; I use this because it
      -- usually results in a fixed-width font being used.
      --
      -- When writing the end of the listing file these tags need to be closed,
      -- as do the <BODY> and <HTML> tags.  This should be done by calling
      -- the WriteHTMLListingFooter procedure.
      --
      -- Error trapping:
      --
      -- All error trapping is performed within SPARK_IO.
      -- The SPARK_IO routines called do not return status parameters and so
      -- no error trapping can be done here.

      procedure WriteHTMLListingHeader (FileName : in EStrings.T)
         --# global in     CommandLineData.Content;
         --#        in     HTMLListingFile;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives SPARK_IO.FILE_SYS from *,
         --#                                CommandLineData.Content,
         --#                                FileName,
         --#                                HTMLListingFile;

      is
         LocalFileName : EStrings.T;
      begin
         LocalFileName := FileName;

         SPARK_IO.Put_Line (HTMLListingFile,
                            "<!DOCTYPE HTML PUBLIC ""-//W3C//DTD HTML 4.0 Transitional//EN""", 61);
         SPARK_IO.Put_Line (HTMLListingFile,
                            "            ""http://www.w3.org/TR/REC-html40/loose.dtd"">", 56);
         SPARK_IO.Put_Line (HTMLListingFile, "<html>", 6);
         SPARK_IO.Put_Line (HTMLListingFile, "  <head>", 8);
         SPARK_IO.Put_String (HTMLListingFile, "    <title>SPARK Examiner HTML Listing File: ", 45);
         if CommandLineData.Content.PlainOutput then
            LocalFileName := EStrings.LowerCase (FileSystem.JustFile (LocalFileName, True));
         end if;
         EStrings.PutString (HTMLListingFile, LocalFileName);
         SPARK_IO.Put_Line (HTMLListingFile, "</title>", 8);
         SPARK_IO.Put_Line (HTMLListingFile, "  </head>", 9);
         SPARK_IO.Put_Line (HTMLListingFile, "  <body>", 8);
         SPARK_IO.Put_Line (HTMLListingFile, "    <pre>", 9);
         SPARK_IO.Put_Line (HTMLListingFile, "      <tt>", 10);

      end WriteHTMLListingHeader;


      -- This subprocedure writes HTML content to the end of the listing file.
      -- It assumes that the file handle HTMLListingFile is the open HTML
      -- listing file.  This procedure will not open or close the file.
      --
      -- The HTML written closes all the formatting tags that were opened
      -- by the call to WriteHTMLListingHeader.
      --
      -- Error trapping:
      --
      -- All error trapping is performed within SPARK_IO.
      -- The SPARK_IO routines called do not return status parameters and so
      -- no error trapping can be done here.

      procedure WriteHTMLListingFooter
         --# global in     HTMLListingFile;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives SPARK_IO.FILE_SYS from *,
         --#                                HTMLListingFile;

      is

      begin

         SPARK_IO.Put_Line (HTMLListingFile, "      </tt>", 11);
         SPARK_IO.Put_Line (HTMLListingFile, "    </pre>", 10);
         SPARK_IO.Put_Line (HTMLListingFile, "  </body>", 9);
         SPARK_IO.Put_Line (HTMLListingFile, "</html>", 7);

      end WriteHTMLListingFooter;

      -- This procedure is used to convert a line of the listing file into HTML.
      -- It is effectively a parser for the listing file.
      --
      -- The procedure design is based on a state machine.  The global variable
      -- ListingFileState records our current location within the listing file (in
      -- terms of what information has already been read).
      --
      -- Given a line, we can determine the line's meaning from our current state
      -- and the contents of the line (all blank lines are ignored).
      --
      -- We can use this method to parse almost all the information in the source   --***
      -- file.  Those bits that we don't parse are usually optional (such as the    --***
      -- flag "rtc" in the options list) and require no translation to HTML anyway. --***
      --
      -- Once the procedure understands what a line represents it updates the
      -- ListingFileState and processes the line by calling an appropriate
      -- subroutine.
      --
      -- The procedure contains a debugging feature which reports each line that it
      -- finds to the screen along with a message for each line that it recognises.
      -- This only happens if the -debug switch is given on the commandline.

      procedure ProcessListingLine (Line      : in out ELStrings.T;
                                    Lookahead : in     ELStrings.T)
         --# global in     CommandLineData.Content;
         --#        in out ListingFileState;
         --#        in out SavedErrorLink;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives Line,
         --#         SavedErrorLink    from Line,
         --#                                ListingFileState,
         --#                                Lookahead,
         --#                                SavedErrorLink &
         --#         ListingFileState  from *,
         --#                                Line &
         --#         SPARK_IO.FILE_SYS from *,
         --#                                CommandLineData.Content,
         --#                                Line,
         --#                                ListingFileState,
         --#                                Lookahead;
      is

         StartPos : ELStrings.Positions;
         CompareString : ELStrings.T;
         DebugMessage : ELStrings.T;

         -- This procedure processes a listing source line.  This line should
         -- consist of some spaces followed by a number followed by some more
         -- spaces and the text of the source line.
         --
         -- The line number is extracted and put into an HTML tag of the form
         -- <A NAME="lineXXX">XXX</A> where XXX is the line number.  The rest of the
         -- source line is unchanged.

         procedure ProcessListingSourceLine (Line : in out ELStrings.T)
            --# derives Line from *;
         is

            OutString : ELStrings.T;
            I         : ELStrings.Positions;
            LineNo    : ELStrings.T;
            Link      : ELStrings.T;

         begin
            OutString := ELStrings.EmptyString;
            I         := ELStrings.Positions'First;
            LineNo     := ELStrings.EmptyString;

            -- Copy up to the first non-space.
            loop
               exit when Line.Content (I) /= ' ';
               AppendCharWithoutCheck (OutString, Line.Content (I));
               I := I + 1;
            end loop;

            -- Copy digits to LineNo
            while Digit (Line.Content (I)) loop
               AppendCharWithoutCheck (LineNo, Line.Content (I));
               I := I + 1;
            end loop;

            if LineNo.Length > 0 then
               -- Create anchor based on number
               ELStrings.CopyString (Link, "<A NAME=""line");
               ELStrings.AppendExaminerLongString (Link, LineNo);
               ELStrings.AppendString (Link, """>");
               ELStrings.AppendExaminerLongString (Link, LineNo);
               ELStrings.AppendString (Link, "</A>");

               -- Append link to OutString
               ELStrings.AppendExaminerLongString (OutString, Link);

            end if;

            -- Append rest of line
            ELStrings.AppendExaminerLongString (
               OutString, LongStringUtilities.Section (Line, I, (Line.Length - I) + 1));

            Line := OutString;

         end ProcessListingSourceLine;

      begin  -- ProcessListingLine
         if Line.Length > 0 then

            StartPos := FirstChar (Line);
            DebugMessage := ELStrings.EmptyString;

            if CommandLineData.Content.Debug.HTML then
               ELStrings.PutLine (SPARK_IO.Standard_Output, Line);  -- Line used for debugging.
            end if;

            case ListingFileState is
               when ListingJustStarted =>
                  if ELStrings.Eq1String (LongStringUtilities.Section (Line, StartPos, 5),
                                                    "*****") then
                     ListingFileState := ListingBannerStarted;
                     ELStrings.CopyString (DebugMessage,
                                                     "ListingBannerStarted");
                  end if;
               when ListingBannerStarted =>
                  if ELStrings.Eq1String (LongStringUtilities.Section (Line, StartPos, 5),
                                                    "*****") then
                     ListingFileState := ListingBannerFinished;
                     ELStrings.CopyString (DebugMessage, "ListingBannerFinished");
                  end if;
               when ListingBannerFinished =>
                  if ELStrings.Eq1String (LongStringUtilities.Section (Line, StartPos, 7),
                                                    "DATE : ") then
                     ListingFileState := ListingDateFound;
                     ELStrings.CopyString (DebugMessage, "ListingDateFound");
                  end if;
               when ListingDateFound =>
                  if ELStrings.Eq1String (LongStringUtilities.Section (Line, StartPos, 4),
                                                    "Line") then
                     ListingFileState := ListingLineHeadingFound;
                     ELStrings.CopyString (DebugMessage, "ListingLineHeadingFound");
                  end if;
               when ListingLineHeadingFound | ListingSourceLineFound =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 1);
                  if ELStrings.Eq1String (CompareString,  "^") then
                     -- Process Error Pointer Line (do nothing?)
                     ListingFileState := ListingErrorSourcePointerFound;
                     ELStrings.CopyString (DebugMessage, "ListingErrorSourcePointerFound");
                  else  -- Some errors don't have pointers
                     CompareString := LongStringUtilities.Section (Line, StartPos, 3);
                     if ELStrings.Eq1String (CompareString,  "!!!") or else
                        ELStrings.Eq1String (CompareString,  "***") or else
                        ELStrings.Eq1String (CompareString,  "---") or else
                        ELStrings.Eq1String (CompareString,  "???") then

                        ProcessFirstErrorMessageLine (Line, Lookahead, SavedErrorLink);

                        ListingFileState := ListingErrorMessageFound;
                        ELStrings.CopyString (DebugMessage, "ListingErrorMessageFound");
                     else
                        ProcessListingSourceLine (Line);
                        ListingFileState := ListingSourceLineFound;
                        ELStrings.CopyString (DebugMessage, "ListingSourceLineFound");
                     end if;
                  end if;
               when ListingErrorSourcePointerFound =>
                  CompareString := LongStringUtilities.Section (Line, StartPos, 3);
                  if ELStrings.Eq1String (CompareString,  "!!!") or else
                     ELStrings.Eq1String (CompareString,  "***") or else
                     ELStrings.Eq1String (CompareString,  "---") or else
                     ELStrings.Eq1String (CompareString,  "???") then

                     ProcessFirstErrorMessageLine (Line, Lookahead, SavedErrorLink);
                     ListingFileState := ListingErrorMessageFound;
                     ELStrings.CopyString (DebugMessage, "ListingErrorMessageFound");
                  end if;
               when ListingErrorMessageFound =>
                  if StartPos > 5 then -- Extra lines of the error message will be indented by 11 characters
                     ProcessNextErrorMessageLine (Line, SavedErrorLink);
                     ELStrings.CopyString (DebugMessage, "ListingNextErrorMessageLineFound");
                  else
                     CompareString := LongStringUtilities.Section (Line, StartPos, 3);
                     if ELStrings.Eq1String (CompareString,  "!!!") or else
                        ELStrings.Eq1String (CompareString,  "***") or else
                        ELStrings.Eq1String (CompareString,  "---") or else
                        ELStrings.Eq1String (CompareString,  "???") then

                        ProcessFirstErrorMessageLine (Line, Lookahead, SavedErrorLink);
                        ListingFileState := ListingErrorMessageFound;
                        ELStrings.CopyString (DebugMessage, "ListingErrorMessageFound");
                     else  -- error source line or summarized warnings
                        ProcessListingSourceLine (Line);
                        ListingFileState := ListingSourceLineFound;
                        ELStrings.CopyString (DebugMessage, "ListingNextSourceLineFound");
                     end if;
                  end if;
               when ListingEndOfListingFile =>
                  ScreenEcho.Put_Line ("An error occurred during HTML listing file generation: Invalid ReportFileState");
            end case;

            if CommandLineData.Content.Debug.HTML then
               ELStrings.PutLine (SPARK_IO.Standard_Output,
                                            DebugMessage);  -- Line used for debugging.
            end if;

         end if;
      end ProcessListingLine;

   begin

      if GenerateHTML then

         -- Get name of listing file.
         ContextManager.GetListingFileName (FileDescriptor, PlainListingFileName);

         if CommandLineData.Content.Echo and not CommandLineData.Content.Brief then
            -- Echo message to screen.
            EStrings.CopyString (Message, "Generating listing file HTML for ");

            if CommandLineData.Content.PlainOutput then
               EchoedListingFileName := EStrings.LowerCase (PlainListingFileName);
            else
               EchoedListingFileName := PlainListingFileName;
            end if;
            EStrings.AppendExaminerString (Message, EchoedListingFileName);
            ScreenEcho.Echo (Message);
         end if;

         -- Initialise SavedErrorLink.
         SavedErrorLink := ELStrings.EmptyString;

         -- Initialise file handles.
         HTMLListingFile  := SPARK_IO.Null_File;
         PlainListingFile := SPARK_IO.Null_File;

         -- Generate filename of the form HTML/<listing_filename>.htm
         HTMLListingFileName := Translate (
            ELStrings.ToExaminerLongString (PlainListingFileName), '.', '_');
         ELStrings.AppendString (HTMLListingFileName, ".htm");
         HTMLListingFileName := FileSystem.CaseOfFilesForCreateL (HTMLFileName (HTMLListingFileName));

         -- Create HTML listing file.
         SPARK_IO.Create (File          => HTMLListingFile,
                          Name_Length   => HTMLListingFileName.Length,
                          Name_Of_File  => HTMLListingFileName.Content,
                          Form_Of_File  => "",
                          Status        => HTMLListingFileCreatedOK);

         -- Check for errors.
         if HTMLListingFileCreatedOK /= SPARK_IO.Ok then

            ScreenEcho.Put_String ("An error occurred while creating the HTML listing file ");
            ELStrings.PutLine (SPARK_IO.Standard_Output, HTMLListingFileName);

         else  -- file created successfully.

            CommandLineData.Normalize_FileName_To_Output_Directory (PlainListingFileName);

            -- Open listing file for input
            SPARK_IO.Open (File         => PlainListingFile,
                           Mode_Of_File => SPARK_IO.In_File,
                           Name_Length  => PlainListingFileName.Length,
                           Name_Of_File => PlainListingFileName.Content,
                           Form_Of_File => "",
                           Status       => PlainListingFileOpenOK);

            -- Check for errors.
            if PlainListingFileOpenOK /= SPARK_IO.Ok then

               ScreenEcho.Put_String ("An error occurred while opening the listing file ");
               EStrings.PutString (SPARK_IO.Standard_Output, PlainListingFileName);
               ScreenEcho.Put_Line   (" for HTML generation.");

            else  -- file opened successfully.

               WriteHTMLListingHeader (PlainListingFileName);

               -- Fill LineBuffer and LookaheadBuffer
               if not SPARK_IO.End_Of_File (PlainListingFile) then
                  ELStrings.GetLine (File => PlainListingFile,
                                               EStr => LineBuffer);
                  LineBuffer := ConvertSpecialHTMLChars (LineBuffer);
                  if not SPARK_IO.End_Of_File (PlainListingFile) then
                     ELStrings.GetLine (File => PlainListingFile,
                                                  EStr => LookaheadBuffer);
                     LookaheadBuffer := ConvertSpecialHTMLChars (LookaheadBuffer);

                     -- Process first line
                     ProcessListingLine (LineBuffer, LookaheadBuffer);

                     -- Write line and process rest of file.
                     loop
                        ELStrings.PutLine (File => HTMLListingFile,
                                                     EStr => LineBuffer);

                        if SPARK_IO.End_Of_File (PlainListingFile) then

                           -- Process and output the lookahead buffer.
                           --# accept Flow, 10, ListingFileState, "Expected ineffective assignment to ListingFileState" &
                           --#        Flow, 10, SavedErrorLink, "Expected ineffective assignment to SavedErrorLink";
                           ProcessListingLine (LookaheadBuffer,
                                               ELStrings.EmptyString);  -- Flow errors expected
                           --#end accept;
                           -- this is the last call and so the saved values will not be used.
                           ELStrings.PutLine (File => HTMLListingFile,
                                                        EStr => LookaheadBuffer);
                           exit;
                        end if;

                        LineBuffer := LookaheadBuffer;

                        ELStrings.GetLine (File => PlainListingFile,
                                                     EStr => LookaheadBuffer);
                        LookaheadBuffer := ConvertSpecialHTMLChars (LookaheadBuffer);

                        ProcessListingLine (LineBuffer, LookaheadBuffer);

                     end loop;

                  end if;

               end if;

               WriteHTMLListingFooter;

               -- Close input listing file.
               --# accept Flow, 10, PlainListingFile, "Expected ineffective assignment to PlainListingFile";
               SPARK_IO.Close (File   => PlainListingFile,
                               Status => PlainListingFileClosedOK);
               --# end accept;


               -- Check for errors.
               if PlainListingFileClosedOK /= SPARK_IO.Ok then
                  ScreenEcho.Put_String ("An error occurred while closing the listing file ");
                  EStrings.PutString (SPARK_IO.Standard_Output, PlainListingFileName);
                  ScreenEcho.Put_Line (" after HTML generation.");
               end if;
               --  We don't use an else here as we need to try and close the HTML file too.

               -- Close HTML output file.
               --# accept Flow, 10, HTMLListingFile, "Expected ineffective assignment to HTMLListingFile";
               SPARK_IO.Close (File   => HTMLListingFile,
                               Status => HTMLListingFileClosedOK);
               --# end accept;

               -- Check for errors.
               if HTMLListingFileClosedOK /= SPARK_IO.Ok then
                  ScreenEcho.Put_String ("An error occurred while closing the HTML listing file ");
                  ELStrings.PutString (SPARK_IO.Standard_Output, HTMLListingFileName);
                  ScreenEcho.Put_Line (" after HTML generation.");
               end if;

            end if;

         end if;


      end if;

   end GenListingHTML;

end SparkHTML;
