-- $Id: spark_xml.ads 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with EStrings,
     ELStrings;

--# inherit Ada.Characters.Latin_1,
--#         ELStrings,
--#         EStrings,
--#         SPARK_IO;

package SPARK_XML
is


   MaxTags : constant Integer := 100;
   MaxAttributesPerTag : constant Integer := 7;
   MaxAttributes : constant Integer := MaxTags * MaxAttributesPerTag;


   subtype TagID is Integer range 0 .. MaxTags;
   subtype AttributeID is Integer range 0 .. MaxAttributes;


   NullTag : constant TagID := TagID'First;
   NullAttribute : constant AttributeID := AttributeID'First;


   -- Simple attribute types.
   type AttributeType is (ATNULL,
                          ATString,
                          ATInteger,
                          ATFloat);

   subtype XMLString is EStrings.T;
   EmptyString : constant XMLString := EStrings.EmptyString;

   subtype XMLLongString is ELStrings.T;
   EmptyLongString : constant XMLLongString := ELStrings.EmptyString;

   type SchemaStatus is (SSOK,
                         SSInvalidAttribute,
                         SSInvalidTag,
                         SSToManyAttributes,
                         SSWrongContentType,
                         SSStackFull,
                         SSStackEmpty,
                         SSTagIncomplete,
                         SSInvalidDepth,
                         SSNoSuchTag,
                         SSTagNotFound);

   type UpOrDown is (Up, Down);


   MaxTagDepth : constant Integer := 100;

   type TagDepth is range 0 .. MaxTagDepth;


   type SchemaRecord is private;
   type SchemaStateRecord is private;


   function XStr (Str : in String) return XMLString;
   function FilterString (Str : in XMLString) return XMLString;
   function FilterLongString (Str : in XMLLongString) return XMLLongString;

   ---------------------
   -- Schema creation --
   ---------------------

   procedure InitSchema (Schema : out SchemaRecord);
   --# derives Schema from ;

   procedure AddTag (Schema : in out SchemaRecord;
                     Name   : in     XMLString;
                     ID     :    out TagID);
   --# derives ID     from Schema &
   --#         Schema from *,
   --#                     Name;

   function IsNullTag (TID : in TagID) return Boolean;

   procedure AddAttributeToTag (Schema      : in out SchemaRecord;
                                TID         : in     TagID;
                                Name        : in     XMLString;
                                ContentType : in     AttributeType;
                                Required    : in     Boolean;
                                ID          :    out AttributeID;
                                Success     :    out Boolean);
   --# derives ID      from Schema &
   --#         Schema,
   --#         Success from ContentType,
   --#                      Name,
   --#                      Required,
   --#                      Schema,
   --#                      TID;



   procedure AddChildTag (Schema   : in out SchemaRecord;
                          TID      : in     TagID;
                          Child    : in     TagID;
                          Required : in     Boolean;
                          Success  :    out Boolean);
   --# derives Schema  from *,
   --#                      Child,
   --#                      Required,
   --#                      TID &
   --#         Success from Schema,
   --#                      TID;


   procedure AddCDATA (Schema : in out SchemaRecord;
                       TID    : in     TagID);
   --# derives Schema from *,
   --#                     TID;


   function CDATA (Schema : in SchemaRecord;
                   TID    : in TagID) return Boolean;



   ------------------
   -- Tag Creation --
   ------------------

   procedure InitSchemaState (SchemaState : out SchemaStateRecord);
   --# derives SchemaState from ;



   -- Opening tags:
   -- 1) Initialise the opening tag
   -- 2) Add attributes to it
   -- 3) Call OutputOpeningTag to return the string.
   procedure InitOpeningTag (Schema      : in     SchemaRecord;
                             SchemaState : in out SchemaStateRecord;
                             Name        : in     XMLString;
                             Status      : out    SchemaStatus);
   --# derives SchemaState,
   --#         Status      from Name,
   --#                          Schema,
   --#                          SchemaState;

   procedure InitOpeningTagByID (Schema      : in     SchemaRecord;
                                 SchemaState : in out SchemaStateRecord;
                                 TID         : in     TagID;
                                 Status      : out    SchemaStatus);
   --# derives SchemaState,
   --#         Status      from Schema,
   --#                          SchemaState,
   --#                          TID;

   procedure InitOpeningTagNOCHECK (SchemaState : in out SchemaStateRecord;
                                    TID         : in     TagID;
                                    Status      : out    SchemaStatus);
   --# derives SchemaState,
   --#         Status      from SchemaState,
   --#                          TID;


   procedure AddAttributeInt (Schema      : in     SchemaRecord;
                              SchemaState : in out SchemaStateRecord;
                              Name        : in     XMLString;
                              Value       : in     Integer;
                              Status      :    out SchemaStatus);
   --# derives SchemaState from *,
   --#                          Name,
   --#                          Schema,
   --#                          Value &
   --#         Status      from Name,
   --#                          Schema,
   --#                          SchemaState;


   procedure AddAttributeStr (Schema      : in     SchemaRecord;
                              SchemaState : in out SchemaStateRecord;
                              Name        : in     XMLString;
                              Value       : in     XMLString;
                              Status      :    out SchemaStatus);
   --# derives SchemaState from *,
   --#                          Name,
   --#                          Schema,
   --#                          Value &
   --#         Status      from Name,
   --#                          Schema,
   --#                          SchemaState;


   procedure OutputOpeningTag (Schema      : in     SchemaRecord;
                               SchemaState : in out SchemaStateRecord;
                               XML         :    out XMLString;
                               Depth       :    out TagDepth;
                               Status      :    out SchemaStatus);
   --# derives Depth,
   --#         SchemaState,
   --#         Status,
   --#         XML         from Schema,
   --#                          SchemaState;


   -- Closing tags
   procedure CloseTag (Schema      : in     SchemaRecord;
                       SchemaState : in out SchemaStateRecord;
                       Depth       : in     TagDepth;
                       XML         :    out XMLString;
                       Status      :    out SchemaStatus);
   --# derives SchemaState,
   --#         Status      from Depth,
   --#                          SchemaState &
   --#         XML         from Depth,
   --#                          Schema,
   --#                          SchemaState;

   procedure CloseTagByID (Schema      : in     SchemaRecord;
                           SchemaState : in out SchemaStateRecord;
                           TID         : in     TagID;
                           XML         :    out XMLString;
                           Status      :    out SchemaStatus);
   --# derives SchemaState,
   --#         Status      from SchemaState,
   --#                          TID &
   --#         XML         from Schema,
   --#                          SchemaState,
   --#                          TID;

   procedure CloseTopTagByID (Schema      : in     SchemaRecord;
                              SchemaState : in out SchemaStateRecord;
                              TID         : in     TagID;
                              XML         :    out XMLString;
                              Status      :    out SchemaStatus);
   --# derives SchemaState,
   --#         Status      from SchemaState,
   --#                          TID &
   --#         XML         from Schema,
   --#                          SchemaState,
   --#                          TID;


   procedure CloseTagByName (Schema      : in     SchemaRecord;
                             SchemaState : in out SchemaStateRecord;
                             Name        : in     XMLString;
                             XML         :    out XMLString;
                             Status      :    out SchemaStatus);
   --# derives SchemaState,
   --#         Status,
   --#         XML         from Name,
   --#                          Schema,
   --#                          SchemaState;



   -----------
   -- Debug --
   -----------

   function IsError (Error : in SchemaStatus) return Boolean;

   procedure PrintSchemaError (Error : in SchemaStatus);
   --# derives null from Error;

   procedure PrintWorkingState (Schema      : in SchemaRecord;
                                SchemaState : in SchemaStateRecord);
   --# derives null from Schema,
   --#                   SchemaState;

private







   -----------------------
   -- Schema Structures --
   -----------------------

   subtype TagAttributeArrayIndex is Integer range 0 .. MaxAttributesPerTag;

   MaxChildTags : constant Integer := 10;

   subtype ChildTagArrayIndex is Integer range 0 .. MaxChildTags;

   type ChildTag is
      record
         Child : TagID;
         Required : Boolean;
      end record;


   type TagAttributeArray is array (TagAttributeArrayIndex) of AttributeID;

   type ChildTagArray is array (ChildTagArrayIndex) of ChildTag;

   type Tag is
      record
         Name : XMLString;
         TagAttributes : TagAttributeArray;
         LastTagAttribute : TagAttributeArrayIndex;
         ChildTags : ChildTagArray;
         LastChild : ChildTagArrayIndex;
         AllowCDATA : Boolean;
      end record;

   EmptyTag : constant Tag := Tag'(Name => EStrings.EmptyString,
                                   TagAttributes => TagAttributeArray'(others => NullAttribute),
                                   LastTagAttribute => TagAttributeArrayIndex'First,
                                   ChildTags => ChildTagArray'(others => ChildTag'(Child => NullTag,
                                                                                   Required => False)),
                                   LastChild => ChildTagArrayIndex'First,
                                   AllowCDATA => False);

   type TagArrayType is array (TagID) of Tag;

   type TagList is
      record
        TagArray : TagArrayType;
        LastTag : TagID;
      end record;

   EmptyTagList : constant TagList := TagList'(TagArray =>
                                              TagArrayType'(others => EmptyTag),
                                            LastTag => 0);


   type Attribute is
      record
         Name : XMLString;
         ContentType : AttributeType;
         Required : Boolean;
      end record;

   type AttributeArrayType is array (AttributeID) of Attribute;

   type AttributeList is
      record
         AttributeArray : AttributeArrayType;
         LastAttribute : AttributeID;
      end record;

   EmptyAttributeList : constant AttributeList :=
     AttributeList'(AttributeArray =>
                      AttributeArrayType'(others =>
                                            Attribute'(Name => EStrings.EmptyString,
                                                       ContentType => ATNULL,
                                                       Required => False)),
                    LastAttribute => 0);


   type SchemaRecord is record
      Attributes : AttributeList;
      Tags : TagList;
   end record;

   EmptySchemaRecord : constant SchemaRecord :=
     SchemaRecord'(Attributes => EmptyAttributeList,
                   Tags => EmptyTagList);



   ----------------------------
   -- SchemaState Structures --
   ----------------------------

   -- TagStack records the hierarcy from the present tag to the root.
   -- This allows us to enforce child tag relations.
   -- If a tag is closed that is not the emmediate parent, we can itterate through the stack
   -- until we find which one it was and close all the intermediate tags.
   -- This isn't perfect, it will have problems with cycles, but will be fine with simple
   -- recursion with a single tag that can be the child of itself (A -> B -> B -> B) but not
   -- (A -> B -> A -> B ->).

   type TagStackArray is array (TagDepth) of TagID;

   type TagStackType is
      record
         Stack : TagStackArray;
         Current : TagDepth;
      end record;

   EmptyTagStack : constant TagStackType := TagStackType'(Stack => TagStackArray'(others => 0),
                                                          Current => 0);


   subtype TagCount is Integer range 0 .. 100;
   type TagCountArray is array (TagID) of TagCount;



   type WorkingAttribute is
      record
         AID : AttributeID;
         Val : XMLString;
      end record;

   type WorkingAttributeArray is array (TagAttributeArrayIndex) of WorkingAttribute;

   type WorkingTagType is
      record
         TID : TagID;
         Attribs : WorkingAttributeArray;
      end record;

   EmptyWorkingTag : constant WorkingTagType :=
     WorkingTagType'(TID => NullTag,
                     Attribs => WorkingAttributeArray'(others =>
                                                         WorkingAttribute'(AID => NullAttribute,
                                                                           Val => EStrings.EmptyString)));


   type SchemaStateRecord is record
      TagStack : TagStackType;
      WorkingTag : WorkingTagType;
   end record;

   EmptySchemaStateRecord : constant SchemaStateRecord := SchemaStateRecord'(TagStack => EmptyTagStack,
                                                                             WorkingTag => EmptyWorkingTag);


end SPARK_XML;
