-- $Id: spark.adb 12696 2009-03-12 13:14:05Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with Ada.Exceptions,
     Ada.Command_Line,
     Cells,
     CommandLineData,
     GNAT.Traceback.Symbolic,
     Heap,
     ScreenEcho,
     SparkMain;

procedure Spark
is
   ---------------------------------------------------------------
   -- Some hosts (notably Apple's OSX/Darwin )place an arbitrary
   -- and unchangeable limit on the stack usage of the environment
   -- task, which is not sufficient to run the Examiner, where
   -- the flow analyser Heap and VCG Heap can be large.
   --
   -- Therefore, we start a local task to do all the
   -- hard work, since such tasks are not subject to such
   -- limits.
   ---------------------------------------------------------------

   function Calculate_Storage_Size return Natural
   is
      Page_Size : constant := 2**12; -- 4 kbytes

      --                                    MBytes
      Other_Stack_Usage : constant := 64 * (2 ** 20);

      Total_Size : Natural;
   begin
      Total_Size := Heap.HeapSize + Cells.VCG_Heap_Size + Other_Stack_Usage;
      return ((Total_Size + Page_Size) / Page_Size) * Page_Size;
   end Calculate_Storage_Size;

   task RunSpark is
      pragma Storage_Size (Calculate_Storage_Size);
   end RunSpark;

   task body RunSpark
   is
   begin
      SparkMain;
   exception
      when E : others =>
         -- Traceback information will vary between different platforms. Thus it
         -- is suppressed where operating in plain mode.
         if CommandLineData.Content.PlainOutput then
            ScreenEcho.New_Line (1);
            ScreenEcho.Put_Line ("Unhandled Exception in Spark.");
            ScreenEcho.Put_Line ("Operating in plain mode: Exception and traceback information suppressed.");
         else
            ScreenEcho.New_Line (1);
            ScreenEcho.Put_Line ("Unhandled Exception in Spark.  Exception information:");
            ScreenEcho.Put_Line (Ada.Exceptions.Exception_Information (E));
            ScreenEcho.Put_Line ("Traceback:");
            ScreenEcho.Put_Line (GNAT.Traceback.Symbolic.Symbolic_Traceback (E));
         end if;

         Ada.Command_Line.Set_Exit_Status (9);
   end RunSpark;

begin
   null;
end Spark;
