-- $Id: sem-compunit-wf_subtype_declaration.adb 12351 2009-02-02 15:03:51Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.CompUnit)

procedure wf_subtype_declaration (Node  : in STree.SyntaxNode;
                                  Scope : in Dictionary.Scopes)
is
   type RealType is (IsFloating, IsFixed);

   TypeNode,
   IdentNode,
   ConstraintNode  : STree.SyntaxNode;
   ConstraintType  : SPSymbols.SPSymbol;
   IdStr           : LexTokenManager.LexString;
   ConstraintFound,
   OkToAdd         : Boolean;
   AllIndexesOK    : Boolean;
   SubtypeSym,
   TypeSym         : Dictionary.Symbol;
   SubtypeIsStatic : Boolean;
   Lower,
   Upper,
   Accuracy        : LexTokenManager.LexString;

   -------------------------------------------------------------------

   function IsNamedAssociation (Node : STree.SyntaxNode) return Boolean
   --# global in STree.Table;
   is
   begin
      return SyntaxNodeType (Child_Node (Child_Node (Node))) = SPSymbols.named_argument_association;
   end IsNamedAssociation;

   -------------------------------------------------------------------

   procedure WalkExpressionAsTypeMark (ExpNode       : in     STree.SyntaxNode;
                                       Scope         : in     Dictionary.Scopes;
                                       ConstraintSym :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#           out AggregateStack.State;
   --#           out TheHeap;
   --# derives AggregateStack.State,
   --#         ConstraintSym,
   --#         Dictionary.Dict,
   --#         LexTokenManager.StringTable,
   --#         STree.Table,
   --#         TheHeap                     from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ExpNode,
   --#                                          LexTokenManager.StringTable,
   --#                                          Scope,
   --#                                          STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          ExpNode,
   --#                                          LexTokenManager.StringTable,
   --#                                          Scope,
   --#                                          SPARK_IO.FILE_SYS,
   --#                                          STree.Table &
   --#         Statistics.TableUsage       from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ExpNode,
   --#                                          LexTokenManager.StringTable,
   --#                                          Scope,
   --#                                          STree.Table;
   is
      UnwantedSeq         : SeqAlgebra.Seq;
      UnusedComponentData : ComponentManager.ComponentData;
      ConstraintRecord    : ExpRecord;

   begin
      Heap.Initialize (TheHeap);
      SeqAlgebra.CreateSeq (TheHeap, UnwantedSeq);
      ComponentManager.Initialise (UnusedComponentData);
      --# accept Flow, 10, UnusedComponentData, "Expected ineffective assignment";
      WalkExpression (ExpNode,
                      Scope,
                      Dictionary.GetUnknownTypeMark,
                      False,
                        --to get
                      ConstraintRecord,
                      UnwantedSeq,
                      UnusedComponentData);
      --# end accept;
      SeqAlgebra.DisposeOfSeq (TheHeap, UnwantedSeq);

      if ConstraintRecord.IsARange then
         ConstraintSym := ConstraintRecord.TypeSymbol;
      else
         ErrorHandler.SemanticError (95,
                                     ErrorHandler.NoReference,
                                     NodePosition (ExpNode),
                                     LexTokenManager.NullString);
         ConstraintSym := Dictionary.GetUnknownTypeMark;
      end if;
      Heap.ReportUsage (TheHeap);
   end WalkExpressionAsTypeMark;

   -------------------------------------------------------------------

   procedure CheckIndexConstraints (IsString     : in     Boolean;
                                    AllIndexesOK :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ConstraintNode;
   --#        in     Scope;
   --#        in     SubTypeSym;
   --#        in     TypeSym;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#           out AggregateStack.State;
   --#           out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.StringTable,
   --#         STree.Table,
   --#         TheHeap                     from CommandLineData.Content,
   --#                                          ConstraintNode,
   --#                                          Dictionary.Dict,
   --#                                          LexTokenManager.StringTable,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          SubTypeSym,
   --#                                          TypeSym &
   --#         AllIndexesOK                from CommandLineData.Content,
   --#                                          ConstraintNode,
   --#                                          Dictionary.Dict,
   --#                                          IsString,
   --#                                          LexTokenManager.StringTable,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          SubTypeSym,
   --#                                          TypeSym &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS           from CommandLineData.Content,
   --#                                          ConstraintNode,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          IsString,
   --#                                          LexTokenManager.StringTable,
   --#                                          Scope,
   --#                                          SPARK_IO.FILE_SYS,
   --#                                          STree.Table,
   --#                                          SubTypeSym,
   --#                                          TypeSym &
   --#         Statistics.TableUsage       from *,
   --#                                          CommandLineData.Content,
   --#                                          ConstraintNode,
   --#                                          Dictionary.Dict,
   --#                                          LexTokenManager.StringTable,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          SubTypeSym,
   --#                                          TypeSym;
   is
      IndexIterator         : Dictionary.Iterator;
      CurrentConstraintNode : STree.SyntaxNode;
      IndexSym,
      ConstraintSym         : Dictionary.Symbol;
      IndexesFinished,
      ConstraintsFinished   : Boolean;

      ResultOfFirstCheck    : Maths.Value;
      ResultOfLastCheck     : Maths.Value;

      ConstraintFirst,
      ConstraintLast        : LexTokenManager.LexString;

      procedure GetFirstIndex (IndexSym        : out Dictionary.Symbol;
                               IndexesFinished : out Boolean)
      --# global in     Dictionary.Dict;
      --#        in     TypeSym;
      --#           out IndexIterator;
      --# derives IndexesFinished,
      --#         IndexIterator,
      --#         IndexSym        from Dictionary.Dict,
      --#                              TypeSym;
      is
      begin
         IndexIterator := Dictionary.FirstArrayIndex (TypeSym);
         if Dictionary.IsNullIterator (IndexIterator) then
            IndexSym := Dictionary.NullSymbol;
            IndexesFinished := True;
         else
            IndexSym := Dictionary.CurrentSymbol (IndexIterator);
            IndexesFinished := False;
         end if;
      end GetFirstIndex;

      procedure GetNextIndex (IndexSym        : out Dictionary.Symbol;
                              IndexesFinished : out Boolean)
      --# global in     Dictionary.Dict;
      --#        in out IndexIterator;
      --# derives IndexesFinished,
      --#         IndexIterator,
      --#         IndexSym        from Dictionary.Dict,
      --#                              IndexIterator;
      is
      begin
         IndexIterator := Dictionary.NextSymbol (IndexIterator);
         if Dictionary.IsNullIterator (IndexIterator) then
            IndexSym := Dictionary.NullSymbol;
            IndexesFinished := True;
         else
            IndexSym := Dictionary.CurrentSymbol (IndexIterator);
            IndexesFinished := False;
         end if;
      end GetNextIndex;

      procedure GetFirstConstraint (ConstraintSym       : out Dictionary.Symbol;
                                    ConstraintsFinished : out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     ConstraintNode;
      --#        in     Scope;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LexTokenManager.StringTable;
      --#        in out SPARK_IO.FILE_SYS;
      --#        in out Statistics.TableUsage;
      --#        in out STree.Table;
      --#           out AggregateStack.State;
      --#           out CurrentConstraintNode;
      --#           out TheHeap;
      --# derives AggregateStack.State,
      --#         ConstraintSym,
      --#         Dictionary.Dict,
      --#         LexTokenManager.StringTable,
      --#         STree.Table,
      --#         TheHeap                     from CommandLineData.Content,
      --#                                          ConstraintNode,
      --#                                          Dictionary.Dict,
      --#                                          LexTokenManager.StringTable,
      --#                                          Scope,
      --#                                          STree.Table &
      --#         ConstraintsFinished         from  &
      --#         CurrentConstraintNode       from ConstraintNode,
      --#                                          STree.Table &
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS           from CommandLineData.Content,
      --#                                          ConstraintNode,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          LexTokenManager.StringTable,
      --#                                          Scope,
      --#                                          SPARK_IO.FILE_SYS,
      --#                                          STree.Table &
      --#         Statistics.TableUsage       from *,
      --#                                          CommandLineData.Content,
      --#                                          ConstraintNode,
      --#                                          Dictionary.Dict,
      --#                                          LexTokenManager.StringTable,
      --#                                          Scope,
      --#                                          STree.Table;
      is
      begin
         ConstraintsFinished := False;
         CurrentConstraintNode := Child_Node (Child_Node (ConstraintNode));
         while SyntaxNodeType (CurrentConstraintNode) /= SPSymbols.expression
         loop
            CurrentConstraintNode := Child_Node (CurrentConstraintNode);
         end loop;

         WalkExpressionAsTypeMark (CurrentConstraintNode,
                                   Scope,
                                    --to get
                                   ConstraintSym);
      end GetFirstConstraint;

      procedure GetNextConstraint (ConstraintSym       : out Dictionary.Symbol;
                                   ConstraintsFinished : out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     Scope;
      --#        in out AggregateStack.State;
      --#        in out CurrentConstraintNode;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LexTokenManager.StringTable;
      --#        in out SPARK_IO.FILE_SYS;
      --#        in out Statistics.TableUsage;
      --#        in out STree.Table;
      --#        in out TheHeap;
      --# derives AggregateStack.State,
      --#         Dictionary.Dict,
      --#         LexTokenManager.StringTable,
      --#         Statistics.TableUsage,
      --#         STree.Table,
      --#         TheHeap                     from *,
      --#                                          CommandLineData.Content,
      --#                                          CurrentConstraintNode,
      --#                                          Dictionary.Dict,
      --#                                          LexTokenManager.StringTable,
      --#                                          Scope,
      --#                                          STree.Table &
      --#         ConstraintsFinished,
      --#         CurrentConstraintNode       from CurrentConstraintNode,
      --#                                          STree.Table &
      --#         ConstraintSym               from CommandLineData.Content,
      --#                                          CurrentConstraintNode,
      --#                                          Dictionary.Dict,
      --#                                          LexTokenManager.StringTable,
      --#                                          Scope,
      --#                                          STree.Table &
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS           from CommandLineData.Content,
      --#                                          CurrentConstraintNode,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          LexTokenManager.StringTable,
      --#                                          Scope,
      --#                                          SPARK_IO.FILE_SYS,
      --#                                          STree.Table;
      is
      begin
         CurrentConstraintNode := Next_Sibling (ParentNode (CurrentConstraintNode));
         if CurrentConstraintNode = STree.NullNode then
            ConstraintsFinished := True;
            ConstraintSym := Dictionary.NullSymbol;
         else
            ConstraintsFinished := False;
            WalkExpressionAsTypeMark (CurrentConstraintNode,
                                      Scope,
                                       --to get
                                      ConstraintSym);
         end if;
      end GetNextConstraint;

   begin -- CheckIndexConstraints
      AllIndexesOK := True;
      GetFirstIndex (IndexSym, IndexesFinished);
      GetFirstConstraint (ConstraintSym, ConstraintsFinished);
      while not (IndexesFinished or ConstraintsFinished) loop
         if not Dictionary.CompatibleTypes (Scope, IndexSym, ConstraintSym) then
            ErrorHandler.SemanticErrorSym2 (107,
                                            ErrorHandler.NoReference,
                                            NodePosition (CurrentConstraintNode),
                                            ConstraintSym,
                                            IndexSym,
                                            Scope);
            AllIndexesOK := False;
         end if;

         ConstraintFirst := Dictionary.GetScalarAttributeValue (False,
                                                                LexTokenManager.FirstToken,
                                                                ConstraintSym);
         ConstraintLast  := Dictionary.GetScalarAttributeValue (False,
                                                                LexTokenManager.LastToken,
                                                                ConstraintSym);

         -- Check that ConstraintSym'First is OK wrt IndexSym'First
         --# accept F, 41,         "Expect stable expression here";
         if IsString then
            if ConstraintFirst /= LexTokenManager.OneValue then
               ErrorHandler.SemanticError (417,
                                           ErrorHandler.NoReference,
                                           NodePosition (ConstraintNode),
                                           LexTokenManager.NullString);
            end if;
         else
            ConstraintCheck (Maths.ValueRep (ConstraintFirst),
                             ResultOfFirstCheck,
                             False, -- cannot be in annotation here
                             IndexSym,
                             NodePosition (CurrentConstraintNode));
            if ResultOfFirstCheck = Maths.NoValue then
               AllIndexesOK := False;
            end if;

         end if;
         --# end accept;

         -- Check that ConstraintSym'Last is OK wrt IndexSym'Last
         ConstraintCheck (Maths.ValueRep (ConstraintLast),
                          ResultOfLastCheck,
                          False, -- cannot be in annotation here
                          IndexSym,
                          NodePosition (CurrentConstraintNode));
         if ResultOfLastCheck = Maths.NoValue then
            AllIndexesOK := False;
         end if;

         Dictionary.AddArrayIndex (SubtypeSym,
                                   ConstraintSym,
                                   Dictionary.Location'(NodePosition (CurrentConstraintNode),
                                                        NodePosition (CurrentConstraintNode)));
         GetNextIndex (IndexSym, IndexesFinished);
         GetNextConstraint (ConstraintSym, ConstraintsFinished);
      end loop;
      if not (IndexesFinished and ConstraintsFinished) then
         ErrorHandler.SemanticError (93,
                                     ErrorHandler.NoReference,
                                     NodePosition (ConstraintNode),
                                     LexTokenManager.NullString);
         AllIndexesOK := False;
      end if;
   end CheckIndexConstraints;

   -----------------------------------------------------------------------

   procedure CheckRealAccuracy (Node   : in     STree.SyntaxNode;
                                Scope  : in     Dictionary.Scopes;
                                Sort   : in     RealType;
                                Static :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#           out Accuracy;
   --#           out TheHeap;
   --# derives Accuracy                    from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          Sort,
   --#                                          STree.Table &
   --#         AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.StringTable,
   --#         Statistics.TableUsage,
   --#         STree.Table                 from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          Sort,
   --#                                          SPARK_IO.FILE_SYS,
   --#                                          STree.Table &
   --#         Static,
   --#         TheHeap                     from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table;
   is
      ExpNode             : STree.SyntaxNode;
      AccuracyType        : ExpRecord;
      UnwantedSeq         : SeqAlgebra.Seq;
      UnusedComponentData : ComponentManager.ComponentData;

      -------------------------------

      function TypeCorrect (TypeSym : Dictionary.Symbol;
                            Scope   : Dictionary.Scopes;
                            Sort    : RealType) return Boolean
      --# global in Dictionary.Dict;
      is
         Result : Boolean;

      begin
         if Sort = IsFloating then
            Result := Dictionary.IsIntegerTypeMark (TypeSym,
                                                    Scope);
         else
            Result := Dictionary.IsRealTypeMark (TypeSym,
                                                 Scope);
         end if;
         return Result or Dictionary.IsUnknownTypeMark (TypeSym);
      end TypeCorrect;

      -------------------------------

   begin --CheckRealAccuracy
      Heap.Initialize (TheHeap);
      ComponentManager.Initialise (UnusedComponentData);
      if CommandLineData.IsSpark83 then
         ExpNode :=  Child_Node (Child_Node (Node));
         SeqAlgebra.CreateSeq (TheHeap, UnwantedSeq);
         --# accept Flow, 10, UnusedComponentData, "Expected ineffective assignment";
         WalkExpression (ExpNode,
                         Scope,
                         Dictionary.GetUnknownTypeMark,
                         True,
                           --to get
                         AccuracyType,
                         UnwantedSeq,
                         UnusedComponentData);
         --# end accept;
         SeqAlgebra.DisposeOfSeq (TheHeap, UnwantedSeq);
         Maths.StorageRep (AccuracyType.Value, Accuracy);
         if not TypeCorrect (AccuracyType.TypeSymbol,
                             Scope,
                             Sort)
         then
            Accuracy := LexTokenManager.NullString;
            ErrorHandler.SemanticError (38,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        LexTokenManager.NullString);
         end if;
         Static := AccuracyType.IsStatic;
      else
         --reduced accuracy subtypes of reals are not allowed in SPARK95
         Static := True; -- to reduce knock-on errors
         Accuracy := LexTokenManager.NullString;
         ErrorHandler.SemanticError (608,
                                     9,
                                     NodePosition (Node),
                                     LexTokenManager.NullString);

      end if;
      Heap.ReportUsage (TheHeap);
   end CheckRealAccuracy;

   ------------------------------------------------------------------------

   procedure CheckRange (Node    : in     STree.SyntaxNode;
                           -- this is the arange node
                         TypeSym : in     Dictionary.Symbol;
                         Scope   : in     Dictionary.Scopes;
                         Static  : in out Boolean;
                         Lower,
                         Upper   :    out LexTokenManager.LexString)
   --# global in     CommandLineData.Content;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#           out AggregateStack.State;
   --#           out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.StringTable,
   --#         Lower,
   --#         STree.Table,
   --#         TheHeap,
   --#         Upper                       from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TypeSym &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          SPARK_IO.FILE_SYS,
   --#                                          STree.Table,
   --#                                          TypeSym &
   --#         Static,
   --#         Statistics.TableUsage       from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TypeSym;
   is
      RangeResult : ExpRecord;
      UnwantedSeq  : SeqAlgebra.Seq;
      UnusedComponentData : ComponentManager.ComponentData;

      UpperAfterConstraintCheck,
      LowerAfterConstraintCheck : Maths.Value;
      LowerLocal,
      UpperLocal : LexTokenManager.LexString;
      RHSNode    : STree.SyntaxNode;

   begin
      Heap.Initialize (TheHeap);
      SeqAlgebra.CreateSeq (TheHeap, UnwantedSeq);
      ComponentManager.Initialise (UnusedComponentData);
      --# accept Flow, 10, UnusedComponentData, "Expected ineffective assignment";
      WalkExpression (Node,
                      Scope,
                      TypeSym,
                      False,
                        --to get
                      RangeResult,
                      UnwantedSeq,
                      UnusedComponentData);
      --# end accept;
      SeqAlgebra.DisposeOfSeq (TheHeap, UnwantedSeq);
      Static := Static and RangeResult.IsStatic;
      Maths.StorageRep (RangeResult.Value, LowerLocal);
      Maths.StorageRep (RangeResult.RangeRHS, UpperLocal);

      -- check that range is constant
      if not RangeResult.IsConstant then
         LowerLocal := LexTokenManager.NullString;
         UpperLocal := LexTokenManager.NullString;
         ErrorHandler.SemanticError (43,
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     LexTokenManager.NullString);
      end if;
      if not Dictionary.CompatibleTypes (Scope,
                                         RangeResult.TypeSymbol,
                                         TypeSym)
      then
         LowerLocal := LexTokenManager.NullString;
         UpperLocal := LexTokenManager.NullString;
         ErrorHandler.SemanticErrorSym2 (107,
                                         ErrorHandler.NoReference,
                                         NodePosition (Node),
                                         RangeResult.TypeSymbol,
                                         TypeSym,
                                         Scope);
      end if;

      -- checks for bounds outside type being constrained
      -- see whether node is attribute or X..Y form and select suitable place
      -- to report errors on upper range bound
      if SyntaxNodeType (Child_Node (Node)) =
         SPSymbols.attribute
      then
         RHSNode := Node;
      else --must be of form X..Y
         RHSNode := Next_Sibling (Child_Node (Node));
      end if;


      ConstraintCheck (Maths.ValueRep (LowerLocal),
                       LowerAfterConstraintCheck,
                       False, -- can't be in annotation here
                       TypeSym,
                       NodePosition (Node));
      Maths.StorageRep (LowerAfterConstraintCheck, LowerLocal);

      ConstraintCheck (Maths.ValueRep (UpperLocal),
                       UpperAfterConstraintCheck,
                       False, -- can't be in annotation here
                       TypeSym,
                       NodePosition (RHSNode));
      Maths.StorageRep (UpperAfterConstraintCheck, UpperLocal);

      Lower := LowerLocal;
      Upper := UpperLocal;
      Heap.ReportUsage (TheHeap);
   end CheckRange;

   -----------------------------------------------------------------------

   procedure CheckRealRange (Node    : in     STree.SyntaxNode;
                              --this is the constraint node
                             TypeSym : in     Dictionary.Symbol;
                             Scope   : in     Dictionary.Scopes;
                             Static  : in out Boolean;
                             Lower,
                             Upper   :    out LexTokenManager.LexString)
   --# global in     CommandLineData.Content;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.StringTable,
   --#         Static,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                     from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TypeSym &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          SPARK_IO.FILE_SYS,
   --#                                          STree.Table,
   --#                                          TypeSym &
   --#         Lower,
   --#         Upper                       from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TypeSym;
   is
      RangeNode   : STree.SyntaxNode;

   begin
      RangeNode := Next_Sibling (Child_Node (Node));
      if RangeNode = STree.NullNode then
         --no range supplied so range is unchanged from parent type
         Upper := Dictionary.GetScalarAttributeValue (False, --no 'base
                                                      LexTokenManager.LastToken,
                                                      TypeSym);
         Lower := Dictionary.GetScalarAttributeValue (False, --no 'base
                                                      LexTokenManager.FirstToken,
                                                      TypeSym);
      else --a range is supplied
         CheckRange (Child_Node (RangeNode),
                     TypeSym,
                     Scope,
                     Static,
                     Lower,
                     Upper);
      end if;
   end CheckRealRange;

   -----------------------------------------------------------------------

   procedure AddScalarSubtype
   --# global in     Accuracy;
   --#        in     IdentNode;
   --#        in     IdStr;
   --#        in     LexTokenManager.StringTable;
   --#        in     Lower;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in     SubTypeIsStatic;
   --#        in     TypeNode;
   --#        in     TypeSym;
   --#        in     Upper;
   --#        in out Dictionary.Dict;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict   from *,
   --#                                Accuracy,
   --#                                IdStr,
   --#                                Lower,
   --#                                Scope,
   --#                                SubTypeIsStatic,
   --#                                TypeSym,
   --#                                Upper &
   --#         SPARK_IO.FILE_SYS from *,
   --#                                Dictionary.Dict,
   --#                                IdentNode,
   --#                                IdStr,
   --#                                LexTokenManager.StringTable,
   --#                                Scope,
   --#                                STree.Table,
   --#                                TypeNode,
   --#                                TypeSym;
   --#
   --#
   is
   begin
      if Dictionary.TypeIsInteger (TypeSym) then
         Dictionary.AddIntegerSubtype (IdStr,
                                       SubtypeIsStatic,
                                       TypeSym,
                                       Dictionary.Location'(NodePosition (TypeNode),
                                                            NodePosition (TypeNode)),
                                       Lower,
                                       Upper,
                                       Dictionary.Location'(NodePosition (IdentNode),
                                                            NodePosition (IdentNode)),
                                       Scope,
                                       Dictionary.ProgramContext);
      elsif Dictionary.TypeIsModular (TypeSym) then
         Dictionary.AddModularSubtype (IdStr,
                                       TypeSym,
                                       Dictionary.Location'(NodePosition (TypeNode),
                                                            NodePosition (TypeNode)),
                                       Lower,
                                       Upper,
                                       Dictionary.Location'(NodePosition (IdentNode),
                                                            NodePosition (IdentNode)),
                                       Scope,
                                       Dictionary.ProgramContext);

      elsif Dictionary.TypeIsEnumeration (TypeSym) then
         Dictionary.AddEnumerationSubtype (IdStr,
                                           SubtypeIsStatic,
                                           TypeSym,
                                           Dictionary.Location'(NodePosition (TypeNode),
                                                                NodePosition (TypeNode)),
                                           Lower,
                                           Upper,
                                           Dictionary.Location'(NodePosition (IdentNode),
                                                                NodePosition (IdentNode)),
                                           Scope,
                                           Dictionary.ProgramContext);
      elsif Dictionary.TypeIsFloatingPoint (TypeSym) then
         Dictionary.AddFloatingPointSubtype (IdStr,
                                             SubtypeIsStatic,
                                             TypeSym,
                                             Dictionary.Location'(NodePosition (TypeNode),
                                                                  NodePosition (TypeNode)),
                                             Lower,
                                             Upper,
                                             Accuracy,
                                             Dictionary.Location'(NodePosition (IdentNode),
                                                                  NodePosition (IdentNode)),
                                             Scope,
                                             Dictionary.ProgramContext);

      elsif Dictionary.TypeIsFixedPoint (TypeSym) then
         Dictionary.AddFixedPointSubtype (IdStr,
                                          SubtypeIsStatic,
                                          TypeSym,
                                          Dictionary.Location'(NodePosition (TypeNode),
                                                               NodePosition (TypeNode)),
                                          Lower,
                                          Upper,
                                          Accuracy,
                                          Dictionary.Location'(NodePosition (IdentNode),
                                                               NodePosition (IdentNode)),
                                          Scope,
                                          Dictionary.ProgramContext);
      end if;
   end AddScalarSubtype;

   procedure AddRecordSubtype
   --# global in     IdentNode;
   --#        in     IdStr;
   --#        in     LexTokenManager.StringTable;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in     TypeNode;
   --#        in     TypeSym;
   --#        in out Dictionary.Dict;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict   from *,
   --#                                IdStr,
   --#                                Scope,
   --#                                TypeSym &
   --#         SPARK_IO.FILE_SYS from *,
   --#                                Dictionary.Dict,
   --#                                IdentNode,
   --#                                IdStr,
   --#                                LexTokenManager.StringTable,
   --#                                Scope,
   --#                                STree.Table,
   --#                                TypeNode,
   --#                                TypeSym;
   is
   begin
      Dictionary.AddRecordSubtype (IdStr,
                                   TypeSym,
                                   Dictionary.Location'(NodePosition (TypeNode),
                                                        NodePosition (TypeNode)),
                                   Dictionary.Location'(NodePosition (IdentNode),
                                                        NodePosition (IdentNode)),
                                   Scope,
                                   Dictionary.ProgramContext);
   end AddRecordSubtype;

   -----------------------------------------------------------------------

   function AlreadyDefined (IdentStr : LexTokenManager.LexString;
                            Scope    : Dictionary.Scopes) return Boolean
   --# global in Dictionary.Dict;
   is
      Sym : Dictionary.Symbol;
   begin
      Sym := Dictionary.LookupItem (IdentStr,
                                    Scope,
                                    Dictionary.ProofContext);

      return not (Sym = Dictionary.NullSymbol or else
                  (Dictionary.IsTypeMark (Sym) and then
                   Dictionary.TypeIsAnnounced (Sym) and then
                   not Dictionary.IsDeclared (Sym)));
   end AlreadyDefined;

   -----------------------------------------------------------------------

   function SelectIndexOrDiscriminantError (ConstraintNode : STree.SyntaxNode) return Natural
   --# global in CommandLineData.Content;
   --#        in STree.Table;
   is
      Result : Natural;
   begin
      -- if an index_or_discriminant_constraint is applied to an inapplicable type we need to report and
      -- error.  This function tries to narrow the scope of the error message returned.
      if CommandLineData.RavenscarSelected then
         -- we could be expecting an array, task or protected type
         if IsNamedAssociation (ConstraintNode) then
            -- must be Task or protected
            Result := 891;
         else
            -- could be any of Task, Protected, Array
            Result := 892;
         end if;
      else
         -- can only be an array
         Result := 41;
      end if;
      return Result;
   end SelectIndexOrDiscriminantError;

   -----------------------------------------------------------------------

   procedure Wf_Ravenscar_Subtype (IdStr          : in LexTokenManager.LexString;
                                   TypeSym        : in Dictionary.Symbol;
                                   Scope          : in Dictionary.Scopes;
                                   IdNode         : in STree.SyntaxNode;
                                   ConstraintNode : in STree.SyntaxNode)

   --# global in     CommandLineData.Content;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.StringTable,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                     from *,
   --#                                          CommandLineData.Content,
   --#                                          ConstraintNode,
   --#                                          Dictionary.Dict,
   --#                                          IdStr,
   --#                                          LexTokenManager.StringTable,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TypeSym &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS           from CommandLineData.Content,
   --#                                          ConstraintNode,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          IdNode,
   --#                                          IdStr,
   --#                                          LexTokenManager.StringTable,
   --#                                          Scope,
   --#                                          SPARK_IO.FILE_SYS,
   --#                                          STree.Table,
   --#                                          TypeSym;
   is separate;

   -----------------------------------------------------------------------

begin --wf_subtype_declaration
      -- ASSUME Node = subtype_declaration

   Lower    := LexTokenManager.NullString;
   Upper    := LexTokenManager.NullString;
   Accuracy := LexTokenManager.NullString;

   IdentNode := Child_Node (Node);
   IdStr  := NodeLexString (IdentNode);
   TypeNode := Child_Node (Next_Sibling (IdentNode));
   ConstraintNode := Child_Node (Next_Sibling (TypeNode));
   ConstraintFound := ConstraintNode /= STree.NullNode;
   if AlreadyDefined (IdStr,
                      Scope)
   then
      OkToAdd := False;
      ErrorHandler.SemanticError (10,
                                  ErrorHandler.NoReference,
                                  NodePosition (IdentNode),
                                  IdStr);
   else
      OkToAdd := True;
   end if;

   wf_type_mark (TypeNode,
                 Scope,
                 Dictionary.ProgramContext,
                  --to get
                 TypeSym);

   if Dictionary.IsPrivateType (TypeSym, Scope) then
      OkToAdd := False;
      ErrorHandler.SemanticError (48,
                                  ErrorHandler.NoReference,
                                  NodePosition (TypeNode),
                                  LexTokenManager.NullString);
   end if;

   -- Subtypes of generic types are not allowed because we can't check whether the boudns will be valid
   -- when they are instantiated
   if Dictionary.TypeIsGeneric (TypeSym) then
      OkToAdd := False;
      ErrorHandler.SemanticError (652,
                                  ErrorHandler.NoReference,
                                  NodePosition (TypeNode),
                                  LexTokenManager.NullString);
   end if;

   -- test to prevent Boolean subtype unless full-range
   if ConstraintFound and
      Dictionary.TypeIsBoolean (TypeSym) then
         OkToAdd := False;
         ErrorHandler.SemanticError (412,
                                     15,
                                     NodePosition (ConstraintNode),
                                     LexTokenManager.NullString);
   end if;

   if OkToAdd and not Dictionary.IsUnknownTypeMark (TypeSym) then
      if ConstraintFound then
         -- there is a constraint node so proceed as before
         ConstraintType := SyntaxNodeType (ConstraintNode);
         if ConstraintType = SPSymbols.index_or_discriminant_constraint then
            if Dictionary.TypeIsArray (TypeSym) then
               if IsNamedAssociation (ConstraintNode) then
                  ErrorHandler.SemanticError (92,
                                              ErrorHandler.NoReference,
                                              NodePosition (ConstraintNode),
                                              LexTokenManager.NullString);
               else -- positional association is ok
                  if Dictionary.IsUnconstrainedArrayType (TypeSym) then
                     Dictionary.AddArraySubtype (IdStr,
                                                 TypeSym,
                                                 Dictionary.Location'(NodePosition (TypeNode),
                                                                      NodePosition (TypeNode)),
                                                 Dictionary.Location'(NodePosition (IdentNode),
                                                                      NodePosition (IdentNode)),
                                                 Scope,
                                                 Dictionary.ProgramContext,
                                                 False,
                                                   --to get
                                                 SubtypeSym);

                     CheckIndexConstraints (Dictionary.IsPredefinedStringType (TypeSym),
                                            AllIndexesOK);

                     Dictionary.SetTypeIsWellformed (SubtypeSym, AllIndexesOK);
                  else -- array already constrained
                     ErrorHandler.SemanticError (99,
                                                 ErrorHandler.NoReference,
                                                 NodePosition (TypeNode),
                                                 LexTokenManager.NullString);
                  end if;
               end if; -- illegal named association fo index_constraint

            elsif Dictionary.IsProtectedType (TypeSym) or else Dictionary.IsTaskType (TypeSym) then
               Wf_Ravenscar_Subtype (IdStr,
                                     TypeSym,
                                     Scope,
                                     IdentNode,
                                     ConstraintNode);

            else
               -- a type has been supplied for which index_or_discriminant_constraint is
               -- not appropriate
               ErrorHandler.SemanticError (SelectIndexOrDiscriminantError (ConstraintNode),
                                           ErrorHandler.NoReference,
                                           NodePosition (TypeNode),
                                           LexTokenManager.NullString);
            end if;

         else --some scalar subtype expected
            if not Dictionary.TypeIsScalar (TypeSym) then
               ErrorHandler.SemanticError (59,
                                           ErrorHandler.NoReference,
                                           NodePosition (TypeNode),
                                           LexTokenManager.NullString);
            else
               SubtypeIsStatic := True; -- default value
               if ConstraintType = SPSymbols.range_constraint then
                  CheckRange (Child_Node (ConstraintNode),
                              TypeSym,
                              Scope,
                              --to get
                              SubtypeIsStatic,
                              Lower,
                              Upper);

                  -- if constraint is a range but type is real then no accuracy
                  -- has been supplied so we need to get it from parent
                  if Dictionary.TypeIsFloatingPoint (TypeSym) then
                     Accuracy := Dictionary.GetScalarAttributeValue
                        (False,
                         LexTokenManager.DigitsToken,
                         TypeSym);
                  elsif Dictionary.TypeIsFixedPoint (TypeSym) then
                     Accuracy := Dictionary.GetScalarAttributeValue
                        (False,
                         LexTokenManager.DeltaToken,
                         TypeSym);
                  end if;
                  SubtypeIsStatic := SubtypeIsStatic and
                     Dictionary.IsStatic (TypeSym, Scope);

               elsif ConstraintType = SPSymbols.floating_point_constraint then
                  if not Dictionary.TypeIsFloatingPoint (TypeSym) then
                     ErrorHandler.SemanticError (100,
                                                 ErrorHandler.NoReference,
                                                 NodePosition (TypeNode),
                                                 LexTokenManager.NullString);
                  else
                     CheckRealAccuracy (ConstraintNode,
                                        Scope,
                                        IsFloating,
                                        SubtypeIsStatic);
                     CheckRealRange (ConstraintNode,
                                     TypeSym,
                                     Scope,
                                     SubtypeIsStatic,
                                     Lower,
                                     Upper);
                  end if;

               elsif ConstraintType = SPSymbols.fixed_point_constraint then
                  if not Dictionary.TypeIsFixedPoint (TypeSym) then
                     ErrorHandler.SemanticError (101,
                                                 ErrorHandler.NoReference,
                                                 NodePosition (TypeNode),
                                                 LexTokenManager.NullString);
                  else
                     CheckRealAccuracy (ConstraintNode,
                                        Scope,
                                        IsFixed,
                                        SubtypeIsStatic);
                     CheckRealRange (ConstraintNode,
                                     TypeSym,
                                     Scope,
                                     SubtypeIsStatic,
                                     Lower,
                                     Upper);
                  end if;
               end if;
               AddScalarSubtype;
            end if;
         end if;
      else  -- no constraint node present
         if Dictionary.TypeIsScalar (TypeSym) then
            --scalars are allowed without constraints so process them here
            SubtypeIsStatic := Dictionary.IsStatic (TypeSym, Scope);
            Lower := Dictionary.GetScalarAttributeValue (False,
                                                         LexTokenManager.FirstToken,
                                                         TypeSym);
            Upper := Dictionary.GetScalarAttributeValue (False,
                                                         LexTokenManager.LastToken,
                                                         TypeSym);
            if Dictionary.TypeIsFloatingPoint (TypeSym) then
               Accuracy := Dictionary.GetScalarAttributeValue (False,
                                                               LexTokenManager.DigitsToken,
                                                               TypeSym);
            elsif Dictionary.TypeIsFixedPoint (TypeSym) then
               Accuracy := Dictionary.GetScalarAttributeValue (False,
                                                               LexTokenManager.DeltaToken,
                                                               TypeSym);
            end if;
            AddScalarSubtype;


         -- Full-range subtypes are also allowed for non-tagged records.
         elsif Dictionary.TypeIsRecord (TypeSym) and then
               not Dictionary.TypeIsTagged (TypeSym) then

            AddRecordSubtype;

         else --not scalar or non-tagged record so must be illegal
            ErrorHandler.SemanticError (406,
                                        ErrorHandler.NoReference,
                                        NodePosition (TypeNode),
                                        LexTokenManager.NullString);
         end if;
      end if;
   end if;
end wf_subtype_declaration;
