-- $Id: sem-compunit-wf_subprogram_declaration.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

-- Overview
-- Checks sub-program declarations from node subprogram_declaration.  These
-- nodes occur only in package declarations therefore well formation of
-- function_ and procedure_specifications are handled here as a special case
-- rather than using the more complex and general-purpose
-- wf_procedure_specification and wf_function_specification.
-- NOTE 11/6/02
-- Declarations also occur in protected types but this procedure can
-- deal with those as well
--------------------------------------------------------------------------------

separate (Sem.CompUnit)

procedure wf_subprogram_declaration (Node          : in     STree.SyntaxNode;
                                     CurrentScope  : in     Dictionary.Scopes;
                                     TheSubprogSym :    out Dictionary.Symbol)
is
   SpecNode,
   AnnoNode,
   ConstraintNode,
   FormalPartNode  : STree.SyntaxNode;
   SubprogSym      : Dictionary.Symbol;

   ------------------------------------------------------------------------

   -- If we are declaring a subprogram in a package spec and the spec contains
   -- protected types we search each of these to detect re-use of the subprogram
   -- name.  If we don't trap such re-use at this point then we end up with a
   -- legal package spec for which no legal body could be written (since its
   -- implementation would inevitably involve overload resolution of calls made from
   -- within the protected body.  e.g. type PT in package P declares operation K.  Package
   -- P also declares an operation K.  From inside the body of PT, a call to K could refer
   -- to either of the two Ks since both are directly visible.
   function IsDefinedInVisibleProtectedType (Name  : LexTokenManager.LexString;
                                             Scope : Dictionary.Scopes) return Boolean
   --# global in Dictionary.Dict;
   is
      Result : Boolean := False;
      It : Dictionary.Iterator;
   begin
      if Dictionary.IsPackage (Dictionary.GetRegion (Scope)) then
         It := Dictionary.FirstVisibleProtectedType (Dictionary.GetRegion (Scope));
         while not Dictionary.IsNullIterator (It) loop
            Result := Dictionary.IsDirectlyDefined (Name,
                                                    Dictionary.VisibleScope (Dictionary.CurrentSymbol (It)),
                                                    Dictionary.ProofContext);
            exit when Result;
            It := Dictionary.NextSymbol (It);
         end loop;
      end if;
      return Result;
   end IsDefinedInVisibleProtectedType;

   -- ditto for protected types in package private scope
   function IsDefinedInPrivateProtectedType (Name  : LexTokenManager.LexString;
                                             Scope : Dictionary.Scopes) return Boolean
   --# global in Dictionary.Dict;
   is
      Result : Boolean := False;
      It : Dictionary.Iterator;
   begin
      if Dictionary.IsPackage (Dictionary.GetRegion (Scope)) then
         It := Dictionary.FirstPrivateProtectedType (Dictionary.GetRegion (Scope));
         while not Dictionary.IsNullIterator (It) loop
            Result := Dictionary.IsDirectlyDefined (Name,
                                                    Dictionary.VisibleScope (Dictionary.CurrentSymbol (It)),
                                                    Dictionary.ProofContext);
            exit when Result;
            It := Dictionary.NextSymbol (It);
         end loop;
      end if;
      return Result;
   end IsDefinedInPrivateProtectedType;

   ------------------------------------------------------------------------

   procedure CheckProcedureSpecification (Node         : in     STree.SyntaxNode;
                                          CurrentScope : in     Dictionary.Scopes;
                                          SubprogSym   :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in     SpecNode;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict           from *,
   --#                                        CurrentScope,
   --#                                        Node,
   --#                                        SpecNode,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         SubprogSym                from CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        Node,
   --#                                        STree.Table;
   is
      IdentStr        : LexTokenManager.LexString;
      LocalSubprogSym : Dictionary.Symbol;

   begin
      IdentStr := NodeLexString (Child_Node (Node));
      if Dictionary.IsDefined (IdentStr,
                               CurrentScope,
                               Dictionary.ProofContext)
      then
         ErrorHandler.SemanticError (10,  ErrorHandler.NoReference, --illegal redeclaration
                                     NodePosition (Node),
                                     IdentStr);
         SubprogSym := Dictionary.NullSymbol;
      elsif IsDefinedInVisibleProtectedType (IdentStr,
                                             CurrentScope) or else
        IsDefinedInPrivateProtectedType (IdentStr,
                                         CurrentScope)
      then
         ErrorHandler.SemanticError (988,  ErrorHandler.NoReference, --illegal redeclaration
                                     NodePosition (Node),
                                     IdentStr);
         SubprogSym := Dictionary.NullSymbol;
      else
         Dictionary.AddSubprogram (IdentStr,
                                   Dictionary.Location'(NodePosition (Node),
                                                        NodePosition (Node)),
                                   CurrentScope,
                                   Dictionary.ProgramContext,
                                    --to get
                                   LocalSubprogSym);
         SubprogSym := LocalSubprogSym;
         if SyntaxNodeType (SpecNode) = SPSymbols.entry_specification then
            Dictionary.SetSubprogramIsEntry (LocalSubprogSym);
         end if;
      end if;
   end CheckProcedureSpecification;

   ------------------------------------------------------------------------

   procedure CheckFunctionSpecification (Node         : in     STree.SyntaxNode;
                                         CurrentScope : in     Dictionary.Scopes;
                                         SubprogSym   :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict           from *,
   --#                                        CommandLineData.Content,
   --#                                        CurrentScope,
   --#                                        Node,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         SubprogSym                from CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        Node,
   --#                                        STree.Table;
   is
      ReturnTypeNode : STree.SyntaxNode;
      IdentStr       : LexTokenManager.LexString;
      SubprogSymLocal,
      TypeSym  : Dictionary.Symbol;

   begin
      SubprogSymLocal := Dictionary.NullSymbol;
      IdentStr := NodeLexString (Child_Node (Child_Node (Node)));
      if Dictionary.IsDefined (IdentStr,
                               CurrentScope,
                               Dictionary.ProofContext)
      then
         ErrorHandler.SemanticError (10, --illegal redeclaration
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     IdentStr);
      elsif IsDefinedInVisibleProtectedType (IdentStr,
                                             CurrentScope)
      then
         ErrorHandler.SemanticError (988,  ErrorHandler.NoReference, --illegal redeclaration
                                     NodePosition (Node),
                                     IdentStr);
      else
         Dictionary.AddSubprogram (IdentStr,
                                   Dictionary.Location'(NodePosition (Node),
                                                        NodePosition (Node)),
                                   CurrentScope,
                                   Dictionary.ProgramContext,
                                    -- to get
                                   SubprogSymLocal);
      end if;
      ReturnTypeNode := LastSiblingOf (Child_Node (Node));
      wf_type_mark (ReturnTypeNode,
                    CurrentScope,
                    Dictionary.ProgramContext,
                     --to get
                    TypeSym);

      if Dictionary.IsUnconstrainedArrayType (TypeSym) then
         TypeSym := Dictionary.GetUnknownTypeMark;
         ErrorHandler.SemanticError (39, --illegal use of unconstrained array type
                                     ErrorHandler.NoReference,
                                     NodePosition (ReturnTypeNode),
                                     NodeLexString (ReturnTypeNode));
      elsif Dictionary.IsPredefinedSuspensionObjectType (TypeSym) or
        Dictionary.TypeIsProtected (TypeSym) then
         TypeSym := Dictionary.GetUnknownTypeMark;
         ErrorHandler.SemanticError (905,
                                     ErrorHandler.NoReference,
                                     NodePosition (ReturnTypeNode),
                                     LexTokenManager.NullString);

      elsif Dictionary.TypeIsTagged (TypeSym) and then
        (Dictionary.GetScope (TypeSym) = CurrentScope) then
         -- attempt to declare primitive function with controlling return result
         TypeSym := Dictionary.GetUnknownTypeMark;
         ErrorHandler.SemanticError (840,
                                     ErrorHandler.NoReference,
                                     NodePosition (ReturnTypeNode),
                                     LexTokenManager.NullString);

      end if;

      if SubprogSymLocal /= Dictionary.NullSymbol then
         Dictionary.AddReturnType (SubprogSymLocal,
                                   TypeSym,
                                   Dictionary.Location'(NodePosition (ReturnTypeNode),
                                                        NodePosition (ReturnTypeNode)));

         -- mark signature as not wellformed if wf_type_mark has returned the unknown type
         if TypeSym = Dictionary.GetUnknownTypeMark then
            Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsAbstract,
                                                            SubprogSymLocal);
         end if;
      end if;
      SubprogSym := SubprogSymLocal;
   end CheckFunctionSpecification;

   ------------------------------------------------------------------------

   procedure GetAnnoAndConNodes (Node           : in     STree.SyntaxNode;
                                 AnnoNode       :    out STree.SyntaxNode;
                                 ConstraintNode :    out STree.SyntaxNode)
   --# global in STree.Table;
   --# derives AnnoNode,
   --#         ConstraintNode from Node,
   --#                             STree.Table;
   --  pre   SyntaxNodeType (Node) =
   --           SPSymbols.procedure_annotation or
   --        SyntaxNodeType (Node) =
   --           SPSymbols.function_annotation;
   is
      ConstraintNodeLocal : STree.SyntaxNode;
      NodeType : SPSymbols.SPSymbol;
   begin
      ConstraintNodeLocal := Child_Node (Node);
      NodeType := SyntaxNodeType (ConstraintNodeLocal);
      if NodeType = SPSymbols.function_constraint or else
         NodeType = SPSymbols.procedure_constraint
      then
         AnnoNode := STree.NullNode;      --only a constraint found
         ConstraintNode := ConstraintNodeLocal;
      else
         AnnoNode := Node;
         ConstraintNode := LastSiblingOf (ConstraintNodeLocal);
      end if;
   end GetAnnoAndConNodes;

   ------------------------------------------------------------------------

begin -- wf_subprog
      -- ASSUME Node = subprogram_declaration OR entry_declaration

   SpecNode := Child_Node (Node);
   FormalPartNode := Next_Sibling (Child_Node (SpecNode));
   GetAnnoAndConNodes (Next_Sibling (SpecNode),
                        --to get
                       AnnoNode,
                       ConstraintNode);
   if SyntaxNodeType (SpecNode) = SPSymbols.procedure_specification or else
     SyntaxNodeType (SpecNode) = SPSymbols.entry_specification
   then
      CheckProcedureSpecification (SpecNode,
                                   CurrentScope,
                                    --to get
                                   SubprogSym);
      TheSubprogSym := SubprogSym; -- pass back to caller
      if SubprogSym /= Dictionary.NullSymbol then
         if FormalPartNode /= STree.NullNode then
            wf_formal_part (FormalPartNode,
                            CurrentScope,
                            SubprogSym,
                            True,    --procedure cannot already be declared
                            Dictionary.ProgramContext);
         end if;
         if AnnoNode /= STree.NullNode then
            wf_procedure_annotation (AnnoNode,
                                     CurrentScope,
                                     SubprogSym,
                                     True);

            -- no anno is always an error of 83 or for 95 if info flow is turned on
         elsif CommandLineData.IsSpark83 or
            CommandLineData.Content.DoInformationFlow
         then
            Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsAbstract,
                                                            SubprogSym);
            ErrorHandler.SemanticError (154,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        Dictionary.GetSimpleName (SubprogSym));
         end if;

         if CommandLineData.IsSpark95 and then
            not CommandLineData.Content.DoInformationFlow
         then
            CreateFullSubProgDependency (Node,
                                         SubprogSym,
                                         Dictionary.IsAbstract);
         end if;

         if ConstraintNode /= STree.NullNode then
            wf_procedure_constraint (ConstraintNode,
                                     Dictionary.LocalScope (SubprogSym),
                                     True);
         end if;
      end if;
      CheckNoOverloadingFromTaggedOps (SpecNode,
                                       SubprogSym,
                                       CurrentScope,
                                       Dictionary.IsAbstract);

   elsif SyntaxNodeType (SpecNode) = SPSymbols.function_specification then
      CheckFunctionSpecification (SpecNode,
                                  CurrentScope,
                                    --to get
                                  SubprogSym);
      TheSubprogSym := SubprogSym; -- pass back to caller
      if SubprogSym /= Dictionary.NullSymbol then
         if FormalPartNode /= STree.NullNode and then
            SyntaxNodeType (FormalPartNode) /= SPSymbols.type_mark then
            wf_formal_part (FormalPartNode,
                            CurrentScope,
                            SubprogSym,
                            True,     --procedure cannot already be declared
                            Dictionary.ProgramContext);
         end if;
         if AnnoNode /= STree.NullNode then
            wf_function_annotation (AnnoNode,
                                    CurrentScope,
                                    SubprogSym,
                                    True);
         end if;
         if ConstraintNode /= STree.NullNode then
            wf_function_constraint (ConstraintNode,
                                    Dictionary.LocalScope (SubprogSym),
                                    True);
         end if;
      end if;
      CheckNoOverloadingFromTaggedOps (SpecNode,
                                       SubprogSym,
                                       CurrentScope,
                                       Dictionary.IsAbstract);
   else
      wf_proof_function_declaration (SpecNode,
                                     CurrentScope,
                                       --to get
                                     TheSubprogSym);
   end if;
end wf_subprogram_declaration;
