-- $Id: sem-compunit-wf_subprogram_body.adb 12351 2009-02-02 15:03:51Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.CompUnit)
procedure wf_subprogram_body (Node     : in     STree.SyntaxNode;
                              Scope    : in out Dictionary.Scopes;
                              NextNode :    out STree.SyntaxNode)
is
   type GenericKinds is (GenericProcedure, GenericFunction);

   NodeType        : SPSymbols.SPSymbol;
   IdentNode,
   SpecNode,
   FormalPartNode,
   ConstraintNode,
   AnnoNode,
   MainNode,
   WithNode,
   SubprogImplemNode,
   EndDesigNode    : STree.SyntaxNode;
   SubprogSym      : Dictionary.Symbol;
   Hidden          : Hidden_Class;
   IsGeneric       : Boolean;
   FirstSeen       : Boolean;
   SubProgScope    : Dictionary.Scopes;
   ScopeForFormalPartCheck : Dictionary.Scopes;

   ------------------------------------------------------------------

   procedure CheckProgramCompleteness (Node       : in STree.SyntaxNode;
                                       SubprogSym : in Dictionary.Symbol;
                                       Scope      : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        SubprogSym;
   is
      InheritIt : Dictionary.Iterator;
      InheritedPackage : Dictionary.Symbol;

      function ContainsTask (ThePackage : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      is
      begin
         return not Dictionary.IsNullIterator (Dictionary.FirstOwnTask (ThePackage));
      end ContainsTask;

      function ContainsInterrupt (ThePackage : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      is
         Result : Boolean := False;
         It     : Dictionary.Iterator;
      begin
         It := Dictionary.FirstOwnVariable (ThePackage);
         while not Dictionary.IsNullIterator (It) loop
            Result := Dictionary.GetHasInterruptProperty (Dictionary.CurrentSymbol (It));
            exit when Result;

            It := Dictionary.NextSymbol (It);
         end loop;
         return Result;
      end ContainsInterrupt;

   begin -- CheckProgramCompleteness
      InheritIt := Dictionary.FirstInheritsClause (SubprogSym);
      while not Dictionary.IsNullIterator (InheritIt) loop
         InheritedPackage := Dictionary.CurrentSymbol (InheritIt);
         if ContainsTask (InheritedPackage) or else
           ContainsInterrupt (InheritedPackage) then
            -- then it must also be WITHed to ensure program completeness
            if not Dictionary.IsWithed (InheritedPackage, Scope) then
               ErrorHandler.SemanticErrorSym (951,
                                              ErrorHandler.NoReference,
                                              NodePosition (Node),
                                              InheritedPackage,
                                              Scope);
            end if;
         end if;
         InheritIt := Dictionary.NextSymbol (InheritIt);
      end loop;
   end CheckProgramCompleteness;

   ------------------------------------------------------------------

   procedure ProcessPartitionAnnotation (MainNode : in STree.SyntaxNode;
                                         Scope    : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives Dictionary.Dict           from *,
   --#                                        CommandLineData.Content,
   --#                                        LexTokenManager.StringTable,
   --#                                        MainNode,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        MainNode,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         Statistics.TableUsage,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        MainNode,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap;
  is separate;

   ------------------------------------------------------------------

   procedure CheckUnprotectedGlobals
     (CheckList                : in Dictionary.Iterator;
      TheThread                : in Dictionary.Symbol;
      AnnotationsAreWellformed : in Boolean;
      Scope                    : in Dictionary.Scopes;
      ErrorNode                : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict           from *,
   --#                                        CheckList,
   --#                                        TheThread &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from AnnotationsAreWellformed,
   --#                                        CheckList,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ErrorNode,
   --#                                        LexTokenManager.StringTable,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TheThread;
   is
      It          : Dictionary.Iterator;
      Sym         : Dictionary.Symbol;
      OtherThread : Dictionary.Symbol;
   begin
      It := CheckList;
      while It /= Dictionary.NullIterator loop
         Sym := Dictionary.CurrentSymbol (It);
         if Sym /= Dictionary.GetNullVariable then
            if not Dictionary.GetOwnVariableProtected (Sym) then
               OtherThread := Dictionary.GetUnprotectedReference (Sym);
               if OtherThread /= Dictionary.NullSymbol then
                  -- This is non-protected global variable that is being
                  -- accessed by more than one thread of control.
                  --
                  ErrorHandler.SemanticErrorSym3 (938,
                                                  ErrorHandler.NoReference,
                                                  NodePosition (ErrorNode),
                                                  Sym,
                                                  OtherThread,
                                                  TheThread,
                                                  Scope);
               else
                  -- Mark this global variable as being accessed by a thread.
                  --
                  Dictionary.SetUnprotectedReference (Sym, TheThread);
               end if;
            end if;
         end if;
         It := Dictionary.NextSymbol (It);
      end loop;
      if not AnnotationsAreWellformed then
         -- The thread has errors in the annotations and so the shared variable check
         -- may not be complete.
         ErrorHandler.SemanticWarningSym (413,
                                          NodePosition (ErrorNode),
                                          TheThread,
                                          Scope);
      end if;
   end CheckUnprotectedGlobals;

   ------------------------------------------------------------------

   procedure SharedVariableCheck (MainProgram : in Dictionary.Symbol;
                                  Scope       : in Dictionary.Scopes;
                                  ErrorNode   : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict           from *,
   --#                                        MainProgram &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ErrorNode,
   --#                                        LexTokenManager.StringTable,
   --#                                        MainProgram,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      InheritedPackageIt  : Dictionary.Iterator;
      InheritedPackageSym : Dictionary.Symbol;
      It                  : Dictionary.Iterator;
      Sym                 : Dictionary.Symbol;
      TypeSym             : Dictionary.Symbol;
   begin
      -- Look for access to unprotected globals by the main procedure
      --
      CheckUnprotectedGlobals
        (CheckList                => Dictionary.FirstGlobalVariable (Dictionary.IsAbstract, MainProgram),
         TheThread                => MainProgram,
         AnnotationsAreWellformed => Dictionary.SubprogramSignatureIsWellformed
           (Dictionary.IsAbstract, MainProgram),
         Scope                    => Scope,
         ErrorNode                => ErrorNode);

      -- Look for access to unprotected globals by all tasks.
      --
      InheritedPackageIt := Dictionary.FirstInheritsClause (MainProgram);
      while InheritedPackageIt /= Dictionary.NullIterator loop
         InheritedPackageSym := Dictionary.CurrentSymbol (InheritedPackageIt);
         It := Dictionary.FirstOwnTask (InheritedPackageSym);
         while It /= Dictionary.NullIterator loop
            Sym := Dictionary.CurrentSymbol (It);
            TypeSym := Dictionary.GetRootType (Dictionary.GetType (Sym));
            if Dictionary.IsDeclared (TypeSym) then
               if Dictionary.UsesUnprotectedVariables (TypeSym) then
                  CheckUnprotectedGlobals
                    (CheckList => Dictionary.FirstGlobalVariable
                       (Dictionary.IsAbstract, TypeSym),
                     TheThread => Sym,
                     AnnotationsAreWellformed => Dictionary.SubprogramSignatureIsWellformed
                       (Dictionary.IsAbstract, TypeSym),
                     Scope => Scope,
                     ErrorNode => ErrorNode);
               end if;
            elsif not Dictionary.IsUnknownTypeMark (TypeSym) then
               -- The task type is not available and hence we cannot perform
               -- the shared variable check for this task.
               ErrorHandler.SemanticWarningSym (411,
                                                NodePosition (ErrorNode),
                                                TypeSym,
                                                Scope);
            end if;
            It := Dictionary.NextSymbol (It);
         end loop;
         InheritedPackageIt := Dictionary.NextSymbol (InheritedPackageIt);
      end loop;
   end SharedVariableCheck;

   ------------------------------------------------------------------

   procedure CheckSuspendsItems
     (CheckList : in Dictionary.Iterator;
      TheThread : in Dictionary.Symbol;
      Scope     : in Dictionary.Scopes;
      ErrorNode : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict           from *,
   --#                                        CheckList,
   --#                                        TheThread &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CheckList,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ErrorNode,
   --#                                        LexTokenManager.StringTable,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TheThread;
   is
      It          : Dictionary.Iterator;
      Sym         : Dictionary.Symbol;
      OtherThread : Dictionary.Symbol;
   begin
      It := CheckList;
      while It /= Dictionary.NullIterator loop
         Sym := Dictionary.CurrentSymbol (It);
         OtherThread := Dictionary.GetSuspendsReference (Sym);
         if OtherThread /= Dictionary.NullSymbol then
            -- This is a suspendable entity that is being
            -- accessed by more than one thread of control.
            --
            ErrorHandler.SemanticErrorSym3 (939,
                                            ErrorHandler.NoReference,
                                            NodePosition (ErrorNode),
                                            Sym,
                                            OtherThread,
                                            TheThread,
                                            Scope);
         else
            -- Mark this suspends item as being accessed by a thread.
            --
            Dictionary.SetSuspendsReference (Sym, TheThread);
         end if;
         It := Dictionary.NextSymbol (It);
      end loop;
   end CheckSuspendsItems;

   ------------------------------------------------------------------

   procedure MaxOneInAQueueCheck (MainProgram : in Dictionary.Symbol;
                                  Scope       : in Dictionary.Scopes;
                                  ErrorNode   : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict           from *,
   --#                                        MainProgram &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ErrorNode,
   --#                                        LexTokenManager.StringTable,
   --#                                        MainProgram,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      InheritedPackageIt  : Dictionary.Iterator;
      InheritedPackageSym : Dictionary.Symbol;
      It                  : Dictionary.Iterator;
      Sym                 : Dictionary.Symbol;
      TypeSym             : Dictionary.Symbol;
   begin
      -- Look for suspendable entities in the main program
      --
      CheckSuspendsItems
        (CheckList => Dictionary.FirstSuspendsListItem (MainProgram),
         TheThread => MainProgram,
         Scope => Scope,
         ErrorNode => ErrorNode);

      -- Look for suspendable entities in all the tasks.
      -- Note. interrupt handlers cannot call operations that suspend.
      --
      InheritedPackageIt := Dictionary.FirstInheritsClause (MainProgram);
      while InheritedPackageIt /= Dictionary.NullIterator loop
         InheritedPackageSym := Dictionary.CurrentSymbol (InheritedPackageIt);
         It := Dictionary.FirstOwnTask (InheritedPackageSym);
         while It /= Dictionary.NullIterator loop
            Sym := Dictionary.CurrentSymbol (It);
            TypeSym := Dictionary.GetRootType (Dictionary.GetType (Sym));
            if Dictionary.IsDeclared (TypeSym) then
               CheckSuspendsItems
                 (CheckList => Dictionary.FirstSuspendsListItem (TypeSym),
                  TheThread => Sym,
                  Scope => Scope,
                  ErrorNode => ErrorNode);
            elsif not Dictionary.IsUnknownTypeMark (TypeSym) then
               -- The task type is not available and hence we cannot perform
               -- the max-one-in-a-queue check for this task.
               ErrorHandler.SemanticWarningSym (412,
                                                NodePosition (ErrorNode),
                                                TypeSym,
                                                Scope);
            end if;
            It := Dictionary.NextSymbol (It);
         end loop;
         InheritedPackageIt := Dictionary.NextSymbol (InheritedPackageIt);
      end loop;
   end MaxOneInAQueueCheck;

   ------------------------------------------------------------------

   procedure wf_main_program (Node         : in STree.SyntaxNode;
                              SubprogSym   : in Dictionary.Symbol;
                              Scope,
                              SubProgScope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives Dictionary.Dict           from *,
   --#                                        CommandLineData.Content,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        SubProgScope,
   --#                                        SubprogSym,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        SubProgScope,
   --#                                        SubprogSym,
   --#                                        TheHeap &
   --#         Statistics.TableUsage,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        SubProgScope,
   --#                                        SubprogSym,
   --#                                        TheHeap;
  is
      ContextNode,
      InheritNode : STree.SyntaxNode;

   begin -- wf_main_program
         -- ASSUME Node = main_program_declaration
      InheritNode := Child_Node (Node);
      if SyntaxNodeType (InheritNode) /= SPSymbols.inherit_clause then
         InheritNode := STree.NullNode;
      end if;

      ContextNode := Child_Node (ParentNode (ParentNode (Node)));
      if SyntaxNodeType (ContextNode) /= SPSymbols.context_clause then
         ContextNode := STree.NullNode;
      end if;

      if not Dictionary.MainProgramExists then
         Dictionary.AddMainProgram (SubprogSym,
                                    Dictionary.Location'(NodePosition (Node),
                                                         NodePosition (Node)));
         if InheritNode /= STree.NullNode then
            wf_inherit_clause (InheritNode,
                               SubprogSym,
                               Scope);
         end if;
         if ContextNode /= STree.NullNode then
            wf_context_clause (ContextNode,
                               SubprogSym,
                               SubProgScope);
         end if;

         -- check here, in Ravencar, that all inherited packages with tasks/interrupts are also WITHed
         if CommandLineData.RavenscarSelected then
            CheckProgramCompleteness (InheritNode,
                                      SubprogSym,
                                      SubProgScope);
         end if;

         -- in Ravencar mode, a main program may have an addition partition flow analysis annotation
         ProcessPartitionAnnotation (Node, Scope);

      else  -- Dictionary.MainProgramExists
         ErrorHandler.SemanticError (313,
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     LexTokenManager.NullString);
      end if;

   end wf_main_program;

   ------------------------------------------------------------------

   function RequiresSecondAnnotation (SubprogSym : Dictionary.Symbol)
                                     return Boolean
   --# global in Dictionary.Dict;
   is
      GlobalVar : Dictionary.Symbol;
      Required  : Boolean;
      GlobalItem : Dictionary.Iterator;
      EnclosingRegion : Dictionary.Symbol;

   begin
      Required := False;
      if not Dictionary.IsGlobalScope (Dictionary.GetScope (SubprogSym)) then
         --EnclosingRegion := Dictionary.GetEnclosingCompilationUnit
         --   (Dictionary.GetScope (SubprogSym));
         EnclosingRegion := Dictionary.GetRegion (Dictionary.GetScope (SubprogSym));
         if Dictionary.IsPackage (EnclosingRegion) or else
           (Dictionary.IsType (EnclosingRegion) and then
              Dictionary.TypeIsProtected (EnclosingRegion)) then
            GlobalItem := Dictionary.FirstGlobalVariable (Dictionary.IsAbstract,
                                                          SubprogSym);
            while GlobalItem /= Dictionary.NullIterator loop
               GlobalVar := Dictionary.CurrentSymbol (GlobalItem);
               if Dictionary.IsAbstractOwnVariable (GlobalVar) and then
                  Dictionary.GetOwner (GlobalVar) = EnclosingRegion
               then
                  Required := True;
                  exit;
               end if;
               GlobalItem := Dictionary.NextSymbol (GlobalItem);
            end loop;
         end if;
      end if;
      return Required;
   end RequiresSecondAnnotation;

   ----------------------------------------------------------------------

   function InSubunit (SubprogBodyNode : in STree.SyntaxNode)
                      return Boolean
   --# global in STree.Table;
   is
   begin
      return SyntaxNodeType (ParentNode (ParentNode (SubprogBodyNode))) =
         SPSymbols.subunit;
   end InSubunit;

   ----------------------------------------------------------------------

   procedure CheckFunctionHasReturn (Node,
                                     EndDesigNode : in    STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EndDesigNode,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      NextNode : STree.SyntaxNode;

   begin
      NextNode := Next_Sibling (Child_Node (LastSiblingOf (Child_Node (Node))));
      if SyntaxNodeType (NextNode) /= SPSymbols.sequence_of_statements then
         NextNode := Next_Sibling (NextNode);
      end if;
      if SyntaxNodeType (NextNode) = SPSymbols.sequence_of_statements then
         NextNode := Child_Node (NextNode);
         if SyntaxNodeType (NextNode) = SPSymbols.sequence_of_statements then
            NextNode := Next_Sibling (NextNode);
         end if;
         if SyntaxNodeType (Child_Node (Child_Node (NextNode))) /=
            SPSymbols.return_statement
         then
            ErrorHandler.ControlFlowError (ErrorHandler.MissingReturn,
                                           NodePosition (EndDesigNode));
         end if;
      end if;
   end CheckFunctionHasReturn;

   ----------------------------------------------------------------------

   procedure GetAnnoAndConNodes (Node           : in     STree.SyntaxNode;
                                 AnnoNode       :    out STree.SyntaxNode;
                                 ConstraintNode :    out STree.SyntaxNode)
   --# global in STree.Table;
   --# derives AnnoNode,
   --#         ConstraintNode from Node,
   --#                             STree.Table;
   --  pre   SyntaxNodeType (Node) =
   --           SPSymbols.procedure_annotation or
   --        SyntaxNodeType (Node) =
   --           SPSymbols.function_annotation;
   is
      ConstraintNodeLocal : STree.SyntaxNode;
      NodeType : SPSymbols.SPSymbol;
   begin
      ConstraintNodeLocal := Child_Node (Node);
      NodeType := SyntaxNodeType (ConstraintNodeLocal);
      -- We are looking for global/derives so any other anno doesn't count
      if NodeType = SPSymbols.function_constraint or else
        NodeType = SPSymbols.procedure_constraint or else
        NodeType = SPSymbols.declare_annotation
      then
         AnnoNode := STree.NullNode;      --only a constraint or declare found
         ConstraintNode := ConstraintNodeLocal;
      else
         AnnoNode := Node;
         ConstraintNode := LastSiblingOf (ConstraintNodeLocal);
      end if;
   end GetAnnoAndConNodes;

   ----------------------------------------------------------------------

   procedure CheckGenericSpecification (Node       : in     STree.SyntaxNode;
                                        Kind       : in     GenericKinds;
                                        Scope      : in out Dictionary.Scopes;
                                        SubProgSym :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict,
   --#         Scope,
   --#         SubprogSym                from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        Kind,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        Kind,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      Sym            : Dictionary.Symbol;
      IdentNode      : STree.SyntaxNode;
      ReturnTypeNode : STree.SyntaxNode := STree.NullNode;
      IdentStr       : LexTokenManager.LexString;

      function ProcedureWhenFunctionExpected (Sym  : Dictionary.Symbol;
                                              Kind : GenericKinds) return Boolean
      --# global in Dictionary.Dict;
      is
      begin
         return Kind = GenericProcedure and then
           Dictionary.IsFunction (Sym);
      end ProcedureWhenFunctionExpected;

      ------------------

      function FunctionWhenProcedureExpected (Sym  : Dictionary.Symbol;
                                              Kind : GenericKinds) return Boolean
      --# global in Dictionary.Dict;
      is
      begin
         return Kind = GenericFunction and then
            Dictionary.IsProcedure (Sym);
      end FunctionWhenProcedureExpected;

      ------------------

      procedure CheckFunctionReturnType (Sym            : in out Dictionary.Symbol;
                                         ReturnTypeNode : in     STree.SyntaxNode)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in     STree.Table;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.StringTable,
      --#                                        ReturnTypeNode,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        Sym &
      --#         Sym                       from *,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ReturnTypeNode,
      --#                                        STree.Table;
      is
         BodyReturnTypeSym : Dictionary.Symbol;
         DeclarationReturnTypeSym : Dictionary.Symbol;
      begin
         -- only do this check for functions
         if ReturnTypeNode /= STree.NullNode then
            DeclarationReturnTypeSym := Dictionary.GetType (Sym);
            wf_type_mark (ReturnTypeNode,
                          Dictionary.LocalScope (Sym),
                          Dictionary.ProgramContext,
                           --to get
                          BodyReturnTypeSym);
            if DeclarationReturnTypeSym /= BodyReturnTypeSym then
               ErrorHandler.SemanticError (22, --inconsistent return type
                                           ErrorHandler.NoReference,
                                           NodePosition (ReturnTypeNode),
                                           Dictionary.GetSimpleName (Sym));

               Sym := Dictionary.NullSymbol; -- signal error back to caller
            end if;
         end if;
      end CheckFunctionReturnType;

      ------------------

   begin -- CheckGenericSpecification
      IdentNode := Child_Node (Node);
      if SyntaxNodeType (IdentNode) = SPSymbols.designator then
         -- its a function
         -- so look below for identifier
         IdentNode := Child_Node (IdentNode);
         -- and find the return type
         ReturnTypeNode := LastSiblingOf (Child_Node (Node));
      end if;

      IdentStr := NodeLexString (IdentNode);

      Sym := Dictionary.LookupItem (IdentStr,
                                    Scope,
                                    Dictionary.ProgramContext);

      -- to be valid, Sym must be a generic unit of the right Kind which
      -- does not already have a body
      if Sym = Dictionary.NullSymbol then
         --no generic dec
         ErrorHandler.SemanticError  (ErrNum   => 641,
                                      Reference => ErrorHandler.NoReference,
                                      Position => NodePosition (IdentNode),
                                      IdStr => IdentStr);
      elsif not Dictionary.IsGenericSubprogram (Sym) then
         --Sym is not a generic subprogram declaration
         ErrorHandler.SemanticError  (ErrNum   => 642,
                                      Reference => ErrorHandler.NoReference,
                                      Position => NodePosition (IdentNode),
                                      IdStr => IdentStr);
         Sym := Dictionary.NullSymbol;
      elsif ProcedureWhenFunctionExpected (Sym, Kind) then
         --wrong unit
         ErrorHandler.SemanticError  (ErrNum   => 643,
                                      Reference => ErrorHandler.NoReference,
                                      Position => NodePosition (IdentNode),
                                      IdStr => IdentStr);
         Sym := Dictionary.NullSymbol;
      elsif FunctionWhenProcedureExpected (Sym, Kind) then
         --wrong unit
         ErrorHandler.SemanticError  (ErrNum   => 644,
                                      Reference => ErrorHandler.NoReference,
                                      Position => NodePosition (IdentNode),
                                      IdStr => IdentStr);
         Sym := Dictionary.NullSymbol;
      elsif Dictionary.HasBody (Sym) then
         --already has body
         ErrorHandler.SemanticError  (ErrNum   => 13,
                                      Reference => ErrorHandler.NoReference,
                                      Position => NodePosition (IdentNode),
                                      IdStr => IdentStr);
         Sym := Dictionary.NullSymbol;
      else
         -- OK so far, just check function return type if applicable
         CheckFunctionReturnType  (Sym, ReturnTypeNode);
         if Sym /= Dictionary.NullSymbol then
            -- check above didn't find an error so add body
            Dictionary.AddBody (Sym,
                                Dictionary.Location'(NodePosition (Node),
                                                     NodePosition (Node)),
                                False);
            -- an enter the local scope of the now legal body
            Scope := Dictionary.LocalScope (Sym);
         end if;
      end if;
      SubProgSym := Sym;
   end CheckGenericSpecification;

   ----------------------------------------------------------------------

begin -- wf_subprogram_body
   IsGeneric := SyntaxNodeType (ParentNode (Node)) = SPSymbols.generic_subprogram_body;
   SubProgScope := Scope;

   -- introduced variable below because the scope we use to check formal parts of
   -- generic bodies is not he same as we need to use for other subprograms.  The
   -- problem occurs when the formal part contains a type which is a generic formal
   -- parameter.  This is only visible in the local scope of the generic subprogram
   -- whereas types in the formal part of a normal subprogram will always be visible
   -- in the scope where the subprogram is being declared.  For all but generics,
   -- the new variable is set as below, for generics it is set to the local scope
   -- of the subprogram (further below).
   ScopeForFormalPartCheck := Scope;

   -- NOTE: Given Ada83 declaration order restrictions, I /think/ that we could always
   -- check formal parts in subprogram local scope rather than, as above, sometimes
   -- doing it the scope in which the subprogram is being declared.  With relaxed ordering
   -- there /might/ be a problem with subunits thus:
   -- spec
   -- stub
   -- declarations that the body can't see -- of course these can't exist in 83
   -- the body (here we might see the declarations we didn't ought to?)
   -- Anyway,  I thought it best to leave the existing code alone and chnage the scope only
   -- for the generic case

   SpecNode := Child_Node (Node);
   GetAnnoAndConNodes (Next_Sibling (SpecNode),
                        --to get
                       AnnoNode,
                       ConstraintNode);
   -- set up identifier for hidden part reporting
   IdentNode := Child_Node (SpecNode);
   if SyntaxNodeType (IdentNode) /= SPSymbols.identifier then
      IdentNode := Child_Node (IdentNode);
   end if;
   --# assert True;
   FormalPartNode := Next_Sibling (Child_Node (SpecNode));
   SubprogImplemNode := LastSiblingOf (SpecNode);
   EndDesigNode := LastSiblingOf (Child_Node (SubprogImplemNode));
   MainNode := ParentNode (Node);
   if SyntaxNodeType (MainNode) /= SPSymbols.main_program_declaration then
      MainNode := STree.NullNode;
   end if;

   --# assert True;
   -- check to look for WITH node in case of subunit
   if SyntaxNodeType (ParentNode (ParentNode (Node))) =
      SPSymbols.subunit
   then --there may be a with node to deal with
      WithNode := Child_Node (Child_Node
                                  (ParentNode (ParentNode
                                               (ParentNode (ParentNode (Node))))));
      if SyntaxNodeType (WithNode) /= SPSymbols.with_clause then
         WithNode := STree.NullNode;
      end if;
   else --not a subunit so there is no with node
      WithNode := STree.NullNode;
   end if;

   --# assert True;
   Hidden := Body_Hidden_Class (SubprogImplemNode);

   NodeType := SyntaxNodeType (SpecNode);
   if NodeType = SPSymbols.procedure_specification then
      if IsGeneric then
         FirstSeen := False;
         CheckGenericSpecification (Node       => SpecNode,
                                    Kind       => GenericProcedure,
                                    Scope      => SubProgScope,
                                    SubProgSym => SubprogSym);
         ScopeForFormalPartCheck := SubProgScope; -- see comment above where var initialized
      else
         wf_procedure_specification (Node       => SpecNode,
                                     Hidden     => (Hidden = All_Hidden),
                                       --using and to get
                                     Scope      => SubProgScope,
                                       --to get
                                     SubProgSym => SubprogSym,
                                     FirstSeen  => FirstSeen);
      end if;
   else --must be a function
      if IsGeneric then
         FirstSeen := False;
         CheckGenericSpecification (Node       => SpecNode,
                                    Kind       => GenericFunction,
                                    Scope      => SubProgScope,
                                    SubProgSym => SubprogSym);
         ScopeForFormalPartCheck := SubProgScope; -- see comment above where var initialized
      else
         wf_function_specification (Node         => SpecNode,
                                    Hidden       => (Hidden = All_Hidden),
                                    --using and to get
                                    Scope        => SubProgScope,
                                    --to get
                                    SubProgSym   => SubprogSym,
                                    FirstSeen    => FirstSeen);
      end if;
   end if;

   --# assert True;

   if SubprogSym = Dictionary.NullSymbol then
      NextNode := STree.NullNode;
   else
      if MainNode /= STree.NullNode then
         wf_main_program (MainNode,
                          SubprogSym,
                          Scope,
                          SubProgScope);
      end if;

      if WithNode /= STree.NullNode then
         wf_context_clause (ParentNode (WithNode),
                            SubprogSym,
                            SubProgScope);
      end if;

      if FormalPartNode /= STree.NullNode then
         wf_formal_part (FormalPartNode,
                         ScopeForFormalPartCheck,
                         SubprogSym,
                         FirstSeen,
                         Dictionary.ProgramContext);

      elsif Dictionary.GetNumberOfSubprogramParameters (SubprogSym) /= 0 then
         ErrorHandler.SemanticError (152,
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     Dictionary.GetSimpleName (SubprogSym));

      end if;

      --# assert True;
      if AnnoNode = STree.NullNode then
         if not InSubunit (Node) then
            if not FirstSeen and then RequiresSecondAnnotation (SubprogSym) then
               ErrorHandler.SemanticError (87,
                                           ErrorHandler.NoReference,
                                           NodePosition (SpecNode),
                                           Dictionary.GetSimpleName (SubprogSym));
               Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsRefined,
                                                               SubprogSym);
            elsif FirstSeen and NodeType = SPSymbols.procedure_specification and
               (CommandLineData.IsSpark83 or
                CommandLineData.Content.DoInformationFlow)
            then
               ErrorHandler.SemanticError (154,
                                           ErrorHandler.NoReference,
                                           NodePosition (SpecNode),
                                           Dictionary.GetSimpleName (SubprogSym));
               Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsAbstract,
                                                               SubprogSym);
            end if;
         end if;
      else --an annotation is present
         if not (FirstSeen or RequiresSecondAnnotation (SubprogSym)) or else
            InSubunit (Node)
         then --annotation not required
            if NodeType = SPSymbols.procedure_specification then
               ErrorHandler.SemanticError (155,
                                           ErrorHandler.NoReference,
                                           NodePosition (AnnoNode),
                                           Dictionary.GetSimpleName (SubprogSym));
            else --must be a function
               --now distinguish between repeated anno and misplaced anno
               if Dictionary.IsNullIterator
                  (Dictionary.FirstGlobalVariable
                   (Dictionary.IsAbstract, SubprogSym)) then --misplaced anno
                  ErrorHandler.SemanticError (335,
                                              ErrorHandler.NoReference,
                                              NodePosition (AnnoNode),
                                              Dictionary.GetSimpleName (SubprogSym));
               else --duplicated anno
                  ErrorHandler.SemanticError (336,
                                              ErrorHandler.NoReference,
                                              NodePosition (AnnoNode),
                                              Dictionary.GetSimpleName (SubprogSym));
               end if;
            end if;
         else --annotation both present and required
            if NodeType = SPSymbols.procedure_specification then
               wf_procedure_annotation (AnnoNode,
                                        Scope,
                                        SubprogSym,
                                        FirstSeen);
            else
               wf_function_annotation (AnnoNode,
                                       Scope,
                                       SubprogSym,
                                       FirstSeen);
            end if;
         end if;
      end if;

      Scope := SubProgScope;
      NextNode := SpecNode;

      --# assert True;
      -- clause for production of "full" dependency clause using modes
      if CommandLineData.IsSpark95 and then
         Dictionary.IsProcedure (SubprogSym) and then
         not CommandLineData.Content.DoInformationFlow and then
         not InSubunit (Node)
      then
         if FirstSeen then
            CreateFullSubProgDependency (Node,
                                         SubprogSym,
                                         Dictionary.IsAbstract);

         elsif RequiresSecondAnnotation (SubprogSym) then
            CreateFullSubProgDependency (Node,
                                         SubprogSym,
                                         Dictionary.IsRefined);
         end if;
      end if;

      --# assert True;
      if Child_Node (ConstraintNode) /= STree.NullNode then
         -- a constraint exists; should it? Check here
         if not (FirstSeen or else
                   RequiresSecondAnnotation (SubprogSym) or else
                   HasParameterOrGlobalOfLocalPrivateType (SubprogSym)) or else
           InSubunit (Node)
         then -- annotation not required
            -- two possible errors: misplaced anno or duplicate anno
            if Dictionary.HasPrecondition (Dictionary.IsAbstract, SubprogSym) or else
              Dictionary.HasPostcondition (Dictionary.IsAbstract, SubprogSym)
            then -- illegal duplicate anno
               ErrorHandler.SemanticError (343,
                                           ErrorHandler.NoReference,
                                           NodePosition (ConstraintNode),
                                           Dictionary.GetSimpleName (SubprogSym));
            else --misplaced anno
               ErrorHandler.SemanticError (342,
                                           ErrorHandler.NoReference,
                                           NodePosition (ConstraintNode),
                                           Dictionary.GetSimpleName (SubprogSym));
            end if;
         else -- annotation is required so continue
            if NodeType = SPSymbols.procedure_specification then
               wf_procedure_constraint (ConstraintNode,
                                        Dictionary.LocalScope (SubprogSym),
                                        FirstSeen);
            else
               wf_function_constraint (ConstraintNode,
                                       Dictionary.LocalScope (SubprogSym),
                                       FirstSeen);
            end if;
         end if;
      end if;
   end if;

   --# assert True;
   case Hidden is
      when All_Hidden =>
         ErrorHandler.HiddenText (NodePosition (EndDesigNode),
                                  NodeLexString (IdentNode),
                                  SPSymbols.subprogram_implementation);
      when Handler_Hidden =>
         ErrorHandler.HiddenHandler (NodePosition (EndDesigNode),
                                     NodeLexString (IdentNode),
                                     SPSymbols.subprogram_implementation);
      when Not_Hidden =>
         null;
   end case;

   -- If a potentially inheritable subprogram of the same name exists then
   -- the new declaration is only legal if it successfully overrides it.
   -- This check is only required if the subprogram has not been previously declared
   -- because, if it has, the check will already have been done in the package spec
   if FirstSeen then
      CheckNoOverloadingFromTaggedOps (SpecNode,
                                       SubprogSym,
                                       Scope,
                                       Dictionary.IsRefined);
   end if;

   if Dictionary.IsMainProgram (SubprogSym) and
     not IsGeneric and
     CommandLineData.RavenscarSelected then
      SharedVariableCheck (MainProgram => SubprogSym,
                           Scope => SubProgScope,
                           ErrorNode => Node);
      MaxOneInAQueueCheck (MainProgram => SubprogSym,
                           Scope => SubProgScope,
                           ErrorNode => Node);
   end if;

   -- Check that function ends with a return statement; this check was previously done
   -- at up_wf_subprogram body where any error detected was too late to stop flow analysis
   -- or VC generation
   if SyntaxNodeType (Child_Node (Node)) = SPSymbols.function_specification then
      CheckFunctionHasReturn (Node, EndDesigNode);
   end if;
end wf_subprogram_body;
