-- $Id: sem-compunit-wf_simple_expression.adb 11946 2008-12-18 16:11:11Z rod chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

-- Overview: Called to check validity of a
-- simple_expression node.  Replaces calls to StaticSimpleExpression,
-- BaseTypeSimpleExpression and CheckTypeSimpleExpression
----------------------------------------------------------------------------

with SPrint;

separate (Sem.CompUnit)

procedure wf_simple_expression (Node                  : in     STree.SyntaxNode;
                                Scope                 : in     Dictionary.Scopes;
                                EStack                : in out ExpStack.ExpStackType;
                                TStack                : in     TypeContextStack.TStackType;
                                ContextRequiresStatic : in     Boolean;
                                IsAnnotation          : in     Boolean)
is
   Left,
   Right,
   Result       : ExpRecord;

   OpNode       : STree.SyntaxNode;
   PNode        : STree.SyntaxNode; -- Parent Node

   Operator   : SPSymbols.SPSymbol;

   POpNode   : STree.SyntaxNode;   -- Parent's Operator Node
   POperator : SPSymbols.SPSymbol; -- Parent's Operator Symbol

   LeftBase,
   RightBase : Dictionary.Symbol;
   LeftPos   : LexTokenManager.TokenPosition;
   RightPos  : LexTokenManager.TokenPosition;

   ResultStringLength : Maths.Value;

   function LengthOf  (S : in ExpRecord) return Maths.Value
   --# global in Dictionary.Dict;
   is
      Result : Maths.Value;
   begin
      if Dictionary.IsPredefinedCharacterType (Dictionary.GetRootType (S.TypeSymbol)) then
         Result := Maths.OneInteger;
      else
         Result := S.RangeRHS;
      end if;
      return Result;
   end LengthOf;

   function LengthSum (Left, Right : ExpRecord) return Maths.Value
   --# global in Dictionary.Dict;
   is
      Unused : Maths.ErrorCode;
      LeftVal,
      RightVal,
      Sum    : Maths.Value;
   begin
      LeftVal  := LengthOf (Left);
      RightVal := LengthOf (Right);

      --# accept Flow, 10, Unused, "Expected ineffective assignment" &
      --#        Flow, 33, Unused, "Expected to be neither referenced nor exported";
      Maths.Add (LeftVal,
                 RightVal,
                  --to get
                 Sum,
                 Unused);
      return Sum;
   end LengthSum;

begin --wf_simple_expression
   OpNode := Child_Node (Next_Sibling (Child_Node (Node)));

   if OpNode /= STree.NullNode then

      LeftPos  := NodePosition (Child_Node (Node));
      RightPos := NodePosition (Next_Sibling (Next_Sibling (Child_Node (Node))));

      --binary_add_op exists
      Operator := SyntaxNodeType (OpNode);
      ExpStack.Pop (Right, EStack);
      ExpStack.Pop (Left, EStack);
      Result := NullTypeRecord; --safety: we may not set all fields below

      -- do static checks first
      Result.IsConstant := Left.IsConstant and Right.IsConstant;

      -- LRM95(4.9) says that & can be a static function.
      -- LRM83(4.9) says that & is never static, so...
      if CommandLineData.IsSpark95 then
         Result.IsStatic := Left.IsStatic and Right.IsStatic;
      else
         Result.IsStatic := Left.IsStatic and Right.IsStatic and
           (Operator /= SPSymbols.ampersand);
      end if;

      Result.HasOperators := True;

      if Left.IsARange or Right.IsARange then
         Result := UnknownTypeRecord;
         ErrorHandler.SemanticError (90, ErrorHandler.NoReference,
                                     NodePosition (OpNode),
                                     LexTokenManager.NullString);
      else -- neither are ranges
         -- now do type compat and operator visibility checks
         if Operator = SPSymbols.ampersand then

            LeftBase  := Dictionary.GetRootType (Left.TypeSymbol);
            RightBase := Dictionary.GetRootType (Right.TypeSymbol);

            if ((Dictionary.IsPredefinedCharacterType (LeftBase) or
                   Dictionary.IsPredefinedStringType (LeftBase)) and
                  (Dictionary.IsPredefinedCharacterType (RightBase) or
                     Dictionary.IsPredefinedStringType (RightBase))) then

               -- "&" expressions in SPARK are always expected to be constant (SR 4.5.3)
               -- As such, we should be able to compute the length of the result.
               if Result.IsConstant then

                  ResultStringLength := LengthSum (Left, Right);
                  Result.RangeRHS := ResultStringLength;
                  Result.TypeSymbol := Dictionary.GetPredefinedStringType;
               else
                  Result := UnknownTypeRecord;
                  ErrorHandler.SemanticError (37,
                                              ErrorHandler.NoReference,
                                              NodePosition (Node),
                                              LexTokenManager.NullString);
               end if;

            else
               -- Types of Left and/or Right are wrong...
               Result := UnknownTypeRecord;
               ErrorHandler.SemanticError (35,
                                           ErrorHandler.NoReference,
                                           NodePosition (OpNode),
                                           LexTokenManager.NullString);
            end if;

         else  --its plus or minus
            CheckBinaryOperator (Operator,
                                 Left,
                                 Right,
                                 Scope,
                                 TStack,
                                 NodePosition (OpNode),
                                 LeftPos,
                                 RightPos,
                                 True, --with implicit type conversion
                                 IsAnnotation,
                                 -- using and to get
                                 Result);

            -- seeding of return type of operator for expression overflow checks in VCG
            STree.AddNodeSymbol (OpNode, Result.TypeSymbol);

            CalcBinaryOperator (Node,
                                Operator,
                                Left.Value,
                                Right.Value,
                                IsAnnotation,
                                 --using and to get
                                Result);

            PNode := ParentNode (Node);

            if (not IsAnnotation) and then
               SyntaxNodeType (PNode) = SPSymbols.simple_expression
            then
               ----------------------------------------------------------------
               -- If the parent is also a simple_expression, then we must have
               -- an unparenthesized expression with two adding operators, such
               -- as A + B + C
               --
               -- Here, we issue warning 302 to warn of potential evaluation
               -- order dependency.
               --
               -- We can reduce false-alarm rate here by suppressing the
               -- warning in two specific cases:
               --  a) If the sub-expression under consideration is static
               --     AND the expression as a whole appears in a context
               --     that requires a static expression.  Example: a type
               --     declaration such as
               --       type T is range B + 2 - 3 .. 10;
               --   or
               --  b) A modular-typed expression where the two operators
               --     under consideration are both the same and
               --     commutative.  For example:
               --       A := A + B + C;
               --     where A, B, and C are all of the same modular
               --     (sub-)type.
               --
               -- The same logic is used in wf_term for multiplying
               -- operators.
               ----------------------------------------------------------------
               POpNode := Child_Node (Next_Sibling (Node));

               POperator := SyntaxNodeType (POpNode);

               if (ContextRequiresStatic and Result.IsStatic)
                 or else
                 (Dictionary.TypeIsModular (Result.TypeSymbol) and
                    Ops_Are_Same_And_Commutative (Operator, POperator)) then
                  null;
               else
                  ErrorHandler.SemanticWarning (302,
                                                NodePosition (Node),
                                                LexTokenManager.NullString);
               end if;
            end if;
         end if;
      end if;
      Result.ErrorsInExpression := Result.ErrorsInExpression or
         Left.ErrorsInExpression or
         Right.ErrorsInExpression;

      -- OtherSymbol may carry a function symbol in the case of uses of unchecked_conversion.
      -- This symbol is used (by wf_Assign) to convery information to the VCG to supress
      -- checks when an unchecked_conversion is assigned to something of the same subtype.
      -- We do not want this mechanism if the unchecked_conversion is sued in any other context
      -- than a direct assignment.  Therefore we clear OtherSymbol here:
      Result.OtherSymbol := Dictionary.NullSymbol;
      ExpStack.Push (Result, EStack);
   end if;


end wf_simple_expression;
