-- $Id: sem-compunit-wf_renaming_declaration.adb 12351 2009-02-02 15:03:51Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with EStrings;

separate (Sem.CompUnit)

procedure wf_renaming_declaration (Node  : in STree.SyntaxNode;
                                   Scope : in Dictionary.Scopes)
is
   ChildNode           : STree.SyntaxNode;

   --========================================================================

   procedure CheckPosition (Node       : in     STree.SyntaxNode;
                            PackString : in     LexTokenManager.LexString;
                            Scope      : in     Dictionary.Scopes;
                            PosOk      :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        PackString,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         PosOk                     from Node,
   --#                                        PackString,
   --#                                        Scope,
   --#                                        STree.Table;
   is
      LastNode : STree.SyntaxNode;
      Ident    : LexTokenManager.LexString;
   begin --CheckPosition
      if Dictionary.IsVisibleScope (Scope) then
         -- If this renaming is part of the visible part of a package, then
         -- it must be in legal position because syntax ensures it is.
         PosOk := True;
      else
         LastNode := ParentNode (Node);
         if SyntaxNodeType (LastNode) = SPSymbols.initial_declarative_item_rep then
            -- If a renaming is part of an initial_declartive_item_rep, then
            -- it should follow be following an embedded package declaration
            Ident := FindPreviousPackage (LastNode);
            if Ident = LexTokenManager.NullString then
               -- There is no preceding package
               ErrorHandler.SemanticError (300,
                                           ErrorHandler.NoReference,
                                           NodePosition (Node),
                                           LexTokenManager.NullString);
               PosOk := False;
            elsif Ident /= PackString then
               -- There is a preceding package, but it's the wrong one!
               ErrorHandler.SemanticError (301,
                                           ErrorHandler.NoReference,
                                           NodePosition (Node),
                                           Ident);
               PosOk := False;
            else
               -- There is a preceding package and it _is_ the one
               -- mentioned in the renaming
               PosOk := True;
            end if;
         else
            -- This renaming is part of a renaming_declaration_rep

            -- Find the enclosing declarative_part node
            loop
               LastNode := ParentNode (LastNode);
               exit when SyntaxNodeType (LastNode) = SPSymbols.declarative_part;
            end loop;

            LastNode := ParentNode (LastNode);
            if SyntaxNodeType (LastNode) = SPSymbols.subprogram_implementation and then
               SyntaxNodeType (ParentNode (ParentNode (LastNode))) /= SPSymbols.main_program_declaration
            then
               -- If a renaming is in a subprogram implementation, but that subprogram
               -- isn't the main program, then error
               ErrorHandler.SemanticError (300,
                                           ErrorHandler.NoReference,
                                           NodePosition (Node),
                                           LexTokenManager.NullString);
               PosOk := False;
            else
               PosOk := True;
            end if;
         end if;
      end if;
   end CheckPosition;

   --========================================================================

   procedure CheckOperatorRenaming (Node  : in STree.SyntaxNode;
                                    Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict           from *,
   --#                                        CommandLineData.Content,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      type ExpectedArgumentType is (None, One, OneOrTwo, Two);
      type ParamCountType is (OneParameter, TwoParameters, TooManyParameters);

      PackNode,
      TypeNode,
      FormalPartNode,
      OpNode,
      ChildNode,
      FirstParamNode,
      SecondParamNode,
      SecondTypeNode,
      FirstTypeNode       : STree.SyntaxNode;
      FirstType,
      SecondType,
      ReturnType,
      ReturnTypeGiven     : Dictionary.Symbol;
      Op1,
      Op2                 : SPSymbols.SPSymbol;
      Expected1,
      Expected2_unused    : ExpectedArgumentType;
      Ok,
      Defined,
      AlreadyVisible      : Boolean;
      NumberFound         : ParamCountType;
      OpName1_unused,
      OpName2,
      FirstIdent,
      SecondIdent         : LexTokenManager.LexString;

      ------------------------------------------------------------------

      procedure wf_operator_symbol (Node   : in     STree.SyntaxNode;
                                    Op     :    out SPSymbols.SPSymbol;
                                    OpName :    out LexTokenManager.LexString;
                                    Params :    out ExpectedArgumentType)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in     STree.Table;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.StringTable,
      --#                                        Node,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table &
      --#         Op,
      --#         OpName,
      --#         Params                    from LexTokenManager.StringTable,
      --#                                        Node,
      --#                                        STree.Table;
      is
         OpString    : EStrings.T;
         OpNameLocal : LexTokenManager.LexString;

      begin
         OpNameLocal := NodeLexString (Child_Node (Node));
         LexTokenManager.LexStringToString (OpNameLocal,
                                            OpString);
         if OpString.Content (2) = '=' and then OpString.Length = 3 then
            Op := SPSymbols.equals;
            Params := Two;
         elsif OpString.Content (2) = '/' and then
            OpString.Content (3) = '=' and then
            OpString.Length = 4
         then
            Op := SPSymbols.not_equal;
            Params := Two;
         elsif OpString.Content (2) = '<' and then OpString.Length = 3 then
            Op := SPSymbols.less_than;
            Params := Two;
         elsif OpString.Content (2) = '<' and then
            OpString.Content (3) = '=' and then
            OpString.Length = 4
         then
            Op := SPSymbols.less_or_equal;
            Params := Two;
         elsif OpString.Content (2) = '>' and then OpString.Length = 3 then
            Op := SPSymbols.greater_than;
            Params := Two;
         elsif OpString.Content (2) = '>' and then
            OpString.Content (3) = '=' and then
            OpString.Length = 4
         then
            Op := SPSymbols.greater_or_equal;
            Params := Two;
         elsif OpString.Content (2) = '+' and then OpString.Length = 3 then
            Op := SPSymbols.plus;
            Params := OneOrTwo;
         elsif OpString.Content (2) = '-' and then OpString.Length = 3 then
            Op := SPSymbols.minus;
            Params := OneOrTwo;
         elsif OpString.Content (2) = '/' and then OpString.Length = 3 then
            Op := SPSymbols.divide;
            Params := Two;
         elsif OpString.Content (2) = '*' and then OpString.Length = 3 then
            Op := SPSymbols.multiply;
            Params := Two;
         elsif OpString.Content (2) = '&' and then OpString.Length = 3 then
            Op := SPSymbols.ampersand;
            Params := Two;
         elsif OpString.Content (2) = '*' and then
            OpString.Content (3) = '*' and then
            OpString.Length = 4
         then
            Op := SPSymbols.double_star;
            Params := Two;
         elsif OpString.Content (2) = 'r' and then
            OpString.Content (3) = 'e' and then
            OpString.Content (4) = 'm' and then
            OpString.Length = 5
         then
            Op := SPSymbols.RWrem;
            Params := Two;
         elsif OpString.Content (2) = 'm' and then
            OpString.Content (3) = 'o' and then
            OpString.Content (4) = 'd' and then
            OpString.Length = 5
         then
            Op := SPSymbols.RWmod;
            Params := Two;
         elsif OpString.Content (2) = 'a' and then
            OpString.Content (3) = 'b' and then
            OpString.Content (4) = 's' and then
            OpString.Length = 5
         then
            Op := SPSymbols.RWabs;
            Params := One;
         elsif OpString.Content (2) = 'a' and then
            OpString.Content (3) = 'n' and then
            OpString.Content (4) = 'd' and then
            OpString.Length = 5
         then
            Op := SPSymbols.RWand;
            Params := Two;
         elsif OpString.Content (2) = 'o' and then
            OpString.Content (3) = 'r' and then
            OpString.Length = 4
         then
            Op := SPSymbols.RWor;
            Params := Two;
         elsif OpString.Content (2) = 'x' and then
            OpString.Content (3) = 'o' and then
            OpString.Content (4) = 'r' and then
            OpString.Length = 5
         then
            Op := SPSymbols.RWxor;
            Params := Two;
         elsif OpString.Content (2) = 'n' and then
            OpString.Content (3) = 'o' and then
            OpString.Content (4) = 't' and then
            OpString.Length = 5
         then
            Op := SPSymbols.RWnot;
            Params := One;
         else
            Op := SPSymbols.SPEND;
            OpNameLocal := LexTokenManager.NullString;
            Params := None;
            ErrorHandler.SemanticError (20,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        LexTokenManager.NullString);
         end if;
         OpName := OpNameLocal;
      end wf_operator_symbol;

      -------------------------------------------------------------------------

      procedure GetParameters (Node           : in     STree.SyntaxNode;
                               Scope          : in     Dictionary.Scopes;
                               NumberFound    :    out ParamCountType;
                               FirstIdent     :    out LexTokenManager.LexString;
                               FirstType      :    out Dictionary.Symbol;
                               SecondIdent    :    out LexTokenManager.LexString;
                               SecondType     :    out Dictionary.Symbol;
                               FirstNode      :    out STree.SyntaxNode;
                               SecondNode     :    out STree.SyntaxNode;
                               FirstTypeNode  :    out STree.SyntaxNode;
                               SecondTypeNode :    out STree.SyntaxNode)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in     STree.Table;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.StringTable,
      --#                                        Node,
      --#                                        Scope,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table &
      --#         FirstIdent,
      --#         FirstNode,
      --#         FirstTypeNode,
      --#         NumberFound,
      --#         SecondIdent,
      --#         SecondNode,
      --#         SecondTypeNode            from Node,
      --#                                        STree.Table &
      --#         FirstType,
      --#         SecondType                from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        Node,
      --#                                        Scope,
      --#                                        STree.Table;
      is
         NextNode         : STree.SyntaxNode;
         NumberFoundLocal : ParamCountType;
         TypeSym          : Dictionary.Symbol;

         -----------------------------------------

         procedure CheckMode (ModeNode : in STree.SyntaxNode)
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     LexTokenManager.StringTable;
         --#        in     STree.Table;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives ErrorHandler.ErrorContext,
         --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        LexTokenManager.StringTable,
         --#                                        ModeNode,
         --#                                        SPARK_IO.FILE_SYS,
         --#                                        STree.Table;
         is
            NextNode : STree.SyntaxNode;

         begin
            NextNode := Child_Node (ModeNode);
            if NextNode /= STree.NullNode and then
               SyntaxNodeType (NextNode) /= SPSymbols.in_mode
            then
               ErrorHandler.SemanticError (64,
                                           ErrorHandler.NoReference,
                                           NodePosition (ParentNode (ModeNode)),
                                           LexTokenManager.NullString);
            end if;
         end CheckMode;

         -----------------------------------------

         procedure CheckExtraBranches (Node  : in     STree.SyntaxNode;
                                       Count : in out ParamCountType)
         --# global in STree.Table;
         --# derives Count from *,
         --#                    Node,
         --#                    STree.Table;
         is
            NextNode : STree.SyntaxNode;

         begin
            NextNode := Node;
            loop
               NextNode := ParentNode (NextNode);
               exit when SyntaxNodeType (NextNode) =
                  SPSymbols.formal_part;

               if Next_Sibling (NextNode) /=
                  STree.NullNode
               then
                  Count := TooManyParameters;
                  exit;
               end if;
            end loop;
         end CheckExtraBranches;

         -----------------------------------------

      begin --GetParameters
         FirstType   := Dictionary.GetUnknownTypeMark;
         FirstTypeNode := STree.NullNode;
         SecondTypeNode := STree.NullNode;
         SecondIdent := LexTokenManager.NullString;
         SecondType  := Dictionary.GetUnknownTypeMark;
         SecondNode  := STree.NullNode;
         NumberFoundLocal := OneParameter;

         NextNode := LastChildOf (Node);
         FirstIdent := NodeLexString (NextNode);
         FirstNode  := NextNode;
         NextNode := Next_Sibling (ParentNode (NextNode));
         if SyntaxNodeType (NextNode) = SPSymbols.identifier then
            --" (LEFT, RIGHT : TYPE)" construction
            NumberFoundLocal := TwoParameters;
            SecondIdent := NodeLexString (NextNode);
            SecondNode := NextNode;
            NextNode    := Next_Sibling (ParentNode (NextNode));
            if SyntaxNodeType (NextNode) /= SPSymbols.mode then
               NumberFoundLocal := TooManyParameters;
            else
               CheckMode (NextNode);
               NextNode := Next_Sibling (NextNode);
               FirstTypeNode := NextNode;
               SecondTypeNode := NextNode;
               wf_type_mark (NextNode,
                             Scope,
                             Dictionary.ProgramContext,
                              --to get
                             TypeSym);
               FirstType := TypeSym;
               SecondType := TypeSym;
               CheckExtraBranches (NextNode,
                                    --using and to to get
                                   NumberFoundLocal);
            end if;
         else --" (LEFT : TYPE; RIGHT : TYPE)" or " (RIGHT : TYPE)" construction
            CheckMode (NextNode);
            NextNode := Next_Sibling (NextNode);
            FirstTypeNode := NextNode;
            SecondTypeNode := NextNode;
            wf_type_mark (NextNode,
                          Scope,
                          Dictionary.ProgramContext,
                           --to get
                          FirstType);
            NextNode := ParentNode (ParentNode (NextNode));
            if Next_Sibling (NextNode) =
               STree.NullNode
            then --" (RIGHT : TYPE)" construction
               CheckExtraBranches (NextNode,
                                    --using and to to get
                                   NumberFoundLocal);
            else --" (LEFT : TYPE; RIGHT : TYPE)" construction
               NumberFoundLocal := TwoParameters;
               NextNode := LastChildOf (Next_Sibling (NextNode));
               SecondIdent := NodeLexString (NextNode);
               SecondNode := NextNode;
               NextNode    := Next_Sibling (ParentNode (NextNode));
               if SyntaxNodeType (NextNode) /= SPSymbols.mode then
                  NumberFoundLocal := TooManyParameters;
               else
                  CheckMode (NextNode);
                  NextNode := Next_Sibling (NextNode);
                  SecondTypeNode := NextNode;
                  wf_type_mark (NextNode,
                                Scope,
                                Dictionary.ProgramContext,
                                 --to get
                                SecondType);
                  CheckExtraBranches (NextNode,
                                       --using and to to get
                                      NumberFoundLocal);
               end if;
            end if;
         end if;
         NumberFound := NumberFoundLocal;
      end GetParameters;

      -------------------------------------------------------------------------

      function NumberCorrect (Expected : ExpectedArgumentType;
                              Found    : ParamCountType) return Boolean
      is
         Ok : Boolean;
      begin
         if Found = TooManyParameters                  or else
            (Found = TwoParameters and Expected = One) or else
            (Found = OneParameter  and Expected = Two)
         then
            Ok := False;
         else
            Ok := True;
         end if;
         return Ok;
      end NumberCorrect;

      -------------------------------------------------------------------------

      procedure CheckNamesRight (NumberFound : in ParamCountType;
                                 FirstIdent,
                                 SecondIdent : in LexTokenManager.LexString;
                                 FirstNode   : in STree.SyntaxNode;
                                 SecondNode  : in STree.SyntaxNode)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in     STree.Table;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        FirstIdent,
      --#                                        FirstNode,
      --#                                        LexTokenManager.StringTable,
      --#                                        NumberFound,
      --#                                        SecondIdent,
      --#                                        SecondNode,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table;
      is
      begin
         if NumberFound = OneParameter then
            if FirstIdent /= LexTokenManager.RightToken then
               ErrorHandler.SemanticError (65,
                                           ErrorHandler.NoReference,
                                           NodePosition (FirstNode),
                                           LexTokenManager.NullString);
            end if;
         else --two parameters
            if FirstIdent /= LexTokenManager.LeftToken then
               ErrorHandler.SemanticError (65,
                                           ErrorHandler.NoReference,
                                           NodePosition (FirstNode),
                                           LexTokenManager.NullString);
            end if;
            if SecondIdent /= LexTokenManager.RightToken then
               ErrorHandler.SemanticError (65,
                                           ErrorHandler.NoReference,
                                           NodePosition (SecondNode),
                                           LexTokenManager.NullString);
            end if;
         end if;
      end CheckNamesRight;

      -------------------------------------------------------------------------

      procedure CheckTypes (NumberFound : in     ParamCountType;
                            Op          : in     SPSymbols.SPSymbol;
                            Scope       : in     Dictionary.Scopes;
                            FirstType,
                            SecondType  : in     Dictionary.Symbol;
                            ChildNode   : in     STree.SyntaxNode;
                            Ok          :    out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in     STree.Table;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from ChildNode,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        FirstType,
      --#                                        LexTokenManager.StringTable,
      --#                                        NumberFound,
      --#                                        Op,
      --#                                        Scope,
      --#                                        SecondType,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table &
      --#         Ok                        from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        FirstType,
      --#                                        NumberFound,
      --#                                        Op,
      --#                                        Scope,
      --#                                        SecondType;
      is
      begin
         Ok := True;
         if NumberFound = OneParameter then
            if not (Dictionary.IsType (FirstType) or
                    Dictionary.IsUnknownTypeMark (FirstType))
            then
               Ok := False;
               ErrorHandler.SemanticError (68,
                                           ErrorHandler.NoReference,
                                           NodePosition (ChildNode),
                                           LexTokenManager.NullString);
            elsif Dictionary.TypeIsLimited (FirstType, Scope) then
               Ok := False;
               ErrorHandler.SemanticError (35,
                                           ErrorHandler.NoReference,
                                           NodePosition (Next_Sibling (ChildNode)),
                                           Dictionary.GetSimpleName (FirstType));
            elsif Dictionary.IsPrivateType (FirstType, Scope) and then
               -- For private type Time_Span, unary "-" and "abs" can be renamed.
               -- No other unary operators are renamable for any private type.
               -- Note that we do not need to do a test to see if the operator is
               -- defined here, since that is covered by CheckOperator.
              (not CommandLineData.RavenscarSelected or else
               FirstType /= Dictionary.GetPredefinedTimeSpanType)
            then
               Ok := False;
               ErrorHandler.SemanticError (35,
                                           ErrorHandler.NoReference,
                                           NodePosition (Next_Sibling (ChildNode)),
                                           Dictionary.GetSimpleName (FirstType));

            elsif (Dictionary.TypeIsModular (FirstType) and
                   (Op = SPSymbols.RWabs or
                    Op = SPSymbols.plus or
                    Op = SPSymbols.minus)) then

               -- Unary arithmetic operators are not allowed for modular types,
               -- so their renamings must also be illegal.  Note we _do_ allow
               -- logical unary "not" on modular types.
               Ok := False;
               ErrorHandler.SemanticError (803,
                                           ErrorHandler.NoReference,
                                           NodePosition (ChildNode),
                                           LexTokenManager.NullString);
            end if;
         else --two parameters
            if not ((Dictionary.IsType (FirstType) or
                     Dictionary.IsUnknownTypeMark (FirstType))
                    and
                    (Dictionary.IsType (SecondType) or
                     Dictionary.IsUnknownTypeMark (SecondType)))
            then
               Ok := False;
               ErrorHandler.SemanticError (68,
                                           ErrorHandler.NoReference,
                                           NodePosition (ChildNode),
                                           LexTokenManager.NullString);
            elsif Dictionary.TypeIsLimited (FirstType, Scope) then
               Ok := False;
               ErrorHandler.SemanticError (35,
                                           ErrorHandler.NoReference,
                                           NodePosition (Next_Sibling (ChildNode)),
                                           Dictionary.GetSimpleName (FirstType));
            elsif Dictionary.IsPrivateType (FirstType, Scope) and then
               -- For private types Time and Time_Span, all defined binary operators
               -- can be renamed. For other private types, only "=" can be renamed.
               Op /= SPSymbols.equals and then
               (not CommandLineData.RavenscarSelected or else
                not Dictionary.IsPredefinedTimeType (FirstType))
            then
               Ok := False;
               ErrorHandler.SemanticError (35,
                                           ErrorHandler.NoReference,
                                           NodePosition (Next_Sibling (ChildNode)),
                                           Dictionary.GetSimpleName (FirstType));
            elsif Dictionary.TypeIsLimited (SecondType, Scope) then
               Ok := False;
               ErrorHandler.SemanticError (35,
                                           ErrorHandler.NoReference,
                                           NodePosition (Next_Sibling (ChildNode)),
                                           Dictionary.GetSimpleName (SecondType));
            elsif Dictionary.IsPrivateType (SecondType, Scope) and then
               Op /= SPSymbols.equals and then
               (not CommandLineData.RavenscarSelected or else
                not Dictionary.IsPredefinedTimeType (SecondType))
            then
               Ok := False;
               ErrorHandler.SemanticError (35,
                                           ErrorHandler.NoReference,
                                           NodePosition (Next_Sibling (ChildNode)),
                                           Dictionary.GetSimpleName (SecondType));
            end if;
         end if;
      end CheckTypes;

      -------------------------------------------------------------------------

      procedure CheckOperator (NumberFound    : in     ParamCountType;
                               Op             : in     SPSymbols.SPSymbol;
                               Defined,
                               AlreadyVisible :    out Boolean;
                               ReturnType     :    out Dictionary.Symbol)
      --# global in CommandLineData.Content;
      --#        in Dictionary.Dict;
      --#        in FirstType;
      --#        in Scope;
      --#        in SecondType;
      --# derives AlreadyVisible from CommandLineData.Content,
      --#                             Dictionary.Dict,
      --#                             FirstType,
      --#                             NumberFound,
      --#                             Op,
      --#                             Scope,
      --#                             SecondType &
      --#         Defined,
      --#         ReturnType     from CommandLineData.Content,
      --#                             Dictionary.Dict,
      --#                             FirstType,
      --#                             NumberFound,
      --#                             Op,
      --#                             SecondType;
      is
         DefinedLocal : Boolean;

      begin
         AlreadyVisible := False;
         ReturnType := Dictionary.GetUnknownTypeMark;

         if NumberFound = OneParameter then
            DefinedLocal := Dictionary.UnaryOperatorIsDefined (Op,
                                                               FirstType);
            if DefinedLocal then
               ReturnType := FirstType;
               AlreadyVisible := Dictionary.UnaryOperatorIsVisible (Op,
                                                                    FirstType,
                                                                    Scope);
            end if;
         else
            DefinedLocal := Dictionary.BinaryOperatorIsDefined (Op,
                                                                FirstType,
                                                                SecondType);
            if DefinedLocal then
               ReturnType := Dictionary.GetBinaryOperatorType (Op,
                                                               FirstType,
                                                               SecondType);
               AlreadyVisible := Dictionary.BinaryOperatorIsVisible (Op,
                                                                     FirstType,
                                                                     SecondType,
                                                                     Scope);
            end if;
         end if;
         Defined := DefinedLocal;
      end CheckOperator;

      ------------------------------------------------------------------------

      function RenamedFromRightPlace (PackNode,
                                         FirstTypeNode,
                                         SecondTypeNode : STree.SyntaxNode)
                                     return Boolean
      --# global in STree.Table;
      is
         Ok : Boolean;

         ------------------------------------------------------------------------

         function NamesMatch (PackNode, TypeNode : STree.SyntaxNode)
                             return Boolean
         --# global in STree.Table;
         is
            CurrPackNode, CurrTypeNode : STree.SyntaxNode;
            Match : Boolean;
            PackDone, TypeDone : Boolean;
         begin
            if TypeNode = STree.NullNode or else
               SyntaxNodeType (Child_Node (Child_Node (TypeNode))) /=
               SPSymbols.dotted_simple_name
            then -- no prefix on type mark
               Match := False;
            else
               CurrPackNode :=  LastChildOf (PackNode);
               CurrTypeNode :=  LastChildOf (TypeNode);

               loop
                  Match := NodeLexString (CurrPackNode) =
                     NodeLexString (CurrTypeNode);

                  exit when not Match;

                  CurrPackNode := ParentNode (CurrPackNode);
                  PackDone := CurrPackNode = ParentNode (PackNode);

                  CurrTypeNode := ParentNode (CurrTypeNode);
                  TypeDone := CurrTypeNode = Child_Node (Child_Node (TypeNode));

                  if PackDone or TypeDone then
                     Match := PackDone and TypeDone;
                     exit;
                  end if;

                  -- move on to next identifiers in names:
                  CurrPackNode := Next_Sibling (CurrPackNode);
                  CurrTypeNode := Next_Sibling (CurrTypeNode);
               end loop;
            end if;

            return Match;
         end NamesMatch;
         ------------------------------------------------------------------------

      begin -- RenamedFromRightPlace
         Ok := NamesMatch (PackNode, FirstTypeNode);
         if not Ok then
            Ok := NamesMatch (PackNode, SecondTypeNode);
         end if;
         return Ok;
      end RenamedFromRightPlace;

      ------------------------------------------------------------------------

      procedure DoRename (NumberFound  : in ParamCountType;
                          Op           : in SPSymbols.SPSymbol;
                          FirstType,
                          SecondType   : in Dictionary.Symbol;
                          Scope        : in Dictionary.Scopes)
      --# global in     LexTokenManager.StringTable;
      --#        in out Dictionary.Dict;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives Dictionary.Dict   from *,
      --#                                FirstType,
      --#                                NumberFound,
      --#                                Op,
      --#                                Scope,
      --#                                SecondType &
      --#         SPARK_IO.FILE_SYS from *,
      --#                                Dictionary.Dict,
      --#                                FirstType,
      --#                                LexTokenManager.StringTable,
      --#                                NumberFound,
      --#                                Op,
      --#                                Scope,
      --#                                SecondType;
      is
      begin
         if NumberFound = OneParameter then
            Dictionary.RenameUnaryOperator (Op,
                                            FirstType,
                                            Scope);
         else
            Dictionary.RenameBinaryOperator (Op,
                                             FirstType,
                                             SecondType,
                                             Scope);
         end if;
      end DoRename;

      ------------------------------------------------------------------------

   begin  -- CheckOperatorRenaming
      ChildNode := Child_Node (Node);
      FormalPartNode := Next_Sibling (ChildNode);
      TypeNode := Next_Sibling (FormalPartNode);
      PackNode := Child_Node (Next_Sibling (TypeNode));
      OpNode := Next_Sibling (Next_Sibling (TypeNode));
      -- Check for dotted package name:
      if SyntaxNodeType (PackNode) /= SPSymbols.identifier then
         if CommandLineData.IsSpark95 then
            CheckPosition (Node,
                           LexTokenManager.NullString,
                           Scope,
                           --to get
                           Ok);
         else
            --report error at second identifier in name
            PackNode := Next_Sibling (ParentNode (LastChildOf (PackNode)));
            ErrorHandler.SemanticError (610,
                                        ErrorHandler.NoReference,
                                        NodePosition (PackNode),
                                        LexTokenManager.NullString);
            Ok := False;
         end if;
      else
         CheckPosition (Node,
                        NodeLexString (PackNode),
                        Scope,
                        --to get
                        Ok);
      end if;

      if Ok then
         --# accept Flow, 10, OpName1_unused, "Expected ineffective assignment";
         wf_operator_symbol (ChildNode,
                              --to get
                             Op1,
                             OpName1_unused,
                             Expected1);
         --# end accept;
         --# accept Flow, 10, Expected2_unused, "Expected ineffective assignment";
         wf_operator_symbol (OpNode,
                              --to get
                             Op2,
                             OpName2,
                             Expected2_unused);
         --# end accept;
         if Op1 /= Op2 then
            ErrorHandler.SemanticError (303,
                                        ErrorHandler.NoReference,
                                        NodePosition (OpNode),
                                        LexTokenManager.NullString);
            -- check that /= is not being renamed
         elsif Op1 = SPSymbols.not_equal then
            ErrorHandler.SemanticError (304,
                                        ErrorHandler.NoReference,
                                        NodePosition (ChildNode),
                                        NodeLexString (Child_Node (ChildNode)));
         else --operator strings are the same (and neither is /=)
            GetParameters (FormalPartNode,
                           Scope,
                           --to get
                           NumberFound,
                           FirstIdent,
                           FirstType,
                           SecondIdent,
                           SecondType,
                           FirstParamNode,
                           SecondParamNode,
                           FirstTypeNode,
                           SecondTypeNode);
            if NumberCorrect (Expected1, NumberFound) then
               CheckNamesRight (NumberFound,
                                FirstIdent,
                                SecondIdent,
                                FirstParamNode,
                                SecondParamNode);
               CheckTypes (NumberFound,
                           Op1,
                           Scope,
                           FirstType,
                           SecondType,
                           ChildNode,
                           --to get
                           Ok);
               if Ok then
                  CheckOperator (NumberFound,
                                 Op1,
                                 --to get
                                 Defined,
                                 AlreadyVisible,
                                 ReturnType);
                  if not Defined then
                     ErrorHandler.SemanticError (35,
                                                 ErrorHandler.NoReference,
                                                 NodePosition (FormalPartNode),
                                                 LexTokenManager.NullString);
                  elsif AlreadyVisible then
                     ErrorHandler.SemanticError (306,
                                                 ErrorHandler.NoReference,
                                                 NodePosition (OpNode),
                                                 OpName2);
                  else
                     wf_type_mark (TypeNode,
                                   Scope,
                                   Dictionary.ProgramContext,
                                    --to get
                                   ReturnTypeGiven);
                     if ReturnType /= ReturnTypeGiven then
                        ErrorHandler.SemanticError (305,
                                                    ErrorHandler.NoReference,
                                                    NodePosition (TypeNode),
                                                    OpName2);
                     elsif RenamedFromRightPlace (PackNode,
                                                  FirstTypeNode,
                                                  SecondTypeNode)
                     then
                        DoRename (NumberFound,
                                  Op1,
                                  FirstType,
                                  SecondType,
                                  Scope);
                     else
                        if SyntaxNodeType (PackNode) /=
                           SPSymbols.identifier
                        then
                           -- use last identifier in package name to report error
                           PackNode := Next_Sibling (PackNode);
                        end if;
                        ErrorHandler.SemanticError (307,
                                                    ErrorHandler.NoReference,
                                                    NodePosition (OpNode),
                                                    NodeLexString (PackNode));
                     end if;
                  end if;
               end if;
            else -- number of parameters wrong
               ErrorHandler.SemanticError (305,
                                           ErrorHandler.NoReference,
                                           NodePosition (FormalPartNode),
                                           OpName2);
            end if;
         end if;
      end if;
      --# accept Flow, 33, Expected2_unused, "Expected to be neither referenced nor exported" &
      --#        Flow, 33, OpName1_unused, "Expected to be neither referenced nor exported";
   end CheckOperatorRenaming;

   --========================================================================

   procedure CheckSubprogramRenaming (Node  : in STree.SyntaxNode;
                                      Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict           from *,
   --#                                        CommandLineData.Content,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      type SubprogSorts is (Func, Proc);

      PackSym,
      SubprogSym  : Dictionary.Symbol;
      ChildNode,
      FormalPartNode,
      PackNode,
      SubprogNode : STree.SyntaxNode;
      Ok          : Boolean;
      SubprogSort : SubprogSorts;

      --------------------------------------------

      procedure CheckNamesSame (Node,
                                SubprogNode : in     STree.SyntaxNode;
                                Ok          :    out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in     STree.Table;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.StringTable,
      --#                                        Node,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        SubprogNode &
      --#         Ok                        from Node,
      --#                                        STree.Table,
      --#                                        SubprogNode;
      is
      begin
         if NodeLexString (SubprogNode) /=
            NodeLexString (LastChildOf (Node))
         then
            Ok := False;
            ErrorHandler.SemanticError (312,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        LexTokenManager.NullString);
         else
            Ok := True;
         end if;
      end CheckNamesSame;

      --------------------------------------------

      procedure FindPackage (Scope    : in     Dictionary.Scopes;
                             PackNode : in     STree.SyntaxNode;
                             PackSym  :    out Dictionary.Symbol)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in     STree.Table;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.StringTable,
      --#                                        PackNode,
      --#                                        Scope,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table &
      --#         PackSym                   from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        PackNode,
      --#                                        Scope,
      --#                                        STree.Table;
      is
         CurrNode : STree.SyntaxNode;
         Sym      : Dictionary.Symbol;
         PrefixOk : Boolean;

      begin
         CurrNode := LastChildOf (PackNode);
         Sym := Dictionary.LookupItem (NodeLexString (CurrNode),
                                       Scope,
                                       Dictionary.ProgramContext);

         loop -- loop to handle multiple prefixes
            if Sym = Dictionary.NullSymbol then
               ErrorHandler.SemanticError (1,
                                           ErrorHandler.NoReference,
                                           NodePosition (CurrNode),
                                           NodeLexString (CurrNode));
               exit;
            end if;

            if not Dictionary.IsPackage (Sym) then
               ErrorHandler.SemanticError (18,
                                           ErrorHandler.NoReference,
                                           NodePosition (CurrNode),
                                           NodeLexString (CurrNode));
               Sym := Dictionary.NullSymbol;
               exit;
            end if;

            CheckPackagePrefix (CurrNode,
                                Sym,
                                Scope,
                                 --to get
                                PrefixOk);
            if not PrefixOk then
               Sym := Dictionary.NullSymbol;
               exit;
            end if;

            CurrNode := ParentNode (CurrNode);
            -- finished if processed all identifiers under PackNode
            exit when CurrNode = ParentNode (PackNode);

            CurrNode := Next_Sibling (CurrNode);
            Sym := Dictionary.LookupSelectedItem (Sym,
                                                  NodeLexString (CurrNode),
                                                  Scope,
                                                  Dictionary.ProgramContext);
         end loop;
         PackSym := Sym;
      end FindPackage;

      --------------------------------------------------------

      procedure FindSubprogram (Scope       : in     Dictionary.Scopes;
                                PackSym     : in     Dictionary.Symbol;
                                SubprogNode : in     STree.SyntaxNode;
                                Sort        : in     SubprogSorts;
                                SubprogSym  :    out Dictionary.Symbol)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in     Node;
      --#        in     STree.Table;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.StringTable,
      --#                                        Node,
      --#                                        PackSym,
      --#                                        Scope,
      --#                                        Sort,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        SubprogNode &
      --#         SubprogSym                from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        PackSym,
      --#                                        Scope,
      --#                                        STree.Table,
      --#                                        SubprogNode;
      is
         Sym : Dictionary.Symbol;

         function SelectError (S : SubprogSorts) return Natural
         is
            Res : Natural;
         begin
            if S = Func then
               Res := 334;
            else
               Res := 19;
            end if;
            return Res;
         end SelectError;

      begin
         if Dictionary.IsDefined (NodeLexString (SubprogNode),
                                  Scope,
                                  Dictionary.ProofContext)
         then
            ErrorHandler.SemanticError (10,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        NodeLexString (SubprogNode));
            Sym := Dictionary.NullSymbol;
         else
            Sym := Dictionary.LookupSelectedItem (PackSym,
                                                  NodeLexString (SubprogNode),
                                                  Scope,
                                                  Dictionary.ProgramContext);
            if Sym = Dictionary.NullSymbol then
               ErrorHandler.SemanticError (1,
                                           ErrorHandler.NoReference,
                                           NodePosition (SubprogNode),
                                           NodeLexString (SubprogNode));
            elsif not Dictionary.IsSubprogram (Sym) then
               ErrorHandler.SemanticError (SelectError (Sort),
                                           ErrorHandler.NoReference,
                                           NodePosition (SubprogNode),
                                           NodeLexString (SubprogNode));
               Sym := Dictionary.NullSymbol;
               -- check to prevent 2nd renames
            elsif Dictionary.IsRenamed (Sym, Dictionary.LocalScope (PackSym)) or else
               Dictionary.IsRenamed (Sym, Dictionary.VisibleScope (PackSym))
            then
               ErrorHandler.SemanticError2 (339,
                                            ErrorHandler.NoReference,
                                            NodePosition (SubprogNode),
                                            NodeLexString (SubprogNode),
                                            Dictionary.GetSimpleName (PackSym));
               Sym := Dictionary.NullSymbol;
            end if;
         end if;
         SubprogSym := Sym;
      end FindSubprogram;

      --------------------------------------------------------

      procedure CheckConsistency (Scope       : in     Dictionary.Scopes;
                                  SubprogSym  : in     Dictionary.Symbol;
                                  ChildNode,
                                  FormalNode,
                                  SubprogNode : in     STree.SyntaxNode;
                                  Ok          :    out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in     STree.Table;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from ChildNode,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        FormalNode,
      --#                                        LexTokenManager.StringTable,
      --#                                        Scope,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        SubprogNode,
      --#                                        SubprogSym &
      --#         Ok                        from ChildNode,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        FormalNode,
      --#                                        Scope,
      --#                                        STree.Table,
      --#                                        SubprogSym;
      is
         It                 : Dictionary.Iterator;
         ParamCount         : Natural;
         CurrentFormalMode,
         CurrentMode        : Dictionary.Modes;
         CurrentFormal,
         CurrentFormalType,
         CurrentType        : Dictionary.Symbol;
         ListNode,
         SpecNode,
         IdentNode          : STree.SyntaxNode;
         SpecsIt            : STree.Iterator;
         IdentIt            : STree.Iterator;
         Consistent         : Boolean;
         CurrentFormalIdent : LexTokenManager.LexString;

         ---------------------------------------------------

         function CheckRightSortOfSubprogram (Sym  : Dictionary.Symbol;
                                              Node : STree.SyntaxNode)
                                             return Boolean
         --# global in Dictionary.Dict;
         --#        in STree.Table;
         is
            RightSort : Boolean;

         begin
            if Dictionary.IsFunction (Sym) then
               RightSort := SyntaxNodeType (Child_Node (Node)) =
                  SPSymbols.function_specification;
            else
               RightSort := SyntaxNodeType (Child_Node (Node)) =
                  SPSymbols.procedure_specification;
            end if;
            return RightSort;
         end CheckRightSortOfSubprogram;

         ---------------------------------------------------

         function GetMode (Node : STree.SyntaxNode)
                          return Dictionary.Modes
         --# global in STree.Table;
         is
            ModeNode : STree.SyntaxNode;
            Result   : Dictionary.Modes;

         begin
            --Node is parameter_specification
            ModeNode := Child_Node (Next_Sibling (Child_Node (Node)));
            if ModeNode = STree.NullNode then
               Result := Dictionary.InMode;
            else
               case SyntaxNodeType (ModeNode) is
                  when SPSymbols.in_mode    => Result := Dictionary.InMode;
                  when SPSymbols.inout_mode => Result := Dictionary.InOutMode;
                  when SPSymbols.outmode    => Result := Dictionary.OutMode;
                  when others => null; -- not possible
               end case;
            end if;
            --# accept Flow, 501, Result, "result is defined, non-exec path exists" &
            --#        Flow, 602, Result, "result is defined, non-exec path exists";
            return Result; --result is defined, non-exec path exists
         end GetMode;  --result is defined, non-exec path exists

         ---------------------------------------------------

         procedure GetType (Node    : in     STree.SyntaxNode;
                            Scope   : in     Dictionary.Scopes;
                            TypeSym :    out Dictionary.Symbol)
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     LexTokenManager.StringTable;
         --#        in     STree.Table;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives ErrorHandler.ErrorContext,
         --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        LexTokenManager.StringTable,
         --#                                        Node,
         --#                                        Scope,
         --#                                        SPARK_IO.FILE_SYS,
         --#                                        STree.Table &
         --#         TypeSym                   from CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        Node,
         --#                                        Scope,
         --#                                        STree.Table;
         is
            TypeNode : STree.SyntaxNode;

         begin
            --Node is parameter_specification
            TypeNode := Next_Sibling (Next_Sibling (Child_Node (Node)));
            wf_type_mark (TypeNode,
                          Scope,
                          Dictionary.ProgramContext,
                           --to get
                          TypeSym);
         end GetType;

         ---------------------------------------------------

         function ModesEquivalent (FormalMode,
                                   RenamedMode : Dictionary.Modes)
                                  return Boolean
         is
            Equivalent : Boolean;

         begin
            if FormalMode = Dictionary.DefaultMode then
               Equivalent := RenamedMode = Dictionary.InMode;
            else
               Equivalent := FormalMode = RenamedMode;
            end if;
            return Equivalent;
         end ModesEquivalent;

         ---------------------------------------------------

         procedure CheckReturnType (SubprogSym : in     Dictionary.Symbol;
                                    FormalNode : in     STree.SyntaxNode;
                                    Scope      : in     Dictionary.Scopes;
                                    Consistent : in out Boolean)
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     LexTokenManager.StringTable;
         --#        in     STree.Table;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives Consistent                from *,
         --#                                        CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        FormalNode,
         --#                                        Scope,
         --#                                        STree.Table,
         --#                                        SubprogSym &
         --#         ErrorHandler.ErrorContext,
         --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        FormalNode,
         --#                                        LexTokenManager.StringTable,
         --#                                        Scope,
         --#                                        SPARK_IO.FILE_SYS,
         --#                                        STree.Table;
         is
            TypeNode : STree.SyntaxNode;
            TypeSym  : Dictionary.Symbol;

         begin
            if SyntaxNodeType (FormalNode) = SPSymbols.type_mark then
               TypeNode := FormalNode;
            else
               TypeNode := Next_Sibling (FormalNode);
            end if;
            wf_type_mark (TypeNode,
                          Scope,
                          Dictionary.ProgramContext,
                           --to get
                          TypeSym);
            Consistent := Consistent and then
               TypeSym = Dictionary.GetType (SubprogSym);
         end CheckReturnType;

         ---------------------------------------------------

      begin  --CheckConsistency
         Consistent := CheckRightSortOfSubprogram (SubprogSym,
                                                   ChildNode);
         if Consistent then
            if FormalNode = STree.NullNode or else
               SyntaxNodeType (FormalNode) /=
               SPSymbols.formal_part
            then
               Consistent := Dictionary.GetNumberOfSubprogramParameters (SubprogSym) = 0;
            else
               ParamCount := 0;
               It := Dictionary.FirstSubprogramParameter (SubprogSym);
               SpecsIt := FindFirstNode (NodeKind    => SPSymbols.parameter_specification,
                                         FromRoot    => FormalNode,
                                         InDirection => STree.Down);

               while not STree.IsNull (SpecsIt) loop
                  SpecNode := GetNode (SpecsIt);
                  CurrentMode := GetMode (SpecNode);
                  GetType (SpecNode,
                           Scope,
                           --to get
                           CurrentType);
                  ListNode := Child_Node (SpecNode);

                  IdentIt := FindFirstNode
                    (NodeKind    => SPSymbols.identifier,
                     FromRoot    => ListNode,
                     InDirection => STree.Down);

                  while not STree.IsNull (IdentIt) loop
                     IdentNode := GetNode (IdentIt);
                     ParamCount := ParamCount + 1;
                     if Dictionary.IsNullIterator (It) then
                        Consistent := False;
                     else
                        CurrentFormal     := Dictionary.CurrentSymbol (It);
                        CurrentFormalType := Dictionary.GetType (CurrentFormal);
                        CurrentFormalMode := Dictionary.GetSubprogramParameterMode (CurrentFormal);
                        CurrentFormalIdent := Dictionary.GetSimpleName (CurrentFormal);
                        Consistent := Consistent                       and then
                           CurrentType = CurrentFormalType  and then
                           ModesEquivalent (CurrentFormalMode,
                                            CurrentMode)     and then
                           CurrentFormalIdent =
                           NodeLexString (IdentNode);
                        It := Dictionary.NextSymbol (It);
                     end if;
                     IdentIt := STree.NextNode (IdentIt);
                  end loop; -- Idents
                  SpecsIt := STree.NextNode (SpecsIt);
               end loop; -- Specs
               Consistent := Consistent and
                  (ParamCount =
                   Dictionary.GetNumberOfSubprogramParameters (SubprogSym));
            end if;

            if Dictionary.IsFunction (SubprogSym) then
               CheckReturnType (SubprogSym,
                                FormalNode,
                                Scope,
                                 --using and to get
                                Consistent);
            end if;
         end if;

         if not Consistent then
            ErrorHandler.SemanticError (302,
                                        ErrorHandler.NoReference,
                                        NodePosition (SubprogNode),
                                        NodeLexString (SubprogNode));
         end if;

         Ok := Consistent;

      end CheckConsistency;

      --------------------------------------------------------

   begin --CheckSubProgramRenaming
         -- check that we are not in package spec
      if Dictionary.IsVisibleScope (Scope) then
         ErrorHandler.SemanticError (340,
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     LexTokenManager.NullString);
      else
         ChildNode := Child_Node (Node);
         FormalPartNode := Next_Sibling (Child_Node (Child_Node (ChildNode)));
         PackNode := Child_Node (Next_Sibling (ChildNode));
         SubprogNode := Child_Node (Next_Sibling (Next_Sibling (ChildNode)));
         -- check for dotted package name:
         if SyntaxNodeType (PackNode) /= SPSymbols.identifier then
            if CommandLineData.IsSpark95 then
               CheckPosition (Node,
                              LexTokenManager.NullString,
                              Scope,
                              --to get
                              Ok);
            else
               --report error at second identifier in name
               PackNode := Next_Sibling (ParentNode (LastChildOf (PackNode)));
               ErrorHandler.SemanticError (610,
                                           ErrorHandler.NoReference,
                                           NodePosition (PackNode),
                                           LexTokenManager.NullString);
               Ok := False;
            end if;
         else
            CheckPosition (Node,
                           NodeLexString (PackNode),
                           Scope,
                           --to get
                           Ok);
         end if;
         if Ok then
            CheckNamesSame (Node,
                            SubprogNode,
                              --to get
                            Ok);
            if Ok then
               FindPackage (Scope,
                            PackNode,
                            PackSym);
               if PackSym /= Dictionary.NullSymbol then
                  -- detect sort of subprogram so that FindSubprogram can report
                  -- correct error
                  if SyntaxNodeType (
                                     Child_Node (ChildNode)) =
                     SPSymbols.procedure_specification
                  then
                     SubprogSort := Proc;
                  else
                     SubprogSort := Func;
                  end if;

                  FindSubprogram (Scope,
                                  PackSym,
                                  SubprogNode,
                                  SubprogSort,
                                  SubprogSym);
                  if SubprogSym /= Dictionary.NullSymbol then
                     CheckConsistency (Scope,
                                       SubprogSym,
                                       ChildNode,
                                       FormalPartNode,
                                       SubprogNode,
                                       --to get
                                       Ok);
                     if Ok then
                        Dictionary.RenameSubprogram (SubprogSym,
                                                     Dictionary.Location'(NodePosition (SubprogNode),
                                                                          NodePosition (SubprogNode)),
                                                     Dictionary.Location'(NodePosition (Node),
                                                                          NodePosition (Node)),
                                                     Scope);
                     end if;  --consistency check
                  end if;  --names not same
               end if;  --no such subprogram
            end if;  --not a package
         end if;  --wrong place
      end if;  --in package spec
   end CheckSubprogramRenaming;

   --========================================================================

begin --wf_renaming_declaration
   ChildNode := Child_Node (Node);
   case SyntaxNodeType (ChildNode) is
      when SPSymbols.subprogram_specification =>
         CheckSubprogramRenaming (Node, Scope);
      when SPSymbols.operator_symbol =>
         CheckOperatorRenaming (Node, Scope);
      when SPSymbols.package_renaming_declaration =>
         ErrorHandler.SemanticError (111,
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     LexTokenManager.NullString);
      when others =>
         SystemErrors.FatalError (SystemErrors.InvalidSyntaxTree, "in WFF.CU.wf_renaming_declaration");
   end case;
end wf_renaming_declaration;
