-- $Id: sem-compunit-wf_protected_body.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.CompUnit)
procedure wf_protected_body (Node     : in     STree.SyntaxNode;
                             Scope    : in out Dictionary.Scopes;
                             NextNode :    out STree.SyntaxNode)
is

   -- Grammar--------------------------------------------------------------
   -- protected_body :
   --        |
   --    identifier --- protected_operation_item --- identifier
   --                                |
   --                   protected_operation_item --- subprogram_body
   --                                |
   --                   protected_operation_item --- entry_body
   --
   -- Assume Node is protected_body
   --
   ------------------------------------------------------------------------
   -- Checks required:
   -- 1.  A protected type of declaration of the same name exists
   -- 2.  No body for it exists already (and, if subunit, a stub does exist)
   -- 3.  The closing identifier matches the initial
   -- 4.  Each operation in the spec has a body
   -- 5.  The second annotations on the operation bodies are refined correctly
   -- 6.  Add body if wellformed
   --
   ------------------------------------------------------------------------

   IdentNode,
   ProtectedOperationItemNode,
   ClosingIdentNode,
   WithNode                     : STree.SyntaxNode;
   ProtectedTypeSym             : Dictionary.Symbol;
   IdentStr,
   ClosingStr                   : LexTokenManager.LexString;
   InSubunit,
   OkToAdd                      : Boolean;
   ProtectedScope               : Dictionary.Scopes;

   procedure CheckOkToAdd (TypeSym    : in     Dictionary.Symbol;
                           InSubunit  : in     Boolean;
                           OkToAdd    :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     IdentNode;
   --#        in     IdentStr;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        IdentNode,
   --#                                        IdentStr,
   --#                                        InSubunit,
   --#                                        LexTokenManager.StringTable,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TypeSym &
   --#         OkToAdd                   from Dictionary.Dict,
   --#                                        InSubunit,
   --#                                        TypeSym;
   is
   begin
      OkToAdd := True;
      if InSubunit then
         -- we require a stub but must not have a previous body
         if Dictionary.HasBody (TypeSym) then
            OkToAdd := False;
            ErrorHandler.SemanticError (997,
                                        ErrorHandler.NoReference,
                                        NodePosition (IdentNode),
                                        IdentStr);
         end if;

         if not Dictionary.HasBodyStub (TypeSym) then
            OkToAdd := False;
            ErrorHandler.SemanticError (15,
                                        ErrorHandler.NoReference,
                                        NodePosition (IdentNode),
                                        IdentStr);
         end if;

      else
         -- we must have neither stub nor previous body
         if Dictionary.HasBody (TypeSym) or else
           Dictionary.HasBodyStub (TypeSym) then
            OkToAdd := False;
            ErrorHandler.SemanticError (997,
                                        ErrorHandler.NoReference,
                                        NodePosition (IdentNode),
                                        IdentStr);
         end if;
      end if;
   end CheckOkToAdd;


begin
   -- Set up key nodes
   IdentNode := Child_Node (Node);
   ProtectedOperationItemNode := Next_Sibling (IdentNode);
   ClosingIdentNode := Next_Sibling (ProtectedOperationItemNode);

   -- see if we are a subunit or an ordinary in-line declaration
   InSubunit := SyntaxNodeType (ParentNode (ParentNode (Node))) = SPSymbols.subunit;
   -- find any context clause present
   if InSubunit then
      WithNode := Child_Node             -- with_clause
        (Child_Node                      -- context_clause
           (ParentNode                       -- compilation_unit
              (ParentNode                    -- secondary_unit
                 (ParentNode                 -- subunit
                    (ParentNode (Node)))))); -- proper_body
      if SyntaxNodeType (WithNode) /= SPSymbols.with_clause then
         WithNode := STree.NullNode;
      end if;
   else
      WithNode := STree.NullNode;
   end if;

   -- Node is set to NullNode if there is an error in the protected body declaration
   -- and so stops the Compunit tree walk at that point.  If the body is ok then we
   -- set the Node to the ProtectedOperationNode so that the main tree walk will
   -- find the various declarations.  We set Node to NullNode here as a default.
   NextNode := STree.NullNode;

   -- find identifiers
   IdentStr := NodeLexString (IdentNode);
   ClosingStr := NodeLexString (ClosingIdentNode);

   -- see if already declared
   ProtectedTypeSym := Dictionary.LookupItem (IdentStr,
                                              Scope,
                                              Dictionary.ProgramContext);
   if Dictionary.IsType (ProtectedTypeSym) and then
     Dictionary.TypeIsProtected (ProtectedTypeSym) then
      -- potentially ok

      -- see if a body has already been declared etc.
      CheckOkToAdd (ProtectedTypeSym,
                    InSubunit,
                    OkToAdd);

      if OkToAdd then
         Dictionary.AddBody (CompilationUnit => ProtectedTypeSym,
                             TheBody => Dictionary.Location'(NodePosition (IdentNode),
                                                             NodePosition (IdentNode)),
                             Hidden => False);

         -- enter local scope of newly-added protected body
         ProtectedScope := Dictionary.LocalScope (ProtectedTypeSym);

         -- process context clause if present
         if WithNode /= STree.NullNode then
            wf_context_clause (ParentNode (WithNode),
                               ProtectedTypeSym,
                               ProtectedScope);
         end if;

         NextNode := ProtectedOperationItemNode;
         Scope := ProtectedScope;
         -- now check each declared operation in main Compunit tree walk
      end if;
   else
      -- either there is no spec to match the body or it not a protected type
      ErrorHandler.SemanticError (998,  ErrorHandler.NoReference,
                                  NodePosition (IdentNode),
                                  IdentStr);
   end if;

   -- Closing identifier check
      if IdentStr /= ClosingStr then
         ErrorHandler.SemanticError (58,
                                     ErrorHandler.NoReference,
                                     NodePosition (ClosingIdentNode),
                                     IdentStr);
      end if;
end wf_protected_body;
