-- $Id: sem-compunit-wf_property_list.adb 12351 2009-02-02 15:03:51Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with SystemErrors;

separate (Sem.CompUnit)

procedure wf_property_list (Node     : in     STree.SyntaxNode;
                            TypeNode : in     STree.SyntaxNode;
                            Scope    : in     Dictionary.Scopes;
                            TheOwner : in     Dictionary.Symbol)
is

   PropertyNode             : STree.SyntaxNode;
   It                       : STree.Iterator;

   --------------------------------------------------------------------------------------

   procedure wf_delay_property (Node         : in     STree.SyntaxNode;
                                Scope        : in     Dictionary.Scopes;
                                TheProcedure : in     Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict           from *,
   --#                                        Scope,
   --#                                        TheProcedure &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TheProcedure;
   is
   begin
      if not Dictionary.IsProcedure (TheProcedure) then

         -- Only procedures can have the delay property.
         --
         ErrorHandler.SemanticError (918,
                                     ErrorHandler.NoReference,
                                     NodePosition (ParentNode (Node)),
                                     Dictionary.GetSimpleName (TheProcedure));

      elsif Dictionary.IsOrIsInProtectedScope (Scope) then

         -- Blocking annotation not allowed in protected scope.
         --
         ErrorHandler.SemanticError (908,
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     LexTokenManager.NullString);

      elsif Dictionary.HasDelayProperty (TheProcedure) then

         -- The applicable compilation unit already has a delay statement.
         --
         ErrorHandler.SemanticError (888,
                                     ErrorHandler.NoReference,
                                     NodePosition (ParentNode (Node)),
                                     Dictionary.GetSimpleName (TheProcedure));
      else
         Dictionary.SetHasDelayProperty (TheProcedure);
      end if;
   end wf_delay_property;

   --------------------------------------------------------------------------------------

   procedure wf_suspendable_property (Node           : in     STree.SyntaxNode;
                                      Scope          : in     Dictionary.Scopes;
                                      TheOwnVariable : in     Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict           from *,
   --#                                        CommandLineData.Content,
   --#                                        Scope,
   --#                                        TheOwnVariable &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TheOwnVariable;
   is
      TypeSym    : Dictionary.Symbol;
      Consistent : Boolean;
   begin
      if Dictionary.IsOwnVariable (TheOwnVariable) and then
        Dictionary.OwnVariableHasType (TheOwnVariable, Scope) then
         TypeSym := Dictionary.GetType (TheOwnVariable);
      else
         TypeSym := Dictionary.GetUnknownTypeMark;
      end if;

      CheckSuspendablePropertyConsistency
        (Sym              => TheOwnVariable,
         TypeSym          => TypeSym,
         IsInSuspendsList => True,
         ErrorNode        => Node,
         Consistent       => Consistent);

      if Consistent then
         Dictionary.SetIsSuspendable (TheOwnVariable);
      end if;
   end wf_suspendable_property;

   --------------------------------------------------------------------------------------

   procedure wf_interrupt_property (NameNode       : in     STree.SyntaxNode;
                                    ValueNode      : in     STree.SyntaxNode;
                                    Scope          : in     Dictionary.Scopes;
                                    TheOwnVariable : in     Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict             from *,
   --#                                          CommandLineData.Content,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheOwnVariable,
   --#                                          ValueNode &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          LexTokenManager.StringTable,
   --#                                          NameNode,
   --#                                          Scope,
   --#                                          SPARK_IO.FILE_SYS,
   --#                                          STree.Table,
   --#                                          TheOwnVariable,
   --#                                          ValueNode &
   --#         LexTokenManager.StringTable from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          Scope,
   --#                                          TheOwnVariable;
   is
      Consistent     : Boolean;
      UnusedValueRep : LexTokenManager.LexString;
      It             : STree.Iterator;

      procedure ProcessOneElement (Node     : in     STree.SyntaxNode;
                                   TheOwner : in     Dictionary.Symbol)
      --# global in     CommandLineData.Content;
      --#        in     LexTokenManager.StringTable;
      --#        in     STree.Table;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives Dictionary.Dict           from *,
      --#                                        Node,
      --#                                        STree.Table,
      --#                                        TheOwner &
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.StringTable,
      --#                                        Node,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        TheOwner;
      is
         It                    : Dictionary.Iterator;
         HandlerLexStr         : LexTokenManager.LexString;
         InterruptStreamLexStr : LexTokenManager.LexString;
         OkToAdd               : Boolean;
      begin
         -- Node is annotation_aggregate_choice_rep

         -- Extract the handler name
         HandlerLexStr :=
           NodeLexString (GetNode
                            (FindFirstNode (NodeKind    => SPSymbols.identifier,
                                            FromRoot    => Node,
                                            InDirection => STree.Down)));

         -- Extract the stream name
         InterruptStreamLexStr :=
           NodeLexString (GetNode
                            (FindFirstNode (NodeKind    => SPSymbols.identifier,
                                            FromRoot    => Next_Sibling (Node),
                                            InDirection => STree.Down)));

         -- Check the handler is not already in the list
         OkToAdd := True;
         It := Dictionary.FirstInterruptStreamMapping (TheOwner);
         while OkToAdd and not Dictionary.IsNullIterator (It) loop
            OkToAdd :=
              Dictionary.GetInterruptStreamMappingHandler
              (Dictionary.CurrentSymbol (It)) /= HandlerLexStr;
            It := Dictionary.NextSymbol (It);
         end loop;

         if OkToAdd then
            -- This is not a duplicate
            -- Record the mapping of handler onto stream
            Dictionary.AddInterruptStreamMapping
              (Subject             => TheOwner,
               TheInterruptHandler => HandlerLexStr,
               TheInterruptStream  => InterruptStreamLexStr);

            -- If the protected type is declared in another package we can create the
            -- implicit interrupt stream variable now.
            if Dictionary.IsDeclared (Dictionary.GetType (TheOwner)) then
               CreateInterruptStreamVariable
                 (ForPO             => TheOwner,
                  TheHandler        => HandlerLexStr,
                  TheStreamVariable => InterruptStreamLexStr,
                  ErrorNode         => Node);
            end if;
         else
            -- This is a duplicate
            ErrorHandler.SemanticError (956,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        HandlerLexStr);
         end if;
      end ProcessOneElement;

   begin
      if not (Dictionary.IsOwnVariable (TheOwnVariable) and then
                Dictionary.GetOwnVariableProtected (TheOwnVariable) and then
                Dictionary.OwnVariableHasType (TheOwnVariable, Scope)) then

         -- The interrupt property can only apply to protected own variables that
         -- are type announced.
         --
         ErrorHandler.SemanticError (936,
                                     ErrorHandler.NoReference,
                                     NodePosition (ParentNode (NameNode)),
                                     Dictionary.GetSimpleName (TheOwnVariable));

      else
         CheckInterruptPropertyConsistency
           (HasInterruptProperty => True,
            Sym                  => TheOwnVariable,
            TypeSym              => Dictionary.GetType (TheOwnVariable),
            ErrorNode            => NameNode,
            Consistent           => Consistent);

         if Consistent then

            if Dictionary.GetPriorityProperty (TheOwnVariable) /=
               LexTokenManager.NullString then
               --# accept Flow, 10, UnusedValueRep, "Expected ineffective assignment";
                  CheckPriorityRange
                     (ErrorSym   => TheOwnVariable,
                      Scope      => Scope,
                      PragmaKind => Dictionary.AttachHandler,
                      ErrPos     => NodePosition (ParentNode (NameNode)),
                      Value      => Maths.ValueRep (Dictionary.GetPriorityProperty
                                                  (TheOwnVariable)),
                      ValueRep   => UnusedValueRep);
               --# end accept;
            end if;

            Dictionary.SetHasInterruptProperty (TheOwnVariable);

            -- Parse the interrupt stream list if there is one
            if ValueNode /= STree.NullNode then

               It := FindFirstNode (NodeKind    => SPSymbols.annotation_aggregate_choice_rep,
                                    FromRoot    => ValueNode,
                                    InDirection => STree.Down);

               while not STree.IsNull (It) loop
                  ProcessOneElement (GetNode (It),
                                     TheOwnVariable);
                  It := STree.NextNode (It);
               end loop;
            end if;
         end if;
      end if;
      --# accept Flow, 33, UnusedValueRep, "Expected to be neither referenced nor exported";
   end wf_interrupt_property;

   --------------------------------------------------------------------------------------

   procedure wf_name_property (Node     : in     STree.SyntaxNode;
                               Scope    : in     Dictionary.Scopes;
                               TheOwner : in     Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict,
   --#         LexTokenManager.StringTable from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheOwner &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          SPARK_IO.FILE_SYS,
   --#                                          STree.Table,
   --#                                          TheOwner;
   is
      NameNode : STree.SyntaxNode;
      NameStr  : EStrings.T;
   begin
      NameNode := Child_Node (Node);

      if SyntaxNodeType (NameNode) = SPSymbols.RWdelay then

         wf_delay_property (Node => NameNode,
                            Scope => Scope,
                            TheProcedure => TheOwner);

      else -- we have an identifier
         LexTokenManager.LexStringToString
           (LexStr => NodeLexString (NameNode),
            Str    => NameStr);

         if EStrings.Eq1String (NameStr, "suspendable") then

            wf_suspendable_property (Node => NameNode,
                                     Scope => Scope,
                                     TheOwnVariable => TheOwner);

         elsif EStrings.Eq1String (NameStr, "interrupt") then

            wf_interrupt_property (NameNode       => NameNode,
                                   ValueNode      => STree.NullNode,
                                   Scope          => Scope,
                                   TheOwnVariable => TheOwner);

         else
            ErrorHandler.SemanticError (921,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        NodeLexString (NameNode));
         end if;
      end if;
   end wf_name_property;

   --------------------------------------------------------------------------------------

   procedure wf_priority_property (NameNode  : in     STree.SyntaxNode;
                                   ValueNode : in     STree.SyntaxNode;
                                   TheOwner  : in     Dictionary.Symbol;
                                   Scope     : in     Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                     from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          LexTokenManager.StringTable,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap,
   --#                                          ValueNode &
   --#         Dictionary.Dict,
   --#         LexTokenManager.StringTable from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          LexTokenManager.StringTable,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap,
   --#                                          TheOwner,
   --#                                          ValueNode &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          LexTokenManager.StringTable,
   --#                                          NameNode,
   --#                                          Scope,
   --#                                          SPARK_IO.FILE_SYS,
   --#                                          STree.Table,
   --#                                          TheHeap,
   --#                                          TheOwner,
   --#                                          ValueNode;
   is
      VarType        : Dictionary.Symbol;
      ValueRep       : LexTokenManager.LexString;
      ExpressionNode : STree.SyntaxNode;
      Valid          : Boolean;

      function DeterminePragmaKind return Dictionary.RavenscarPragmas
      --# global in Dictionary.Dict;
      --#        in TheOwner;
      is
         Result : Dictionary.RavenscarPragmas;
      begin
         if Dictionary.IsOwnVariable (TheOwner) and then
           Dictionary.GetHasInterruptProperty (TheOwner) then
            Result := Dictionary.AttachHandler;
         else
            Result := Dictionary.InterruptPriority;
         end if;
         return Result;
      end DeterminePragmaKind;

   begin
      ExpressionNode := Child_Node (ValueNode);
      if SyntaxNodeType (ExpressionNode) /= SPSymbols.annotation_expression then

         -- The grammar allows an aggregate here but for the priority property
         -- value we must have a static expression.
         --
         ErrorHandler.SemanticError (945,
                                     ErrorHandler.NoReference,
                                     NodePosition (ValueNode),
                                     NodeLexString (NameNode));
      else

         wf_priority_value
           (Node       => ExpressionNode, -- expect ineffective assignment to TheHeap
            PragmaKind => DeterminePragmaKind,
            Context    => Dictionary.ProofContext,
            ErrorSym   => TheOwner,
            Scope      => Scope,
            ValueRep   => ValueRep,
            Compatible => Valid);

         if Valid then

            if Dictionary.IsOwnVariable (TheOwner) and then
              Dictionary.OwnVariableHasType (TheOwner, Scope) then
               VarType := Dictionary.GetType (TheOwner);
            else
               VarType := Dictionary.GetUnknownTypeMark;
            end if;

            CheckPriorityPropertyConsistency
              (Sym                   => TheOwner,
               TypeSym               => VarType,
               PriorityPropertyValue => ValueRep,
               ErrorNode             => NameNode,
               Consistent            => Valid);

            if Valid then
               Dictionary.SetPriorityProperty (OwnVariable => TheOwner,
                                               TheValue => ValueRep);
            end if;
         end if;
      end if;
   end wf_priority_property;

   --------------------------------------------------------------------------------------

   procedure wf_suspends_property (NameNode  : in     STree.SyntaxNode;
                                   ValueNode : in     STree.SyntaxNode;
                                   Scope     : in     Dictionary.Scopes;
                                   TheOwner  : in     Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict           from *,
   --#                                        CommandLineData.Content,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheOwner,
   --#                                        ValueNode &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        NameNode,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TheOwner,
   --#                                        ValueNode;
   is

      It : STree.Iterator;

      procedure ProcessOnePOorSO (ValueNode : in     STree.SyntaxNode;
                                  Scope     : in     Dictionary.Scopes;
                                  TheOwner  : in     Dictionary.Symbol)
      --# global in     CommandLineData.Content;
      --#        in     LexTokenManager.StringTable;
      --#        in     STree.Table;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives Dictionary.Dict           from *,
      --#                                        CommandLineData.Content,
      --#                                        Scope,
      --#                                        STree.Table,
      --#                                        TheOwner,
      --#                                        ValueNode &
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.StringTable,
      --#                                        Scope,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        TheOwner,
      --#                                        ValueNode;
      is
         VarSym       : Dictionary.Symbol;
         Unused       : Boolean;
         ScopeToCheck : Dictionary.Scopes;
      begin
         if Dictionary.IsOwnTask (TheOwner) then
            ScopeToCheck := Scope;
         else
            ScopeToCheck := Dictionary.LocalScope (TheOwner);
         end if;

         --# accept Flow, 10, Unused, "Expected ineffective assignment";
         wf_entire_variable
           (ValueNode,                  -- expect ineffective assignment for Unused
            ScopeToCheck,
            In_Suspends_List,
            VarSym,
            Unused);
         --# end accept;

         if VarSym /= Dictionary.NullSymbol then

            if Dictionary.SuspendsOn (TheTaskOrProc => TheOwner,
                                      ThePOorSO => VarSym) then
               ErrorHandler.SemanticErrorSym (890,
                                              ErrorHandler.NoReference,
                                              NodePosition (ValueNode),
                                              VarSym,
                                              Scope);
            elsif Dictionary.GetIsSuspendable (VarSym) then
               Dictionary.AddPOorSOToSuspendsList (TheTaskOrProc => TheOwner,
                                                   ThePOorSO => VarSym);
            else
               -- This symbol cannot suspend.
               ErrorHandler.SemanticErrorSym (909,
                                              ErrorHandler.NoReference,
                                              NodePosition (ValueNode),
                                              VarSym,
                                              Scope);
            end if;
         end if; -- not entire var
         --# accept Flow, 33, Unused, "Expected to be neither referenced nor exported";
      end ProcessOnePOorSO; -- expect flow error for Unused

   begin
      if not ((Dictionary.IsTypeMark (TheOwner) and then
                 Dictionary.TypeIsTask (TheOwner)) or

                (Dictionary.IsProcedure (TheOwner))) then

         -- The suspends list is not being applied to a valid construct.
         --
         ErrorHandler.SemanticError (920,
                                     ErrorHandler.NoReference,
                                     NodePosition (NameNode),
                                     Dictionary.GetSimpleName (TheOwner));

      elsif Dictionary.IsProcedure (TheOwner) and then
        Dictionary.IsOrIsInProtectedScope (Scope) then

         -- Blocking annotation not allowed in protected scope.
         --
         ErrorHandler.SemanticError (908,
                                     ErrorHandler.NoReference,
                                     NodePosition (NameNode),
                                     LexTokenManager.NullString);

      else
         It := FindFirstNode (NodeKind    => SPSymbols.annotation_primary,
                              FromRoot    => ValueNode,
                              InDirection => STree.Down);

         while not STree.IsNull (It) loop
            ProcessOnePOorSO (GetNode (It),
                              Scope,
                              TheOwner);
            It := STree.NextNode (It);
         end loop;
      end if;
   end wf_suspends_property;

   --------------------------------------------------------------------------------------

   procedure wf_protects_property (NameNode  : in     STree.SyntaxNode;
                                   ValueNode : in     STree.SyntaxNode;
                                   TypeNode  : in     STree.SyntaxNode;
                                   Scope     : in     Dictionary.Scopes;
                                   TheOwnVar : in     Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict           from *,
   --#                                        CommandLineData.Content,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheOwnVar,
   --#                                        ValueNode &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        NameNode,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        TheOwnVar,
   --#                                        TypeNode,
   --#                                        ValueNode;
   is
      TheOwnVarType : Dictionary.Symbol;
      It            : STree.Iterator;

      procedure ProcessOneElement (ValueNode     : in     STree.SyntaxNode;
                                   Scope         : in     Dictionary.Scopes;
                                   TheOwnVar     : in     Dictionary.Symbol;
                                   TheOwnVarType : in     Dictionary.Symbol)
      --# global in     CommandLineData.Content;
      --#        in     LexTokenManager.StringTable;
      --#        in     STree.Table;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives Dictionary.Dict           from *,
      --#                                        CommandLineData.Content,
      --#                                        Scope,
      --#                                        STree.Table,
      --#                                        TheOwnVar,
      --#                                        TheOwnVarType,
      --#                                        ValueNode &
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.StringTable,
      --#                                        Scope,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        TheOwnVar,
      --#                                        ValueNode;
      is
         ProtectsItemSym : Dictionary.Symbol;
         IsDotted        : Boolean;
      begin
         wf_entire_variable (ValueNode,
                             Scope,
                             In_Suspends_List,
                             ProtectsItemSym,
                             IsDotted);

         if ProtectsItemSym /= Dictionary.NullSymbol then

            if not Dictionary.IsOwnVariable (ProtectsItemSym) or else
              IsDotted or else
              Dictionary.GetOwnVariableProtected (ProtectsItemSym) then

               -- Items in protects properties must be local, unprotected own variables.
               --
               ErrorHandler.SemanticErrorSym
                 (943,
                  ErrorHandler.NoReference,
                  NodePosition (ValueNode),
                  ProtectsItemSym,
                  Scope);

            elsif Dictionary.GetOwnVariableMode (TheOwnVar) /=
              Dictionary.DefaultMode and then
              Dictionary.GetOwnVariableMode (TheOwnVar) /=
              Dictionary.GetOwnVariableMode (ProtectsItemSym) then

               -- Refinement constituent mode mismatch
               --
               ErrorHandler.SemanticErrorSym
                 (701,
                  ErrorHandler.NoReference,
                  NodePosition (ValueNode),
                  ProtectsItemSym,
                  Scope);

            elsif Dictionary.IsVirtualElement (ProtectsItemSym) then

               -- This item has already appeared in another protects list.
               --
               ErrorHandler.SemanticErrorSym2
                 (944,
                  ErrorHandler.NoReference,
                  NodePosition (ValueNode),
                  ProtectsItemSym,
                  Dictionary.GetVirtualElementOwner (ProtectsItemSym),
                  Scope);

            else

               -- Add this variable to the protects list for the type.
               --
               Dictionary.AddVirtualElement
                 (ToProtectedType => TheOwnVarType,
                  TheVirtualElement => ProtectsItemSym,
                  TheOwner => TheOwnVar);

            end if; -- not entire var
         end if;
      end ProcessOneElement; -- expect flow error for Unused

      function ValueIsCorrectFormat
        (ValueNode : STree.SyntaxNode) return Boolean
      --# global in STree.Table;
      -- ValueNode is annotation_aggregate_or_expression.
      -- The allowable values are:
      -- (1) protects => X
      -- (2) protects => (X)
      -- (3) protects => (X, Y, Z)
      -- Hence the only branch allowed in the syntax tree for the above is at nodes
      -- of type annotation_positional_association_rep vis (3).
      -- However, we also allow
      -- (4) protects => X.Y
      -- i.e. a branch at nodes of type annotation_selected_component.
      -- This ensures we get the more meaningful error message (943).
      is
         It     : STree.Iterator;
         Result : Boolean := True;
      begin
         It := FindFirstBranchNode (FromRoot    => ValueNode,
                                    InDirection => STree.Down);
         while not STree.IsNull (It) loop
            if SyntaxNodeType (GetNode (It)) /=
              SPSymbols.annotation_positional_association_rep and then
              SyntaxNodeType (GetNode (It)) /=
              SPSymbols.annotation_selected_component then
               Result := False;
               exit;
            end if;
            It := STree.NextNode (It);
         end loop;
         return Result;
      end ValueIsCorrectFormat;

   begin

      if Dictionary.IsOwnVariable (TheOwnVar) and then
        Dictionary.GetOwnVariableProtected (TheOwnVar) and then
        Dictionary.OwnVariableHasType (OwnVariable => TheOwnVar,
                                       Scope => Scope)
      then
         TheOwnVarType := Dictionary.GetRootType (Dictionary.GetType (TheOwnVar));

         if Dictionary.IsDeclared (TheOwnVarType) then

            -- The type of the own variable must be local to this package.
            --
            ErrorHandler.SemanticError (941,
                                        ErrorHandler.NoReference,
                                        NodePosition (TypeNode),
                                        Dictionary.GetSimpleName (TheOwnVar));

         elsif not ValueIsCorrectFormat (ValueNode) then

            -- The format must be a simple expression or a position associated
            -- aggregate.
            --
            ErrorHandler.SemanticError (961,
                                        ErrorHandler.NoReference,
                                        NodePosition (ValueNode),
                                        Dictionary.GetSimpleName (TheOwnVar));

         else
            It := FindFirstNode (NodeKind    => SPSymbols.annotation_primary,
                                 FromRoot    => ValueNode,
                                 InDirection => STree.Down);

            while not STree.IsNull (It) loop
               ProcessOneElement (GetNode (It),
                                  Scope,
                                  TheOwnVar,
                                  TheOwnVarType);
               It := STree.NextNode (It);
            end loop;
         end if;

      else
         -- The protects property must be applied to protected own variables
         --
         ErrorHandler.SemanticError (937,
                                     ErrorHandler.NoReference,
                                     NodePosition (NameNode),
                                     Dictionary.GetSimpleName (TheOwnVar));
      end if;
   end wf_protects_property;

   procedure wf_integrity_property (NameNode       : in     STree.SyntaxNode;
                                    ValueNode      : in     STree.SyntaxNode;
                                    Scope          : in     Dictionary.Scopes;
                                    TheOwnVariable : in     Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         LexTokenManager.StringTable,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                     from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          LexTokenManager.StringTable,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap,
   --#                                          ValueNode &
   --#         Dictionary.Dict             from *,
   --#                                          CommandLineData.Content,
   --#                                          LexTokenManager.StringTable,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap,
   --#                                          TheOwnVariable,
   --#                                          ValueNode &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          LexTokenManager.StringTable,
   --#                                          NameNode,
   --#                                          Scope,
   --#                                          SPARK_IO.FILE_SYS,
   --#                                          STree.Table,
   --#                                          TheHeap,
   --#                                          ValueNode;
   is
      ExpressionNode      : STree.SyntaxNode;
      TypeSym             : Dictionary.Symbol;
      Result              : ExpRecord;
      ResultRep           : LexTokenManager.LexString;
      UnusedSeq           : SeqAlgebra.Seq;
      UnusedComponentData : ComponentManager.ComponentData;

      function IsNaturalInteger (V : in Maths.Value) return Boolean
      is
         Comp   : Maths.Value;
         Result : Boolean;
         Unused : Maths.ErrorCode;
      begin
         if Maths.IsIntegerValue (V) then
            --# accept Flow, 10, Unused, "Expected ineffective assignment" &
            --#        Flow, 33, Unused, "Expected to be neither referenced nor exported";
            Maths.GreaterOrEqual (V,
                                  Maths.ZeroInteger,
                                  Comp,
                                  Unused);
            Maths.ValueToBool (Comp, Result, Unused);
         else
            Result := False;
         end if;
         return Result;
      end IsNaturalInteger;


   begin
      -- Value should be static expression, type Integer, and  >= 0

      ExpressionNode := Child_Node (ValueNode);
      if SyntaxNodeType (ExpressionNode) /= SPSymbols.annotation_expression then

         -- The grammar allows an aggregate here but for the integrity property
         -- value we must have a static expression.
         --
         ErrorHandler.SemanticError (945,
                                     ErrorHandler.NoReference,
                                     NodePosition (ValueNode),
                                     NodeLexString (NameNode));
      else
         -- 1 walk the expression
         SeqAlgebra.CreateSeq (TheHeap, UnusedSeq);
         ComponentManager.Initialise (UnusedComponentData);
         --# accept Flow, 10, UnusedComponentData, "Expected ineffective assignment";
         WalkAnnotationExpression -- expect UnusedComponentData
           (ExpNode        => ExpressionNode,
            Scope          => Scope,
            TypeContext    => Dictionary.GetPredefinedIntegerType,
            Context        => Precondition, -- ~ never allowed here, so use Precondition context
            Result         => Result,
            ComponentData  => UnusedComponentData);
         --# end accept;

         SeqAlgebra.DisposeOfSeq (TheHeap, UnusedSeq);

         -- 2 check it's constant
         if Result.IsConstant then

            -- 3 check type
            TypeSym := Dictionary.GetRootType (Result.TypeSymbol);
            if Dictionary.CompatibleTypes
              (Scope, TypeSym, Dictionary.GetPredefinedIntegerType) and then

               -- 4 check >= 0
              IsNaturalInteger (Result.Value) then

               Maths.StorageRep (Result.Value, ResultRep);
               Dictionary.SetIntegrityProperty (TheOwnVariable, ResultRep);
            else
               ErrorHandler.SemanticError (882,
                                           ErrorHandler.NoReference,
                                           NodePosition (ExpressionNode),
                                           LexTokenManager.NullString);
            end if;
         else
            ErrorHandler.SemanticError (37, -- not constant
                                        ErrorHandler.NoReference,
                                        NodePosition (ExpressionNode),
                                        LexTokenManager.NullString);
         end if;
      end if;
   end wf_integrity_property;

   --------------------------------------------------------------------------------------

   procedure wf_name_value_property (Node     : in     STree.SyntaxNode;
                                     TypeNode : in     STree.SyntaxNode;
                                     Scope    : in     Dictionary.Scopes;
                                     TheOwner : in     Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                     from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         Dictionary.Dict,
   --#         LexTokenManager.StringTable from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap,
   --#                                          TheOwner &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          SPARK_IO.FILE_SYS,
   --#                                          STree.Table,
   --#                                          TheHeap,
   --#                                          TheOwner,
   --#                                          TypeNode;
   is
      NameNode  : STree.SyntaxNode;
      ValueNode : STree.SyntaxNode;
      NameStr   : EStrings.T;
   begin
      NameNode := Child_Node (Node);
      ValueNode := Next_Sibling (NameNode);

      LexTokenManager.LexStringToString
        (LexStr => NodeLexString (NameNode),
         Str    => NameStr);

      if EStrings.Eq1String (NameStr, "priority") then

         wf_priority_property (NameNode => NameNode,
                               ValueNode => ValueNode,
                               TheOwner => TheOwner,
                               Scope => Scope);

      elsif EStrings.Eq1String (NameStr, "suspends") then

         wf_suspends_property (NameNode => NameNode,
                               ValueNode => ValueNode,
                               Scope => Scope,
                               TheOwner => TheOwner);

      elsif EStrings.Eq1String (NameStr, "protects") then

         wf_protects_property (NameNode  => NameNode,
                               ValueNode => ValueNode,
                               TypeNode  => TypeNode,
                               Scope     => Scope,
                               TheOwnVar => TheOwner);

      elsif EStrings.Eq1String (NameStr, "interrupt") then

         wf_interrupt_property (NameNode       => NameNode,
                                ValueNode      => ValueNode,
                                Scope          => Scope,
                                TheOwnVariable => TheOwner);

      elsif EStrings.Eq1String (NameStr, "integrity") then

         wf_integrity_property (NameNode       => NameNode,
                                ValueNode      => ValueNode,
                                Scope          => Scope,
                                TheOwnVariable => TheOwner);

      else
         ErrorHandler.SemanticError (921,
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     NodeLexString (NameNode));
      end if;
   end wf_name_value_property;


begin -- wf_property_list

   It := FindFirstNode (NodeKind    => SPSymbols.property,
                        FromRoot    => Node,
                        InDirection => STree.Down);

   while not STree.IsNull (It) loop

      PropertyNode := Child_Node (GetNode (It));

      if SyntaxNodeType (PropertyNode) = SPSymbols.name_property then

         wf_name_property (Node => PropertyNode,
                           Scope => Scope,
                           TheOwner => TheOwner);

      else

         wf_name_value_property (Node => PropertyNode,
                                 TypeNode => TypeNode,
                                 Scope => Scope,
                                 TheOwner => TheOwner);

      end if;

      It := STree.NextNode (It);

   end loop;
end wf_property_list;
