-- $Id: sem-compunit-wf_pragma.adb 12351 2009-02-02 15:03:51Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.CompUnit)
procedure wf_pragma (Node  : in STree.SyntaxNode;
                     Scope : in Dictionary.Scopes)
is
   PackIdentNode,
   PackSpecNode,
   IdNode,
   SubprogOrVariableNode : STree.SyntaxNode;

   StatementOK  : Boolean;
   ErrorToRaise : Natural;

   ------------------------------------------------------------------

   procedure Wf_Pragma_Atomic (PragmaNode : STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict           from *,
   --#                                        CommandLineData.Content,
   --#                                        PragmaNode,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        PragmaNode,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table;
   is
      ArgAssNode,
      ExpNode,
      AtomicArg  : STree.SyntaxNode;
      ErrorFound : Boolean;
      IsChain    : Boolean;
      SubjectSym : Dictionary.Symbol;

      procedure CheckArgCount
      --# global in     STree.Table;
      --#        in out ArgAssNode;
      --#           out AtomicArg;
      --#           out ErrorFound;
      --# derives ArgAssNode,
      --#         AtomicArg,
      --#         ErrorFound from ArgAssNode,
      --#                         STree.Table;
      is
         ArgCount   : Natural := 0;
      begin
         ErrorFound := False;
         AtomicArg := STree.NullNode;
         while SyntaxNodeType (ArgAssNode) /=
            SPSymbols.argument_association
         loop
            ArgAssNode := Child_Node (ArgAssNode);
         end loop;
         -- now pointing at leftmost argument association
         while ArgAssNode /= STree.NullNode
         loop
            if ArgCount = 1 then
               ErrorFound := True;
               exit;
            end if;
            ArgCount := ArgCount + 1;
            AtomicArg := Child_Node (ArgAssNode);
            ArgAssNode := Next_Sibling (ParentNode (ArgAssNode));
         end loop;
      end CheckArgCount;

      procedure CheckIsChain
      --# global in     STree.Table;
      --#        in out ExpNode;
      --#           out IsChain;
      --# derives ExpNode,
      --#         IsChain from ExpNode,
      --#                      STree.Table;
      is
         NextNode : STree.SyntaxNode;
      begin
         -- Walk down the syntax tree; if there's ever a Next_Sibling
         -- on the way down, then this isn't a chain - so we can stop the
         -- walk. Otherwise, find the node at the end of the chain.
         loop
            IsChain := Next_Sibling (ExpNode) =
              STree.NullNode;
            NextNode := Child_Node (ExpNode);
            exit when not IsChain or NextNode = STree.NullNode;
            ExpNode := NextNode;
         end loop;
      end CheckIsChain;

      function CheckLocation (NodeToCheck : STree.SyntaxNode)
                             return Boolean
      --# global in STree.Table;
      is
         NodeType : SPSymbols.SPSymbol;
      begin
         NodeType := SyntaxNodeType (NodeToCheck);
         -- allows use in declarative parts of subprogram and package bodies,
         -- package visible parts, and package private parts.
         return NodeType = SPSymbols.initial_declarative_item_rep or else
           NodeType = SPSymbols.basic_declarative_item_rep or else
           NodeType = SPSymbols.visible_part_rep;
      end CheckLocation;

      -------------------------------------------------------------------------
      -- SEPR 2253 introduced the need for objects which are atomic but of a
      -- predefined type like Boolean, Character, or System.Address.
      -- RavenSPARK forbids Atomic on objects, and we can't apply pragma Atomic
      -- to these types, since they are predefined.
      -- Therefore, we allow a record type to be Atomic if it has a single
      -- field which ((is predefined and a scalar basetype) or (is System.Address))
      -------------------------------------------------------------------------
      function IsPotentiallyAtomicRecordType (Sym   : in Dictionary.Symbol;
                                              Scope : in Dictionary.Scopes) return Boolean
      --# global in CommandLineData.Content;
      --#        in Dictionary.Dict;
      is
         Result         : Boolean;
         The_Component  : Dictionary.Symbol;
         The_Type       : Dictionary.Symbol;
         Package_System : Dictionary.Symbol;
         Type_Address   : Dictionary.Symbol;
      begin
         if Dictionary.IsRecordTypeMark (Sym, Scope) then
            if Dictionary.GetNumberOfComponents (Sym) = 1 and
              not Dictionary.TypeIsTagged (Sym) then

               The_Component := Dictionary.GetRecordComponent (Sym, 1);
               The_Type      := Dictionary.GetType (The_Component);

               -- Acceptable if it's predefined, scalar, and not a subtype
               if Dictionary.IsPredefined (The_Type) and then
                 Dictionary.TypeIsScalar (The_Type) and then
                 not Dictionary.IsSubtype (The_Type) then

                  Result := True;
               else
                  -- Not predefined and scalar, so check for special
                  -- case of System.Address
                  Package_System := Dictionary.LookupItem (LexTokenManager.SystemToken,
                                                           Dictionary.GlobalScope,
                                                           Dictionary.ProgramContext);
                  if Package_System /= Dictionary.NullSymbol then
                     Type_Address :=
                       Dictionary.LookupSelectedItem (Package_System,
                                                      LexTokenManager.AddressToken,
                                                      Dictionary.VisibleScope (Package_System),
                                                      Dictionary.ProgramContext);
                     Result := (The_Type = Type_Address);
                  else
                     -- can't find package System, so
                     Result := False;
                  end if;
               end if;

            else
               -- Record with 0 or >= 2 fields, or a tagged record
               Result := False;
            end if;
         else
            -- Not a record type at all
            Result := False;
         end if;
         return Result;
      end IsPotentiallyAtomicRecordType;

   begin
      ArgAssNode := Next_Sibling (Child_Node (PragmaNode));
      -- should be top of argument_asociation_rep chain
      if SyntaxNodeType (ArgAssNode) =
         SPSymbols.argument_association_rep
      then
         CheckArgCount;
         if not ErrorFound then
            if SyntaxNodeType (AtomicArg) = SPSymbols.identifier then
               -- The parameter to pragma Atomic must be a simple_name,
               -- not a named association.
               ErrorHandler.SemanticError (851,
                                           ErrorHandler.NoReference,
                                           NodePosition (AtomicArg),
                                           LexTokenManager.NullString);
            else
               ExpNode := AtomicArg;
               CheckIsChain;
               if IsChain and then
                 SyntaxNodeType (ExpNode) = SPSymbols.identifier then
                  -- happy; found a simple_name; have the identifier
                  SubjectSym := Dictionary.LookupItem (NodeLexString (ExpNode),
                                                       Scope,
                                                       Dictionary.ProgramContext);
                  if SubjectSym = Dictionary.NullSymbol then
                     -- the pragma refers to an identifier which isn't visible
                     -- at this point.
                     ErrorHandler.SemanticError (1,
                                                 ErrorHandler.NoReference,
                                                 NodePosition (ExpNode),
                                                 NodeLexString (ExpNode));
                  else
                     if (Dictionary.IsTypeMark (SubjectSym) and then
                           not Dictionary.IsSubtype (SubjectSym) and then
                           (Dictionary.IsScalarTypeMark (SubjectSym, Scope) or
                            IsPotentiallyAtomicRecordType (SubjectSym, Scope)))
                     then
                        -- OK; the pragma refers to a scalar base type or to
                        -- a record type that may be Atomic
                        if CheckLocation (ParentNode (PragmaNode)) and then
                          Dictionary.GetScope (SubjectSym) = Scope then
                           -- it's also in a legitimate part of the syntax tree,
                           -- and in the same scope as the variable or type it identifies.
                           Dictionary.SetTypeAtomic (SubjectSym);
                        else
                           -- the pragma isn't legal here
                           ErrorHandler.SemanticError (852,
                                                       ErrorHandler.NoReference,
                                                       NodePosition (ExpNode),
                                                       LexTokenManager.NullString);
                        end if;
                     else
                        -- the argument to pragma Atomic must be a scalar base type,
                        -- or a record type that is potentially Atomic.
                        -- it's not here, issue an error
                        ErrorHandler.SemanticError (853,
                                                    ErrorHandler.NoReference,
                                                    NodePosition (ExpNode),
                                                    LexTokenManager.NullString);
                     end if;
                  end if;
               else
                  -- The parameter to pragma Atomic must be a simple_name
                  ErrorHandler.SemanticError (851,
                                              ErrorHandler.NoReference,
                                              NodePosition (ExpNode),
                                              LexTokenManager.NullString);
               end if;
            end if;
         else
            -- pragma atomic takes exactly one argument, more than one was found
            -- in this case, so issue an error
            ErrorHandler.SemanticError (854,
                                        ErrorHandler.NoReference,
                                        NodePosition (ArgAssNode),
                                        LexTokenManager.NullString);
         end if;
      else
         -- no argument association was found; so no parameters were supplied.
         ErrorHandler.SemanticError (854,
                                     ErrorHandler.NoReference,
                                     NodePosition (PragmaNode),
                                     LexTokenManager.NullString);
      end if;
   end Wf_Pragma_Atomic;

   --------------------------------------------------------------------

   function WeAreNotInAProtectedType return Boolean
   --# global in Dictionary.Dict;
   --#        in Scope;
   is
   begin
      return not Dictionary.IsType (Dictionary.GetRegion (Scope));
   end WeAreNotInAProtectedType;

   --------------------------------------------------------------------

   procedure wf_attach_handler (PragmaNode : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict           from *,
   --#                                        PragmaNode,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        PragmaNode,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table;
      is separate;

      --------------------------------------------------------------------

   procedure wf_main_program_priority (IdNode : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Node;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict,
   --#         LexTokenManager.StringTable from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          IdNode,
   --#                                          LexTokenManager.StringTable,
   --#                                          Scope,
   --#                                          STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          IdNode,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          SPARK_IO.File_Sys,
   --#                                          STree.Table;
   is
      -- Grammar:
      --          identifier --- [Argument_Association_Rep
      --                                      |
      --                           Argument_Association
      --                                      |
      --                               Ada_Expression
      --                                     ...
      --                                 Ada_Primary etc]
      --

      Iterator,
      ErrNode     : STree.SyntaxNode;
      Sym         : Dictionary.Symbol;
      Error       : Boolean;
      PriorityVal : Maths.Value := Maths.NoValue;
      PriorityLex : LexTokenManager.LexString;

   begin
      -- Reject if pragma Priority has already been given.
      if not Dictionary.MainProgramPrioritySupplied then
         --
         -- We restrict the argument to pragma Priority to be a single ADA_primary
         -- over one of the following:
         -- (a) numeric_literal that resolves to integer_number
         --     numeric_literal
         --            |
         --     decimal_literal
         --            |
         --     integer_number
         -- (b) ADA_Name that resolves to a simple identifier in the current scope
         --     that is either a named number or a constant of type System.Priority.
         --     ADA_Name
         --        |
         --     identifier
         -- This is done by walking down the Child_Nodes of the
         -- Argument_Association_Rep until the ADA_primary is found, ensuring at
         -- each step of the way that NextDerivative is null (to ensure a single
         -- argument which has no operators), and then ensuring that the child of
         -- the ADA_primary is one of the allowed kinds of node.
         --
         Iterator := Next_Sibling (IdNode); -- Arg_Assoc_Rep or null
         -- Set the error reporting node.
         if Iterator = STree.NullNode then
            ErrNode := IdNode;
         else
            ErrNode := Iterator;
         end if;
         loop
            Error := Iterator = STree.NullNode or else
              Next_Sibling (Iterator) /= STree.NullNode;
            exit when Error or else SyntaxNodeType (Iterator) = SPSymbols.ADA_primary;
            Iterator := Child_Node (Iterator);
         end loop;

         if not Error then
            -- Iterator is the ADA_Primary. Check that we have the right kind of Primary.
            Iterator := Child_Node (Iterator);
            case SyntaxNodeType (Iterator) is
               when SPSymbols.numeric_literal =>
                  Iterator := Child_Node (Child_Node (Iterator));
                  if SyntaxNodeType (Iterator) = SPSymbols.integer_number then
                     GetLiteralValue (Iterator, PriorityVal);
                  else
                     Error := True;
                  end if;
               when SPSymbols.ADA_name =>
                  Iterator := Child_Node (Iterator);
                  if SyntaxNodeType (Iterator) = SPSymbols.identifier then
                     Sym := Dictionary.LookupItem (NodeLexString (Iterator),
                                                   Scope,
                                                   Dictionary.ProgramContext);
                     -- Ensure we have a local constant with a static integer value
                     if Sym /= Dictionary.NullSymbol and then
                       Dictionary.IsConstant (Sym) and then Dictionary.IsStatic (Sym, Scope) and then
                       Dictionary.CompatibleTypes (Scope,
                                                   Dictionary.GetType (Sym),
                                                   Dictionary.GetPredefinedIntegerType) then
                        PriorityVal := Maths.ValueRep (Dictionary.GetValue (Sym));
                     else
                        Error := True;
                     end if;
                  else
                     Error := True;
                  end if;
               when others => Error := True;
            end case;
         end if;

         if not Error then
            -- We may be able to do a range check on PriorityVal, if package System is provided.
            -- The value must be in the range of System.Priority (see RM D.1(8)).
            CheckPriorityRange (ErrorSym => Dictionary.GetMainProgram,
                                Scope => Scope,
                                PragmaKind => Dictionary.Priority,
                                ErrPos => NodePosition (ErrNode),
                                Value => PriorityVal,
                                 -- to get
                                ValueRep => PriorityLex);
            -- CheckPriorityRange will report any out of range value. The returned PriorityLex
            -- will be NullString if known to be out of range, or else the input value, so we
            -- add this result to the Dictionary.
            Dictionary.SetMainProgramPriority (PriorityLex);
         else
            -- Invalid argument for pragma Priority. Must be integer literal or local constant
            -- whose value is static integer.
            ErrorHandler.SemanticError (911,
                                        ErrorHandler.NoReference,
                                        NodePosition (ErrNode),
                                        LexTokenManager.NullString);
         end if;
      else
         ErrorHandler.SemanticError (879,
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     NodeLexString (IdNode));
      end if;
   end wf_main_program_priority;

   --------------------------------------------------------------------

   procedure HandleInterfaceOnSubprogram (SubprogDeclarationNode : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in     Node;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict           from *,
   --#                                        CommandLineData.Content,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        SubprogDeclarationNode &
   --#         ErrorHandler.ErrorContext from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        SubprogDeclarationNode &
   --#         SPARK_IO.File_Sys         from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        SubprogDeclarationNode;
   is
      NameNode          : STree.SyntaxNode;
      SubprogSym        : Dictionary.Symbol;
      Name              : LexTokenManager.LexString;
      ErrorInPragma     : Boolean;
   begin
      -- find name of subprogram starting from subprogram_declaration
      -- grammar
      --
      -- subprogram_declaration
      --            |
      --  procedure_specification OR function_specification
      --            |                          |
      --        identifier                designator
      --                                       |
      --                                   identifier
      NameNode := Child_Node (SubprogDeclarationNode);
      NameNode := Child_Node (NameNode);
      if SyntaxNodeType (NameNode) = SPSymbols.designator then
         NameNode := Child_Node (NameNode);
      end if;
      Name := NodeLexString (NameNode);
      SubprogSym := Dictionary.LookupItem (Name,
                                           Scope,
                                           Dictionary.ProgramContext);
      wf_external_interface (Node,
                             Name,
                              -- to get
                             ErrorInPragma);
      if not ErrorInPragma then
         Dictionary.AddBody (SubprogSym,
                             Dictionary.Location'(NodePosition (Node),
                                                  NodePosition (Node)),
                             True); -- treat interface procs as hidden
      end if;

   end HandleInterfaceOnSubprogram;

   --------------------------------------------------------------------

   procedure HandleImportOnVariable (BasicDeclarationNode : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     IdNode;
   --#        in     LexTokenManager.StringTable;
   --#        in     Node;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict           from *,
   --#                                        BasicDeclarationNode,
   --#                                        CommandLineData.Content,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from BasicDeclarationNode,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        IdNode,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table;
   is
      VarDecNode    : STree.SyntaxNode;
      NameNode      : STree.SyntaxNode;
      VarName       : LexTokenManager.LexString;
      VarSym        : Dictionary.Symbol;
      ErrorInPragma : Boolean;
   begin -- HandleImportOnVariable

      -- on entry, BasicDeclarationNode is a basic_declarative_item if there is any chance that we
      -- have found a variable declaration
      if SyntaxNodeType (BasicDeclarationNode) = SPSymbols.basic_declarative_item then
         --
         -- Grammar:
         -- basic_declarative_item
         --            |
         --    basic_declaration
         --            |
         --   object_declaration
         --            |
         --   variable_declaration
         --            |
         --       identifier_list
         --            |
         --       identifier (assumes V : T, not V1, V2, V3 ; T, otherwise there are N identifier_lists)
         --
         -- Checks:
         -- 1. illegal in Ada 83
         -- 2. must be a variable declaration
         -- 3. name consistency
         -- 4. no explicit initialization allowed
         -- 5. mode, init rules and warnings

         -- Check 1
         if CommandLineData.IsSpark95 then
            -- Check 2
            VarDecNode := Child_Node (Child_Node (Child_Node (BasicDeclarationNode)));
            if SyntaxNodeType (VarDecNode) = SPSymbols.variable_declaration then
               -- check 3
               NameNode := LastChildOf (VarDecNode);
               VarName := NodeLexString (NameNode);
               -- use existing wf to check name consistency
               wf_external_interface (Node,
                                      VarName,
                                       -- to get
                                      ErrorInPragma);
               if not ErrorInPragma then
                  -- Potentially OK to add pragma to variable in dictionary,
                  -- for which we will need a symbol
                  VarSym := Dictionary.LookupItem (VarName,
                                                   Scope,
                                                   Dictionary.ProgramContext);

                  -- If the pragma Import "looks right" in that it names the
                  -- variable declared immediately above it, LookupItem still might
                  -- return NullSymbol if the variable declaration itself was illegal -
                  -- for example, in the case of a missing own annotation.  If LookupItem
                  -- does return NullSymbol, then do nothing since an error message on the
                  -- offending variable declaration will have already been issued.
                  if VarSym /= Dictionary.NullSymbol then

                     -- since the pragma import has just been checked to ensure it directly
                     -- follows a variable declaration, the abov elook up must always succeed
                     SystemErrors.RTAssert (Dictionary.IsVariable (VarSym),
                                            SystemErrors.AssertionFailure,
                                            "Variable not found in HandleImportOnVariable");

                     -- Check 4
                     if Dictionary.VariableIsInitialized (VarSym) then
                        ErrorHandler.SemanticError (120,
                                                    ErrorHandler.NoReference,
                                                    NodePosition (Node),
                                                    VarName);

                     else
                        -- Now we really can add the pragma to the Dictionary
                        -- Following call also marks variable as being initialized "at declaration"
                        Dictionary.AddVariablePragmaImport (VarSym);

                        -- Check 5
                        -- First deal with own variables that have a pragma import but aren't in an
                        -- initializes clause and don't have a mode, an existing function gives the desired answer
                        if UnexpectedInitialization (VarSym) then
                           ErrorHandler.SemanticError (333,
                                                       ErrorHandler.NoReference,
                                                       NodePosition (Node),
                                                       VarName);
                        end if;
                        -- Now warn in the case where we have ANY variable that lacks a "stream" mode (even if it
                        -- passes the previous test because it is an initialized own variable)
                        if Dictionary.GetOwnVariableOrConstituentMode (VarSym) =
                          Dictionary.DefaultMode then
                           ErrorHandler.SemanticWarning (350,
                                                         NodePosition (Node),
                                                         VarName);
                        end if;
                     end if;
                  end if;
               end if;
            else -- not a variable_declaration
               ErrorHandler.SemanticError (72,
                                           ErrorHandler.NoReference,
                                           NodePosition (Node),
                                           NodeLexString (IdNode));
            end if;
         else -- Ada 83
            -- will report unexpected pragma INTERFACE.
            ErrorHandler.SemanticError (72,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        NodeLexString (IdNode));
         end if;
      else -- wasn't a basic_declarative_item on entry so no chance it can be a variable
         ErrorHandler.SemanticError (72,
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     NodeLexString (IdNode));
      end if;
   end HandleImportOnVariable;

   --------------------------------------------------------------------

   -------------------------------------------------
   -- This function checks the number of arguments
   -- present in a pragma statement by counting the
   -- number of argument_association_rep symbols.
   -- This is then compared against known acceptable
   -- numbers of arguments taken from the LRM.
   --
   -- Return:  True if number of arguments OK,
   --          False otherwise
   -------------------------------------------------


      function NumberOfArgs (PragmaNode : STree.SyntaxNode)
               return Natural
      --# global in STree.Table;
      is
         ArgAssNode  : STree.SyntaxNode;
         NumArgs     : Natural := 0;
      begin
         ArgAssNode := Next_Sibling (Child_Node (PragmaNode));
         while SyntaxNodeType (ArgAssNode) =
            SPSymbols.argument_association_rep
         loop
            NumArgs := NumArgs + 1;
            ArgAssNode := Child_Node (ArgAssNode);
         end loop;

         return NumArgs;
      end NumberOfArgs;




   procedure CheckArgCount (Node         : in  STree.SyntaxNode;
                            StatementOK  : out Boolean;
                            ErrorToRaise : out Natural)
   --# global in CommandLineData.Content;
   --#        in STree.Table;
   --# derives ErrorToRaise,
   --#         StatementOK  from CommandLineData.Content,
   --#                           Node,
   --#                           STree.Table;
   is
      procedure CheckArgCount95 (Node         : in  STree.SyntaxNode;
                                 StatementOK  : out Boolean;
                                 ErrorToRaise : out Natural)
      --# global in STree.Table;
      --# derives ErrorToRaise,
      --#         StatementOK  from Node,
      --#                           STree.Table;
      is
         PragmaType  : LexTokenManager.LexString;
      begin
         PragmaType := NodeLexString (Child_Node (Node));

         if PragmaType = LexTokenManager.Normalize_ScalarsToken or
            PragmaType = LexTokenManager.PageToken or
            PragmaType = LexTokenManager.ReviewableToken then
               -- Check for 0 args
               StatementOK  := (NumberOfArgs (Node) = 0);
               ErrorToRaise := 365;

         elsif PragmaType = LexTokenManager.All_Calls_RemoteToken or
               PragmaType = LexTokenManager.Discard_NamesToken or
               PragmaType = LexTokenManager.Elaborate_BodyToken or
               PragmaType = LexTokenManager.Interrupt_PriorityToken or
               PragmaType = LexTokenManager.PreelaborateToken or
               PragmaType = LexTokenManager.PureToken or
               PragmaType = LexTokenManager.Remote_Call_InterfaceToken or
               PragmaType = LexTokenManager.Remote_TypesToken or
               PragmaType = LexTokenManager.Shared_PassiveToken then
               -- Check for 0/1 arg
               StatementOK  := (NumberOfArgs (Node) = 0) or
                               (NumberOfArgs (Node) = 1);
               ErrorToRaise := 360;

         elsif PragmaType = LexTokenManager.AsynchronousToken or
               PragmaType = LexTokenManager.AtomicToken or
               PragmaType = LexTokenManager.Atomic_ComponentsToken or
               PragmaType = LexTokenManager.ControlledToken or
               PragmaType = LexTokenManager.Interrupt_HandlerToken or
               PragmaType = LexTokenManager.Linker_OptionsToken or
               PragmaType = LexTokenManager.ListToken or
               PragmaType = LexTokenManager.Locking_PolicyToken or
               PragmaType = LexTokenManager.OptimizeToken or
               PragmaType = LexTokenManager.PackToken or
               PragmaType = LexTokenManager.PriorityToken or
               PragmaType = LexTokenManager.Queueing_PolicyToken or
               PragmaType = LexTokenManager.Storage_SizeToken or
               PragmaType = LexTokenManager.Task_Dispatching_PolicyToken or
               PragmaType = LexTokenManager.VolatileToken or
               PragmaType = LexTokenManager.Volatile_ComponentsToken then
               -- Check for 1 arg
               StatementOK  := (NumberOfArgs (Node) = 1);
               ErrorToRaise := 361;

         elsif PragmaType = LexTokenManager.SuppressToken then
               -- Check for 1/2 args
               StatementOK  := (NumberOfArgs (Node) = 1) or
                               (NumberOfArgs (Node) = 2);
               ErrorToRaise := 366;

         elsif PragmaType = LexTokenManager.ElaborateToken or
               PragmaType = LexTokenManager.Elaborate_AllToken or
               PragmaType = LexTokenManager.InlineToken or
               PragmaType = LexTokenManager.RestrictionsToken then
               -- Check for >=1 args
               StatementOK  := (NumberOfArgs (Node) /= 0);
               ErrorToRaise := 363;

         elsif
               PragmaType = LexTokenManager.Attach_HandlerToken or
               PragmaType = LexTokenManager.ConventionToken then
               -- Check for 2 args
               StatementOK  := (NumberOfArgs (Node) = 2);
               ErrorToRaise := 362;

         elsif PragmaType = LexTokenManager.ExportToken or
               PragmaType = LexTokenManager.ImportToken then
               -- Check for 2-4 args
               StatementOK  := (NumberOfArgs (Node) >= 2) and
                               (NumberOfArgs (Node) <= 4);
               ErrorToRaise := 364;

         elsif PragmaType = LexTokenManager.Inspection_PointToken then
               -- Any number of arguments
               StatementOK  := True;
               ErrorToRaise := 0;

         else
               -- Unknown pragma! Ada 2005? Implementation defined?
               StatementOK  := True;
               ErrorToRaise := 0;
         end if;
      end CheckArgCount95;

      procedure CheckArgCount83 (Node         : in  STree.SyntaxNode;
                                 StatementOK  : out Boolean;
                                 ErrorToRaise : out Natural)
      --# global in STree.Table;
      --# derives ErrorToRaise,
      --#         StatementOK  from Node,
      --#                           STree.Table;
      is
         PragmaType  : LexTokenManager.LexString;
      begin
         PragmaType := NodeLexString (Child_Node (Node));

         if PragmaType = LexTokenManager.PageToken then
               -- Check for 0 args
               StatementOK  := (NumberOfArgs (Node) = 0);
               ErrorToRaise := 365;

         elsif PragmaType = LexTokenManager.ControlledToken or
               PragmaType = LexTokenManager.ListToken or
               PragmaType = LexTokenManager.Memory_SizeToken or
               PragmaType = LexTokenManager.OptimizeToken or
               PragmaType = LexTokenManager.PackToken or
               PragmaType = LexTokenManager.PriorityToken or
               PragmaType = LexTokenManager.SharedToken or
               PragmaType = LexTokenManager.Storage_UnitToken or
               PragmaType = LexTokenManager.System_NameToken then
               -- Check for 1 arg
               StatementOK  := (NumberOfArgs (Node) = 1);
               ErrorToRaise := 361;

         elsif PragmaType = LexTokenManager.SuppressToken then
               -- Check for 1/2 args
               StatementOK  := (NumberOfArgs (Node) = 1) or
                               (NumberOfArgs (Node) = 2);
               ErrorToRaise := 366;

         elsif PragmaType = LexTokenManager.ElaborateToken or
               PragmaType = LexTokenManager.InlineToken then
               -- Check for >=1 args
               StatementOK  := (NumberOfArgs (Node) /= 0);
               ErrorToRaise := 363;

         -- InterfaceToken requires 2 arguments, but always checked
         -- before entry to this subprogram

         else
               -- Unknown pragma! Ada 2005? Implementation defined?
               StatementOK  := True;
               ErrorToRaise := 0;
         end if;
      end CheckArgCount83;

   begin
      if CommandLineData.IsSpark95 then
         CheckArgCount95 (Node, StatementOK, ErrorToRaise);
      else
         CheckArgCount83 (Node, StatementOK, ErrorToRaise);
      end if;
   end CheckArgCount;



begin -- wf_pragma
   IdNode := Child_Node (Node);
   if SyntaxNodeType (IdNode) = SPSymbols.assert_pragma then
      ErrorHandler.APragma (LexTokenManager.AssertToken,
                            NodePosition (Node));
   else
      if IsExternalInterface (Node) then
         --Name := LexTokenManager.NullString;
         if WeAreNotInAProtectedType then -- Import/Interface MAY be ok

            -- Grammar (let XXX = visible_part_rep OR basic_declarative_item_rep
            -- Subprogram case
            --
            -- XXX --- subprogram_declaration
            --  |
            -- XXX --- apragma
            --
            -- Variable cases
            --
            -- XXX --- basic_declarative_item
            --  |
            -- XXX --- apragma
            --
            -- OR -----------------------
            --
            -- initial_declarative_item_rep --- basic_declarative_item
            --              |
            -- initial_declarative_item_rep --- apragma
            --

            -- Look for the preceeding declaration
            SubprogOrVariableNode := Child_Node (Child_Node (ParentNode (Node)));
            if SyntaxNodeType (SubprogOrVariableNode) = SPSymbols.visible_part_rep or
               SyntaxNodeType (SubprogOrVariableNode) = SPSymbols.basic_declarative_item_rep then
               SubprogOrVariableNode := Next_Sibling (SubprogOrVariableNode);
            end if;

            if SyntaxNodeType (SubprogOrVariableNode) = SPSymbols.subprogram_declaration then
               -- It's a subprogram_declaration, note that pragma Import must immediately follow
               -- the subprogram declaration so going one step up the syntax tree will find the subprogram
               -- if one is there.
               HandleInterfaceOnSubprogram (SubprogOrVariableNode);

            elsif SyntaxNodeType (SubprogOrVariableNode) = SPSymbols.basic_declarative_item then
               -- it might be a variable declaration
               HandleImportOnVariable (SubprogOrVariableNode);

            elsif SyntaxNodeType (SubprogOrVariableNode) = SPSymbols.initial_declarative_item_rep then
               HandleImportOnVariable (Next_Sibling (SubprogOrVariableNode)); -- basic_declarative_item

            else
               -- none of the things where pragma import allowed has been found so it's an error
               ErrorHandler.SemanticError (72,
                                           ErrorHandler.NoReference,
                                           NodePosition (Node),
                                           NodeLexString (IdNode));
            end if;
         else -- we ARE in a protected type, so pragma import is unexpected
            ErrorHandler.SemanticError (72,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        NodeLexString (IdNode));
         end if; -- in protected type

         -- handle pragma elaborate_body which must be the first
         -- visible_part_rep in a package specification to be acceptable
      elsif CommandLineData.IsSpark95 and then
        NodeLexString (IdNode) = LexTokenManager.Elaborate_BodyToken
      then  -- A pragma Elaborate_Body has been found and must be processed.
            -- First check it is correctly positioned
         PackSpecNode := ParentNode (Node);
         if  SyntaxNodeType (PackSpecNode) = SPSymbols.visible_part_rep and then
           Child_Node (Child_Node (PackSpecNode)) = STree.NullNode
         then  -- Potentially legal.
               -- Work up chain to package specification node
            while SyntaxNodeType (PackSpecNode) /= SPSymbols.package_specification
            loop
               PackSpecNode := ParentNode (PackSpecNode);
            end loop;
            -- find identifier of package
            PackIdentNode := Child_Node (Child_Node (PackSpecNode));
            while SyntaxNodeType (PackIdentNode) /= SPSymbols.identifier
            loop
               PackIdentNode := Next_Sibling (PackIdentNode);
            end loop;
            wf_elaborate_body (Node,
                               Dictionary.LookupItem (NodeLexString (PackIdentNode),
                                                      Scope,
                                                      Dictionary.ProgramContext));
         else -- unexpected in this position
            ErrorHandler.SemanticError (72,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        NodeLexString (IdNode));
         end if;

      elsif CommandLineData.RavenscarSelected  then
         if NodeLexString (IdNode) = LexTokenManager.AtomicToken then
            -- special handling for Pragma Atomic (...) in Ravenscar mode
            Wf_Pragma_Atomic (Node);
         elsif (NodeLexString (IdNode) = LexTokenManager.PriorityToken) or else
           (NodeLexString (IdNode) = LexTokenManager.Interrupt_PriorityToken)then
            -- Other than pragma Priority in the declarative part of the main program,
            -- we have an out-of-place priority pragma; these are handled by
            -- wf_priority_pragma which is called from grammar-specific locations
            -- where a priority pragma is allowed.
            if Dictionary.MainProgramExists and then
              Dictionary.IsMainProgram (Dictionary.GetRegion (Scope)) and then
              SyntaxNodeType (ParentNode (Node)) /= SPSymbols.statement and then
              NodeLexString (IdNode) = LexTokenManager.PriorityToken then
               wf_main_program_priority (IdNode);
            else
               ErrorHandler.SemanticError (879,
                                           ErrorHandler.NoReference,
                                           NodePosition (Node),
                                           NodeLexString (IdNode));
            end if;
         elsif NodeLexString (IdNode) = LexTokenManager.Interrupt_HandlerToken then
            ErrorHandler.SemanticError (883,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        LexTokenManager.NullString);
         elsif NodeLexString (IdNode) = LexTokenManager.Atomic_ComponentsToken then
            ErrorHandler.SemanticError (842,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        LexTokenManager.NullString);
         elsif NodeLexString (IdNode) = LexTokenManager.Volatile_ComponentsToken then
            ErrorHandler.SemanticError (843,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        LexTokenManager.NullString);
         elsif NodeLexString (IdNode) = LexTokenManager.Attach_HandlerToken then
            wf_attach_handler (Node);
         else
            ErrorHandler.APragma (NodeLexString (IdNode),
                                  NodePosition (Node));
         end if;

      else -- not a "special" pragma so handle with normal warning
         -- Check the number of arguments associated with
         -- particular pragmas.
         CheckArgCount (Node, StatementOK, ErrorToRaise);
         if not StatementOK then
            ErrorHandler.SemanticWarning (ErrorToRaise,
                                          NodePosition (Node),
                                          LexTokenManager.NullString);
         else
            ErrorHandler.APragma (NodeLexString (IdNode),
                                  NodePosition (Node));
         end if;
      end if;
   end if;

end wf_pragma;
