-- $Id: sem-compunit-wf_pragma-wf_attach_handler.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.CompUnit.wf_pragma)
procedure wf_attach_handler (PragmaNode : in STree.SyntaxNode)
is
   -- Grammar: (wellformed case shown)
   --
   --          PragmaNode
   --              |
   --          identifier --- argument_association_rep
   --                                   |
   --                         argument_association_rep --- argument_association
   --                                   |                          |
   --                          argument_association           ADA_expression
   --                                   |
   --                             ADA_expression
   -- Checks:
   --       (1) Positional association
   --       (2) Exactly 2 arguments
   --       (3) First is the procedure name
   --       (4) 2nd is ignore for now
   --       (5) pragma immediately follows procedure
   --       -- Rule 6 removed after design rethink (6) only one attach_handler per PT
   --       (7) procedure must be parameterless
   --       (8) must be in PT

   TheRegion               : Dictionary.Symbol;
   ProcSpecNode            : STree.SyntaxNode;
   ProcIdent               : LexTokenManager.LexString;
   ProcSym                 : Dictionary.Symbol;
   ErrorFound              : Boolean := False;

   procedure FindProcSpec
   --# global in     PragmaNode;
   --#        in     STree.Table;
   --#           out ProcSpecNode;
   --# derives ProcSpecNode from PragmaNode,
   --#                           STree.Table;
   -- locates the place where a procedure_specification should be if the pragma is
   -- correctly placed.
   is
      CurrentNode : STree.SyntaxNode;
   begin
      -- There are two cases to consider: the attach_handler follows the first subprogram in the PT; or
      -- it follows some later subprogram declaration.
      -- Note that the protected_operation_declaration_rep grammar means that the sequence of declarations
      -- is "upside down" with the first declaration at the bottom.
      CurrentNode := Child_Node (ParentNode (PragmaNode));
      -- protected_operation_declaration_rep to left of pragma

      if Child_Node (CurrentNode) = STree.NullNode then
         -- The pragma is at the bottom of the sequence of protected_operation_declaration_reps and
         -- so we are dealing with FIRST subprogram in the PT (immediately after the priority pragma)

         -- Go to the top of the list of protected_operation_declaration_reps
         loop
            CurrentNode := ParentNode (CurrentNode);
            exit when SyntaxNodeType (CurrentNode) = SPSymbols.protected_operation_declaration;
         end loop;
         -- Move to procedure spec
         ProcSpecNode := Child_Node (Child_Node
                                           (Next_Sibling (Child_Node (CurrentNode))));
      else
         -- we are dealing with a potential subprogram in the
         -- sequence of declarations in the PT declarative part
         ProcSpecNode := Child_Node (Next_Sibling (Child_Node (CurrentNode)));
      end if;
   end FindProcSpec;

   --------------------------------------------------------------------------------------

   procedure CheckArguments (PragmaNode : in STree.SyntaxNode;
                             EntityName : in LexTokenManager.LexString)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out ErrorFound;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorFound                from *,
   --#                                        EntityName,
   --#                                        PragmaNode,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EntityName,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        PragmaNode,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table;
   is
      ArgAssocRepNode : STree.SyntaxNode;
      SubprogNameNode : STree.SyntaxNode;


      procedure CheckRepresentSameName (ExpNode : in STree.SyntaxNode;
                                        Name    : in LexTokenManager.LexString)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in     STree.Table;
      --#        in out ErrorFound;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.File_Sys;
      --# derives ErrorFound                from *,
      --#                                        ExpNode,
      --#                                        Name,
      --#                                        STree.Table &
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.File_Sys         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        ExpNode,
      --#                                        LexTokenManager.StringTable,
      --#                                        Name,
      --#                                        SPARK_IO.File_Sys,
      --#                                        STree.Table;
      is
         IsChain : Boolean;
         IdNode,
         NextNode : STree.SyntaxNode;
      begin
         IdNode := ExpNode;
         loop
            IsChain := Next_Sibling (IdNode) =
              STree.NullNode;
            NextNode := Child_Node (IdNode);
            exit when not IsChain or NextNode = STree.NullNode;

            IdNode := NextNode;
         end loop;

         if IsChain and then
           SyntaxNodeType (IdNode) = SPSymbols.identifier and then
           NodeLexString (IdNode) = Name
         then
            null;
         else -- Rule 3 failure
            ErrorFound := True;
            ErrorHandler.SemanticError (71,
                                        ErrorHandler.NoReference,
                                        NodePosition (ExpNode),
                                        Name);
         end if;
      end CheckRepresentSameName;

      --------------------------

   begin --CheckArguments
      ArgAssocRepNode := Child_Node (Next_Sibling (Child_Node (PragmaNode)));
      if SyntaxNodeType (Child_Node (ArgAssocRepNode)) =
         SPSymbols.argument_association
      then --pragma has two arguments
         SubprogNameNode := Child_Node (Child_Node (ArgAssocRepNode));

         if SyntaxNodeType (SubprogNameNode) /=
            SPSymbols.ADA_expression
         then --form of expression wrong - Rule 1 failure
            ErrorFound := True;
            ErrorHandler.SemanticError (71,
                                        ErrorHandler.NoReference,
                                        NodePosition (SubprogNameNode),
                                        EntityName);
         else --form of expression ok so check name actually matches
            CheckRepresentSameName (SubprogNameNode,
                                    EntityName);
         end if;

      else --pragma does nor have exactly 2 arguments -- Rule 2 failure
         ErrorFound := True;
         ErrorHandler.SemanticError (69,
                                     ErrorHandler.NoReference,
                                     NodePosition (PragmaNode),
                                     LexTokenManager.Attach_HandlerToken);
      end if;
   end CheckArguments;

   -----------------------------------

begin --wf_attach_handler
   TheRegion := Dictionary.GetRegion (Scope);
   -- attach_Handler can only appear in the spec of a protected type
   if Dictionary.IsType (TheRegion) and then Dictionary.TypeIsProtected (TheRegion) then
      FindProcSpec;
      if SyntaxNodeType (ProcSpecNode) = SPSymbols.procedure_specification then
         ProcIdent := NodeLexString (Child_Node (ProcSpecNode));
         CheckArguments (PragmaNode,
                         ProcIdent);
         if not ErrorFound then
            ProcSym := Dictionary.LookupItem (ProcIdent,
                                              Scope,
                                              Dictionary.ProgramContext);
            if Dictionary.GetNumberOfSubprogramParameters (ProcSym) = 0 then
                  Dictionary.SetIsInterruptHandler (ProcSym);
                  Dictionary.SetTypeHasPragma (TheRegion,
                                               Dictionary.AttachHandler);
            else -- rule 7 failure
               ErrorHandler.SemanticError (885,
                                           ErrorHandler.NoReference,
                                           NodePosition (PragmaNode),
                                           LexTokenManager.NullString);

            end if;
         end if; -- ErrorFound
      else -- rule 5 failure
         ErrorHandler.SemanticError (884,
                                     ErrorHandler.NoReference,
                                     NodePosition (PragmaNode),
                                     LexTokenManager.NullString);
      end if;
   else -- not in PT (Rule 8)
      ErrorHandler.SemanticError (884,
                                  ErrorHandler.NoReference,
                                  NodePosition (PragmaNode),
                                  LexTokenManager.NullString);
   end if;
end wf_attach_handler;
