-- $Id: sem-compunit-wf_package_specification-checkmodes.adb 12351 2009-02-02 15:03:51Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.CompUnit.wf_package_specification)

procedure CheckModes (Node    : in STree.SyntaxNode;
                      PackSym : in Dictionary.Symbol)
is
   --Node is visible_part node of the package declaration we are in

   --# inherit Heap,
   --#         LexTokenManager,
   --#         SeqAlgebra,
   --#         SPARK_IO,
   --#         Statistics;
   package SubprogList
   --# own State;
   --# initializes State;
   is
      procedure AddPair (TheHeap      : in out Heap.HeapRecord;
                         SubprogStr,
                         ParamStr     : in     LexTokenManager.LexString);
      --# global in out State;
      --#        in out Statistics.TableUsage;
      --# derives State                 from *,
      --#                                    SubprogStr,
      --#                                    TheHeap &
      --#         Statistics.TableUsage,
      --#         TheHeap               from *,
      --#                                    ParamStr,
      --#                                    State,
      --#                                    SubprogStr,
      --#                                    TheHeap;

      procedure SelectSubprog (TheHeap    : in     Heap.HeapRecord;
                               SubprogStr : in     LexTokenManager.LexString;
                               IsAffected :    out Boolean);
      --# global in out State;
      --# derives IsAffected,
      --#         State      from State,
      --#                         SubprogStr,
      --#                         TheHeap;

      --NB. Select subprogam using above call before using this function
      function IsAffectedParameter (TheHeap  : Heap.HeapRecord;
                                    ParamStr : LexTokenManager.LexString)
                                   return Boolean;
      --# global in State;
   end SubprogList;

   ---------------------------------------------------------------------------

   PrivTypeIt,
   SubprogIt,
   ParamIt            : Dictionary.Iterator;
   TypeSym,
   SubprogSym,
   ParamSym           : Dictionary.Symbol;
   VisPartRepNode,
   ProcSpecNode       : STree.SyntaxNode;
   SubprogramsToMark,
   IsAffected         : Boolean;

   ---------------------------------------------------------------------------

   package body SubprogList
   --# own State is CurrentParamList,
   --#              CurrentSubProgStr,
   --#              SubPrgList;
   is

      CurrentParamList  : SeqAlgebra.Seq;
      CurrentSubProgStr : LexTokenManager.LexString;
      SubPrgList        : Heap.Atom;

      -----------------------------------------------------------------------

      procedure SelectSubprog (TheHeap    : in     Heap.HeapRecord;
                               SubprogStr : in     LexTokenManager.LexString;
                               IsAffected :    out Boolean)
      --# global in     SubPrgList;
      --#        in out CurrentParamList;
      --# derives CurrentParamList from *,
      --#                               SubPrgList,
      --#                               SubprogStr,
      --#                               TheHeap &
      --#         IsAffected       from SubPrgList,
      --#                               SubprogStr,
      --#                               TheHeap;
      is
         SeekCell : Heap.Atom;
         SeekVal  : Natural;
      begin
         IsAffected := False;
         SeekVal := LexTokenManager.LexStringRef (SubprogStr);
         SeekCell := SubPrgList;
         while not Heap.IsNullPointer (Heap.APointer (TheHeap, SeekCell))
         loop
            SeekCell := Heap.APointer (TheHeap, SeekCell);
            if Heap.AValue (TheHeap, SeekCell) = SeekVal then
               IsAffected := True;
               CurrentParamList := SeqAlgebra.NaturalToSeq
                  (Natural (Heap.BPointer (TheHeap, SeekCell)));
               exit;
            end if;
         end loop;
      end SelectSubprog;

      ---------------------------------------------------------------------

      procedure AddPair (TheHeap    : in out Heap.HeapRecord;
                         SubprogStr,
                         ParamStr   : in     LexTokenManager.LexString)
      --# global in out CurrentParamList;
      --#        in out CurrentSubProgStr;
      --#        in out Statistics.TableUsage;
      --#        in out SubPrgList;
      --# derives CurrentParamList,
      --#         SubPrgList            from *,
      --#                                    CurrentSubProgStr,
      --#                                    SubPrgList,
      --#                                    SubprogStr,
      --#                                    TheHeap &
      --#         CurrentSubProgStr     from *,
      --#                                    SubprogStr &
      --#         Statistics.TableUsage,
      --#         TheHeap               from *,
      --#                                    CurrentParamList,
      --#                                    CurrentSubProgStr,
      --#                                    ParamStr,
      --#                                    SubPrgList,
      --#                                    SubprogStr,
      --#                                    TheHeap;
      is
         SubProgCell : Heap.Atom;
         Found       : Boolean;
      begin
         if SubprogStr /= CurrentSubProgStr then
            if CurrentSubProgStr = LexTokenManager.NullString then
               --first call so need to create list
               Heap.CreateAtom (TheHeap,
                                 --to get
                                SubPrgList);
            end if;
            CurrentSubProgStr := SubprogStr;

            --first see if we already have an entry for this subprogram
            SelectSubprog (TheHeap,
                           SubprogStr,
                           --to get
                           Found);

            --if we haven't found it then we need to create a suitable entry
            if not Found then
               --create and link in new subprogram cell
               Heap.CreateAtom (TheHeap,
                                 --to get
                                SubProgCell);
               Heap.UpdateAValue (TheHeap,
                                  SubProgCell,
                                  LexTokenManager.LexStringRef (SubprogStr));
               Heap.UpdateAPointer (TheHeap,
                                    SubProgCell,
                                    Heap.APointer (TheHeap, SubPrgList));
               Heap.UpdateAPointer (TheHeap,
                                    SubPrgList,
                                    SubProgCell);
               --create new list of parameters and associate it with new subprogram cell
               SeqAlgebra.CreateSeq (TheHeap,
                                     CurrentParamList);
               Heap.UpdateBPointer (TheHeap,
                                    SubProgCell,
                                    Heap.Atom (SeqAlgebra.SeqToNatural (CurrentParamList)));
            end if;
         end if;
         SeqAlgebra.AddMember (TheHeap,
                               CurrentParamList,
                               LexTokenManager.LexStringRef (ParamStr));
      end AddPair;

      --------------------------------------------------------------------

      function IsAffectedParameter (TheHeap  : Heap.HeapRecord;
                                    ParamStr : LexTokenManager.LexString)
                                   return Boolean
      --# global in CurrentParamList;
      is
      begin
         return SeqAlgebra.IsMember (TheHeap,
                                     CurrentParamList,
                                     LexTokenManager.LexStringRef (ParamStr));
      end IsAffectedParameter;

      --------------------------------------------------------------------

   begin -- initialization
      CurrentSubProgStr := LexTokenManager.NullString;
      -- other two components of State do not need initialization
      -- because they will be given values in AddPair if they are
      -- undefined when it is called
      --# accept Flow, 31, CurrentParamList, "Will be defined when necessary" &
      --#        Flow, 32, CurrentParamList, "Will be defined when necessary" &
      --#        Flow, 31, SubPrgList, "Will be defined when necessary" &
      --#        Flow, 32, SubPrgList, "Will be defined when necessary";
   end SubprogList; -- ignore DFA on incomplete init of abstract state

   ---------------------------------------------------------------------------

   procedure ProcessProcedure (Node : in     STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in     SubProgList.State;
   --#        in     TheHeap;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        SubProgList.State,
   --#                                        TheHeap;
   is
      It          : STree.Iterator;
      IdNode      : STree.SyntaxNode;

   begin --ProcessProcedure (Node is formal_part)

      It := FindFirstNode (NodeKind    => SPSymbols.identifier,
                           FromRoot    => Node,
                           InDirection => STree.Down);

      while not STree.IsNull (It) loop
         IdNode := GetNode (It);
         if SubprogList.IsAffectedParameter (TheHeap,
                                             NodeLexString (IdNode))
         then
            ErrorHandler.SemanticError (338,
                                        ErrorHandler.NoReference,
                                        NodePosition (IdNode),
                                        NodeLexString (IdNode));
         end if;
         It := STree.NextNode (It);
      end loop;
   end ProcessProcedure;

   ---------------------------------------------------------------------------

begin --CheckModes
   SubprogramsToMark := False;
   PrivTypeIt := Dictionary.FirstPrivateType (PackSym);
   while not Dictionary.IsNullIterator (PrivTypeIt) loop
      TypeSym := Dictionary.CurrentSymbol (PrivTypeIt);
      if Dictionary.IsDeclared (TypeSym) and then
         Dictionary.TypeIsScalar (TypeSym)
      then --we have a scalar private type which may affect subprog params
         SubprogIt := Dictionary.FirstVisibleSubprogram (PackSym);
         while not Dictionary.IsNullIterator (SubprogIt) loop
            SubprogSym := Dictionary.CurrentSymbol (SubprogIt);

            ParamIt := Dictionary.FirstSubprogramParameter (SubprogSym);
            while not Dictionary.IsNullIterator (ParamIt) loop
               ParamSym := Dictionary.CurrentSymbol (ParamIt);
               if Dictionary.GetType (ParamSym) = TypeSym and then
                  Dictionary.GetSubprogramParameterMode (ParamSym) =
                  Dictionary.InOutMode and then
                  not Dictionary.IsImport (Dictionary.IsAbstract,
                                           SubprogSym,
                                           ParamSym)
               then
                  SubprogramsToMark := True;
                  SubprogList.AddPair (TheHeap,
                                       Dictionary.GetSimpleName (SubprogSym),
                                       Dictionary.GetSimpleName (ParamSym));
               end if;
               ParamIt := Dictionary.NextSymbol (ParamIt);
            end loop;

            SubprogIt := Dictionary.NextSymbol (SubprogIt);
         end loop;
      end if;
      PrivTypeIt := Dictionary.NextSymbol (PrivTypeIt);
   end loop;

   -- At this point we have created in SubprogList a data structure listing
   -- all the procedures made illegal by the private types' full declarations
   -- and for each of them a list of affected parameters.  We now walk the
   -- syntax tree marking each parameter occurrence found.
   if SubprogramsToMark then
      VisPartRepNode := Child_Node (Node);
      while VisPartRepNode /= STree.NullNode
      loop
         ProcSpecNode := Child_Node (Next_Sibling (VisPartRepNode));
         if SyntaxNodeType (ProcSpecNode) =
            SPSymbols.procedure_specification
         then
            SubprogList.SelectSubprog (TheHeap,
                                       NodeLexString (Child_Node (ProcSpecNode)),
                                       --to get
                                       IsAffected);
            if IsAffected then
               ProcessProcedure (Next_Sibling (Child_Node (ProcSpecNode)));
            end if;
         end if;
         VisPartRepNode := Child_Node (VisPartRepNode);
      end loop;
   end if;
end CheckModes;
