-- $Id: sem-compunit-wf_package_body-wf_refine-wf_clause.adb 12351 2009-02-02 15:03:51Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.CompUnit.wf_package_body.wf_refine)
procedure wf_clause (Node  : in STree.SyntaxNode;
                     Scope : in Dictionary.Scopes)
is
   type Modes is
      record
         Value   : Dictionary.Modes;
         IsValid : Boolean;
      end record;

   SubjectSym      : Dictionary.Symbol;
   SubjectNode     : STree.SyntaxNode;
   SubjectLoc      : LexTokenManager.TokenPosition;
   ConstituentNode : STree.SyntaxNode;

   -----------------------------------------------------------------

   procedure DoSubject (SubjectNode : in     STree.SyntaxNode;
                        Scope       : in     Dictionary.Scopes;
                        SubjectSym  :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        SubjectNode &
   --#         SubjectSym                from Dictionary.Dict,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        SubjectNode;

   is
      VarSym : Dictionary.Symbol;
      VarPos : LexTokenManager.TokenPosition;
      VarStr : LexTokenManager.LexString;

   begin -- DoSubject
      VarStr := NodeLexString (SubjectNode);
      VarPos := NodePosition (SubjectNode);
      VarSym := Dictionary.LookupItem (VarStr,
                                       Dictionary.VisibleScope (Dictionary.GetRegion (Scope)),
                                       Dictionary.ProofContext);
      if VarSym = Dictionary.NullSymbol then --not declared or visible
         ErrorHandler.SemanticError (136,  ErrorHandler.NoReference, VarPos, VarStr);
         SubjectSym := Dictionary.NullSymbol;

      elsif Dictionary.IsOwnVariable (VarSym) then     -- possibly ok, it's an own var
         if not Dictionary.GetOwnVariableProtected (VarSym) then -- cannot refine protected state
            if not Dictionary.IsVirtualElement (VarSym) then -- cannot refine virtual elements
               if Dictionary.IsDeclared (VarSym) or else     -- declared as concrete
                 Dictionary.OwnVariableIsAbstract (VarSym)  -- or already refined away
               then                                         -- so raise error 73
                  ErrorHandler.SemanticError (73, ErrorHandler.NoReference,  VarPos, VarStr);
                  SubjectSym := Dictionary.NullSymbol;

               elsif  (Dictionary.OwnVariableHasType (VarSym,
                                                      Scope))
               then  -- type announced so refinement valid only if type is abstract proof type
                  if  Dictionary.IsDeclared (Dictionary.GetType (VarSym)) then
                     -- type is concrete type so refinement definitely illegal
                     ErrorHandler.SemanticError (73, ErrorHandler.NoReference,  VarPos, VarStr);
                     SubjectSym := Dictionary.NullSymbol;

                  else
                     -- type not declared so we don't know whether problem is illegal refinement
                     -- or missing abstract type declaration.  Need special error message here
                     ErrorHandler.SemanticError (273, ErrorHandler.NoReference,  VarPos, VarStr);
                     SubjectSym := Dictionary.NullSymbol;
                  end if;

               else  -- this is the valid case
                  SubjectSym := VarSym;
               end if;
            else -- it's a virtual element which cannot be refined
               ErrorHandler.SemanticError (858, ErrorHandler.NoReference, VarPos, VarStr);
               SubjectSym := Dictionary.NullSymbol;
            end if;
         else -- it's a protected own variable which cannot be refined
            ErrorHandler.SemanticError (859, ErrorHandler.NoReference, VarPos, VarStr);
            SubjectSym := Dictionary.NullSymbol;
         end if;
      else -- it's not an own variable so it can't be refined
         ErrorHandler.SemanticError (74,  ErrorHandler.NoReference, VarPos, VarStr);
         SubjectSym := Dictionary.NullSymbol;
      end if;
   end DoSubject;

   -----------------------------------------------------------------

   procedure DoConstituentList (ConstituentListNode : in STree.SyntaxNode;
                                Scope               : in Dictionary.Scopes;
                                SubjectSym          : in Dictionary.Symbol;
                                SubjectLoc          : in LexTokenManager.TokenPosition)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict           from *,
   --#                                        CommandLineData.Content,
   --#                                        ConstituentListNode,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        SubjectSym &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        ConstituentListNode,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        SubjectLoc,
   --#                                        SubjectSym;
   is
      It                  : STree.Iterator;
      ConstituentNode     : STree.SyntaxNode;
      Mode                : Modes;

      -----------------------------------------------------------------

      procedure DoConstituent (ConstituentNode : in STree.SyntaxNode;
                               Scope           : in Dictionary.Scopes;
                               SubjectSym      : in Dictionary.Symbol;
                               Mode            : in Modes;
                               SubjectLoc      : in LexTokenManager.TokenPosition)
      --# global in     CommandLineData.Content;
      --#        in     LexTokenManager.StringTable;
      --#        in     STree.Table;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives Dictionary.Dict           from *,
      --#                                        CommandLineData.Content,
      --#                                        ConstituentNode,
      --#                                        Mode,
      --#                                        Scope,
      --#                                        STree.Table,
      --#                                        SubjectSym &
      --#         ErrorHandler.ErrorContext from *,
      --#                                        CommandLineData.Content,
      --#                                        ConstituentNode,
      --#                                        Dictionary.Dict,
      --#                                        LexTokenManager.StringTable,
      --#                                        Mode,
      --#                                        Scope,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table,
      --#                                        SubjectSym &
      --#         SPARK_IO.FILE_SYS         from *,
      --#                                        CommandLineData.Content,
      --#                                        ConstituentNode,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.StringTable,
      --#                                        Mode,
      --#                                        Scope,
      --#                                        STree.Table,
      --#                                        SubjectLoc,
      --#                                        SubjectSym;
      is
         FirstNode  : STree.SyntaxNode;
         SecondNode : STree.SyntaxNode;

         ----------------------------------------------------------------

         procedure DoConcreteConstituent (VarNode    : in STree.SyntaxNode;
                                          Scope      : in Dictionary.Scopes;
                                          SubjectSym : in Dictionary.Symbol;
                                          SubjectLoc : in LexTokenManager.TokenPosition)
         --# global in     CommandLineData.Content;
         --#        in     LexTokenManager.StringTable;
         --#        in     Mode;
         --#        in     STree.Table;
         --#        in out Dictionary.Dict;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives Dictionary.Dict           from *,
         --#                                        Mode,
         --#                                        Scope,
         --#                                        STree.Table,
         --#                                        SubjectSym,
         --#                                        VarNode &
         --#         ErrorHandler.ErrorContext from *,
         --#                                        CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        LexTokenManager.StringTable,
         --#                                        Scope,
         --#                                        SPARK_IO.FILE_SYS,
         --#                                        STree.Table,
         --#                                        VarNode &
         --#         SPARK_IO.FILE_SYS         from *,
         --#                                        CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        LexTokenManager.StringTable,
         --#                                        Mode,
         --#                                        Scope,
         --#                                        STree.Table,
         --#                                        SubjectLoc,
         --#                                        SubjectSym,
         --#                                        VarNode;
         is
            VarPos           : LexTokenManager.TokenPosition;
            VarStr           : LexTokenManager.LexString;

         begin --DoConcreteConstituent
            VarStr := NodeLexString (VarNode);
            VarPos := NodePosition (VarNode);
            if Dictionary.IsDefined (VarStr,
                                     Scope,
                                     Dictionary.ProofContext)
            then
               ErrorHandler.SemanticError (10, ErrorHandler.NoReference,  VarPos, VarStr);
            elsif Mode.IsValid then
               Dictionary.AddConstituent (VarStr,
                                          SubjectSym,
                                          Mode.Value,
                                          Dictionary.Location'(SubjectLoc,
                                                               SubjectLoc),
                                          Dictionary.Location'(VarPos, VarPos));
            end if;
         end DoConcreteConstituent;

         -----------------------------------------------------------------

         procedure DoEmbeddedConstituent
            (PackDSimpleNameNode : in STree.SyntaxNode;
             VarIdentNode        : in STree.SyntaxNode;
             Scope               : in Dictionary.Scopes;
             SubjectSym          : in Dictionary.Symbol;
             SubjectLoc          : in LexTokenManager.TokenPosition)
         --# global in     CommandLineData.Content;
         --#        in     LexTokenManager.StringTable;
         --#        in     Mode;
         --#        in     STree.Table;
         --#        in out Dictionary.Dict;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives Dictionary.Dict           from *,
         --#                                        CommandLineData.Content,
         --#                                        Mode,
         --#                                        PackDSimpleNameNode,
         --#                                        Scope,
         --#                                        STree.Table,
         --#                                        SubjectSym,
         --#                                        VarIdentNode &
         --#         ErrorHandler.ErrorContext from *,
         --#                                        CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        LexTokenManager.StringTable,
         --#                                        PackDSimpleNameNode,
         --#                                        Scope,
         --#                                        SPARK_IO.FILE_SYS,
         --#                                        STree.Table,
         --#                                        VarIdentNode &
         --#         SPARK_IO.FILE_SYS         from *,
         --#                                        CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        LexTokenManager.StringTable,
         --#                                        Mode,
         --#                                        PackDSimpleNameNode,
         --#                                        Scope,
         --#                                        STree.Table,
         --#                                        SubjectLoc,
         --#                                        SubjectSym,
         --#                                        VarIdentNode;
         is
            PackSym          : Dictionary.Symbol;
            PackPos          : LexTokenManager.TokenPosition;
            PackStr          : LexTokenManager.LexString;
            PackNode         : STree.SyntaxNode;
            VarSym           : Dictionary.Symbol;
            VarPos           : LexTokenManager.TokenPosition;
            VarStr           : LexTokenManager.LexString;
            VarNode          : STree.SyntaxNode;
            OkToAdd          : Boolean;

         begin
            OkToAdd := False;

            PackNode := Child_Node (PackDSimpleNameNode);
            PackStr := NodeLexString (PackNode);
            PackPos := NodePosition (PackNode);

            VarNode := VarIdentNode;
            VarStr := NodeLexString (VarNode);
            VarPos := NodePosition (VarNode);

            PackSym := Dictionary.LookupItem (PackStr,
                                              Scope,
                                              Dictionary.ProofContext);
            if PackSym = Dictionary.NullSymbol then
               OkToAdd := True;
            elsif Dictionary.IsPackage (PackSym) then
               if Dictionary.GetContext (PackSym) = Dictionary.ProgramContext then
                  ErrorHandler.SemanticError (75,
                                              ErrorHandler.NoReference,
                                              PackPos,
                                              PackStr);
               else
                  VarSym := Dictionary.LookupSelectedItem (PackSym,
                                                           VarStr,
                                                           Scope,
                                                           Dictionary.ProofContext);
                  if VarSym = Dictionary.NullSymbol then
                     OkToAdd := True;
                  else
                     ErrorHandler.SemanticError (10,
                                                 ErrorHandler.NoReference,
                                                 VarPos,
                                                 VarStr);
                  end if;
               end if;
            else
               ErrorHandler.SemanticError (76,  ErrorHandler.NoReference, PackPos, PackStr);
            end if;

            if OkToAdd and Mode.IsValid then
               VarNode := VarIdentNode;
               VarStr := NodeLexString (VarNode);
               VarPos := NodePosition (VarNode);
               Dictionary.AddEmbeddedConstituent (PackStr,
                                                  VarStr,
                                                  SubjectSym,
                                                  Mode.Value,
                                                  Dictionary.Location'(SubjectLoc,
                                                                       SubjectLoc),
                                                  Dictionary.Location'(VarPos,
                                                                       VarPos));
            end if;
         end DoEmbeddedConstituent;

         ----------------------------------------------------------------

         procedure DoChildConstituent
            (PackDSimpleNameNode : in STree.SyntaxNode;
             VarIdentNode        : in STree.SyntaxNode;
             Scope               : in Dictionary.Scopes;
             SubjectSym          : in Dictionary.Symbol;
             SubjectLoc          : in LexTokenManager.TokenPosition)

         --# global in     CommandLineData.Content;
         --#        in     LexTokenManager.StringTable;
         --#        in     Mode;
         --#        in     STree.Table;
         --#        in out Dictionary.Dict;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives Dictionary.Dict           from *,
         --#                                        CommandLineData.Content,
         --#                                        Mode,
         --#                                        PackDSimpleNameNode,
         --#                                        Scope,
         --#                                        STree.Table,
         --#                                        SubjectSym,
         --#                                        VarIdentNode &
         --#         ErrorHandler.ErrorContext from *,
         --#                                        CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        LexTokenManager.StringTable,
         --#                                        Mode,
         --#                                        PackDSimpleNameNode,
         --#                                        Scope,
         --#                                        SPARK_IO.FILE_SYS,
         --#                                        STree.Table,
         --#                                        SubjectSym,
         --#                                        VarIdentNode &
         --#         SPARK_IO.FILE_SYS         from *,
         --#                                        CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        LexTokenManager.StringTable,
         --#                                        Mode,
         --#                                        PackDSimpleNameNode,
         --#                                        Scope,
         --#                                        STree.Table,
         --#                                        SubjectLoc,
         --#                                        SubjectSym,
         --#                                        VarIdentNode;
         is
            PackSym          : Dictionary.Symbol;
            PackPos          : LexTokenManager.TokenPosition;
            PackStr          : LexTokenManager.LexString;
            PackNode         : STree.SyntaxNode;
            VarSym           : Dictionary.Symbol;
            VarPos           : LexTokenManager.TokenPosition;
            VarStr           : LexTokenManager.LexString;
            Owner            : Dictionary.Symbol;
            ErrNum           : Natural;

         begin
            PackNode := LastChildOf (PackDSimpleNameNode);
            PackStr := NodeLexString (PackNode);
            PackPos := NodePosition (PackNode);

            PackSym := Dictionary.LookupItem (PackStr,
                                              Dictionary.GlobalScope,
                                              Dictionary.ProofContext);

            loop
               if PackSym = Dictionary.NullSymbol then
                  ErrorHandler.SemanticError (137,  ErrorHandler.NoReference,
                                              PackPos, PackStr);
                  exit;
               end if;

               if not Dictionary.IsPackage (PackSym) then
                  ErrorHandler.SemanticError (18,  ErrorHandler.NoReference,
                                              PackPos, PackStr);
                  PackSym := Dictionary.NullSymbol;
                  exit;
               end if;

               PackNode :=  Next_Sibling (ParentNode (PackNode));

               exit when PackNode = VarIdentNode;

               PackStr := NodeLexString (PackNode);
               PackPos := NodePosition (PackNode);
               PackSym := Dictionary.LookupSelectedItem (PackSym,
                                                         PackStr,
                                                         Dictionary.GlobalScope,
                                                         Dictionary.ProofContext);
            end loop;

            if PackSym /= Dictionary.NullSymbol then
               Owner := Dictionary.GetRegion (Scope);
               if Dictionary.GetPackageOwner (PackSym) /= Owner then
                  -- package not owned by subject package
                  ErrorHandler.SemanticError (619,  ErrorHandler.NoReference,
                                              PackPos, PackStr);
               else
                  VarStr := NodeLexString (VarIdentNode);
                  VarPos := NodePosition (VarIdentNode);
                  VarSym := Dictionary.LookupSelectedItem (PackSym,
                                                           VarStr,
                                                           Scope,
                                                           Dictionary.ProofContext);
                  if VarSym = Dictionary.NullSymbol then
                     -- constituent own variable not found
                     ErrorHandler.SemanticError (138,  ErrorHandler.NoReference,
                                                 VarPos, VarStr);
                  elsif not Dictionary.IsOwnVariable (VarSym) then
                     ErrorHandler.SemanticError (74,  ErrorHandler.NoReference,
                                                 VarPos, VarStr);
                  elsif Dictionary.IsRefinementConstituent (Owner, VarSym) then
                     -- already declared as constituent
                     ErrorHandler.SemanticError (10,  ErrorHandler.NoReference,
                                                 VarPos, VarStr);
                  else -- valid constituent
                     if Dictionary.OwnVariableIsInitialized (SubjectSym) /=
                        Dictionary.OwnVariableIsInitialized (VarSym)
                     then -- subject and constituent have different initialization
                        if Dictionary.OwnVariableIsInitialized (SubjectSym) then
                           -- if subject is initialized and constituent is not then
                           -- it is only an error if the constituent is not a stream var
                           if Dictionary.GetOwnVariableOrConstituentMode (VarSym) =
                              Dictionary.DefaultMode then
                              ErrorHandler.SemanticError (622,  ErrorHandler.NoReference,
                                                          VarPos, VarStr);
                           end if;
                        else
                           ErrorHandler.SemanticError (623,  ErrorHandler.NoReference,
                                                       VarPos, VarStr);
                        end if;
                     end if;

                     -- constituent must have same mode as own variable it being
                     -- associated with
                     if Mode.Value /= Dictionary.GetOwnVariableMode (VarSym) then
                        if Dictionary.GetOwnVariableMode (VarSym) = Dictionary.InMode then
                           ErrNum := 705; -- must be IN
                        elsif Dictionary.GetOwnVariableMode (VarSym) = Dictionary.OutMode then
                           ErrNum := 706; -- must be OUT
                        else
                           ErrNum := 707; -- no mode permitted
                        end if;
                        ErrorHandler.SemanticError (ErrNum,
                                                    ErrorHandler.NoReference,
                                                    VarPos,
                                                    VarStr);
                     end if;
                     -- end of mode checks

                     if Mode.IsValid then
                        Dictionary.AddChildConstituent
                           (VarSym,
                            SubjectSym,
                            Mode.Value,
                            Dictionary.Location'(SubjectLoc, SubjectLoc),
                            Dictionary.Location'(VarPos, VarPos));
                     end if;
                  end if;
               end if;
            end if;
         end DoChildConstituent;

         ----------------------------------------------------------------

      begin --DoConstituent
         FirstNode := Child_Node (Child_Node (ConstituentNode));
         SecondNode := Next_Sibling (FirstNode);
         if SecondNode = STree.NullNode then
            DoConcreteConstituent (FirstNode,
                                   Scope,
                                   SubjectSym,
                                   SubjectLoc);
         elsif SyntaxNodeType (Child_Node (FirstNode))
            = SPSymbols.identifier
         then
            DoEmbeddedConstituent (FirstNode,
                                   SecondNode,
                                   Scope,
                                   SubjectSym,
                                   SubjectLoc);
         elsif CommandLineData.IsSpark95 then
            DoChildConstituent (FirstNode,
                                SecondNode,
                                Scope,
                                SubjectSym,
                                SubjectLoc);
         else
            ErrorHandler.SemanticError (156, -- entire var expected
                                        ErrorHandler.NoReference,
                                        NodePosition (ConstituentNode),
                                        LexTokenManager.NullString);
         end if;
      end DoConstituent;

      -----------------------------------------------------------------

      procedure CheckModifier (Node        : in     STree.SyntaxNode;
                               SubjectSym  : in     Dictionary.Symbol;
                               Mode        :    out Modes)
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     LexTokenManager.StringTable;
         --#        in     STree.Table;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out SPARK_IO.FILE_SYS;
         --# derives ErrorHandler.ErrorContext,
         --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        LexTokenManager.StringTable,
         --#                                        Node,
         --#                                        SPARK_IO.FILE_SYS,
         --#                                        STree.Table,
         --#                                        SubjectSym &
         --#         Mode                      from Dictionary.Dict,
         --#                                        Node,
         --#                                        STree.Table,
         --#                                        SubjectSym;
      is
         ModifierNode,
         ErrNode,
         ModeNode    : STree.SyntaxNode;
         SubjectMode : Dictionary.Modes;

      begin --CheckMode
         -- NodeType (Node)is  own_variable_modifier

         SubjectMode := Dictionary.GetOwnVariableMode (SubjectSym);

         Mode := Modes'(Dictionary.DefaultMode, False); -- default, invalid answers

         ModifierNode := Child_Node (Node);
         ErrNode := Next_Sibling (Node); -- can't point at modes

         case SyntaxNodeType (ModifierNode) is
            when SPSymbols.mode =>
               ModeNode := Child_Node (ModifierNode);
               if ModeNode /= STree.NullNode then -- mode is present
                  case SyntaxNodeType (ModeNode) is
                     when SPSymbols.in_mode =>
                        -- IN is legal if subject is IN or subject if Default
                        if SubjectMode = Dictionary.DefaultMode or else
                          SubjectMode = Dictionary.InMode
                        then
                           Mode := Modes'(Dictionary.InMode, True);
                        else
                           ErrorHandler.SemanticError (701,
                                                       ErrorHandler.NoReference,
                                                       NodePosition (ErrNode),
                                                       Dictionary.GetSimpleName (SubjectSym));
                        end if;

                     when SPSymbols.outmode =>
                        -- OUT is legal if subject is OUT or subject if Default
                        if SubjectMode = Dictionary.DefaultMode or else
                          SubjectMode = Dictionary.OutMode
                        then
                           Mode := Modes'(Dictionary.OutMode, True);
                        else
                           ErrorHandler.SemanticError (701,
                                                       ErrorHandler.NoReference,
                                                       NodePosition (ErrNode),
                                                       Dictionary.GetSimpleName (SubjectSym));
                        end if;

                     when SPSymbols.inout_mode =>
                        ErrorHandler.SemanticError (700,
                                                    ErrorHandler.NoReference,
                                                    NodePosition (ErrNode),
                                                    LexTokenManager.NullString);

                     when others =>
                        null; -- can't happen
                  end case;
               else -- no mode supplied so wff for default mode case
                     -- Default is legal only if subject is Default
                  if SubjectMode = Dictionary.DefaultMode then
                     Mode := Modes'(Dictionary.DefaultMode, True);
                  else
                     ErrorHandler.SemanticError (701,
                                                 ErrorHandler.NoReference,
                                                 NodePosition (ErrNode),
                                                 Dictionary.GetSimpleName (SubjectSym));
                  end if;
               end if;
            when SPSymbols.protected_modifier =>
               -- Refinement constituents cannot be protected.
               ErrorHandler.SemanticError (859,
                                           ErrorHandler.NoReference,
                                           NodePosition (ErrNode),
                                           Dictionary.GetSimpleName (SubjectSym));

            when others =>
               null;
         end case;
      end CheckModifier;

      ---------------------------------------------

   begin --DoConstituentList
      It := FindFirstNode (NodeKind    => SPSymbols.own_variable_modifier,
                           FromRoot    => ConstituentListNode,
                           InDirection => STree.Down);

      while not STree.IsNull (It) loop
         ConstituentNode := GetNode (It);
         CheckModifier (ConstituentNode,
                        SubjectSym,
                        --to get
                        Mode);

         DoConstituent (Next_Sibling (ConstituentNode),
                        Scope,
                        SubjectSym,
                        Mode,
                        SubjectLoc);
         It := STree.NextNode (It);
      end loop;
   end DoConstituentList;

   -----------------------------------------------------------------

begin -- wf_refinement_clause

   SubjectNode     := Child_Node (Node);
   SubjectLoc      := NodePosition (SubjectNode);
   ConstituentNode := Next_Sibling (SubjectNode);

   if ConstituentNode = STree.NullNode then
      -- There is _no_ constituent node at all.  This used to be
      -- rejected by the gammar alone, but this resulted in a poor
      -- error message, and it's a common "beginner" mistake.
      -- The grammar now accepts this, so we can detect and reject it
      -- here with a much better message.
      ErrorHandler.SemanticError (115,  ErrorHandler.NoReference,
                                  SubjectLoc, LexTokenManager.NullString);
   else
      DoSubject (SubjectNode,
                 Scope,
                  --to get
                 SubjectSym);
      if SubjectSym /= Dictionary.NullSymbol then
         DoConstituentList (ConstituentNode,
                            Scope,
                            SubjectSym,
                            SubjectLoc);
      end if;
   end if;

end wf_clause;
