-- $Id: sem-compunit-wf_named_argument_association.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.CompUnit)

procedure wf_named_argument_association (Node         : in     STree.SyntaxNode;
                                         Scope        : in     Dictionary.Scopes;
                                         EStack       : in out ExpStack.ExpStackType;
                                         HeapParam    : in out Lists.ListHeap;
                                         IsAnnotation : in     Boolean)
is
   SIMPLE_LOOKUP : constant Annotation_Symbol_Table :=
      Annotation_Symbol_Table'(False => SPSymbols.simple_name,
                               True  => SPSymbols.annotation_simple_name);

   ExpResult,
   FunInfo             : ExpRecord;
   FunSym,
   ParamSym            : Dictionary.Symbol;
   IdentNode           : STree.SyntaxNode;
   IdentStr            : LexTokenManager.LexString;
   AlreadyPresent,
   NameIsParameterName : Boolean;
   UnusedValue         : Maths.Value;
   ErrorFound          : Boolean := False;

   --------------------------------------------------------------

   function ExpressionLocation (Node : STree.SyntaxNode)
                               return STree.SyntaxNode
   --# global in IsAnnotation;
   --#        in STree.Table;
   is
      ExpLoc : STree.SyntaxNode;

   begin
      if SyntaxNodeType (Child_Node (Node)) = SIMPLE_LOOKUP (IsAnnotation) then
         ExpLoc := Next_Sibling (Child_Node (Node));
      else
         ExpLoc := Next_Sibling (Next_Sibling (Child_Node (Node)));
      end if;
      return ExpLoc;
   end ExpressionLocation;

   --------------------------------------------------------------

   function FindIdentifier (Node : STree.SyntaxNode)
                           return STree.SyntaxNode
   --# global in IsAnnotation;
   --#        in STree.Table;
   is
      IdentNode : STree.SyntaxNode;
   begin
      if SyntaxNodeType (Child_Node (Node)) = SIMPLE_LOOKUP (IsAnnotation) then
         IdentNode := Child_Node (Child_Node (Node));
      else
         IdentNode := Child_Node (Next_Sibling (Child_Node (Node)));
      end if;
      return IdentNode;
   end FindIdentifier;

   --------------------------------------------------------------

   procedure RangeCheck (ARange : in Boolean;
                         Node   : in STree.SyntaxNode)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     IsAnnotation;
      --#        in     LexTokenManager.StringTable;
      --#        in     STree.Table;
      --#        in out ErrorFound;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorFound                from *,
      --#                                        ARange &
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from ARange,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        IsAnnotation,
      --#                                        LexTokenManager.StringTable,
      --#                                        Node,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table;
   is
   begin
      if ARange then
         ErrorFound := True;
         ErrorHandler.SemanticError (341,
                                     ErrorHandler.NoReference,
                                     NodePosition (ExpressionLocation (Node)),
                                     LexTokenManager.NullString);
      end if;
   end RangeCheck;

   --------------------------------------------------------------

begin --wf_named_argument_association
   ExpStack.Pop (ExpResult, EStack);
   ExpStack.Pop (FunInfo, EStack);
   FunSym := FunInfo.OtherSymbol;

   FindNamedArgumentAssociationParameter
     (Node                => Node,
      SubprogSym          => FunSym,
      IsAnnotation        => IsAnnotation,
      NameIsParameterName => NameIsParameterName,
      ParamSym            => ParamSym);

   IdentNode := FindIdentifier (Node);
   IdentStr := NodeLexString (IdentNode);

   if NameIsParameterName then
      -- Seed syntax tree with expected type for run-time check;
      -- but, don't do this for instantiation of unchecked_conversion
      -- because we don't want any RTCs for association of those parameters
      -- (provided the function parameter subtype and actual subtype match)
      if not (Dictionary.IsAnUncheckedConversion (FunSym) and then
                ExpResult.TypeSymbol = Dictionary.GetType (ParamSym)) then
         STree.AddNodeSymbol (Node, Dictionary.GetType (ParamSym));
      end if;

      -- There is a special case involving functions an stream variables.  We allow a stream
      -- variable to be a parameter to an Unchecked_Conversion but need to ensure that
      -- the function inherits the restrictions associated with referencing a stream
      -- (e.g. cannot be used in gernal expression).  We can do this here by checking
      -- the StreamSymbol of the parameter expression (there will only be one if we are
      -- talking about an unchecked conversion) and if it is non-null then setting the
      -- stream symbol of the function result record (now an object) to the function symbol.
      -- Note that this clause will only be executed for an unchecked conversion because
      -- a parameter which is a stream would hav ebeen rejected at wf_primary in all other
      -- cases
      if ExpResult.StreamSymbol /= Dictionary.NullSymbol then
         FunInfo.StreamSymbol := FunSym;
      end if;

      AddName (IdentStr,
               FunInfo.ParamList,
               HeapParam,
               AlreadyPresent);
      if AlreadyPresent then
         ErrorFound := True;
         ErrorHandler.SemanticError (4,
                                     ErrorHandler.NoReference,
                                     NodePosition (IdentNode),
                                     IdentStr);
      else --not already present so do further checks

         RangeCheck (ExpResult.IsARange, Node);

         -- function is deemed constant if it is predefined and all its parameters
         -- are constant.
         FunInfo.IsConstant := FunInfo.IsConstant and ExpResult.IsConstant;

         if (FunInfo.TaggedParameterSymbol = ExpResult.TypeSymbol or else
             (FunInfo.TaggedParameterSymbol = Dictionary.NullSymbol and then
              Dictionary.CompatibleTypes (Scope,
                                          Dictionary.GetType (ParamSym),
                                          ExpResult.TypeSymbol)) or else
             (not Dictionary.IsAnExtensionOf (ExpResult.TypeSymbol, FunInfo.TaggedParameterSymbol) and then
              Dictionary.CompatibleTypes (Scope,
                                          Dictionary.GetType (ParamSym),
                                          ExpResult.TypeSymbol)))
         then
            TaggedActualMustBeObjectCheck (NodePos         => NodePosition (ExpressionLocation (Node)),
                                           FormalType      => Dictionary.GetType (ParamSym),
                                           ActualType      => ExpResult.TypeSymbol,
                                           ControllingType => Dictionary.GetSubprogramControllingType (FunSym),
                                           IsAVariable     => ExpResult.IsAVariable,
                                           IsAConstant     => ExpResult.IsConstant,
                                           ErrorFound      => ErrorFound);
            -- Following call will deal with scalar value constraint checking
            --# accept Flow, 10, UnusedValue, "Expected ineffective assignment";
            ConstraintCheck (ExpResult.Value,
                             UnusedValue,
                             IsAnnotation,
                             Dictionary.GetType (ParamSym),
                             NodePosition (ExpressionLocation (Node)));
            --# end accept;
            -- Check array bounds etc.
            if Dictionary.TypeIsArray (Dictionary.GetType (ParamSym)) and then
              not Dictionary.IsUnconstrainedArrayType (Dictionary.GetType (ParamSym)) then
               -- Formal is a constrained subtype of an unconstrained array
               if Dictionary.IsUnconstrainedArrayType (ExpResult.TypeSymbol) then

                  -- Actual is unconstrained.  In SPARK95, this is OK if
                  -- the actual is a static String expression, but illegal
                  -- otherwise.
                  if CommandLineData.IsSpark95 and then
                    Dictionary.IsPredefinedStringType (ExpResult.TypeSymbol) then
                     -- Formal must be a constrained String subtype, so we need
                     -- to check the upper bound of the actual against the expected
                     -- upper bound of the formal.
                     if ExpResult.RangeRHS = Maths.NoValue then
                        -- Actual is not static, so must be illegal
                        ErrorHandler.SemanticError (39,
                                                    ErrorHandler.NoReference,
                                                    NodePosition (ExpressionLocation (Node)),
                                                    LexTokenManager.NullString);
                     else
                        -- Actual is static, so check upper-bound against that expected
                        if ExpResult.RangeRHS /= Maths.ValueRep
                          (Dictionary.GetScalarAttributeValue (False,
                                                               LexTokenManager.LastToken,
                                                               Dictionary.GetType (ParamSym))) then
                           ErrorHandler.SemanticError (418,
                                                       ErrorHandler.NoReference,
                                                       NodePosition (ExpressionLocation (Node)),
                                                       LexTokenManager.NullString);

                        end if;
                     end if;
                  else
                     -- SPARK83 or not a String type, so illegal
                     ErrorHandler.SemanticError (39,
                                                 ErrorHandler.NoReference,
                                                 NodePosition (ExpressionLocation (Node)),
                                                 LexTokenManager.NullString);
                  end if;

               elsif Illegal_Unconstrained (ExpResult.TypeSymbol,
                                            Dictionary.GetType (ParamSym)) then
                  -- Although both formal and actual are constrained their bounds don't match
                  ErrorHandler.SemanticError (418,
                                              ErrorHandler.NoReference,
                                              NodePosition (ExpressionLocation (Node)),
                                              LexTokenManager.NullString);
               end if;
            end if;

            -- To help the VCG with generating checks involving unconstrained formal parameters, we
            -- seed the syntax tree with a constraining type mark.  The positional_argument_association
            -- node is already used for RTC purposes, so we seed the expression node instead.
            if not IsAnnotation then
               PlantConstrainingType (ExpResult.TypeSymbol,
                                      ExpResult.RangeRHS,
                                      STree.ExpressionFromNamedArgumentAssociation (Node));
            end if;
         else
            ErrorFound := True;
            ErrorHandler.SemanticError (38,
                                        ErrorHandler.NoReference,
                                        NodePosition (ExpressionLocation (Node)),
                                        LexTokenManager.NullString);
         end if;
      end if;
   else
      ErrorFound := True;
      ErrorHandler.SemanticErrorLex1Sym1 (2,
                                          ErrorHandler.NoReference,
                                          NodePosition (IdentNode),
                                          IdentStr,
                                          FunSym,
                                          Scope);

   end if;
   FunInfo.ErrorsInExpression := ErrorFound or
      FunInfo.ErrorsInExpression or
      ExpResult.ErrorsInExpression;
   ExpStack.Push (FunInfo, EStack);

   --# accept Flow, 33, UnusedValue, "Expected to be neither referenced nor exported";
end wf_named_argument_association;
