-- $Id: sem-compunit-wf_justification_statement.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with Debug;
separate (Sem.Compunit)
procedure wf_justification_statement (Node     : in STree.SyntaxNode;
                                      Scope    : in Dictionary.Scopes)
is
   procedure wf_start_justification (Node     : in STree.SyntaxNode;
                                     Scope    : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives Dictionary.Dict,
   --#         GlobalComponentData,
   --#         Statistics.TableUsage,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        GlobalComponentData,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table,
   --#                                        TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        GlobalComponentData,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheHeap;
   is
      It : STree.Iterator;

      procedure wf_justification_clause (StartLine : in LexTokenManager.LineNumbers;
                                         Node      : in STree.SyntaxNode;
                                         Scope     : in Dictionary.Scopes)
      --# global in     CommandLineData.Content;
      --#        in     LexTokenManager.StringTable;
      --#        in     STree.Table;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out GlobalComponentData;
      --#        in out SPARK_IO.File_Sys;
      --#        in out Statistics.TableUsage;
      --#        in out TheHeap;
      --# derives Dictionary.Dict,
      --#         GlobalComponentData,
      --#         Statistics.TableUsage,
      --#         TheHeap                   from *,
      --#                                        CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        GlobalComponentData,
      --#                                        Node,
      --#                                        Scope,
      --#                                        STree.Table,
      --#                                        TheHeap &
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.File_Sys         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        GlobalComponentData,
      --#                                        LexTokenManager.StringTable,
      --#                                        Node,
      --#                                        Scope,
      --#                                        SPARK_IO.File_Sys,
      --#                                        StartLine,
      --#                                        STree.Table,
      --#                                        TheHeap;
      is
         ErrorFound  : Boolean;
         CurrentNode : STree.SyntaxNode;
         ErrNumNode  : STree.SyntaxNode;
         Valid       : Boolean;
         Kind        : ErrorHandler.JustificationKinds;
         Val         : Maths.Value;
         MathsValid  : Maths.ErrorCode;
         ErrNum      : Natural;
         Explanation : LexTokenManager.LexString;
         Identifiers : ErrorHandler.JustificationIdentifiers;
         MaximumJustificationsReached : Boolean;

         procedure CheckKind (LexString : in     LexTokenManager.LexString;
                              Kind      :    out ErrorHandler.JustificationKinds;
                              Valid     :    out Boolean)
         --# global in LexTokenManager.StringTable;
         --# derives Kind,
         --#         Valid from LexString,
         --#                    LexTokenManager.StringTable;
         is
            ExString  : EStrings.T;
            Flow   : constant EStrings.T :=
              EStrings.T'(Length => 12,
                                              Content => EStrings.Contents'
                                                ('F', 'L', 'O', 'W', '_', 'M', 'E', 'S', 'S', 'A', 'G', 'E',
                                                 others => ' '));
            Warn   : constant EStrings.T :=
              EStrings.T'(Length => 15,
                                              Content => EStrings.Contents'
                                                ('W', 'A', 'R', 'N', 'I', 'N', 'G', '_', 'M', 'E', 'S', 'S', 'A', 'G', 'E',
                                                 others => ' '));
            Found    : Boolean;
            StartPos : EStrings.Positions;
         begin
            -- The kind of message (Flow or Warning) is in the form of an identifier and therefore is extracted
            -- from the syntax tree as a LexString.  We first convert it to an ExaminerString
            LexTokenManager.LexStringToString (LexString,
                                                --to get
                                               ExString);
            -- Then we see if it a unique subset of either "Flow_Message" or "Warning_Message"
            -- Ignore case
            ExString := EStrings.UpperCase (ExString);
            -- Try "flow" first
            StringUtilities.FindExaminerSubString (Flow,
                                                   ExString,
                                                   -- to get
                                                   Found,
                                                   StartPos);
            -- To get a match we need Found and StartPos = 1
            if Found and then StartPos = 1 then
               Kind := ErrorHandler.FlowMessage;
               Valid  := True;

            else
               -- Try "warn"
               StringUtilities.FindExaminerSubString (Warn,
                                                      ExString,
                                                      -- to get
                                                      Found,
                                                      StartPos);
               -- To get a match we need Found and StartPos = 1
               if Found and then StartPos = 1 then
                  Kind := ErrorHandler.WarningMessage;
                  Valid  := True;
               else
                  Kind := ErrorHandler.FlowMessage; -- not used, for DF purposes only
                  Valid  := False;
               end if;
            end if;
         end CheckKind;

         function IsDisallowedWarning (Kind   : ErrorHandler.JustificationKinds;
                                       ErrNum : Natural) return Boolean
         is
         begin
            -- Initially only prohibit warnings generated by the justification system itself.
            -- Extend here as necessary.

            return Kind = ErrorHandler.WarningMessage and then
              (ErrNum = 120 or else ErrNum = 121 or else ErrNum = 122);
         end IsDisallowedWarning;

         procedure CheckIdentifiers (OptNode     : in     STree.SyntaxNode;
                                     Identifiers :    out ErrorHandler.JustificationIdentifiers;
                                     Valid       :    out Boolean)
         --# global in     CommandLineData.Content;
         --#        in     ErrNum;
         --#        in     Kind;
         --#        in     LexTokenManager.StringTable;
         --#        in     Scope;
         --#        in     STree.Table;
         --#        in out Dictionary.Dict;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out GlobalComponentData;
         --#        in out SPARK_IO.File_Sys;
         --#        in out Statistics.TableUsage;
         --#        in out TheHeap;
         --# derives Dictionary.Dict,
         --#         GlobalComponentData,
         --#         Statistics.TableUsage,
         --#         TheHeap                   from *,
         --#                                        CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        GlobalComponentData,
         --#                                        OptNode,
         --#                                        Scope,
         --#                                        STree.Table,
         --#                                        TheHeap &
         --#         ErrorHandler.ErrorContext,
         --#         SPARK_IO.File_Sys         from CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        ErrNum,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        GlobalComponentData,
         --#                                        Kind,
         --#                                        LexTokenManager.StringTable,
         --#                                        OptNode,
         --#                                        Scope,
         --#                                        SPARK_IO.File_Sys,
         --#                                        STree.Table,
         --#                                        TheHeap &
         --#         Identifiers               from CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        GlobalComponentData,
         --#                                        OptNode,
         --#                                        Scope,
         --#                                        STree.Table,
         --#                                        TheHeap &
         --#         Valid                     from CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        ErrNum,
         --#                                        GlobalComponentData,
         --#                                        Kind,
         --#                                        OptNode,
         --#                                        Scope,
         --#                                        STree.Table,
         --#                                        TheHeap;
         is
            It              : STree.Iterator;
            IdentifierCount : Natural := 0;
            ValidSimpleName : Boolean;
            IdStr           : LexTokenManager.LexString;
            Sym             : Dictionary.Symbol;
            NameError       : Natural;

            procedure ProcessDottedSimpleName (DSNnode : in     STree.SyntaxNode;
                                               Str     :    out LexTokenManager.LexString;
                                               Sym     :    out Dictionary.Symbol;
                                               Valid   :    out Boolean)
            --# global in     CommandLineData.Content;
            --#        in     LexTokenManager.StringTable;
            --#        in     Scope;
            --#        in     STree.Table;
            --#        in out Dictionary.Dict;
            --#        in out ErrorHandler.ErrorContext;
            --#        in out GlobalComponentData;
            --#        in out SPARK_IO.File_Sys;
            --#        in out Statistics.TableUsage;
            --#        in out TheHeap;
            --# derives Dictionary.Dict,
            --#         GlobalComponentData,
            --#         Str,
            --#         Sym,
            --#         TheHeap,
            --#         Valid                     from CommandLineData.Content,
            --#                                        Dictionary.Dict,
            --#                                        DSNnode,
            --#                                        GlobalComponentData,
            --#                                        Scope,
            --#                                        STree.Table,
            --#                                        TheHeap &
            --#         ErrorHandler.ErrorContext,
            --#         SPARK_IO.File_Sys         from CommandLineData.Content,
            --#                                        Dictionary.Dict,
            --#                                        DSNnode,
            --#                                        ErrorHandler.ErrorContext,
            --#                                        GlobalComponentData,
            --#                                        LexTokenManager.StringTable,
            --#                                        Scope,
            --#                                        SPARK_IO.File_Sys,
            --#                                        STree.Table,
            --#                                        TheHeap &
            --#         Statistics.TableUsage     from *,
            --#                                        CommandLineData.Content,
            --#                                        Dictionary.Dict,
            --#                                        DSNnode,
            --#                                        GlobalComponentData,
            --#                                        Scope,
            --#                                        STree.Table,
            --#                                        TheHeap;
            is
               It         : STree.Iterator;
               Dotted     : Boolean;
               IdNode     : STree.SyntaxNode;
               PIdStr,
               IdStr      : LexTokenManager.LexString;
               LocalSym,
               SymSoFar : Dictionary.Symbol;

               function SelectorAllowedFor (Sym : Dictionary.Symbol) return Boolean
               --# global in Dictionary.Dict;
               is
               begin
                  return Dictionary.IsPackage (Sym) or else
                    (Dictionary.IsTypeMark (Sym) and then
                       (Dictionary.TypeIsRecord (Sym) or else
                          Dictionary.TypeIsProtected (Sym))) or else
                    Dictionary.IsRecordComponent (Sym) or else
                    (Dictionary.IsObject (Sym) and then Dictionary.TypeIsRecord (Dictionary.GetType (Sym))) or else
                    (Dictionary.IsFunction (Sym) and then Dictionary.TypeIsRecord (Dictionary.GetType (Sym))) or else
                    (Dictionary.IsObject (Sym) and then Dictionary.IsProtectedType (Dictionary.GetType (Sym)));
               end SelectorAllowedFor;

            begin
               -- local grammar
               --
               -- dotted_simple_name         or     dotted_simple_name
               --         |                                 |
               --     identifier                    dotted_simple_name --- identifier
               --                                           |
               --                                       identifier

               Valid := True; -- default
               -- See whether it is a simple identifier or not.  If it is we return a LexString and a Symbol
               -- otherwise just a Symbol.  Dotted gets set True if we loop through >1 identifiers
               Dotted := False;

               -- Loop through identifiers.  Loop exits prematurely for simple identifier case
               It := FindFirstNode (NodeKind    => SPSymbols.identifier,
                                    FromRoot    => DSNnode,
                                    InDirection => STree.Down);

               IdNode := GetNode (It);
               IdStr := NodeLexString (IdNode);
               PIdStr := LexTokenManager.NullString;
               -- Debug.PrintLexStr ("String sought in CheckIdentifiers is ", IdStr);
               -- Note that the lookup uses ProofContext because we may be trying to justify a flow error
               -- or warning involving an identifier that is not visible in ProgramContext (eg an abstract
               -- own variable).
               LocalSym := Dictionary.LookupItem (IdStr,
                                                  Scope,
                                                  Dictionary.ProofContext);
               loop
                  -- Debug.PrintSym ("LocalSym in loop in CheckIdentifiers is ", LocalSym);
                  -- any time we fail to find something it is an error failure
                  if LocalSym = Dictionary.NullSymbol then
                     ErrorHandler.SemanticError2 (1,
                                                  ErrorHandler.NoReference,
                                                  NodePosition (IdNode),
                                                  IdStr,
                                                  PIdStr);
                     Valid := False;
                     exit;
                  end if;

                  -- set up next iteration
                  It := STree.NextNode (It);

                  exit when STree.IsNull (It);

                  -- If we get to here then there is more than one identifier

                  -- If there is more than one identifier and we are processing a record object or
                  -- record subcomponent, then there is
                  -- an extra step required:  we do not add symbols for all the components of records to the
                  -- dictionary all the time, but only where they are needed.  Therefore if we try and look
                  -- up R.F here (where R is record object) then the look up will fail because there is no
                  -- subcomponent symbol for R.F.  Therefore we must add the symbols now so that the
                  -- LookUpSelectedItem below will succeed.
                  if Dictionary.IsVariableOrSubcomponent (LocalSym) and then
                    Dictionary.TypeIsRecord (Dictionary.GetType (LocalSym)) then
                     -- Debug.PrintMsg ("Adding subcomponents in wf_justification_statement", True);
                     AddRecordSubComponents (LocalSym,
                                             Dictionary.GetType (LocalSym),
                                             GlobalComponentData);
                  end if;
                  -- end of sub component addition

                  -- Because there is more than identifier we save some context for next time round the loop
                  Dotted := True;
                  PIdStr := IdStr;
                  IdNode := GetNode (It);
                  IdStr := NodeLexString (IdNode);
                  SymSoFar := LocalSym;  -- needed for trapping P.P.P.P.X case later on

                  -- At this point we have a prefix in LocalSym and we are about to process
                  -- a selector.  LocalSym had better be the kind of thing that can have a
                  -- selector.
                  if not SelectorAllowedFor (LocalSym) then
                     ErrorHandler.SemanticErrorSym (9,
                                                    ErrorHandler.NoReference,
                                                    NodePosition (IdNode),
                                                    LocalSym,
                                                    Scope);

                     LocalSym := Dictionary.NullSymbol;
                     exit;
                  end if;

                  -- Debug.PrintLexStr ("String sought in loop in CheckIdentifiers is ", IdStr);
                  -- Debug.PrintSym ("             looking it up in ", LocalSym);
                  -- Note that the lookup uses ProofContext because we may be trying to justify a flow error
                  -- or warning involving an identifier that is not visible in ProgramContext (eg an abstract
                  -- own variable).
                  LocalSym := Dictionary.LookupSelectedItem (LocalSym,
                                                             IdStr,
                                                             Scope,
                                                             Dictionary.ProofContext);
                  -- check to see if we are getting the same symbol over and again
                  if LocalSym = SymSoFar then            -- P.P.P.P.X case
                     LocalSym := Dictionary.NullSymbol;  -- to cause "Not visible" error at top of loop
                  end if;

               end loop;

               -- return results
               if Dotted then
                  Str := LexTokenManager.NullString;
               else
                  Str := IdStr;
               end if;
               Sym := LocalSym;
            end ProcessDottedSimpleName;

            -- This function checks the number of names in the accept
            -- annotation against the error number. It returns the error message
            -- to report, or 0 if the clause is semantically correct.
            --
            -- Unfortunately Flow Error 10 can have either 0 or 1 identifiers.
            --
            -- Flow Errors 50 and 602 can have 1 or 2 identifiers, depending
            -- on whether the enclosing program unit is a function (1 identifier needed)
            -- or not (2 identifiers needed for procedures or task bodies).
            function JustificationNameLengthError (EnclosingRegionIsAFunction : in Boolean) return Natural
            --# global in ErrNum;
            --#        in IdentifierCount;
            --#        in Kind;
            is
               RetVal : Natural := 0;
            begin
               case Kind is
                  when ErrorHandler.FlowMessage =>
                     case ErrNum is
                        -- These flow errors require exactly zero names to be justified
                        when  22 |  40 |  41 =>
                           if IdentifierCount /= 0 then
                              RetVal := 124;
                           end if;
                        -- These flow errors require exactly two names to be justified
                        when   3 |   4 | 601 | 605 | 606 =>
                           if IdentifierCount /= 2 then
                              RetVal := 126;
                           end if;
                        -- Flow Error 10 (ineffective expression or statement) can require
                        -- either zero or one name
                        when  10 =>
                           if IdentifierCount = 0 or IdentifierCount = 1 then
                              RetVal := 0;
                           else
                              RetVal := 127;
                           end if;

                        -- Flow errors 50 and 602 can require one or two names,
                        -- depending on EnclosingRegionIsAFunction
                        when  50 | 602 =>
                           if EnclosingRegionIsAFunction then
                              -- function - 1 identifier needed
                              if IdentifierCount = 1 then
                                 RetVal := 0;
                              else
                                 RetVal := 125;
                              end if;
                           else
                              -- procedure or task body - 2 identifiers needed
                              if IdentifierCount = 2 then
                                 RetVal := 0;
                              else
                                 RetVal := 126;
                              end if;
                           end if;

                        -- All other flow errors require exactly one name
                        when others =>
                           if IdentifierCount /= 1 then
                              RetVal := 125;
                           end if;
                     end case;
                  when ErrorHandler.WarningMessage =>
                     case ErrNum is
                        -- The following warnings require exactly 1 name to be justified
                        when   1 |   5 |   9 |  10 |  12 |  13 | 169 | 311 | 312 | 313 | 314 |
                          350 | 351 | 391 | 392 | 393 | 394 | 395 | 396 | 397 | 398 | 400 | 403 |
                          410 | 411 | 412 | 413 =>
                           if IdentifierCount /= 1 then
                              RetVal := 125;
                           end if;

                        -- All other warnings require exactly zero names
                        when others =>
                           if IdentifierCount /= 0 then
                              RetVal := 124;
                           end if;
                     end case;
               end case;
               return RetVal;
            end JustificationNameLengthError;


         begin -- CheckIdentifiers

            -- local grammar
            -- justification_opt        or       justification_opt
            --       |                                   |
            --    nothing                        dotted_simple_name_list
            --                                           |
            --                                   dotted_simple_name_list---dotted_simple_name
            --                                                 |
            --                                   dotted_simple_name
            --
            -- Rules:
            --   (1) Between 0 and ErrorHandler.MaxJustificationIdentifierLength identifiers found
            --   (2) Each identifier must be declared and visible in Scope
            --   (3) Identifiers (I) is populated with each legal identifier thus:
            --        (a) If the identifier has no dots in it, then we store the LexString AND the looked-up symbol
            --        (b) If it has dots then we store a null lex string and the looked-up symbol
            --            (this complexity is because we don't know whether warnings will be passed to the errohandler
            --             using, for example, SemanticWarning or SemanticWarningSym so we need to match either)

            -- Establish default result
            Identifiers := ErrorHandler. NullJustificationIdentifiers;
            Valid := True;

            -- Iterate through dotted_simple_name nodes
            It := FindFirstNode (NodeKind    => SPSymbols.dotted_simple_name,
                                 FromRoot    => OptNode,
                                 InDirection => STree.Down);
            if not STree.IsNull (It) then
               loop
                  IdentifierCount := IdentifierCount + 1;

                  -- We have a single dotted simple name to process.  If it is OK then it will go in the
                  -- identifier list at index position IdentifierCount
                  ProcessDottedSimpleName (GetNode (It),
                                             -- to get
                                           IdStr,
                                           Sym,
                                           ValidSimpleName);
                  if ValidSimpleName then
                     if IdentifierCount in ErrorHandler.JustificationIdentifierIndex then
                        Identifiers (IdentifierCount) := ErrorHandler.JustificationIdentifier'(StringForm => IdStr,
                                                                                               SymbolForm => Sym);
                     end if;
                  else
                     Valid := False; -- don't add clause at all if any part is malformed
                  end if;

                  if STree.IsNull (STree.NextNode (It)) then
                     exit;
                  end if;

                  It := STree.NextNode (It);
               end loop;
            end if;

            NameError := JustificationNameLengthError
              (EnclosingRegionIsAFunction => Dictionary.IsFunction (Dictionary.GetRegion (Scope)));

            if NameError /= 0 then
               ErrorHandler.SemanticError (NameError,
                                           ErrorHandler.NoReference,
                                           NodePosition (Next_Sibling
                                                            (Child_Node (ParentNode (OptNode)))),
                                           LexTokenManager.NullString);
               Valid := False; -- don't add clause at all if any part is malformed
            end if;
         end CheckIdentifiers;

         procedure HandleFunctionReturn (Kind        : in     ErrorHandler.JustificationKinds;
                                         ErrNum      : in     Natural;
                                         Identifiers : in out ErrorHandler.JustificationIdentifiers)
         --# derives Identifiers from *,
         --#                          ErrNum,
         --#                          Kind;
         is
            function MessageIsIFA (ErrNum : Natural) return Boolean
            is
            begin
               return ErrNum = 50 or else ErrNum = 602;
            end MessageIsIFA;

         begin
            -- If the users has tried to justify an information flow error where the "export" is the function
            -- return result, then there will only be one variable name in the message (which will say, e.g.,
            -- "The function value is not derived from the imported value(s) of Y.") but the pattern matching
            -- in ErrorHandler.Justification.CheckWhetherJustified will still be expecting an two symbols, an
            -- export followed by an import.  The Examiner's flow analyser uses NullSymbol to represent the
            -- function return value.  In this procedure we:
            --    (1) Detect cases where only one argument has been supplied for an IFA msg that needs two
            --    (2) Assume in that case that a function return in implicitly intended
            --    (3) Move the given variable to the second, import, slot.
            --    (4) Insert a null identifier in the first slot (this will match NullSymbol).
            -- Note that this transformation is "safe" even if the user has simply forgotten a variable name
            -- because the transformed annotation will not pattern match any more than it would have before.
            -- e.g. User in intends "F, 50, X, Y" but types "F, 50, X" by mistake.  Transformation gives
            -- "F, 50, NullSym, X".  Neither original incorrect form nor transformed form will pattern match
            -- so behaviour is unaltered

            if Kind = ErrorHandler.FlowMessage and then
               MessageIsIFA (ErrNum) then
               -- possibly something to do
               if Identifiers (2) = ErrorHandler. NullJustificationIdentifier then
                  -- only one identifier was supplied so transformation is needed
                  Identifiers (2) := Identifiers (1); -- move given variable to second place
                  Identifiers (1) := ErrorHandler. NullJustificationIdentifier; -- to match NullSym
               end if;
            end if;
         end HandleFunctionReturn;

      begin --wf_justification_clause
         -- grammar: justification_clause
         --               |
         --          identifier--- numeric_literal---justification_opt---string_literal
         --                                                 |
         --                                          dotted_simple_name_list (or null)
         --                                                 |
         --                                          dotted_simple_name_list---dotted_simple_name
         --                                                 |
         --                                          dotted_simple_name
         ErrorFound := False;

         -- Check whether we are dealing with Flow_Message or Warning_Message --------
         CurrentNode := Child_Node (Node);
         SystemErrors.RTAssert (SyntaxNodeType (CurrentNode) = SPSymbols.identifier,
                                SystemErrors.AssertionFailure,
                                "Failed to find indentifier giving error kind in justification clause");
         CheckKind (NodeLexString (CurrentNode),
                     --to get
                    Kind,
                    Valid);

         if not Valid then
            ErrorFound := True;
            ErrorHandler.SemanticError (121,
                                        ErrorHandler.NoReference,
                                        NodePosition (CurrentNode),
                                        LexTokenManager.NullString);
         end if;

         -- Check error number ---------------------------------------------------------
         CurrentNode := Next_Sibling (CurrentNode);
         SystemErrors.RTAssert (SyntaxNodeType (CurrentNode) = SPSymbols.numeric_literal,
                                SystemErrors.AssertionFailure,
                                "Failed to find error number in justification clause");
         -- Use existing wff to get literal
         -- local grammar
         -- numeric_literal
         --       |
         -- decimal_literal
         --       |
         -- integer_number
         ErrNumNode := Child_Node (Child_Node (CurrentNode));
         if SyntaxNodeType (ErrNumNode) = SPSymbols.integer_number then
            GetLiteralValue (Child_Node (Child_Node (CurrentNode)), -- drill down to the integer_number
                              -- to get
                             Val);
            Maths.ValueToInteger (Val,
                                  ErrNum,
                                  MathsValid);

            Valid := MathsValid = Maths.NoError;
            if not Valid then
               ErrorFound := True;
               ErrNum := 0;
            end if;
         else -- wrong kind of number
            ErrorFound := True;
            ErrNum := 0;
         end if;
         -- We should have a valid positive integer value for ErrNum by here.  If not, raise error
         if ErrNum = 0 then
            ErrorHandler.SemanticError (122,
                                        ErrorHandler.NoReference,
                                        NodePosition (ErrNumNode),
                                        LexTokenManager.NullString);

         elsif IsDisallowedWarning (Kind, ErrNum) then
            -- we have a wellformed warning number but we may want to disallow certain warning numbers
            ErrorHandler.SemanticError (123,
                                        ErrorHandler.NoReference,
                                        NodePosition (ErrNumNode),
                                        LexTokenManager.NullString);
            ErrorFound := True;
         end if;

         -- Check identifiers --------- ---------------------------------------------------------
         CurrentNode := Next_Sibling (CurrentNode);
         SystemErrors.RTAssert (SyntaxNodeType (CurrentNode) = SPSymbols.justification_opt,
                                SystemErrors.AssertionFailure,
                                "Failed to find identifier list location in justification clause");

         CheckIdentifiers (CurrentNode,
                           -- to get
                           Identifiers,
                           Valid);
         if not Valid then  -- I think this is clearer that ErrorFOund := ErrorFound or not Valid;
            ErrorFound := True;
         end if;

         -- Check explanation --------- ---------------------------------------------------------
         CurrentNode := Next_Sibling (CurrentNode);
         SystemErrors.RTAssert (SyntaxNodeType (CurrentNode) = SPSymbols.string_literal,
                                SystemErrors.AssertionFailure,
                                "Failed to find explanation string in justification clause");
         Explanation := NodeLexString (CurrentNode);

         -- Insert justification data in error handler data table
         if not ErrorFound then

            -- See whether Identifiers needs transforming to handle IFA errors on function return
            HandleFunctionReturn (Kind,
                                  ErrNum,
                                    -- using and to get
                                  Identifiers);

            -- Finally, add it to table of justification
            ErrorHandler.StartJustification (NodePosition (Node),
                                             StartLine,
                                             Kind,
                                             ErrNum,
                                             Identifiers,
                                             Explanation,
                                             -- to get
                                             MaximumJustificationsReached);
            if MaximumJustificationsReached then
               ErrorHandler.SemanticWarning (122,
                                             NodePosition (Node),
                                             LexTokenManager.NullString);
            end if;
         end if;
      end wf_justification_clause;

   begin -- wf_start_justification
      It := FindFirstNode (NodeKind    => SPSymbols.justification_clause,
                           FromRoot    => Node,
                           InDirection => STree.Down);
      while not STree.IsNull (It) loop
         wf_justification_clause (NodePosition (Node).StartLineNo,
                                  GetNode (It),
                                  Scope);
         It := STree.NextNode (It);
      end loop;
   end wf_start_justification;

   ----------------------------

   procedure wf_end_justification (Node     : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table;
   is
      UnmatchedEnd : Boolean;
   begin
      ErrorHandler.EndJustification (NodePosition (Node).StartLineNo,
                                       -- to get
                                     UnmatchedEnd);
      if UnmatchedEnd then
         ErrorHandler.SemanticWarning (120,
                                       NodePosition (Node),
                                       LexTokenManager.NullString);
      end if;
   end wf_end_justification;

   ----------------------------

begin -- wf_justification_statement
   if SyntaxNodeType (Child_Node (Node)) = SPSymbols.start_justification then
      wf_start_justification (Node, Scope);
   elsif SyntaxNodeType (Child_Node (Node)) = SPSymbols.end_justification then
      wf_end_justification (Node);
   else
      SystemErrors.FatalError (SystemErrors.InvalidSyntaxTree,
                               "Unexpected node type found in wf_justification_statement");
   end if;
end wf_justification_statement;
