-- $Id: sem-compunit-wf_inherit_clause.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

-- Overview:
-- Checks an inherit clause for Sem starting at node inherit_clause.
-- Directly capable of rasing errors for: undeclared item in inherit list,
-- duplicate item in inherit list or inheriting of something which is not a
-- package.
--------------------------------------------------------------------------------

separate (Sem.CompUnit)

procedure wf_inherit_clause (Node    : in STree.SyntaxNode;
                             CompSym : in Dictionary.Symbol;
                             Scope   : in Dictionary.Scopes)
is
   It : STree.Iterator;

   procedure ProcessDottedSimpleName (Node : in     STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     CompSym;
   --#        in     LextokenManager.StringTable;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict           from *,
   --#                                        CommandLineData.Content,
   --#                                        CompSym,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        CompSym,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LextokenManager.StringTable,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      PrefixSym         : Dictionary.Symbol := Dictionary.NullSymbol;
      CurrentSym        : Dictionary.Symbol;
      CurrentNode       : STree.SyntaxNode;
      ExplicitDuplicate : Boolean;
      Ok                : Boolean;
      ----------------------------

      function DottedIdentifierFound return Boolean
      --# global in Node;
      --#        in STree.Table;
      --  pre Node is top-most dotted_simple_name node;
      is
      begin
         return SyntaxNodeType (Child_Node (Node)) =
            SPSymbols.dotted_simple_name;
      end DottedIdentifierFound;

      -------------------

      function IsLastIdentifierNode (Node : STree.SyntaxNode) return Boolean
      --# global in STree.Table;
      --  pre    SyntaxNodeType (Node) = SPSymbols.identifier;
      is
      begin
         return SyntaxNodeType (ParentNode (ParentNode (Node))) /=
            SPSymbols.dotted_simple_name;
      end IsLastIdentifierNode;

      --------------------

      function LookUp (Prefix  : in     Dictionary.Symbol;
                       Str     : in     LexTokenManager.LexString) return Dictionary.Symbol
      --# global in CommandLineData.Content;
      --#        in Dictionary.Dict;
      --#        in Scope;
      is
         Sym : Dictionary.Symbol;
      begin
         if Prefix = Dictionary.NullSymbol then
            Sym := Dictionary.LookupItem (Str,
                                          Scope,
                                          Dictionary.ProofContext);
         else
            Sym := Dictionary.LookupSelectedItem (Prefix,
                                                  Str,
                                                  Scope,
                                                  Dictionary.ProofContext);
         end if;
         return Sym;
      end LookUp;

      --------------------

      procedure CheckPackageOwner (CompSym, CurrentSym : in Dictionary.Symbol;
                                   Ok                  : out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     CurrentNode;
      --#        in     Dictionary.Dict;
      --#        in     LextokenManager.StringTable;
      --#        in     STree.Table;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        CompSym,
      --#                                        CurrentNode,
      --#                                        CurrentSym,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LextokenManager.StringTable,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table &
      --#         Ok                        from CompSym,
      --#                                        CurrentSym,
      --#                                        Dictionary.Dict;
      is
         Owner : Dictionary.Symbol;
      begin
         Ok := True;
         Owner := Dictionary.GetPackageOwner (CompSym);
         if Owner /= Dictionary.NullSymbol and then CurrentSym /= Owner then
            if Dictionary.IsProperDescendent (CurrentSym, Owner) then
               if not Dictionary.IsPrivateDescendent (CurrentSym, Owner) then
                  ErrorHandler.SemanticError (617,
                                              ErrorHandler.NoReference,
                                              NodePosition (CurrentNode),
                                              NodeLexString (CurrentNode));
                  Ok := False;
               end if;
            elsif not Dictionary.IsInherited (CurrentSym, Owner) then
               ErrorHandler.SemanticError (618,
                                           ErrorHandler.NoReference,
                                           NodePosition (CurrentNode),
                                           NodeLexString (CurrentNode));
               Ok := False;
            end if;
         end if;
      end CheckPackageOwner;

      --------------------

      --    detects eg P.Q inheriting both R and P.R
      procedure CheckForRedeclaration (CompSym, CurrentSym : in Dictionary.Symbol;
                                       Ok                  : out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     CurrentNode;
      --#        in     Dictionary.Dict;
      --#        in     LextokenManager.StringTable;
      --#        in     STree.Table;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        CompSym,
      --#                                        CurrentNode,
      --#                                        CurrentSym,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LextokenManager.StringTable,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table &
      --#         Ok                        from CompSym,
      --#                                        CurrentNode,
      --#                                        CurrentSym,
      --#                                        Dictionary.Dict,
      --#                                        STree.Table;
      is
         ParentSym  : Dictionary.Symbol;
         VisibleSym : Dictionary.Symbol;
      begin
         Ok := True;
         if not Dictionary.IsEmbeddedPackage (CompSym) and then
           Dictionary.IsPackage (CurrentSym) then -- guard for next line's call
            ParentSym := Dictionary.GetPackageParent (CurrentSym);
            if ParentSym = Dictionary.NullSymbol or else
               Dictionary.IsProperDescendent (CompSym, ParentSym)
            then -- CurrentSym will be directly visible
               VisibleSym := Dictionary.LookupItem (NodeLexString (CurrentNode),
                                                    Dictionary.VisibleScope (CompSym),
                                                    Dictionary.ProofContext);
               if VisibleSym /= Dictionary.NullSymbol and then
                  VisibleSym /= CurrentSym
               then -- name is already directly visible (and not duplicate)
                  ErrorHandler.SemanticError (10,
                                              ErrorHandler.NoReference,
                                              NodePosition (CurrentNode),
                                              NodeLexString (CurrentNode));
                  Ok := False;
               end if;
            end if;
         end if;
      end CheckForRedeclaration;

      --------------------

   begin --ProcessDottedSimpleName
      if DottedIdentifierFound and then CommandLineData.IsSpark83 then
         ErrorHandler.SemanticError (610,
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     LexTokenManager.NullString);
      else
         CurrentNode := LastChildOf (Node); --first prefix identifier
         loop
            CurrentSym := LookUp (PrefixSym,
                                  NodeLexString (CurrentNode));
            PrefixSym := CurrentSym; --ready for next lookup
            if CurrentSym = Dictionary.NullSymbol then
               ErrorHandler.SemanticError (135,
                                           ErrorHandler.NoReference,
                                           NodePosition (CurrentNode),
                                           NodeLexString (CurrentNode));
               exit;
            end if;

            if not Dictionary.IsPackage (CurrentSym) and then
              not Dictionary.IsGenericSubprogram (CurrentSym) then -- can't be inherited
               ErrorHandler.SemanticError (18,
                                           ErrorHandler.NoReference,
                                           NodePosition (CurrentNode),
                                           NodeLexString (CurrentNode));
               exit;
            end if;

            if CurrentSym = CompSym then --trying to inherit self
               ErrorHandler.SemanticError (134,
                                           ErrorHandler.NoReference,
                                           NodePosition (CurrentNode),
                                           NodeLexString (CurrentNode));
               exit;
            end if;

            -- check for valid inheriting of private packages
            if Dictionary.IsPackage (CurrentSym) and then --guard for next call
              Dictionary.IsPrivatePackage (CurrentSym) and then
              Dictionary.GetPackageParent (CurrentSym) /= Dictionary.NullSymbol and then
              (Dictionary.IsMainProgram (CompSym) or else
                 not (Dictionary.IsEmbeddedPackage (CompSym) or else
                        Dictionary.IsDescendentOfPrivateSibling (CompSym,
                                                                 CurrentSym)))
            then
               ErrorHandler.SemanticError (616,
                                           ErrorHandler.NoReference,
                                           NodePosition (CurrentNode),
                                           NodeLexString (CurrentNode));
               exit;
            end if;


            -- check rules for what a child package may and may not inherit (i.e. siblings of
            -- same kind (public/private) etc).  Note guard so that we don't do this if the
            -- inherited things is a generic subprogram since these are library-level units that
            -- aren't covered by the child package hierarchy rules
            Ok := True;
            if CommandLineData.IsSpark95 and then
               Dictionary.IsPackage (CompSym) and then
              not Dictionary.IsGenericSubprogram (CurrentSym)
            then
               CheckPackageOwner (CompSym, CurrentSym, Ok);

               if Ok then
                  CheckForRedeclaration (CompSym, CurrentSym, Ok);
               end if;
            end if;
            exit when not Ok;

            Dictionary.AddInheritsReference
               (CompSym,
                CurrentSym,
                IsLastIdentifierNode (CurrentNode),
                Dictionary.Location'(NodePosition (CurrentNode),
                                     NodePosition (CurrentNode)),
                  --to get
                ExplicitDuplicate);
            if ExplicitDuplicate then
               ErrorHandler.SemanticErrorSym (190,
                                              ErrorHandler.NoReference,
                                              NodePosition (Node),
                                              CurrentSym,
                                              Scope);
            end if;

            exit when IsLastIdentifierNode (CurrentNode);

            CurrentNode := Next_Sibling (ParentNode (CurrentNode));
         end loop;
      end if;
   end ProcessDottedSimpleName;

   -----------------------------

begin  --wf_inherit_clause
   It := FindFirstNode (NodeKind    => SPSymbols.dotted_simple_name,
                        FromRoot    => Node,
                        InDirection => STree.Down);

   while not STree.IsNull (It) loop
      ProcessDottedSimpleName (GetNode (It));
      It := STree.NextNode (It);
   end loop;

end wf_inherit_clause;
