-- $Id: sem-compunit-wf_generic_subprogram_instantiation.adb 12351 2009-02-02 15:03:51Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


-- Grammar (with punctuation tokens removed)
--
--  generic_subprogram_instantiation :
--         generic_procedure_instantiation
--       | generic_function_instantiation
--       | generic_child_function_instantiation ;

--  generic_procedure_instantiation :
--         procedure_specification procedure_annotation identifier
--            generic_actual_part semicolon
--       | procedure_specification procedure_annotation identifier semicolon ;

--  generic_function_instantiation :
--         identifier [is new] identifier
--            generic_actual_part semicolon
--       | identifier [is new] identifier semicolon ;

--  generic_child_function_instantiation :
--         identifier [is new] identifier point identifier
--            generic_actual_part semicolon
--       | identifier [is new] identifier point identifier semicolon ;

--  generic_actual_part :
--         left_paren name_argument_list right_paren ;

separate (Sem.Compunit)
procedure wf_generic_subprogram_instantiation (Node  : in STree.SyntaxNode;
                                               Scope : in Dictionary.Scopes)
is
   type ExpectedGenericKind is (GenericProcedure, GenericFunction);

   procedure CheckValidInstantiationIdent (IdentNode : in     STree.SyntaxNode;
                                           Scope     : in     Dictionary.Scopes;
                                           Ok        :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        IdentNode,
   --#                                        LexTokenManager.StringTable,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         Ok                        from Dictionary.Dict,
   --#                                        IdentNode,
   --#                                        Scope,
   --#                                        STree.Table;
   is
      Sym : Dictionary.Symbol;
      IdentStr : LexTokenManager.LexString;
   begin
      -- check that name is not already in use
      IdentStr := NodeLexString (IdentNode);
      Sym :=  Dictionary.LookupItem (IdentStr,
                                     Scope,
                                     Dictionary.ProofContext);

      if Sym = Dictionary.NullSymbol then
         Ok := True;
      else
         Ok := False;
            ErrorHandler.SemanticError (10,
                                        ErrorHandler.NoReference,
                                        NodePosition (IdentNode),
                                        IdentStr);
      end if;

   end CheckValidInstantiationIdent;

   ----------------------------------------------------------

   procedure CheckGeneric (IdentNode  : in     STree.SyntaxNode;
                           Scope      : in     Dictionary.Scopes;
                           Kind       : in     ExpectedGenericKind;
                           GenericSym :    out Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        IdentNode,
   --#                                        Kind,
   --#                                        LexTokenManager.StringTable,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         GenericSym                from Dictionary.Dict,
   --#                                        IdentNode,
   --#                                        Kind,
   --#                                        Scope,
   --#                                        STree.Table;
   is
      -- Checks that the identifier after "is new" represents a visible generic unit of the
      -- appropriate kind.  Returns symbol of this generic unit if legal or a null symbol
      -- otherwise.
      Sym : Dictionary.Symbol;
   begin
      Sym := Dictionary.LookupItem (NodeLexString (IdentNode),
                                    Scope,
                                    Dictionary.ProgramContext);
      -- Kludge Alert
      -- If I am honest, I don't really understand why this is needed.  We have looked
      -- up the generic in program context so I would expect to get the Ada function
      -- symbol.  A test case showed otherwise, hence this is in for now.  PNA 13/1/4
      if Dictionary.IsProofFunction (Sym) then
         Sym := Dictionary.GetAdaFunction (Sym);
      end if;

      -- Validate generic unit symbol
      if Sym = Dictionary.NullSymbol then
         ErrorHandler.SemanticError (ErrNum    => 1,
                                     Reference => ErrorHandler.NoReference,
                                     Position  => NodePosition (IdentNode),
                                     IdStr     => NodeLexString (IdentNode));

      elsif Dictionary.IsGenericSubprogram (Sym) then
         if Kind = GenericProcedure and Dictionary.IsFunction (Sym) then
            -- wrong sort of subprogram
            ErrorHandler.SemanticError (ErrNum    => 631,
                                        Reference => ErrorHandler.NoReference,
                                        Position  => NodePosition (IdentNode),
                                        IdStr     => LexTokenManager.NullString);
            Sym := Dictionary.NullSymbol;
         elsif Kind = GenericFunction and Dictionary.IsProcedure (Sym) then
            -- wrong sort of subprogram
            ErrorHandler.SemanticError (ErrNum    => 632,
                                        Reference => ErrorHandler.NoReference,
                                        Position  => NodePosition (IdentNode),
                                        IdStr     => LexTokenManager.NullString);
            Sym := Dictionary.NullSymbol;

         elsif not Dictionary.SubprogramSignatureIsWellformed (Dictionary.IsAbstract,
                                                               Sym) then
            -- right sort of subprogram, but generic declaration had errors
            ErrorHandler.SemanticWarning (ErrNum    => 390,
                                          Position  => NodePosition (IdentNode),
                                          IdStr     => LexTokenManager.NullString);
            Sym := Dictionary.NullSymbol;
         end if;

      else -- not a generic subprgoram at all
         ErrorHandler.SemanticErrorSym (ErrNum    => 630,
                                        Reference => ErrorHandler.NoReference,
                                        Position  => NodePosition (IdentNode),
                                        Sym       => Sym,
                                        Scope     => Scope);
         Sym := Dictionary.NullSymbol;
      end if;

      GenericSym := Sym;
   end CheckGeneric;

   ----------------------------------------------------------

   procedure ProcessGenericProcedureInstantiation (Node  : in STree.SyntaxNode;
                                                   Scope : in Dictionary.Scopes)


   --# global in     CommandLineData.Content;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.StringTable,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                     from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          SPARK_IO.FILE_SYS,
   --#                                          STree.Table,
   --#                                          TheHeap;
   is
      GenericSym,
      InstantiationSym : Dictionary.Symbol;
      GenericNode : STree.SyntaxNode;
      InstantiationIdentNode : STree.SyntaxNode;
      ErrorsFound : Boolean;
      Ok : Boolean;
   begin
      --  generic_procedure_instantiation :
      --         procedure_specification procedure_annotation identifier
      --            generic_actual_part
      --       | procedure_specification procedure_annotation identifier ;
      GenericNode := Next_Sibling (Next_Sibling (Child_Node (Node)));
      InstantiationIdentNode := Child_Node (Child_Node (Node));
      CheckValidInstantiationIdent (InstantiationIdentNode,
                                    Scope,
                                    -- to get
                                    Ok);
      if Ok then
         CheckGeneric (GenericNode,
                       Scope,
                       GenericProcedure,
                        -- to get
                       GenericSym);

         if GenericSym /= Dictionary.NullSymbol then
            -- check parameters etc.
            -- add the instantiation
            Dictionary.AddSubprogramInstantiation (NodeLexString (Child_Node (Child_Node (Node))),
                                                   GenericSym,
                                                   Dictionary.Location'(NodePosition (Node),
                                                                        NodePosition (Node)),
                                                   Scope,
                                                   Dictionary.ProgramContext,
                                                   -- to get
                                                   InstantiationSym);

            -- check parameters etc.
            wf_generic_actual_part (GenericNode,
                                    GenericSym,
                                    InstantiationSym,
                                    Scope,
                                    -- to get
                                    ErrorsFound);
            if ErrorsFound then
               Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsAbstract,
                                                               InstantiationSym);
            else
               -- do formal/actual substitutions
               Dictionary.InstantiateSubprogramParameters (GenericSym,
                                                           InstantiationSym);
            end if;

         end if;
      end if;
   end ProcessGenericProcedureInstantiation;

   ----------------------------------------------------

   procedure ProcessGenericFunctionInstantiation (Node  : in STree.SyntaxNode;
                                                  Scope : in Dictionary.Scopes)

   --# global in     CommandLineData.Content;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.StringTable,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                     from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          SPARK_IO.FILE_SYS,
   --#                                          STree.Table,
   --#                                          TheHeap;
   is
      GenericSym,
      InstantiationSym : Dictionary.Symbol;
      InstantiationIdentNode : STree.SyntaxNode;
      GenericNode : STree.SyntaxNode;
      ErrorsFound : Boolean;
      Ok : Boolean;
   begin
      --  generic_function_instantiation :
      --         identifier [is new] identifier
      --            generic_actual_part semicolon
      --       | identifier [is new] identifier semicolon ;
      GenericNode := Next_Sibling (Child_Node (Node));
      InstantiationIdentNode := Child_Node (Node);
      CheckValidInstantiationIdent (InstantiationIdentNode,
                                    Scope,
                                    -- to get
                                    Ok);
      if Ok then
         CheckGeneric (GenericNode,
                       Scope,
                       GenericFunction,
                        -- to get
                       GenericSym);
         if GenericSym /= Dictionary.NullSymbol then
            -- add the instantiation
            Dictionary.AddSubprogramInstantiation (NodeLexString (Child_Node (Node)),
                                                   GenericSym,
                                                   Dictionary.Location'(NodePosition (Node),
                                                                        NodePosition (Node)),
                                                   Scope,
                                                   Dictionary.ProgramContext,
                                                   -- to get
                                                InstantiationSym);

            -- check parameters etc.
            wf_generic_actual_part (GenericNode,
                                    GenericSym,
                                    InstantiationSym,
                                    Scope,
                                    -- to get
                                    ErrorsFound);

            if ErrorsFound then
               Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsAbstract,
                                                               InstantiationSym);
            else
               Dictionary.InstantiateSubprogramParameters (GenericSym,
                                                           InstantiationSym);
            end if;

         end if;
      end if;
   end ProcessGenericFunctionInstantiation;

   ----------------------------------------------------

   procedure ProcessGenericChildFunctionInstantiation (Node  : in STree.SyntaxNode;
                                                       Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.StringTable,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                     from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          SPARK_IO.FILE_SYS,
   --#                                          STree.Table,
   --#                                          TheHeap;
   is
      PrefixSym,
      GenericSym,
      InstantiationSym : Dictionary.Symbol;
      PrefixNode,
      GenericNode : STree.SyntaxNode;
      InstantiationIdentNode : STree.SyntaxNode;
      ErrorsFound : Boolean;
      Ok : Boolean;
   begin
      --  generic_child_function_instantiation :
      --         identifier [is new] identifier point identifier
      --            generic_actual_part semicolon
      --       | identifier [is new] identifier point identifier semicolon ;
      PrefixNode := Next_Sibling (Child_Node (Node));
      GenericNode := Next_Sibling (PrefixNode);
      InstantiationIdentNode := Child_Node (Node);
      CheckValidInstantiationIdent (InstantiationIdentNode,
                                    Scope,
                                    -- to get
                                    Ok);
      if Ok then
         -- check prefix, in practice the only thing acceptable here will be package Ada
         PrefixSym := Dictionary.LookupItem (NodeLexString (PrefixNode),
                                             Scope,
                                             Dictionary.ProgramContext);
         if PrefixSym = Dictionary.NullSymbol then
            ErrorHandler.SemanticError (ErrNum    => 1,
                                        Reference => ErrorHandler.NoReference,
                                        Position  => NodePosition (PrefixNode),
                                        IdStr     => NodeLexString (PrefixNode));

         elsif not Dictionary.IsPackage (PrefixSym) then
            ErrorHandler.SemanticError (ErrNum    => 18,
                                        Reference => ErrorHandler.NoReference,
                                        Position  => NodePosition (PrefixNode),
                                        IdStr     => NodeLexString (PrefixNode));


         else
            -- potentially valid prefix
            CheckGeneric (GenericNode,
                          Dictionary.VisibleScope (PrefixSym),
                          GenericFunction,
                           -- to get
                          GenericSym);
            if GenericSym /= Dictionary.NullSymbol then
               -- add the instantiation
               Dictionary.AddSubprogramInstantiation (NodeLexString (Child_Node (Node)),
                                                      GenericSym,
                                                      Dictionary.Location'(NodePosition (Node),
                                                                           NodePosition (Node)),
                                                      Scope,
                                                      Dictionary.ProgramContext,
                                                      -- to get
                                                      InstantiationSym);
               -- check parameters etc.
               wf_generic_actual_part (GenericNode,
                                       GenericSym,
                                       InstantiationSym,
                                       Scope,
                                       -- to get
                                       ErrorsFound);

               if ErrorsFound then
                  Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsAbstract,
                                                                  InstantiationSym);
               else
                  -- do formal/actual substitutions
                  Dictionary.InstantiateSubprogramParameters (GenericSym,
                                                              InstantiationSym);
               end if;

            end if;
         end if;
      end if;
   end ProcessGenericChildFunctionInstantiation;

   ----------------------------------------------------
begin -- wf_generic_subprogram_instantiation
   if SyntaxNodeType (Child_Node (Node)) = SPSymbols.generic_procedure_instantiation then
      ProcessGenericProcedureInstantiation (Child_Node (Node),
                                            Scope);

   elsif SyntaxNodeType (Child_Node (Node)) = SPSymbols.generic_function_instantiation then
      ProcessGenericFunctionInstantiation (Child_Node (Node),
                                           Scope);

   elsif SyntaxNodeType (Child_Node (Node)) = SPSymbols.generic_child_function_instantiation then
      ProcessGenericChildFunctionInstantiation (Child_Node (Node),
                                                Scope);

   else
      SystemErrors.FatalError (SystemErrors.InvalidSyntaxTree,
                               "Unknown generic kind in wf_generic_subprogram_instantiation");
   end if;
end wf_generic_subprogram_instantiation;
