-- $Id: sem-compunit-wf_full_type_declaration.adb 12351 2009-02-02 15:03:51Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.CompUnit)

procedure wf_full_type_declaration (Node  : in STree.SyntaxNode;
                                    Scope : in Dictionary.Scopes)
is
   IdentNode : STree.SyntaxNode;
   IdentStr  : LexTokenManager.LexString;
   Sym       : Dictionary.Symbol;

   TypeDeclaredAsLimited : Boolean;
   TypeDeclaredAsTagged  : Boolean;

   -------------------------------

   function IsPrivateTypeResolution (Sym   : Dictionary.Symbol;
                                     Scope : Dictionary.Scopes) return Boolean
   --# global in Dictionary.Dict;
   is
   begin
      return (not Dictionary.IsDeclared (Sym) and then
              not Dictionary.IsVisibleScope (Scope) and then
              Dictionary.IsType (Sym) and then
              Dictionary.TypeIsPrivate (Sym) and then
              (Dictionary.GetRegion (Scope) =
               Dictionary.GetRegion (Dictionary.GetScope (Sym))));
   end IsPrivateTypeResolution;

   -------------------------------

   procedure EmptyTypeCheck (DecLoc : in     LexTokenManager.TokenPosition;
                             Lower,
                             Upper  : in out LexTokenManager.LexString)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LextokenManager.StringTable;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        DecLoc,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LextokenManager.StringTable,
   --#                                        Lower,
   --#                                        SPARK_IO.File_Sys,
   --#                                        Upper &
   --#         Lower,
   --#         Upper                     from LextokenManager.StringTable,
   --#                                        Lower,
   --#                                        Upper;
   is
      Unused        : Maths.ErrorCode;
      MathsResult   : Maths.Value;
      RangeIsEmpty  : Boolean;
   begin
      if Lower /= LexTokenManager.NullString and then
         Upper /= LexTokenManager.NullString
      then
         --# accept Flow, 10, Unused, "Expected ineffective assignment";
         Maths.Lesser (Maths.ValueRep (Upper),
                       Maths.ValueRep (Lower),
                        --to get
                       MathsResult,
                       Unused);  --not used because it can only be ok or type mismatch
         Maths.ValueToBool (MathsResult,
                              --to get
                            RangeIsEmpty,
                            Unused);
         if RangeIsEmpty then
            Lower := LexTokenManager.NullString;
            Upper := LexTokenManager.NullString;
            ErrorHandler.SemanticError (416,  -- type may not have an empty range
                                        ErrorHandler.NoReference,
                                        DecLoc,
                                        LexTokenManager.NullString);
         end if;
      end if;
      --# accept Flow, 33, Unused, "Expected to be neither referenced nor exported";
   end EmptyTypeCheck;

   -------------------------------

   procedure wf_integer (Node   : in STree.SyntaxNode;
                         Scope  : in Dictionary.Scopes;
                         IdStr  : in LexTokenManager.LexString;
                         DecLoc : in LexTokenManager.TokenPosition)
   --# global in     CommandLineData.Content;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LextokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --#           out AggregateStack.State;
   --# derives AggregateStack.State        from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          LextokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         Dictionary.Dict             from *,
   --#                                          CommandLineData.Content,
   --#                                          IdStr,
   --#                                          LextokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         ErrorHandler.ErrorContext   from *,
   --#                                          CommandLineData.Content,
   --#                                          DecLoc,
   --#                                          Dictionary.Dict,
   --#                                          LextokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          SPARK_IO.File_Sys,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         LextokenManager.StringTable,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                     from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          LextokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         SPARK_IO.File_Sys           from *,
   --#                                          CommandLineData.Content,
   --#                                          DecLoc,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          IdStr,
   --#                                          LextokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap;
      is separate;

   -------------------------------

   procedure wf_modular
      (Node   : in STree.SyntaxNode;
       Scope  : in Dictionary.Scopes;
       IdStr  : in LexTokenManager.LexString;
       DecLoc : in LexTokenManager.TokenPosition)
   --# global in     CommandLineData.Content;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LextokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         LextokenManager.StringTable,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                     from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          LextokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         Dictionary.Dict             from *,
   --#                                          CommandLineData.Content,
   --#                                          IdStr,
   --#                                          LextokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         ErrorHandler.ErrorContext   from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          LextokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          SPARK_IO.File_Sys,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         SPARK_IO.File_Sys           from *,
   --#                                          CommandLineData.Content,
   --#                                          DecLoc,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          IdStr,
   --#                                          LextokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap;
      is separate;

   -------------------------------

   procedure wf_enum (Node   : in STree.SyntaxNode;
                      Scope  : in Dictionary.Scopes;
                      IdStr  : in LexTokenManager.LexString;
                      DecLoc : in LexTokenManager.TokenPosition)
   --# global in     CommandLineData.Content;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LextokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict,
   --#         LextokenManager.StringTable from Dictionary.Dict,
   --#                                          IdStr,
   --#                                          LextokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys           from CommandLineData.Content,
   --#                                          DecLoc,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          IdStr,
   --#                                          LextokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          SPARK_IO.File_Sys,
   --#                                          STree.Table;
      is separate;

   -------------------------------

   procedure wf_record (Node                     : in STree.SyntaxNode;
                        Scope                    : in Dictionary.Scopes;
                        IdStr                    : in LexTokenManager.LexString;
                        DecLoc                   : in LexTokenManager.TokenPosition;
                        Extends                  : in Dictionary.Symbol;
                        PrivateTypeBeingResolved : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     LextokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict           from *,
   --#                                        CommandLineData.Content,
   --#                                        Extends,
   --#                                        IdStr,
   --#                                        Node,
   --#                                        PrivateTypeBeingResolved,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        DecLoc,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        Extends,
   --#                                        IdStr,
   --#                                        LextokenManager.StringTable,
   --#                                        Node,
   --#                                        PrivateTypeBeingResolved,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table;
      is separate;

   -------------------------------

--     procedure wf_array replaced by wf_array_type_definition in compunit

   -------------------------------

   procedure wf_real (Node   : in STree.SyntaxNode;
                      Scope  : in Dictionary.Scopes;
                      IdStr  : in LexTokenManager.LexString;
                      DecLoc : in LexTokenManager.TokenPosition)
   --# global in     CommandLineData.Content;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LextokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --#           out AggregateStack.State;
   --# derives AggregateStack.State        from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          LextokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         Dictionary.Dict             from *,
   --#                                          CommandLineData.Content,
   --#                                          IdStr,
   --#                                          LextokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         ErrorHandler.ErrorContext   from *,
   --#                                          CommandLineData.Content,
   --#                                          DecLoc,
   --#                                          Dictionary.Dict,
   --#                                          LextokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          SPARK_IO.File_Sys,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         LextokenManager.StringTable,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                     from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          LextokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         SPARK_IO.File_Sys           from *,
   --#                                          CommandLineData.Content,
   --#                                          DecLoc,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          IdStr,
   --#                                          LextokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap;
      is separate;

   -------------------------------

   procedure wf_type_extension (Node                     : in STree.SyntaxNode;
                                Scope                    : in Dictionary.Scopes;
                                IdStr                    : in LexTokenManager.LexString;
                                DecLoc                   : in LexTokenManager.TokenPosition;
                                PrivateTypeBeingResolved : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     LextokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict           from *,
   --#                                        CommandLineData.Content,
   --#                                        IdStr,
   --#                                        Node,
   --#                                        PrivateTypeBeingResolved,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        DecLoc,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        IdStr,
   --#                                        LextokenManager.StringTable,
   --#                                        Node,
   --#                                        PrivateTypeBeingResolved,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table;
      is separate;

   -------------------------------

   procedure wf_protected_type_declaration (Node  : in STree.SyntaxNode;
                                            Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LextokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LextokenManager.StringTable,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                     from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          GlobalComponentData,
   --#                                          LextokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          GlobalComponentData,
   --#                                          LextokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          SPARK_IO.File_Sys,
   --#                                          STree.Table,
   --#                                          TheHeap;
      is separate;

   -------------------------------

   procedure wf_task_type_declaration (Node  : in STree.SyntaxNode;
                                       Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LextokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LextokenManager.StringTable,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                     from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          LextokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          LextokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          SPARK_IO.File_Sys,
   --#                                          STree.Table,
   --#                                          TheHeap;
      is separate;

   -------------------------------

   procedure wf_type_definition (Node                     : in STree.SyntaxNode;
                                 Scope                    : in Dictionary.Scopes;
                                 IdStr                    : in LexTokenManager.LexString;
                                 DecLoc                   : in LexTokenManager.TokenPosition;
                                 PrivateTypeBeingResolved : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LextokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                     from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          LextokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         Dictionary.Dict             from *,
   --#                                          CommandLineData.Content,
   --#                                          IdStr,
   --#                                          LextokenManager.StringTable,
   --#                                          Node,
   --#                                          PrivateTypeBeingResolved,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys           from CommandLineData.Content,
   --#                                          DecLoc,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          IdStr,
   --#                                          LextokenManager.StringTable,
   --#                                          Node,
   --#                                          PrivateTypeBeingResolved,
   --#                                          Scope,
   --#                                          SPARK_IO.File_Sys,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         LextokenManager.StringTable from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          IdStr,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap;
   is
      ChildNode : STree.SyntaxNode;
      NodeType  : SPSymbols.SPSymbol;
      UnusedArrayTypeSymbol : Dictionary.Symbol;
      UnusedErrorsFound : Boolean;
   begin
      -- ASSUME Node = type_definition

      ChildNode := Child_Node (Node);
      NodeType := SyntaxNodeType (ChildNode);
      case NodeType is
         when SPSymbols.enumeration_type_definition =>
            wf_enum (ChildNode, Scope, IdStr, DecLoc);

         when SPSymbols.real_type_definition =>
            wf_real (ChildNode, Scope, IdStr, DecLoc);

         when SPSymbols.record_type_definition =>
            wf_record (ChildNode, Scope, IdStr, DecLoc, Dictionary.NullSymbol, PrivateTypeBeingResolved);

         when SPSymbols.integer_type_definition =>
            wf_integer (ChildNode, Scope, IdStr, DecLoc);

         when SPSymbols.modular_type_definition =>
            wf_modular (ChildNode, Scope, IdStr, DecLoc);

         when SPSymbols.array_type_definition =>
            --wf_array (ChildNode, Scope, IdStr, DecLoc);
            --# accept Flow, 10, UnusedErrorsFound, "Expected ineffective assignment" &
            --#        Flow, 10, UnusedArrayTypeSymbol, "Expected ineffective assignment";
            wf_array_type_definition (Node => ChildNode,
                                      Scope => Scope,
                                      IdStr => IdStr,
                                      DecLoc => DecLoc,
                                      IsGeneric => False,
                                      ErrorsFound => UnusedErrorsFound,
                                      TheArray => UnusedArrayTypeSymbol);
            --# end accept;

         when SPSymbols.type_extension =>
            wf_type_extension (ChildNode, Scope, IdStr, DecLoc, PrivateTypeBeingResolved);

         when others =>
            null;        -- error
      end case;
      --# accept Flow, 33, UnusedErrorsFound, "Expected to be neither referenced nor exported" &
      --#        Flow, 33, UnusedArrayTypeSymbol, "Expected to be neither referenced nor exported";
   end wf_type_definition;

   -------------------------------

   function ValidScopeForTaskOrProtectedTypeDeclaration (Scope : Dictionary.Scopes) return Boolean
   --# global in Dictionary.Dict;
   is
   begin
      return (Dictionary.IsVisibleScope (Scope) or else
                Dictionary.IsPrivateScope (Scope)) and then
        Dictionary.IsPackage (Dictionary.GetRegion (Scope));
   end ValidScopeForTaskOrProtectedTypeDeclaration;

   -------------------------------

begin --wf_full_type_declaration
      --ASSUME Node = full_type_declaration

   IdentNode := Child_Node (Node);
   case SyntaxNodeType (IdentNode) is
      when SPSymbols.identifier =>  -- a type other than a task or protected object
         IdentStr := NodeLexString (IdentNode);
         Sym := Dictionary.LookupItem (IdentStr,
                                       Scope,
                                       Dictionary.ProofContext);

         if Sym = Dictionary.NullSymbol               or else
           (Dictionary.IsTypeMark (Sym) and then
            Dictionary.TypeIsAnnounced (Sym) and then
            not Dictionary.IsDeclared (Sym))      or else
           IsPrivateTypeResolution (Sym,
                                    Scope)
         then
            if Sym = Dictionary.NullSymbol then
               --initial type declaration
               wf_type_definition (Next_Sibling (IdentNode),
                                   Scope,
                                   IdentStr,
                                   NodePosition (Node),
                                   Sym);
            else
               TypeDeclaredAsLimited := Dictionary.IsLimitedPrivateType (Sym);
               TypeDeclaredAsTagged := Dictionary.TypeIsTagged (Sym);

               wf_type_definition (Next_Sibling (IdentNode),
                                   Scope,
                                   IdentStr,
                                   NodePosition (Node),
                                   Sym);

               if Dictionary.IsUnconstrainedArrayType (Sym) then
                  if Dictionary.TypeIsAnnounced (Sym) then
                     ErrorHandler.SemanticError (311,
                                                 ErrorHandler.NoReference,
                                                 NodePosition (IdentNode),
                                                 IdentStr);
                  else --private type
                     ErrorHandler.SemanticError (331,
                                                 ErrorHandler.NoReference,
                                                 NodePosition (IdentNode),
                                                 IdentStr);
                  end if;
               end if;

               if not TypeDeclaredAsLimited and then
                 Dictionary.TypeIsLimited (Sym,
                                           Dictionary.GlobalScope)
               then
                  ErrorHandler.SemanticError (332,
                                              ErrorHandler.NoReference,
                                              NodePosition (IdentNode),
                                              IdentStr);
               end if;

               -- initial declaration tagged but completion is not
               if TypeDeclaredAsTagged and then
                 not Dictionary.TypeIsTagged (Sym)
               then
                  ErrorHandler.SemanticError (821,
                                              ErrorHandler.NoReference,
                                              NodePosition (IdentNode),
                                              IdentStr);
               end if;

               -- reverse case; this could be made legal but is not allowed for now
               if not TypeDeclaredAsTagged and then
                 Dictionary.TypeIsTagged (Sym)
               then
                  ErrorHandler.SemanticError (830,
                                              ErrorHandler.NoReference,
                                              NodePosition (IdentNode),
                                              IdentStr);
               end if;

            end if;
         else
            ErrorHandler.SemanticError (10,
                                        ErrorHandler.NoReference,
                                        NodePosition (IdentNode),
                                        IdentStr);
         end if;

      when SPSymbols.task_type_declaration =>
         if CommandLineData.RavenscarSelected then
            if ValidScopeForTaskOrProtectedTypeDeclaration (Scope) then
               wf_task_type_declaration (IdentNode, Scope);
            else
               ErrorHandler.SemanticError (987,
                                           ErrorHandler.NoReference,
                                           NodePosition (IdentNode),
                                           LexTokenManager.NullString);
            end if;
         else -- declaration not allowed
            ErrorHandler.SemanticError (850,
                                        ErrorHandler.NoReference,
                                        NodePosition (IdentNode),
                                        LexTokenManager.NullString);
         end if;

      when SPSymbols.protected_type_declaration =>
         if CommandLineData.RavenscarSelected then
            if ValidScopeForTaskOrProtectedTypeDeclaration (Scope) then
               wf_protected_type_declaration (IdentNode, Scope);
            else
               ErrorHandler.SemanticError (987,
                                           ErrorHandler.NoReference,
                                           NodePosition (IdentNode),
                                           LexTokenManager.NullString);
            end if;
         else -- declaration not allowed
            ErrorHandler.SemanticError (850,
                                        ErrorHandler.NoReference,
                                        NodePosition (IdentNode),
                                        LexTokenManager.NullString);
         end if;

      when others =>
         SystemErrors.FatalError (SystemErrors.InvalidSyntaxTree, "in wf_full_type_declaration");
   end case;
end wf_full_type_declaration;
