-- $Id: sem-compunit-wf_full_type_declaration-wf_task_type_declaration.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.CompUnit.wf_full_type_declaration)
procedure wf_task_type_declaration (Node  : in STree.SyntaxNode;
                                    Scope : in Dictionary.Scopes)
is
   TaskTypeSym                 : Dictionary.Symbol;
   TaskScope                   : Dictionary.Scopes;
   Sym                         : Dictionary.Symbol;
   IdentNode                   : STree.SyntaxNode;
   AnnoNode                    : STree.SyntaxNode;
   ClosingIdentNode            : STree.SyntaxNode;
   DiscriminantNode            : STree.SyntaxNode;
   IdentStr                    : LexTokenManager.LexString;
   PragmaNode                  : STree.SyntaxNode;
   GlobalError                 : Boolean;
   DerivesError                : Boolean := False;


   -----------------------------------------------------------------------------------
   -- Node is "task_type_declaration"
   --
   -- Grammar:
   -- task_type_declaration
   --         |
   --     identifier -- [known_discriminant_part] -- task_type_annotation -- task_definition
   --                                                         |                    |
   --               +-----------------------------------------+                    |
   --               |                                                              |
   --  moded_global_definition -- [dependency_relation] -- [declare_annotation]    |
   --                                                              |               |
   --                      +---------------------------------------+               |
   --                      |                                                       |
   --                property_list                                                 |
   --                      |...                                                    |
   --                property_rep                                                  |
   --                      |                                                       |
   --                  property                                                    |
   --                      |                                                   task_pragma -- identifier
   --              name_value_property  or  delay_property                           |...
   --                      |                      |                            task_pragma -- apragma
   --  identifier -- global_variable_list or     RWDelay
   --  identifier -- integer_number                                            priority_pragma
   --
   --
   --

   function GetDiscriminantNode (TaskTypeDeclarationNode : STree.SyntaxNode)
                                return STree.SyntaxNode
   --# global in STree.Table;
   is
   begin
      return Next_Sibling (Child_Node (TaskTypeDeclarationNode));
   end GetDiscriminantNode;

   ----------

   function GetAnnoNode (TaskTypeDeclarationNode : STree.SyntaxNode)
                                return STree.SyntaxNode
   --# global in STree.Table;
   is
      Result : STree.SyntaxNode;
   begin
      Result := GetDiscriminantNode (TaskTypeDeclarationNode);
      -- Result is known_discriminant_part or task_type_annotation
      if SyntaxNodeType (Result) = SPSymbols.known_discriminant_part then
         Result := Next_Sibling (Result);
      end if;
      -- Result is task_type_annotation
      return Child_Node (Result); -- mode_global_definition
   end GetAnnoNode;

   ----------

   function GetTaskDefinitionNode (TaskTypeDeclarationNode : STree.SyntaxNode)
                                return STree.SyntaxNode
   --# global in STree.Table;
   is
   begin
      return LastSiblingOf (Child_Node (TaskTypeDeclarationNode));
   end GetTaskDefinitionNode;

   ----------

   function GetClosingIdentNode (TaskTypeDeclarationNode : STree.SyntaxNode)
                                return STree.SyntaxNode
   --# global in STree.Table;
   is
   begin
      return Next_Sibling (Child_Node (GetTaskDefinitionNode (TaskTypeDeclarationNode)));
   end GetClosingIdentNode;

   ----------

   function GetPriorityPragmaNode (TaskTypeDeclarationNode : STree.SyntaxNode)
                                  return STree.SyntaxNode
   --# global in STree.Table;
   is
   begin
      return ParentNode (LastChildOf (GetTaskDefinitionNode (TaskTypeDeclarationNode)));
   end GetPriorityPragmaNode;

   ----------

   procedure CheckPragmaValidity (EndNodePosition : in LexTokenManager.TokenPosition)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LextokenManager.StringTable;
   --#        in     TaskTypeSym;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EndNodePosition,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LextokenManager.StringTable,
   --#                                        SPARK_IO.File_Sys,
   --#                                        TaskTypeSym;
   is
      PriorityFound,
      InterruptPriorityFound : Boolean;
   begin
      PriorityFound          := Dictionary.GetTypeHasPragma (TaskTypeSym,
                                                             Dictionary.Priority);
      InterruptPriorityFound := Dictionary.GetTypeHasPragma (TaskTypeSym,
                                                             Dictionary.InterruptPriority);

      -- There must be either Priority or Interrupt_Priority
      if not (PriorityFound or InterruptPriorityFound) then
         ErrorHandler.SemanticError (876,
                                     ErrorHandler.NoReference,
                                     EndNodePosition,
                                     LexTokenManager.NullString);
      end if;
   end CheckPragmaValidity;

   ------------

begin -- wf_task_type_declaration;
   IdentNode := Child_Node (Node);
   IdentStr := NodeLexString (IdentNode);
   DiscriminantNode := GetDiscriminantNode (Node);
   Sym := Dictionary.LookupItem (IdentStr,
                                 Scope,
                                 Dictionary.ProofContext);

   if Sym = Dictionary.NullSymbol or else
     (Dictionary.IsTypeMark (Sym) and then
      Dictionary.TypeIsAnnounced (Sym) and then
      not Dictionary.IsDeclared (Sym))
   then
      Dictionary.AddTaskType (Name        => NodeLexString (IdentNode),
                              Declaration => Dictionary.Location'(NodePosition (Node),
                                                                  NodePosition (Node)),
                              Scope       => Scope,
                              Context     => Dictionary.ProgramContext,
                              Constrained => (SyntaxNodeType (DiscriminantNode) /= SPSymbols.known_discriminant_part),
                              -- to get
                              TaskType    => TaskTypeSym);
      TaskScope := Dictionary.VisibleScope (TaskTypeSym);

      -- wff discriminants here
      if SyntaxNodeType (DiscriminantNode) = SPSymbols.known_discriminant_part then
         wf_known_discriminant_part (DiscriminantNode,
                                     TaskTypeSym,
                                     Scope);
      end if;

      -- handle annotation
      -- global
      AnnoNode := GetAnnoNode (Node);
      wf_global_definition (AnnoNode,
                            Scope, --pna--was TaskScope, but for global look up we should start in enclosing scope
                            TaskTypeSym,
                            True,  --FirstSeen
                              -- to get
                            GlobalError);

      if not CommandLineData.Content.DoInformationFlow then
         CreateFullSubProgDependency (Node,
                                      TaskTypeSym,
                                      Dictionary.IsAbstract);
      end if;

      -- derives
      AnnoNode := Next_Sibling (AnnoNode);
      if SyntaxNodeType (AnnoNode) = SPSymbols.dependency_relation then
         wf_dependency_relation (AnnoNode,
                                 TaskScope,
                                 TaskTypeSym,
                                 True, -- FirstSeen
                                 GlobalError);
      else
         -- no derives so check if its required
         if CommandLineData.Content.DoInformationFlow then
            DerivesError := True;
            ErrorHandler.SemanticError (501,
                                        ErrorHandler.NoReference,
                                        NodePosition (GetAnnoNode (Node)),
                                        LexTokenManager.NullString);
         end if;
      end if;

      -- asserts
      if SyntaxNodeType (AnnoNode) = SPSymbols.dependency_relation then
         AnnoNode := Next_Sibling (AnnoNode);
      end if;
      if SyntaxNodeType (AnnoNode) = SPSymbols.declare_annotation then
         wf_declare_annotation (AnnoNode,
                                TaskScope,
                                TaskTypeSym,
                                True);
      end if;

      -- if there are errors in the task type signature then mark it as malformed
      if GlobalError or DerivesError then
         Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsAbstract,
                                                         TaskTypeSym);
      end if;

      -- deal with priority pragma which should be first
      wf_priority_pragma (GetPriorityPragmaNode (Node),
                          TaskScope);

      -- check any other pragmas
      PragmaNode := GetPriorityPragmaNode (Node);
      loop
         PragmaNode := Next_Sibling (ParentNode (PragmaNode));
         exit when SyntaxNodeType (PragmaNode) = SPSymbols.identifier;
         wf_pragma (PragmaNode,
                    TaskScope);
      end loop;

      -- closing identifier must match initial
      ClosingIdentNode := GetClosingIdentNode (Node);
      if IdentStr /= NodeLexString (ClosingIdentNode) then
         ErrorHandler.SemanticError (58,
                                     ErrorHandler.NoReference,
                                     NodePosition (ClosingIdentNode),
                                     IdentStr);
      end if;

      CheckPragmaValidity (NodePosition (ClosingIdentNode));

   else -- illegal redeclaration
      ErrorHandler.SemanticError (10,
                                  ErrorHandler.NoReference,
                                  NodePosition (IdentNode),
                                  IdentStr);
   end if;
end wf_task_type_declaration;
