-- $Id: sem-compunit-wf_full_type_declaration-wf_record.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.CompUnit.wf_full_type_declaration)
procedure wf_record (Node                     : in STree.SyntaxNode;
                     Scope                    : in Dictionary.Scopes;
                     IdStr                    : in LexTokenManager.LexString;
                     DecLoc                   : in LexTokenManager.TokenPosition;
                     Extends                  : in Dictionary.Symbol;
                     PrivateTypeBeingResolved : in Dictionary.Symbol)
is
   It             : STree.Iterator;
   RecordSym      : Dictionary.Symbol;
   IsTagged,
   IsAbstract     : Boolean;
   HasFields      : Boolean := False;

   -----------------------------------------------------------------
   -- this function finds all the fields in a record including any
   -- non-private ones obtained by inheritance
   function IsExistingField (Fieldname : LexTokenManager.LexString;
                             TheRecord : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   is
      It             : Dictionary.Iterator;
      Result         : Boolean := False;
      CurrentRecord  : Dictionary.Symbol;
      ThisPackage    : Dictionary.Symbol;
      CurrentPackage : Dictionary.Symbol;

      function IsPublicDescendant (RootPackage, ThePackage : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      is
         CurrentPackage : Dictionary.Symbol;
         Result : Boolean := False;
      begin
         CurrentPackage := ThePackage;
         loop
            -- success case, we have got back to root all via public children
            if CurrentPackage = RootPackage then
               Result := True;
               exit;
            end if;
            --fail case, private child found
            exit when Dictionary.IsPrivatePackage (CurrentPackage);

            CurrentPackage := Dictionary.GetPackageParent (CurrentPackage);
            exit when CurrentPackage = Dictionary.NullSymbol;
         end loop;
         return Result;
      end IsPublicDescendant;

   begin -- IsExistingField
      ThisPackage := Dictionary.GetLibraryPackage (Dictionary.GetScope (TheRecord));
      CurrentRecord := TheRecord;
      loop
         CurrentPackage := Dictionary.GetLibraryPackage (Dictionary.GetScope (CurrentRecord));
         if not Dictionary.TypeIsPrivate (CurrentRecord) or else
           IsPublicDescendant (CurrentPackage, ThisPackage) then
            -- not private so search for all fields
            It := Dictionary.FirstRecordComponent (CurrentRecord);
            while not Dictionary.IsNullIterator (It) loop
               if Dictionary.GetSimpleName (Dictionary.CurrentSymbol (It)) = Fieldname then
                  Result := True;
                  exit;
               end if;
               It := Dictionary.NextSymbol (It);
            end loop;
         end if;
         exit when Result;
         CurrentRecord := Dictionary.GetRootOfExtendedType (CurrentRecord);
         exit when CurrentRecord = Dictionary.NullSymbol;
      end loop;
      return Result;
   end IsExistingField;

   -----------------------------------------------------------------

   procedure wf_component_declaration (Node   : in STree.SyntaxNode;
                                       RecSym : in Dictionary.Symbol;
                                       Scope  : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict           from *,
   --#                                        CommandLineData.Content,
   --#                                        Node,
   --#                                        RecSym,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        RecSym,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      TypeNode,
      IdentListNode,
      NextNode       : STree.SyntaxNode;
      It             : STree.Iterator;
      TypeSym        : Dictionary.Symbol;
      ComponentIdent : LexTokenManager.LexString;
      TypePos        : LexTokenManager.TokenPosition;

      TypeMarkIsNotDotted : Boolean;
      TypeMarkSimpleName  : LexTokenManager.LexString;
   begin
      -- ASSUME Node = component_declaration
      TypeNode :=  Child_Node (Next_Sibling (Child_Node (Node)));

      -- If the indicated typemark is not dotted (e.g. just "T" but not "P.T")
      -- then an additional check is required.
      -- Two nodes below type_mark, there will either be a
      -- dotted_simple_name node (dotted case) or an identifier node (not dotted).
      TypeMarkIsNotDotted :=
        SyntaxNodeType (Child_Node (Child_Node (TypeNode))) = SPSymbols.identifier;
      if TypeMarkIsNotDotted then
         TypeMarkSimpleName := NodeLexString (Child_Node (Child_Node (TypeNode)));
      else
         TypeMarkSimpleName := LexTokenManager.NullString;
      end if;


      TypePos := NodePosition (TypeNode);
      wf_type_mark (TypeNode,
                    Scope,
                    Dictionary.ProgramContext,
                     --to get
                    TypeSym);
      if not Dictionary.IsUnknownTypeMark (TypeSym) then
         if TypeSym = RecSym then
            -- Type of field is same type as the record type being declared.
            ErrorHandler.SemanticError (751,
                                        ErrorHandler.NoReference,
                                        TypePos,
                                        Dictionary.GetSimpleName (TypeSym));
         elsif Dictionary.IsUnconstrainedArrayType (TypeSym) then
            ErrorHandler.SemanticError (39,
                                        ErrorHandler.NoReference,
                                        NodePosition (TypeNode),
                                        LexTokenManager.NullString);
         end if;
      end if;

      IdentListNode := Child_Node (Node);

      It := FindFirstNode (NodeKind    => SPSymbols.identifier,
                           FromRoot    => IdentListNode,
                           InDirection => STree.Down);

      while not STree.IsNull (It) loop
         NextNode := GetNode (It);

         ComponentIdent := NodeLexString (NextNode);

         -- if the TypeMark is not dotted, then we need to check for the
         -- illegal case of a record field name which attempts to override
         -- the name of an existing directly visible TypeMake, such as
         --   type R is record
         --      T : T; -- illegal
         --   end record;
         if TypeMarkIsNotDotted and then (ComponentIdent = TypeMarkSimpleName) then
            ErrorHandler.SemanticError (757,
                                        ErrorHandler.NoReference,
                                        NodePosition (TypeNode),
                                        ComponentIdent);

         elsif IsExistingField (ComponentIdent, RecSym) -- catches repeat within dec
            -- which is an existing Examiner bug not to do with tagged types
         then
            ErrorHandler.SemanticError (10,
                                        ErrorHandler.NoReference,
                                        NodePosition (NextNode),
                                        ComponentIdent);
         elsif Dictionary.IsPredefinedSuspensionObjectType (TypeSym) or
           Dictionary.TypeIsProtected (TypeSym) then
            ErrorHandler.SemanticError (906,
                                        ErrorHandler.NoReference,
                                        NodePosition (TypeNode),
                                        LexTokenManager.NullString);
         else
            Dictionary.AddRecordComponent (ComponentIdent,
                                           RecSym,
                                           TypeSym,
                                           False, -- inherited field
                                           Dictionary.Location'(TypePos, TypePos));
         end if;
         It := STree.NextNode (It);
      end loop;
   end wf_component_declaration;

   -----------------------------------------------------------------

   procedure SetTagStatus (TagOptionNode : in STree.SyntaxNode)
   --# global in     STree.Table;
   --#           out IsAbstract;
   --#           out IsTagged;
   --# derives IsAbstract,
   --#         IsTagged   from STree.Table,
   --#                         TagOptionNode;
   is
   begin
      IsAbstract := SyntaxNodeType (TagOptionNode) = SPSymbols.abstract_tagged;
      IsTagged := IsAbstract or SyntaxNodeType (TagOptionNode) = SPSymbols.non_abstract_tagged;
   end SetTagStatus;

   -----------------------------------------------------------------

begin --wf_record_type_definition
      -- ASSUME Node = record_type_definition
   SetTagStatus (Child_Node (Child_Node (Node)));
   -- temporary prevention of use of abstract types
   if IsAbstract then
      ErrorHandler.SemanticError (820,
                                  ErrorHandler.NoReference,

                                  NodePosition (Node),
                                  LexTokenManager.NullString);
   end if;

   --tagged types can only be declared in library package specs
   if (IsTagged or else (Extends /= Dictionary.NullSymbol)) and then
     ((Dictionary.GetLibraryPackage (Scope) /= Dictionary.GetRegion (Scope)) or else
     not (Dictionary.IsVisibleScope (Scope) or Dictionary.IsPrivateScope (Scope))) then
      ErrorHandler.SemanticError (828,
                                  ErrorHandler.NoReference,
                                  NodePosition (Node),
                                  LexTokenManager.NullString);

   elsif IsTagged and then
     (not IsPrivateTypeResolution (PrivateTypeBeingResolved, Scope)) and then
     (Dictionary.PackageDeclaresTaggedType (Dictionary.GetRegion (Scope)) or
        Dictionary.PackageExtendsAnotherPackage (Dictionary.GetRegion (Scope))) then

      -- illegal second root tagged type declaration
      ErrorHandler.SemanticError (839,
                                  ErrorHandler.NoReference,
                                  NodePosition (Node),
                                  LexTokenManager.NullString);


   else -- either not tagged type or correctly declared tagged type
      Dictionary.AddRecordType (Name           => IdStr,
                                IsTaggedType   => IsTagged,
                                IsAbstractType => IsAbstract,
                                Extends        => Extends,
                                Declaration    => Dictionary.Location'(DecLoc, DecLoc),
                                Scope          => Scope,
                                Context        => Dictionary.ProgramContext,
                                 --to get
                                TheRecordType  => RecordSym);

      -- if Extends is not null then we need to add in the fields inherited
      -- from the root type
      if Extends /= Dictionary.NullSymbol then
         Dictionary.AddRecordComponent
           (Name                   => LexTokenManager.InheritToken,
            TheRecordType          => RecordSym,
            TheComponentType       => Extends,
            InheritedField         => True,
            ComponentTypeReference => Dictionary.Location'(DecLoc, DecLoc));
      end if;

      -- search for components unaffected by addition of tag info.  If the grammar
      -- is of the form "null record" then no components get found which is correct
      It := FindFirstNode (NodeKind    => SPSymbols.component_declaration,
                           FromRoot    => Node,
                           InDirection => STree.Down);

      while not STree.IsNull (It) loop

         HasFields := True;
         wf_component_declaration (GetNode (It),
                                   RecordSym,
                                   Scope);
         It := STree.NextNode (It);
      end loop;

      -- SPARK disallows null records unless they are tagged (and maybe abstract as well TBD)
      if not (IsTagged or else HasFields or else Extends /= Dictionary.NullSymbol) then
         ErrorHandler.SemanticError (834,
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     LexTokenManager.NullString);
      end if;
   end if;
end wf_record;
