-- $Id: sem-compunit-wf_full_type_declaration-wf_protected_type_declaration.adb 12518 2009-02-19 15:46:30Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.CompUnit.wf_full_type_declaration)
procedure wf_protected_type_declaration (Node  : in STree.SyntaxNode;
                                         Scope : in Dictionary.Scopes)
is
   ProtectedTypeSym            : Dictionary.Symbol;
   Sym                         : Dictionary.Symbol;
   ProtectedScope              : Dictionary.Scopes;
   ProtectedPrivateScope       : Dictionary.Scopes;
   IdentNode                   : STree.SyntaxNode;
   ProtectedElementDeclNode    : STree.SyntaxNode;
   ProtectedElementNode        : STree.SyntaxNode;
   ClosingIdentNode            : STree.SyntaxNode;
   DiscriminantNode            : STree.SyntaxNode;
   IdentStr                    : LexTokenManager.LexString;
   AtLeastOneOperationDeclared : Boolean;
   It                          : STree.Iterator;

   -----------------------------------------------------------------------------------
   -- Node is "protected_type_declaration"
   --
   -- Grammar:
   -- protected_type_declaration
   --            |
   --        identifier -- [known_discriminant_part] -- protected_definition
   --                                                            |
   --                        +-----------------------------------+
   --                        |                 (v--or hidden part------------------------v)
   --       protected_operation_declaration -- protected_element_declaration -- identifier
   --                        |                                   |
   --       (seq of subprogs/entries/pragmas)     (seq of variable declarations or justification statements)
   --
   ------------------------------------------------------------------------------------

   function GetDiscriminantNode (ProtectedTypeDeclarationNode : STree.SyntaxNode)
                                      return STree.SyntaxNode
   --# global in STree.Table;
   is
   begin
      return Next_Sibling (Child_Node (ProtectedTypeDeclarationNode));
   end GetDiscriminantNode;

   ----------

   function GetProtectedOperationsNode (ProtectedTypeDeclarationNode : STree.SyntaxNode)
                                      return STree.SyntaxNode
   --# global in STree.Table;
   is
   begin
      return Child_Node (LastSiblingOf (Child_Node (ProtectedTypeDeclarationNode)));
   end GetProtectedOperationsNode;

   ----------

   function GetProtectedElementDeclarationNode (ProtectedTypeDeclarationNode : STree.SyntaxNode)
                                    return STree.SyntaxNode
   --# global in STree.Table;
   is
   begin
      return Next_Sibling (GetProtectedOperationsNode (ProtectedTypeDeclarationNode));
   end GetProtectedElementDeclarationNode;

   ----------

   function GetClosingIdentNode (ProtectedTypeDeclarationNode : STree.SyntaxNode)
                                return STree.SyntaxNode
   --# global in STree.Table;
   is
   begin
      return LastSiblingOf (GetProtectedOperationsNode (ProtectedTypeDeclarationNode));
   end GetClosingIdentNode;

   ----------

   procedure wf_protected_op_dec (Node    : in     STree.SyntaxNode;
                                  Scope   : in     Dictionary.Scopes;
                                  OpFound :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.StringTable,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                     from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          SPARK_IO.FILE_SYS,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         OpFound                     from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap;
   is separate;

   ----------

   procedure CheckPragmaValidity (EndNodePosition : in LexTokenManager.TokenPosition)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     ProtectedTypeSym;
   --#        in     Scope;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          EndNodePosition,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          LexTokenManager.StringTable,
   --#                                          ProtectedTypeSym,
   --#                                          Scope,
   --#                                          SPARK_IO.FILE_SYS &
   --#         LexTokenManager.StringTable from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ProtectedTypeSym;
   is
      PriorityFound,
      InterruptPriorityFound,
      AttachHandlerFound     : Boolean;
      UnusedValueRep         : LexTokenManager.LexString;
   begin
      PriorityFound          := Dictionary.GetTypeHasPragma (ProtectedTypeSym,
                                                             Dictionary.Priority);
      InterruptPriorityFound := Dictionary.GetTypeHasPragma (ProtectedTypeSym,
                                                             Dictionary.InterruptPriority);
      AttachHandlerFound     := Dictionary.GetTypeHasPragma (ProtectedTypeSym,
                                                             Dictionary.AttachHandler);

      -- There must be either Priority or Interrupt_Priority
      if not (PriorityFound or InterruptPriorityFound) then
         ErrorHandler.SemanticError (876,
                                     ErrorHandler.NoReference,
                                     EndNodePosition,
                                     LexTokenManager.NullString);
      end if;

      -- If there is one or more Attach_Handler there must be Interrupt_Priority
      if AttachHandlerFound then
         --# accept Flow, 10, UnusedValueRep, "Expected ineffective assignment";
         if InterruptPriorityFound then
            CheckPriorityRange
              (ErrorSym   => ProtectedTypeSym,
               Scope      => Scope,
               PragmaKind => Dictionary.AttachHandler,
               ErrPos     => EndNodePosition,
               Value      => Maths.ValueRep (Dictionary.GetTypePragmaValue
                                               (ProtectedTypeSym, Dictionary.InterruptPriority)),
               ValueRep   => UnusedValueRep);

         else
            ErrorHandler.SemanticError (878,
                                        ErrorHandler.NoReference,
                                        EndNodePosition,
                                        LexTokenManager.NullString);
         end if;
         --# end accept;
      end if;
      --# accept Flow, 33, UnusedValueRep, "Expected to be neither referenced nor exported";
   end CheckPragmaValidity;

   ----------

   procedure AddVirtualElements (TypeSym : in Dictionary.Symbol)
   --# global in     LexTokenManager.StringTable;
   --#        in out Dictionary.Dict;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict   from *,
   --#                                TypeSym &
   --#         SPARK_IO.FILE_SYS from *,
   --#                                Dictionary.Dict,
   --#                                LexTokenManager.StringTable,
   --#                                TypeSym;
   is
      It : Dictionary.Iterator;
   begin
      It := Dictionary.FirstVirtualElement (TypeSym);
      while It /= Dictionary.NullIterator loop
         -- Make the virtual element a refinement constituent of the implicit own
         -- variable associated with this protected type.
         Dictionary.AddConstituentSym
           (Dictionary.CurrentSymbol (It),
            Dictionary.GetProtectedTypeOwnVariable (TypeSym),
            -- Dummy locations as not used by examiner
            Dictionary.Location'(LexTokenManager.TokenPosition'(0, 0),
                                 LexTokenManager.TokenPosition'(0, 0)),
            Dictionary.Location'(LexTokenManager.TokenPosition'(0, 0),
                                 LexTokenManager.TokenPosition'(0, 0)));
         It := Dictionary.NextSymbol (It);
      end loop;
   end AddVirtualElements;

   ----------

   procedure AddImplicitInterruptStreamVariables
     (ProtectedTypeSym : in Dictionary.Symbol;
      Scope            : in Dictionary.Scopes;
      ErrorNode        : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict           from *,
   --#                                        ProtectedTypeSym,
   --#                                        Scope &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ErrorNode,
   --#                                        LexTokenManager.StringTable,
   --#                                        ProtectedTypeSym,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      TheOwnVariable : Dictionary.Symbol;
      It             : Dictionary.Iterator;
   begin
      -- Go through all the own variables of this package
      It := Dictionary.FirstOwnVariable (ThePackage => Dictionary.GetRegion (Scope));
      while It /= Dictionary.NullIterator loop
         TheOwnVariable := Dictionary.CurrentSymbol (It);

         -- If the variable is of type ProtectedTypeSym
         if Dictionary.OwnVariableHasType (OwnVariable => TheOwnVariable,
                                           Scope       => Scope) and then
           Dictionary.GetType (TheOwnVariable) = ProtectedTypeSym then

            -- Create interrupt stream variables as necessary
            CreateInterruptStreamVariables
              (ForPO     => TheOwnVariable,
               ErrorNode => ErrorNode);
         end if;
         It := Dictionary.NextSymbol (It);
      end loop;
   end AddImplicitInterruptStreamVariables;

   ----------

   function GetMode (ForType : in LexTokenManager.LexString;
                     Scope   : in Dictionary.Scopes) return Dictionary.Modes
   --# global in Dictionary.Dict;
   is
      It        : Dictionary.Iterator;
      OwnVarSym : Dictionary.Symbol;
      Result    : Dictionary.Modes;
   begin
      Result := Dictionary.DefaultMode;
      -- Go through all the own variables of this package looking for ones
      -- with this type.
      It := Dictionary.FirstOwnVariable (Dictionary.GetRegion (Scope));
      while not Dictionary.IsNullIterator (It) loop
         OwnVarSym := Dictionary.CurrentSymbol (It);
         if Dictionary.OwnVariableHasType (OwnVariable => OwnVarSym,
                                           Scope       => Scope) and then
           ForType = Dictionary.GetSimpleName (Dictionary.GetType (OwnVarSym)) then
            -- Found an own variable of the type. So use its mode.
            -- Note. If the own variable is moded then there can only be
            -- one instance of that that type. So we can exit when we find
            -- the first one.
            Result := Dictionary.GetOwnVariableMode (OwnVarSym);
            exit;
         end if;
         It := Dictionary.NextSymbol (It);
      end loop;
      return Result;
   end GetMode;


   procedure Check_Element_Initialization
     (Variable_Declaration_Node : in STree.SyntaxNode;
      Error_Node                : in STree.SyntaxNode;
      Current_Scope             : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Current_Scope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        Error_Node,
   --#                                        LexTokenManager.StringTable,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        Variable_Declaration_Node;
   is
      It              : STree.Iterator;
      Ident_List_Node : STree.SyntaxNode;
      Ident_Node      : STree.SyntaxNode;
      Ident_Str       : LexTokenManager.LexString;
      Type_Node       : STree.SyntaxNode;
      Exp_Node        : STree.SyntaxNode;
      Sym             : Dictionary.Symbol;
   begin
      Ident_List_Node := Child_Node (Variable_Declaration_Node);
      Type_Node :=  Next_Sibling (Ident_List_Node);
      if SyntaxNodeType (Type_Node) = SPSymbols.RWaliased then
         Type_Node := Next_Sibling (Type_Node);
      end if;
      Exp_Node := Next_Sibling (Type_Node);

      if Exp_Node = STree.NullNode then
         -- Variable(s) ISN'T initialized - we raise an error
         -- at the node corresponsing to the end of the
         -- enclosing protected type declaration, so that the
         -- error may be justified

         -- Variable_Declaration_Node has a list of identifiers below
         -- it, so we need to raise an error for each of them.
         It := FindFirstNode (NodeKind    => SPSymbols.identifier,
                              FromRoot    => Ident_List_Node,
                              InDirection => STree.Down);

         while not STree.IsNull (It) loop
            Ident_Node := GetNode (It);
            Ident_Str := NodeLexString (Ident_Node);
            Sym := Dictionary.LookupItem (Ident_Str,
                                          Current_Scope,
                                          Dictionary.ProgramContext);

            -- If the protected element had a semantic error in its declaration,
            -- then LookupItem might return NullSymbol, so
            if Sym /= Dictionary.NullSymbol then
               ErrorHandler.UsageError (ErrorHandler.UninitializedProtectedElement,
                                        NodePosition (Error_Node),
                                        Sym,
                                        Current_Scope);
            end if;
            It := STree.NextNode (It);
         end loop;

      end if;

   end Check_Element_Initialization;

begin -- wf_protected_type_declaration;
   IdentNode := Child_Node (Node);
   DiscriminantNode := GetDiscriminantNode (Node);
   ClosingIdentNode := GetClosingIdentNode (Node);
   IdentStr := NodeLexString (IdentNode);
   Sym := Dictionary.LookupItem (IdentStr,
                                 Scope,
                                 Dictionary.ProofContext);

   if Sym = Dictionary.NullSymbol or else
     (Dictionary.IsTypeMark (Sym) and then
      Dictionary.TypeIsAnnounced (Sym) and then
      not Dictionary.IsDeclared (Sym))
   then
      Dictionary.AddProtectedType
        (Name          => IdentStr,
         Declaration   => Dictionary.Location'(NodePosition (Node),
                                               NodePosition (Node)),
         Scope         => Scope,
         Context       => Dictionary.ProgramContext,
         Mode          => GetMode (ForType => IdentStr,
                                   Scope   => Scope),
         Constrained   => (SyntaxNodeType (DiscriminantNode) /=
                             SPSymbols.known_discriminant_part),
         -- to get
         ProtectedType => ProtectedTypeSym);

      -- wff discriminants here
      if SyntaxNodeType (DiscriminantNode) = SPSymbols.known_discriminant_part then
         wf_known_discriminant_part (DiscriminantNode,
                                     ProtectedTypeSym,
                                     Scope);
      end if;

      -- wff protected ops
      ProtectedScope := Dictionary.VisibleScope (ProtectedTypeSym);
      wf_protected_op_dec (GetProtectedOperationsNode (Node),
                           ProtectedScope,
                           -- to get
                           AtLeastOneOperationDeclared);


      -- wff protected elements
      ProtectedElementDeclNode := GetProtectedElementDeclarationNode (Node);
      if SyntaxNodeType (ProtectedElementDeclNode) = SPSymbols.protected_element_declaration then
         if GetMode (IdentStr, Scope) /= Dictionary.DefaultMode then
            ErrorHandler.SemanticErrorSym (928,
                                           ErrorHandler.NoReference,
                                           NodePosition (ProtectedElementDeclNode),
                                           ProtectedTypeSym,
                                           Scope);
         else
            -- Element declarations are not hidden so we need to wf them
            -- Grammar ensures there is at least one declaration
            ProtectedPrivateScope := Dictionary.PrivateScope (ProtectedTypeSym);

            ---------------------------------------------------------------------
            -- We check protected elements in 2 passes.
            --
            -- Pass 1 WFFs each variable_declaration or justification_statement
            --
            -- Pass 2 checks each variable for initialization, raising errors
            --        at the "end PT;" node - this allows for initialization
            --        errors to be justified
            ---------------------------------------------------------------------


            -- Pass 1
            It := FindFirstNode (NodeKind    => SPSymbols.protected_element,
                                 FromRoot    => ProtectedElementDeclNode,
                                 InDirection => STree.Down);

            while not STree.IsNull (It) loop
               ProtectedElementNode := Child_Node (GetNode (It));
               case SyntaxNodeType (ProtectedElementNode) is
                  when SPSymbols.variable_declaration =>
                     -- Here we need to distinguish between the scope of the
                     -- enclosing unit and the scope of the declaration, which
                     -- may be different.
                     wf_variable_declaration (Node               => ProtectedElementNode,
                                              EnclosingUnitScope => Scope,
                                              DeclarationScope   => ProtectedPrivateScope);

                  when SPSymbols.justification_statement =>
                     wf_justification_statement (ProtectedElementNode,
                                                 ProtectedPrivateScope);
                  when others =>
                     null;
               end case;
               It := STree.NextNode (It);
            end loop;
            -- end of Pass 1

            -- Pass 2
            It := FindFirstNode (NodeKind    => SPSymbols.protected_element,
                                 FromRoot    => ProtectedElementDeclNode,
                                 InDirection => STree.Down);

            while not STree.IsNull (It) loop
               ProtectedElementNode := Child_Node (GetNode (It));
               case SyntaxNodeType (ProtectedElementNode) is
                  when SPSymbols.variable_declaration =>
                     Check_Element_Initialization (ProtectedElementNode,
                                                   ClosingIdentNode,
                                                   ProtectedPrivateScope);
                  when others =>
                     null;
               end case;
               It := STree.NextNode (It);
            end loop;
            -- end of Pass 2

         end if;
      end if; -- elements not hidden

      -- Add any virtual elements. The virtual elements are the items in the protects list
      -- for this type. They behave as if they were elements of the protected type and
      -- hence must be made constituents of the implicit own variable associated with the
      -- protected type.
      AddVirtualElements (ProtectedTypeSym);

      -- This call will creates any interrupt stream variables specified by local own variables
      -- of ProtectedTypeSym.
      AddImplicitInterruptStreamVariables (ProtectedTypeSym, Scope, Node);

      -- closing identifier must match initial
      if SyntaxNodeType (ClosingIdentNode) = SPSymbols.identifier then
         -- private part is not hidden so we need to check that the closing identifier is correct
         if IdentStr /= NodeLexString (ClosingIdentNode) then
            ErrorHandler.SemanticError (58,
                                        ErrorHandler.NoReference,
                                        NodePosition (ClosingIdentNode),
                                        IdentStr);
         end if;
      else -- must be hidden
         Dictionary.SetProtectedTypeElementsHidden (ProtectedTypeSym);
         ErrorHandler.HiddenText (NodePosition (ClosingIdentNode),
                                  IdentStr,
                                  SPSymbols.protected_type_declaration);
      end if;

      -- protected type must declare at leat one operation
      if not AtLeastOneOperationDeclared then
         ErrorHandler.SemanticError (870,
                                     ErrorHandler.NoReference,
                                     NodePosition (ClosingIdentNode),
                                     IdentStr);
      end if;

      -- there must be a valid combination of pragmas declared in PT
      CheckPragmaValidity (NodePosition (ClosingIdentNode));

   else -- illegal redeclaration
      ErrorHandler.SemanticError (10,
                                  ErrorHandler.NoReference,
                                  NodePosition (IdentNode),
                                  IdentStr);
   end if;

end wf_protected_type_declaration;
