-- $Id: sem-compunit-wf_full_type_declaration-wf_protected_type_declaration-wf_protected_operation_declaration.adb 12351 2009-02-02 15:03:51Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.Compunit.wf_full_type_declaration.wf_protected_type_declaration)
procedure wf_protected_op_dec (Node    : in     STree.SyntaxNode;
                               Scope   : in     Dictionary.Scopes;
                               OpFound :    out Boolean)
is

   -----------------------------------------------------------------------------------
   -- Node is "protected_operation_declaration"
   --
   -- Grammar:
   --
   --       protected_operation_declaration
   --                        |
   --                   priority_pragma --- entry_or_subprogram --- protected_operation_declaration_rep
   --                        |                    |                            |
   --                       ...                  ...            protected_operation_declaration_rep apragma
   --                                                                          |
   --                                                           protected_operation_declaration_rep subprogram_declaration
   --                                                                          |
   --                                                           protected_operation_declaration_rep entry_declaration
   --                                                                          |
   --                                                                         etc ...
   ------------------------------------------------------------------------------------
   -- Overview:
   --
   -- Process intial pragma and subprogram then loops through all the declarations in the visible part of the
   -- protected type and passes each to an appropriate wellformation check
   ------------------------------------------------------------------------------------

   PragmaNode,
   EntryOrSubprogramNode,
   DeclarationSequenceNode,
   NextNode,
   NodeToCheck      : STree.SyntaxNode;
   NodeToCheckType  : SPSymbols.SPSymbol;
   EntryFound       : Boolean;
   NonProofOpFound  : Boolean := False;
   SubprogSym       : Dictionary.Symbol;
   ProtectedTypeSym : Dictionary.Symbol;


   function ValidAdaOp (Sym : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   is
   begin
      return (Sym /= Dictionary.NullSymbol) and then
        Dictionary.IsSubprogram (Sym) and then
        not Dictionary.IsProofFunction (Sym);
   end ValidAdaOp;

   -----------------------------------------------------------------------

   procedure CheckGlobalValidity (SubprogSym     : in Dictionary.Symbol;
                                  ProtectedType  : in Dictionary.Symbol;
                                  DecNode        : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        DecNode,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        ProtectedType,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        SubprogSym;
   is
      GlobalIt           : Dictionary.Iterator;
      GlobalSym          : Dictionary.Symbol;

      function PositionToReportError return LexTokenManager.TokenPosition
      --# global in DecNode;
      --#        in STree.Table;
      is
         ResultNode : STree.SyntaxNode;
      begin
         -- try and find a global anno to report on, if not use subprogram declaration
         ResultNode := Child_Node (Next_Sibling (Child_Node (DecNode)));
         if SyntaxNodeType (ResultNode) /= SPSymbols.moded_global_definition then
            ResultNode := DecNode;
         end if;
         return NodePosition (ResultNode);
      end PositionToReportError;

      function VarIsProtected (Sym : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      --#        in ProtectedType;
      is
      begin
         return Dictionary.GetOwnVariableProtected (Sym) or else
           (Sym = Dictionary.GetProtectedTypeOwnVariable (ProtectedType)) or else
           Sym = Dictionary.GetNullVariable; -- allow "null" in protected op annos
      end VarIsProtected;

   begin -- CheckGlobalValidity
      if ValidAdaOp (SubprogSym) then
         GlobalIt := Dictionary.FirstGlobalVariable (Dictionary.IsAbstract, SubprogSym);
         while not Dictionary.IsNullIterator (GlobalIt) loop
            GlobalSym := Dictionary.CurrentSymbol (GlobalIt);
            if not VarIsProtected (GlobalSym) then
               -- Global variables used in protected operations must be protected.
               ErrorHandler.SemanticErrorSym (872,
                                              ErrorHandler.NoReference,
                                              PositionToReportError,
                                              GlobalSym,
                                              Scope);
            end if;
            GlobalIt := Dictionary.NextSymbol (GlobalIt);
         end loop;
      end if;
   end CheckGlobalValidity;

   -----------------------------------------------------------------------

begin -- wf_protected_operation_declaration
   ProtectedTypeSym := Dictionary.GetRegion (Scope);
   PragmaNode := Child_Node (Node);
   EntryOrSubprogramNode := Next_Sibling (PragmaNode);
   DeclarationSequenceNode := Next_Sibling (EntryOrSubprogramNode);

   -- Process priority pragma
   wf_priority_pragma (PragmaNode, Scope);

   -- Process initial subprogram or entry
   EntryOrSubprogramNode := Child_Node (EntryOrSubprogramNode);
   wf_subprogram_declaration (EntryOrSubprogramNode,
                              Scope,
                              -- to get
                              SubprogSym);

   EntryFound := SyntaxNodeType (EntryOrSubprogramNode) =
     SPSymbols.entry_declaration;

   if EntryFound then
      Dictionary.SetProtectedTypeEntry (ProtectedTypeSym, SubprogSym);
   end if;

   CheckGlobalValidity (SubprogSym,
                        ProtectedTypeSym,
                        EntryOrSubprogramNode);
   NonProofOpFound := NonProofOpFound or ValidAdaOp (SubprogSym);

   -- Process any subsequent sequence of declarations
   NextNode := LastChildOf (DeclarationSequenceNode);
   while NextNode /= DeclarationSequenceNode loop
      NodeToCheck := Next_Sibling (NextNode);
      NodeToCheckType := SyntaxNodeType (NodeToCheck);
      case NodeToCheckType is
         when SPSymbols.apragma =>
            wf_pragma (NodeToCheck,
                       Scope);
         when SPSymbols.subprogram_declaration =>
            wf_subprogram_declaration (NodeToCheck,
                                       Scope,
                                       -- to get
                                       SubprogSym);
            CheckGlobalValidity (SubprogSym,
                                 ProtectedTypeSym,
                                 NodeToCheck);
            NonProofOpFound := NonProofOpFound or ValidAdaOp (SubprogSym);

         when SPSymbols.entry_declaration =>
            if EntryFound then
               ErrorHandler.SemanticError (869,
                                           ErrorHandler.NoReference,
                                           NodePosition (NodeToCheck),
                                           Dictionary.GetSimpleName (Dictionary.GetRegion (Scope)));
            else
               EntryFound := True;
               -- entry declarations are the same "shape" as subprograms and can be handled
               -- by wf_subprog
               wf_subprogram_declaration (NodeToCheck,
                                          Scope,
                                          -- to get
                                          SubprogSym);
               Dictionary.SetProtectedTypeEntry (ProtectedTypeSym, SubprogSym);
               CheckGlobalValidity (SubprogSym,
                                    ProtectedTypeSym,
                                    NodeToCheck);
               NonProofOpFound := NonProofOpFound or ValidAdaOp (SubprogSym);
            end if;
         when others =>
            SystemErrors.FatalError (SystemErrors.InvalidSyntaxTree, "in wf_protected_operation_declaration");
      end case;
      NextNode := ParentNode (NextNode);
   end loop;
   OpFound := NonProofOpFound;
end wf_protected_op_dec;
