-- $Id: sem-compunit-wf_external_interface.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

--Synopsis
--This procedure checks the validity of a pragma interface (Ada83) or pragma
--import (Ada95).  The checks made are:
-- 1.  Internal consistency of associations used, number of parameters etc.
-- 2.  The Entity/Subprogram name is that expected
--------------------------------------------------------------------------------
separate (Sem.CompUnit)
procedure wf_external_interface (PragmaNode : in     STree.SyntaxNode;
                                 EntityName : in     LexTokenManager.LexString;
                                 ErrorFound :    out Boolean)
is

   procedure CheckRepresentSameName (ExpNode : in STree.SyntaxNode;
                                     Name    : in LexTokenManager.LexString)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in     STree.Table;
      --#        in out ErrorFound;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorFound                from *,
      --#                                        ExpNode,
      --#                                        Name,
      --#                                        STree.Table &
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        ExpNode,
      --#                                        LexTokenManager.StringTable,
      --#                                        Name,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table;
   is
      IsChain : Boolean;
      IdNode,
      NextNode : STree.SyntaxNode;
   begin
      IdNode := ExpNode;
      loop
         IsChain := Next_Sibling (IdNode) =
            STree.NullNode;
         NextNode := Child_Node (IdNode);
         exit when not IsChain or NextNode = STree.NullNode;

         IdNode := NextNode;
      end loop;

      if IsChain and then
         SyntaxNodeType (IdNode) = SPSymbols.identifier and then
         NodeLexString (IdNode) = Name
      then
         null;
      else
         ErrorFound := True;
         ErrorHandler.SemanticError (71,
                                     ErrorHandler.NoReference,
                                     NodePosition (ExpNode),
                                     Name);
      end if;
   end CheckRepresentSameName;

   ------------------------------------------------------------------

   procedure wf_pragma_interface (PragmaNode : in STree.SyntaxNode;
                                  EntityName : in LexTokenManager.LexString)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in     STree.Table;
      --#        in out ErrorFound;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorFound                from *,
      --#                                        EntityName,
      --#                                        PragmaNode,
      --#                                        STree.Table &
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        EntityName,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.StringTable,
      --#                                        PragmaNode,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table;
   is
      ArgAssocRepNode : STree.SyntaxNode;
      SubprogNameNode : STree.SyntaxNode;
   begin
      ArgAssocRepNode := Child_Node (Next_Sibling (Child_Node (PragmaNode)));
      if SyntaxNodeType (Child_Node (ArgAssocRepNode)) =
         SPSymbols.argument_association
      then --pragma has two arguments
         SubprogNameNode := Child_Node (Next_Sibling (ArgAssocRepNode));

         if SyntaxNodeType (SubprogNameNode) /=
            SPSymbols.ADA_expression
         then --form of expression wrong
            ErrorFound := True;
            ErrorHandler.SemanticError (71,
                                        ErrorHandler.NoReference,
                                        NodePosition (SubprogNameNode),
                                        EntityName);
         else --form of expression ok so check name actually matches
            CheckRepresentSameName (SubprogNameNode,
                                    EntityName);
         end if;

      else --pragma does nor have exatcly 2 arguments
         ErrorFound := True;
         ErrorHandler.SemanticError (69,
                                     ErrorHandler.NoReference,
                                     NodePosition (PragmaNode),
                                     LexTokenManager.InterfaceToken);
      end if;
   end wf_pragma_interface;

   ----------------------

   procedure wf_pragma_import (PragmaNode : in STree.SyntaxNode;
                               EntityName : in LexTokenManager.LexString)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in     STree.Table;
      --#        in out ErrorFound;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorFound                from *,
      --#                                        EntityName,
      --#                                        PragmaNode,
      --#                                        STree.Table &
      --#         ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        EntityName,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.StringTable,
      --#                                        PragmaNode,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table;
   is
      type Args is (Illegal, Convention, Entity, ExternalName, LinkName);
      subtype LegalArgs is Args range Convention .. LinkName;
      type Founds is array (LegalArgs) of Boolean;

      Found : Founds := Founds'(LegalArgs => False);
      UsingNamedAssociation : Boolean := False;
      ArgAssNode : STree.SyntaxNode;
      ArgCount : Natural := 0;
      MaxArgs  : constant Natural := 4;

      procedure CheckArgument (Node : in STree.SyntaxNode)
         --# global in     ArgCount;
         --#        in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     EntityName;
         --#        in     LexTokenManager.StringTable;
         --#        in     STree.Table;
         --#        in out ErrorFound;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out Found;
         --#        in out SPARK_IO.FILE_SYS;
         --#        in out UsingNamedAssociation;
         --# derives ErrorFound                from *,
         --#                                        ArgCount,
         --#                                        EntityName,
         --#                                        Found,
         --#                                        Node,
         --#                                        STree.Table,
         --#                                        UsingNamedAssociation &
         --#         ErrorHandler.ErrorContext,
         --#         SPARK_IO.FILE_SYS         from ArgCount,
         --#                                        CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        EntityName,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        Found,
         --#                                        LexTokenManager.StringTable,
         --#                                        Node,
         --#                                        SPARK_IO.FILE_SYS,
         --#                                        STree.Table,
         --#                                        UsingNamedAssociation &
         --#         Found                     from *,
         --#                                        ArgCount,
         --#                                        Node,
         --#                                        STree.Table,
         --#                                        UsingNamedAssociation &
         --#         UsingNamedAssociation     from *,
         --#                                        Node,
         --#                                        STree.Table;

      is
         ExpNode : STree.SyntaxNode;
         Arg     : Args;

         function GetArg (ArgString : LexTokenManager.LexString) return Args
         is
            Result : Args;
         begin
            if ArgString = LexTokenManager.ConventionToken then
               Result := Convention;
            elsif ArgString = LexTokenManager.EntityToken then
               Result := Entity;
            elsif ArgString = LexTokenManager.External_NameToken then
               Result := ExternalName;
            elsif ArgString = LexTokenManager.Link_NameToken then
               Result := LinkName;
            else
               Result := Illegal;
            end if;
            return Result;
         end GetArg;

         ---------------

      begin --CheckArgument
         ExpNode := Child_Node (Node);
         --must be an identifier or an ADA_expression
         if SyntaxNodeType (ExpNode) = SPSymbols.identifier then
            --named association
            UsingNamedAssociation := True;
            Arg := GetArg (NodeLexString (ExpNode));
            if Arg = Illegal then
               ErrorFound := True;
               ErrorHandler.SemanticError (601,
                                           ErrorHandler.NoReference,
                                           NodePosition (ExpNode),
                                           LexTokenManager.NullString);
            elsif Found (Arg) then
               ErrorFound := True;
               ErrorHandler.SemanticError (602,
                                           ErrorHandler.NoReference,
                                           NodePosition (ExpNode),
                                           NodeLexString (ExpNode));
            else
               Found (Arg) := True;
               if Arg = Entity then
                  ExpNode := Next_Sibling (ExpNode);
                  CheckRepresentSameName (ExpNode, EntityName);
               end if;
            end if;

         else --positional association
            if UsingNamedAssociation then --illegal switch form named to positional assoc
               ErrorFound := True;
               ErrorHandler.SemanticError (601,
                                           ErrorHandler.NoReference,
                                           NodePosition (ExpNode),
                                           LexTokenManager.NullString);
            else
               Arg := Args'Val (ArgCount);
               Found (Arg) := True;
               if Arg = Entity then
                  CheckRepresentSameName (ExpNode, EntityName);
               end if;
            end if;
         end if;
      end CheckArgument;

      ----------------------

   begin --wf_pragma_import
      ArgAssNode := Next_Sibling (Child_Node (PragmaNode));
      --should be top of argument_asociation_rep chain
      if SyntaxNodeType (ArgAssNode) =
         SPSymbols.argument_association_rep
      then
         while SyntaxNodeType (ArgAssNode) /=
            SPSymbols.argument_association
         loop
            ArgAssNode := Child_Node (ArgAssNode);
         end loop;
         --now pointing at leftmost argument association
         while ArgAssNode /= STree.NullNode
         loop
            if ArgCount = MaxArgs then
               ErrorFound := True;
               ErrorHandler.SemanticError (600,
                                           ErrorHandler.NoReference,
                                           NodePosition (ArgAssNode),
                                           LexTokenManager.NullString);
               exit;
            end if;
            ArgCount := ArgCount + 1;
            CheckArgument (ArgAssNode);
            ArgAssNode := Next_Sibling (ParentNode (ArgAssNode));
         end loop;
         if ArgCount < 2 then
            ErrorFound := True;
            ErrorHandler.SemanticError (600,
                                        ErrorHandler.NoReference,
                                        NodePosition (PragmaNode),
                                        LexTokenManager.ConventionToken);

         else
            if not Found (Convention) then
               ErrorFound := True;
               ErrorHandler.SemanticError (603,
                                           ErrorHandler.NoReference,
                                           NodePosition (PragmaNode),
                                           LexTokenManager.ConventionToken);
            end if;
            if not Found (Entity) then
               ErrorFound := True;
               ErrorHandler.SemanticError (603,
                                           ErrorHandler.NoReference,
                                           NodePosition (PragmaNode),
                                           LexTokenManager.EntityToken);
            end if;
         end if;
      else --there are no arguments
         ErrorFound := True;
         ErrorHandler.SemanticError (600,
                                     ErrorHandler.NoReference,
                                     NodePosition (PragmaNode),
                                     LexTokenManager.NullString);
      end if;
   end wf_pragma_import;

   ----------------------

begin --wf_external_interface
   ErrorFound := False;
   if CommandLineData.IsSpark83 then
      wf_pragma_interface (PragmaNode, EntityName);
   else
      wf_pragma_import (PragmaNode, EntityName);
   end if;
end wf_external_interface;
