-- $Id: sem-compunit-wf_expression.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

-- Overview: Called to check validity of an
-- expression node.  Replaces calls to StaticExpression,
-- BaseTypeExpression and CheckTypeExpression
----------------------------------------------------------------------------

separate (Sem.CompUnit)

procedure wf_expression (Node         : in     STree.SyntaxNode;
                         Scope        : in     Dictionary.Scopes;
                         EStack       : in out ExpStack.ExpStackType;
                         TStack       : in     TypeContextStack.TStackType;
                         IsAnnotation : in     Boolean)
is
   OpNode   : STree.SyntaxNode;
   Operator : SPSymbols.SPSymbol;
   Left,
   Right,
   Result   : ExpRecord;
   LeftPos  : LexTokenManager.TokenPosition;
   RightPos : LexTokenManager.TokenPosition;

   ---------------------------------------------------------------------

   procedure CheckShortCircuit (Op     : in     SPSymbols.SPSymbol;
                                Node   : in     STree.SyntaxNode;
                                OpPos  : in     LexTokenManager.TokenPosition;
                                EStack : in out ExpStack.ExpStackType)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     IsAnnotation;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        Op,
   --#                                        OpPos,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         EStack                    from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        LexTokenManager.StringTable,
   --#                                        Op;
   is
      Left,
      Right,
      Result : ExpRecord;

   begin
      ExpStack.Pop (Right, EStack);
      ExpStack.Pop (Left,  EStack);
      Result := NullTypeRecord; -- safety: we may not set all fields below

      Result.IsStatic   := CommandLineData.IsSpark95 and then
         Left.IsStatic and then
         Right.IsStatic;
      Result.IsConstant := Left.IsConstant and Right.IsConstant;
      Result.HasOperators := True;
      if (Dictionary.IsBooleanTypeMark (Left.TypeSymbol) and
          Dictionary.IsBooleanTypeMark (Right.TypeSymbol))
      then
         Result.IsARange := False;
         Result.TypeSymbol  := Left.TypeSymbol;
         CalcBinaryOperator (Node,
                             Op,
                             Left.Value,
                             Right.Value,
                             IsAnnotation,
                              --using and to get
                             Result);
      else
         Result := UnknownTypeRecord;
         ErrorHandler.SemanticError (35,
                                     ErrorHandler.NoReference,
                                     OpPos,
                                     LexTokenManager.NullString);
      end if;

      Result.ErrorsInExpression := Result.ErrorsInExpression or
         Left.ErrorsInExpression or
         Right.ErrorsInExpression;
      ExpStack.Push (Result, EStack);

   end CheckShortCircuit;

   -----------------------------------------------------------------

   procedure CheckImplication (OpPos  : in     LexTokenManager.TokenPosition;
                               EStack : in out ExpStack.ExpStackType)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        LexTokenManager.StringTable,
   --#                                        OpPos,
   --#                                        SPARK_IO.FILE_SYS &
   --#         EStack                    from *,
   --#                                        Dictionary.Dict;
   is
      Left,
      Right,
      Result : ExpRecord;

   begin
      ExpStack.Pop (Right, EStack);
      ExpStack.Pop (Left,  EStack);
      Result := NullTypeRecord; -- safety: we may not set all fields below

      Result.IsStatic := Left.IsStatic and Right.IsStatic;
      Result.IsConstant := Left.IsConstant and Right.IsConstant;
      Result.HasOperators := True;
      if (Dictionary.IsBooleanTypeMark (Left.TypeSymbol) and
          Dictionary.IsBooleanTypeMark (Right.TypeSymbol))
      then
         Result.TypeSymbol  := Left.TypeSymbol;
      else
         Result := UnknownTypeRecord;
         ErrorHandler.SemanticError (35,
                                     ErrorHandler.NoReference,
                                     OpPos,
                                     LexTokenManager.NullString);
      end if;

      Result.ErrorsInExpression := Result.ErrorsInExpression or
         Left.ErrorsInExpression or
         Right.ErrorsInExpression;
      ExpStack.Push (Result, EStack);

   end CheckImplication;

   -----------------------------------------------------------------

begin   --wf_expression
   OpNode := Next_Sibling (Child_Node (Node));

   if OpNode /= STree.NullNode then
      --expression_rep exists
      Operator := SyntaxNodeType (OpNode);
      if Operator = SPSymbols.RWandthen or
         Operator = SPSymbols.RWorelse
      then
         CheckShortCircuit (Operator,
                            Node,
                            NodePosition (OpNode),
                            EStack);
      elsif IsAnnotation and then
         (Operator = SPSymbols.implies or
          Operator = SPSymbols.is_equivalent_to)
      then --   -> or <->
         CheckImplication (NodePosition (OpNode), EStack);

      else -- must be "and", "or" or "xor"
         LeftPos  := NodePosition (Child_Node (Node));
         RightPos := NodePosition (Next_Sibling (OpNode));

         ExpStack.Pop (Right, EStack);
         ExpStack.Pop (Left,  EStack);
         Result := NullTypeRecord; -- safety: we may not set all fields below
         Result.IsConstant := Left.IsConstant and Right.IsConstant;
         Result.IsStatic   := Left.IsStatic and Right.IsStatic;
         Result.HasOperators := True;
         CheckBinaryOperator (Operator          => Operator,
                              Left              => Left,
                              Right             => Right,
                              Scope             => Scope,
                              TStack            => TStack,
                              OpPos             => NodePosition (OpNode),
                              LeftPos           => LeftPos,
                              RightPos          => RightPos,
                              -- Implicit type conv so that modular logical ops with
                              -- literal operand work
                              Convert           => True,
                              IsAnnotation      => IsAnnotation,
                              -- using and to get
                              Result            => Result);

         -- check that array bounds match.
         if Result /= UnknownTypeRecord then
            -- check that whole array operation not being performed on unconstrained array
            if Dictionary.IsUnconstrainedArrayType (Left.TypeSymbol) or else
               Dictionary.IsUnconstrainedArrayType (Right.TypeSymbol)
            then
               Result := UnknownTypeRecord;
               ErrorHandler.SemanticError (39,
                                           ErrorHandler.NoReference,
                                           NodePosition (OpNode),
                                           LexTokenManager.NullString);

            elsif Illegal_Unconstrained (Left.TypeSymbol, Right.TypeSymbol) then
               Result := UnknownTypeRecord;
               ErrorHandler.SemanticError (418,
                                           ErrorHandler.NoReference,
                                           NodePosition (OpNode),
                                           LexTokenManager.NullString);
            end if;
            if Result /= UnknownTypeRecord then
               CalcBinaryOperator (Node,
                                   Operator,
                                   Left.Value,
                                   Right.Value,
                                   IsAnnotation,
                                    --using and to get
                                   Result);
            end if;
         end if;

         -- test to prevent result being considered unconstrained
         if Dictionary.TypeIsArray (Result.TypeSymbol) then
            Result.TypeSymbol := Left.TypeSymbol;
         end if;

         -- Plant result type for use by VCG
         -- It will be used to identify cases where a special model is needed for bitwise ops
         -- between arrays or modular types
         STree.AddNodeSymbol (OpNode, Result.TypeSymbol);

         Result.ErrorsInExpression := Result.ErrorsInExpression or
            Left.ErrorsInExpression or
            Right.ErrorsInExpression;

         -- OtherSymbol may carry a function symbol in the case of uses of unchecked_conversion.
         -- This symbol is used (by wf_Assign) to convery information to the VCG to supress
         -- checks when an unchecked_conversion is assigned to something of the same subtype.
         -- We do not want this mechanism if the unchecked_conversion is sued in any other context
         -- than a direct assignment.  Therefore we clear OtherSymbol here:
         Result.OtherSymbol := Dictionary.NullSymbol;
         ExpStack.Push (Result, EStack);

      end if;
   end if;

end wf_expression;
