-- $Id: sem-compunit-wf_entire_variable.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.CompUnit)

procedure wf_entire_variable (Node         : in     STree.SyntaxNode;
                              CurrentScope : in     Dictionary.Scopes;
                              ErrorHint    : in     Visibility_Error_Hint;
                              VarSym       :    out Dictionary.Symbol;
                              Dotted       :    out Boolean)
is
   IdNode,
   NextIdNode : STree.SyntaxNode;
   It         : STree.Iterator;
   PIdStr,
   IdStr      : LexTokenManager.LexString;
   Sym,
   SymSoFar   : Dictionary.Symbol;
   PrefixOk   : Boolean;

begin
   -- ASSUME Node is the root of any variable (dotted or otherwise).
   --
   -- This procedure traverses (depth first) the tree under the root looking for
   -- identifiers. VarSym is made up of all the identifiers found.
   --
   -- Current uses are:
   -- Node = entire_variable
   -- Node = primary.

   It := FindFirstNode (NodeKind    => SPSymbols.identifier,
                        FromRoot    => Node,
                        InDirection => STree.Down);

   IdNode := GetNode (It);
   IdStr := NodeLexString (IdNode);
   PIdStr := LexTokenManager.NullString;
   Sym := Dictionary.LookupItem (IdStr,
                                 CurrentScope,
                                 Dictionary.ProofContext);
   loop
      if Sym = Dictionary.NullSymbol then

         case ErrorHint is
            when No_Hint =>
               ErrorHandler.SemanticError2 (1,
                                            ErrorHandler.NoReference,
                                            NodePosition (Node),
                                            IdStr,
                                            PIdStr);
            when In_Global_List =>
               ErrorHandler.SemanticError2 (144,
                                            ErrorHandler.NoReference,
                                            NodePosition (Node),
                                            IdStr,
                                            PIdStr);
            when In_Derives_Import_List =>
               ErrorHandler.SemanticError2 (752,
                                            ErrorHandler.NoReference,
                                            NodePosition (Node),
                                            IdStr,
                                            PIdStr);
            when In_Derives_Export_List =>
               ErrorHandler.SemanticError2 (753,
                                            ErrorHandler.NoReference,
                                            NodePosition (Node),
                                            IdStr,
                                            PIdStr);
            when In_Suspends_List =>
               ErrorHandler.SemanticError2 (755,
                                            ErrorHandler.NoReference,
                                            NodePosition (Node),
                                            IdStr,
                                            PIdStr);
         end case;

         exit;
      end if;

      It := STree.NextNode (It);
      NextIdNode := GetNode (It);

      if Dictionary.IsVariable (Sym) then
         -- at this point Sym is a variable, final check that there is no dotted
         -- part to the right of it as there would be if a record field was there
         if NextIdNode /= STree.NullNode then
            ErrorHandler.SemanticError (156, --entire var expected
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        LexTokenManager.NullString);
            Sym := Dictionary.NullSymbol;
         end if;
         exit;
      end if;

      if Dictionary.IsConstant (Sym) then
         ErrorHandler.SemanticError (150, --entire var expected not constant
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     LexTokenManager.NullString);
         Sym := Dictionary.NullSymbol;
         exit;
      end if;

      if not Dictionary.IsPackage (Sym) then
         ErrorHandler.SemanticError (156, --entire var expected
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     LexTokenManager.NullString);
         Sym := Dictionary.NullSymbol;
         exit;
      end if;

      if NextIdNode = STree.NullNode then
         -- package without a selected component
         ErrorHandler.SemanticError (156, --entire var expected
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     LexTokenManager.NullString);
         Sym := Dictionary.NullSymbol;
         exit;
      end if;

      CheckPackagePrefix (IdNode,
                          Sym,
                          CurrentScope,
                           --to get
                          PrefixOk);
      if not PrefixOk then
         Sym := Dictionary.NullSymbol;
         exit;
      end if;

      PIdStr := IdStr;
      IdNode := NextIdNode;
      IdStr := NodeLexString (IdNode);
      SymSoFar := Sym;
      Sym := Dictionary.LookupSelectedItem (Sym,
                                            IdStr,
                                            CurrentScope,
                                            Dictionary.ProofContext);
      -- check to see if we are getting the same symbol over and again
      if Sym = SymSoFar then            -- P.P.P.P.X case
         Sym := Dictionary.NullSymbol;  -- to cause "Not visible" error at top of loop
      end if;

   end loop;

   Dotted := PIdStr /= LexTokenManager.NullString and then
      Sym /= Dictionary.NullSymbol;
   VarSym := Sym;
end wf_entire_variable;
