-- $Id: sem-compunit-wf_context_clause-with_clause.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

-- Overview:
-- Checks a with clause for Sem starting at node with_clause.
-- Directly capable of raising errors for: undeclared item in with list,
-- duplicate item in with list or withing of something which is not a
-- package.
--
-- NB.  In present form permits with for something not inherited; this is
--      necessary for withing something to be used solely in hidden part
--      (eg. text_io by spark_io).  However, we wish to issue a
--      semantic warning in such circumstances.
--      It is also necessary to with something not inherited in the case
--      where an inherit cannot be placed; for example where a package
--      body withs a private child package.
--------------------------------------------------------------------------------
-- with Debug;

separate (Sem.CompUnit.wf_context_clause)

procedure with_clause (Node    : in STree.SyntaxNode;
                       CompSym : in Dictionary.Symbol;
                       Scope   : in Dictionary.Scopes)
is
   It       : STree.Iterator;
   --NextNode   : STree.SyntaxNode;

   -----------------------------

   procedure ProcessDottedSimpleName (Node : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     CompSym;
   --#        in     LextokenManager.StringTable;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict           from *,
   --#                                        CommandLineData.Content,
   --#                                        CompSym,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        CompSym,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LextokenManager.StringTable,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
      PrefixSym         : Dictionary.Symbol := Dictionary.NullSymbol;
      CurrentSym        : Dictionary.Symbol;
      CurrentNode       : STree.SyntaxNode;
      ExplicitDuplicate : Boolean;
      WithingDescendent : Boolean := False;
      Discard           : Boolean;
      LibSym            : Dictionary.Symbol;
      SearchString      : LexTokenManager.LexString;

      ----------------------------

      function DottedIdentifierFound return Boolean
      --# global in Node;
      --#        in STree.Table;
         --  pre Node is top-most dotted_simple_name node;
      is
      begin
         return SyntaxNodeType (Child_Node (Node)) =
            SPSymbols.dotted_simple_name;
      end DottedIdentifierFound;

      -------------------

      function IsLastIdentifierNode (Node : STree.SyntaxNode) return Boolean
      --# global in STree.Table;
         --  pre    SyntaxNodeType (Node) = SPSymbols.identifier;
      is
      begin
         return SyntaxNodeType (ParentNode (ParentNode (Node))) /=
            SPSymbols.dotted_simple_name;
      end IsLastIdentifierNode;

      --------------------

      function LookUp (Prefix  : in     Dictionary.Symbol;
                       Str     : in     LexTokenManager.LexString;
                       Scope   : in     Dictionary.Scopes) return Dictionary.Symbol
      --# global in CommandLineData.Content;
      --#        in Dictionary.Dict;
      is
         Sym : Dictionary.Symbol;
      begin
         if Prefix = Dictionary.NullSymbol then
            Sym := Dictionary.LookupItem (Str,
                                          Scope,
                                          Dictionary.ProofContext);

         else
            Sym := Dictionary.LookupSelectedItem (Prefix,
                                                  Str,
                                                  Scope,
                                                  Dictionary.ProofContext);
         end if;

         return Sym;
      end LookUp;

      --------------------

      procedure CheckDescendent (Sym     : in out Dictionary.Symbol;
                                 LibSym  : in     Dictionary.Symbol)
      --# global in     Dictionary.Dict;
      --#           out WithingDescendent;
      --# derives Sym,
      --#         WithingDescendent from Dictionary.Dict,
      --#                                LibSym,
      --#                                Sym;
      is
      begin
         if Dictionary.GetPackageOwner (Sym) /= LibSym then
            Sym := Dictionary.NullSymbol;
            WithingDescendent := False;
         else
            WithingDescendent := True;
         end if;
      end CheckDescendent;

      --------------------

   begin --ProcessDottedSimpleName
      if DottedIdentifierFound and then CommandLineData.IsSpark83 then
         ErrorHandler.SemanticError (610,
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     LexTokenManager.NullString);
      else
         LibSym := Dictionary.GetLibraryPackage (Dictionary.VisibleScope (CompSym));
         CurrentNode := LastChildOf (Node); --first prefix identifier
         loop
            -- look up in global scope first:
            SearchString := NodeLexString (CurrentNode);
            CurrentSym := LookUp (PrefixSym,
                                  SearchString,
                                  Dictionary.GlobalScope);

            if CurrentSym /= Dictionary.NullSymbol and then
               CurrentSym /= LibSym and then
               Dictionary.IsPackage (CurrentSym)
            then -- package exists and is not self
               -- if necessary, check inherited by looking up in current scope
               if Dictionary.IsProperDescendent (CurrentSym, LibSym) then
                  -- withing a descendent
                  CheckDescendent (CurrentSym, LibSym);
               elsif Dictionary.IsProperDescendent (LibSym, CurrentSym) then
                  -- withing an ancestor
                  if IsLastIdentifierNode (CurrentNode) then
                     CurrentSym := LookUp (Dictionary.NullSymbol,
                                           SearchString,
                                           Scope);
                  end if;
               elsif PrefixSym /= Dictionary.NullSymbol and then
                  Dictionary.IsProperDescendent (LibSym, PrefixSym)
               then  -- withing child of ancestor
                  CurrentSym := LookUp (Dictionary.NullSymbol,
                                        SearchString,
                                        Scope);
               else
                  CurrentSym := LookUp (PrefixSym,
                                        SearchString,
                                        Scope);
               end if;
            end if;

            if CurrentSym /= Dictionary.NullSymbol and then
              not Dictionary.IsPackage (CurrentSym) and then
              not Dictionary.IsGenericSubprogram (CurrentSym)
            then -- can't be inherited
               ErrorHandler.SemanticError (18,
                                           ErrorHandler.NoReference,
                                           NodePosition (CurrentNode),
                                           SearchString);
               exit;
            end if;

            if IsLastIdentifierNode (CurrentNode) and then
               CurrentSym = LibSym
            then --trying to with self (or enclosing package)
               ErrorHandler.SemanticError (132,
                                           ErrorHandler.NoReference,
                                           NodePosition (CurrentNode),
                                           SearchString);
               exit;
            end if;

            -- extra check for private root packages,
            -- which cannot be with'd by specs of public packages:
            if CurrentSym /= Dictionary.NullSymbol and then
              Dictionary.IsPackage (CurrentSym) and then -- guard for precon of next line
              Dictionary.IsPrivatePackage (CurrentSym) and then
              Dictionary.GetPackageParent (CurrentSym) = Dictionary.NullSymbol
              and then
              Dictionary.IsVisibleScope (Scope) and then
              Dictionary.GetPackageOwner (LibSym) = Dictionary.NullSymbol and then
              not Dictionary.IsPrivatePackage (Dictionary.GetRootPackage (LibSym))
            then
               ErrorHandler.SemanticError (616,
                                           ErrorHandler.NoReference,
                                           NodePosition (CurrentNode),
                                           SearchString);
               exit;
            end if;

            -- Look up will find generic functions because they have an associated
            -- proof function in proof context.  We want to exclude them unless they have
            -- also been inherited.
            if Dictionary.IsGenericSubprogram (CurrentSym) and then
              not Dictionary.IsInherited (CurrentSym, CompSym) then
               CurrentSym := Dictionary.NullSymbol;
            end if;

            if CurrentSym = Dictionary.NullSymbol then
               if CommandLineData.RavenscarSelected and then
                 SearchString /= LexTokenManager.AdaToken and then
                 SearchString /= LexTokenManager.SystemToken then
                  -- stronger warning for uninherited withs of non-predefined packages in Ravenscar
                  ErrorHandler.SemanticWarning (391,
                                                NodePosition (CurrentNode),
                                                SearchString);
               else
                  ErrorHandler.SemanticWarning (1,
                                                NodePosition (CurrentNode),
                                                SearchString);
               end if;
               exit;
            end if;

            -- check sym found is not a local redeclaration
            if not Dictionary.IsGlobalScope (Dictionary.GetScope (CurrentSym)) then
               -- This semantic error has not been checked with new error number
               -- because unable to find test case which causes the error.
               ErrorHandler.SemanticError (133,
                                           ErrorHandler.NoReference,
                                           NodePosition (CurrentNode),
                                           SearchString);
               exit;
            end if;

            --Debug.PrintSym ("SYM: ", CurrentSym);

            --there is something to add because symbol is not null
            Dictionary.AddWithReference (Scope,
                                         CurrentSym,
                                         IsLastIdentifierNode (CurrentNode),
                                         Dictionary.Location'(NodePosition (CurrentNode),
                                                              NodePosition (CurrentNode)),
                                          --to get
                                         ExplicitDuplicate);

            -- handle the case of a with for a descendent package, which
            -- causes problems with visibility for nested child packages.
            -- only relevant for private child packages, as public child
            -- packages will not have been found by the earlier lookup

            -- Add a 'fake inherit' as well as the 'with'
            if WithingDescendent then
               --# accept Flow, 10, Discard, "Expected ineffective assignment";
               Dictionary.AddInheritsReference -- Discard is unused
                 (CompSym,
                  CurrentSym,
                  False, -- fake, so definitely implicit!
                  Dictionary.Location'(NodePosition (CurrentNode),
                                     NodePosition (CurrentNode)),
                  --to get
                  Discard);   -- can never be explicitly duplicated, as is
                              -- only ever implicit...
               --# end accept;
            end if;

            if ExplicitDuplicate then
               ErrorHandler.SemanticErrorSym (191,
                                              ErrorHandler.NoReference,
                                              NodePosition (Node),
                                              CurrentSym,
                                              Dictionary.GlobalScope);
            end if;

            exit when IsLastIdentifierNode (CurrentNode);

            PrefixSym := CurrentSym; --ready for next lookup

            CurrentNode := Next_Sibling (ParentNode (CurrentNode));
         end loop;
      end if;
      --# accept Flow, 33, Discard, "Expected to be neither referenced nor exported";
   end ProcessDottedSimpleName;

   -----------------------------

begin  --with_clause
   Dictionary.AddWithClause (Scope,
                             Dictionary.Location'(NodePosition (Node),
                                                  NodePosition (Node)));

   It := FindFirstNode (NodeKind    => SPSymbols.dotted_simple_name,
                        FromRoot    => Node,
                        InDirection => STree.Down);

   while not STree.IsNull (It) loop --for each identifier in with list
      ProcessDottedSimpleName (GetNode (It));
      It := STree.NextNode (It);
   end loop;

end with_clause;
