-- $Id: sem-compunit-wf_constant_declaration.adb 11946 2008-12-18 16:11:11Z rod chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with Debug;
separate (Sem.CompUnit)

procedure wf_constant_declaration (Node         : in STree.SyntaxNode;
                                   CurrentScope : in Dictionary.Scopes)
is
   IdentNode,
   TypeNode,
   ExpNode     : STree.SyntaxNode;
   TypeSym     : Dictionary.Symbol;
   ExpType     : ExpRecord;
   UnwantedSeq : SeqAlgebra.Seq;
   StoreVal    : LexTokenManager.LexString;
   UnusedComponentData : ComponentManager.ComponentData;

   -------------------------------------------------------------------------

   function ValidNamedNumberType (Sym : Dictionary.Symbol) return Dictionary.Symbol
   --# global in CommandLineData.Content;
   --#        in Dictionary.Dict;
   --This returns universal_real or universal_integer if a valid (according to tool
   --variant) type is supplied or the unknown type mark if it not
   is
      Result : Dictionary.Symbol;
   begin
      if CommandLineData.IsSpark83 then
         if Dictionary.IsUniversalIntegerType (Sym) then
            Result := Sym;
         elsif Dictionary.IsUniversalRealType (Sym) then
            Result := Sym;
         else
            Result := Dictionary.GetUnknownTypeMark;
         end if;

      else --Ada95 rules apply
         if Dictionary.TypeIsInteger (Sym) or
            Dictionary.TypeIsModular (Sym) then
            Result := Dictionary.GetUniversalIntegerType;
         elsif Dictionary.TypeIsReal (Sym) then
            Result := Dictionary.GetUniversalRealType;
         else
            Result := Dictionary.GetUnknownTypeMark;
         end if;
      end if;
      return Result;
   end ValidNamedNumberType;

   -------------------------------------------------------------------------

   procedure DoIdentifierList (Node,
                               TypeNode        : in STree.SyntaxNode;
                               TypeSym         : in Dictionary.Symbol;
                               CurrentScope    : in Dictionary.Scopes;
                               ExpIsWellFormed : in Boolean;
                               Static          : in Boolean)
   --# global in     CommandLineData.Content;
   --#        in     ExpNode;
   --#        in     LexTokenManager.StringTable;
   --#        in     StoreVal;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Dictionary.Dict           from *,
   --#                                        CommandLineData.Content,
   --#                                        CurrentScope,
   --#                                        ExpIsWellFormed,
   --#                                        ExpNode,
   --#                                        Node,
   --#                                        Static,
   --#                                        StoreVal,
   --#                                        STree.Table,
   --#                                        TypeSym &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        CurrentScope,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        ExpIsWellFormed,
   --#                                        ExpNode,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        Static,
   --#                                        StoreVal,
   --#                                        STree.Table,
   --#                                        TypeNode,
   --#                                        TypeSym;
   is
      NextNode       : STree.SyntaxNode;
      It             : STree.Iterator;
      IdentStr       : LexTokenManager.LexString;
      Sym            : Dictionary.Symbol;
      OkToAdd        : Boolean;
      TypeLocation,
      IdentLocation  : Dictionary.Location;

      -------------------------------

      function IsDeferredConstantResolution (Sym   : Dictionary.Symbol;
                                             Scope : Dictionary.Scopes) return Boolean
      --# global in Dictionary.Dict;
      is
      begin
         return (not Dictionary.IsDeclared (Sym) and then
                 Dictionary.IsPrivateScope (Scope) and then
                 Dictionary.IsDeferredConstant (Sym) and then
                 (Dictionary.GetRegion (Scope) =
                  Dictionary.GetRegion (Dictionary.GetScope (Sym))));
      end IsDeferredConstantResolution;

      -------------------------------

   begin  --DoIdentifierList
      OkToAdd := False;
      TypeLocation := Dictionary.Location'(NodePosition (TypeNode),
                                           NodePosition (TypeNode));

      It := FindFirstNode (NodeKind    => SPSymbols.identifier,
                           FromRoot    => Node,
                           InDirection => STree.Down);

      while not STree.IsNull (It) loop

         NextNode := GetNode (It);
         IdentStr := NodeLexString (NextNode);
         Sym := Dictionary.LookupItem (IdentStr,
                                       CurrentScope,
                                       Dictionary.ProofContext);

         if Sym = Dictionary.NullSymbol then

            OkToAdd := True;

         elsif IsDeferredConstantResolution (Sym, CurrentScope) then

            if TypeSym = Dictionary.GetType (Sym) then
               OkToAdd := True;
            else
               ErrorHandler.SemanticError (22,
                                           ErrorHandler.NoReference,
                                           NodePosition (NextNode),
                                           IdentStr);
            end if;
         else -- already exists but is not a deferred constant completion

            if Dictionary.IsOwnVariable (Sym) or
              Dictionary.IsConstituent (Sym) then
               -- A common mistake - trying to complete an own variable with
               -- a constant declaration.  Spot this to give a better error
               -- message here.
               ErrorHandler.SemanticError (12,
                                           ErrorHandler.NoReference,
                                           NodePosition (NextNode),
                                           IdentStr);
            else
               ErrorHandler.SemanticError (10,
                                           ErrorHandler.NoReference,
                                           NodePosition (NextNode),
                                           IdentStr);
            end if;

         end if;

         if OkToAdd then
            IdentLocation := Dictionary.Location'(NodePosition (NextNode),
                                                  NodePosition (NextNode));
            if Dictionary.IsUnknownTypeMark (TypeSym) or else
               Dictionary.IsScalarTypeMark (TypeSym, CurrentScope)
            then
               Dictionary.AddScalarConstant (IdentStr,
                                             TypeSym,
                                             TypeLocation,
                                             StoreVal,
                                             ExpIsWellFormed,
                                             STree.NodeToRef (ExpNode),
                                             Static,
                                             IdentLocation,
                                             CurrentScope,
                                             Dictionary.ProgramContext);
            elsif Dictionary.IsArrayTypeMark (TypeSym, CurrentScope) then
               Dictionary.AddArrayConstant (IdentStr,
                                            TypeSym,
                                            TypeLocation,
                                            ExpIsWellFormed,
                                            STree.NodeToRef (ExpNode),
                                            Static,
                                            IdentLocation,
                                            CurrentScope,
                                            Dictionary.ProgramContext);
            else -- must be record
               Dictionary.AddRecordConstant (IdentStr,
                                             TypeSym,
                                             TypeLocation,
                                             ExpIsWellFormed,
                                             STree.NodeToRef (ExpNode),
                                             IdentLocation,
                                             CurrentScope,
                                             Dictionary.ProgramContext);
            end if;
         end if;
         It := STree.NextNode (It);
      end loop;
   end DoIdentifierList;

   -------------------------------------------------------------------------

begin  -- wf_constant_declaration
   -- ASSUME Node = constant_declaration

   Heap.Initialize (TheHeap);
   IdentNode := Child_Node (Node);
   TypeNode :=  Next_Sibling (IdentNode);
   if SyntaxNodeType (TypeNode) = SPSymbols.expression then
      ExpNode := TypeNode;
      TypeNode := STree.NullNode;
      TypeSym := Dictionary.GetUnknownTypeMark;
   else
      ExpNode  :=  Next_Sibling (TypeNode);
      wf_type_mark (TypeNode,
                    CurrentScope,
                    Dictionary.ProgramContext,
                     --to get
                    TypeSym);
      if Dictionary.IsUnconstrainedArrayType (TypeSym) and then
        not Dictionary.IsPredefinedStringType (TypeSym) then -- allow string constants
         ErrorHandler.SemanticError (39,
                                     ErrorHandler.NoReference,
                                     NodePosition (TypeNode),
                                     LexTokenManager.NullString);
      elsif Dictionary.IsPredefinedSuspensionObjectType (TypeSym) or
        Dictionary.TypeIsProtected (TypeSym) then
         ErrorHandler.SemanticError (903,
                                     ErrorHandler.NoReference,
                                     NodePosition (TypeNode),
                                     LexTokenManager.NullString);
      elsif Dictionary.TypeIsGeneric (TypeSym) then
         ErrorHandler.SemanticError (653,
                                     ErrorHandler.NoReference,
                                     NodePosition (TypeNode),
                                     LexTokenManager.NullString);
      end if;
   end if;

   SeqAlgebra.CreateSeq (TheHeap, UnwantedSeq);
   ComponentManager.Initialise (UnusedComponentData);
   --# accept Flow, 10, UnusedComponentData, "Expected ineffective assignment";
   WalkExpression (ExpNode               => ExpNode,
                   Scope                 => CurrentScope,
                   TypeContext           => TypeSym,
                   ContextRequiresStatic => False,
                   Result                => ExpType,
                   RefVar                => UnwantedSeq,
                   ComponentData         => UnusedComponentData);
   --# end accept;
   SeqAlgebra.DisposeOfSeq (TheHeap, UnwantedSeq);
   Maths.StorageRep (ExpType.Value, StoreVal);        -- scalar value if needed later

   if TypeNode = STree.NullNode then -- must be a named number
      if ExpType.IsARange then
         ErrorHandler.SemanticError (114,
                                     ErrorHandler.NoReference,
                                     NodePosition (ExpNode),
                                     LexTokenManager.NullString);
      else
         if not ExpType.IsConstant then
            ErrorHandler.SemanticError (37,
                                        13,
                                        NodePosition (ExpNode),
                                        LexTokenManager.NullString);
         end if;
         TypeSym := ValidNamedNumberType (ExpType.TypeSymbol);
         if TypeSym = Dictionary.GetUnknownTypeMark then
            ErrorHandler.SemanticError (38,
                                        10,
                                        NodePosition (ExpNode),
                                        LexTokenManager.NullString);
         end if;
      end if;

   else -- end of named number checks

      if Dictionary.IsPredefinedStringType (TypeSym) then
         -- We have a constant of type string, implicitly constrained by its initializing
         -- string literal.  In this case we create a string subtype of the right length
         -- and substitute this subtype for string before adding the constant.
         if not Maths.HasNoValue (ExpType.RangeRHS) then -- but only create subtype if range known
            CreateImplicitStringSubtype (ExpType.RangeRHS,
                                         Dictionary.Location'(NodePosition (TypeNode),
                                                              NodePosition (TypeNode)),
                                          -- to get replacement
                                         TypeSym);
         end if;
         -- StoreVal set earlier is not used in string case
         StoreVal := LexTokenManager.NullString;
      end if;

      AssignmentCheck (NodePosition (ExpNode),
                       CurrentScope,
                       TypeSym,
                       ExpType);
      if not ExpType.IsConstant then
         ErrorHandler.SemanticError (37,
                                     13,
                                     NodePosition (ExpNode),
                                     LexTokenManager.NullString);
      end if;
   end if;

   DoIdentifierList (IdentNode,
                     TypeNode,
                     TypeSym,
                     CurrentScope,
                     not ExpType.ErrorsInExpression,
                     Dictionary.IsStatic (TypeSym, CurrentScope) and
                     ExpType.IsStatic);

   Heap.ReportUsage (TheHeap);
end wf_constant_declaration;
