-- $Id: sem-compunit-wf_body_stub.adb 12351 2009-02-02 15:03:51Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with EStrings;
separate (Sem.CompUnit)

procedure wf_body_stub (Node  : in STree.SyntaxNode;
                        Scope : in Dictionary.Scopes)

is
   NodeType               : SPSymbols.SPSymbol;
   IdentNode              : STree.SyntaxNode;
   IdentStr               : LexTokenManager.LexString;
   SpecNode               : STree.SyntaxNode;
   FormalNode             : STree.SyntaxNode;
   ConstraintNode         : STree.SyntaxNode;
   PragmaNode             : STree.SyntaxNode;
   AnnoNode               : STree.SyntaxNode;
   PackSym,
   SubprogSym,
   ProtectedSym,
   TaskSym                : Dictionary.Symbol;
   FirstSeen              : Boolean;
   ScopeLocal             : Dictionary.Scopes;
   InterfacingPragmaFound : Boolean := False;
   OtherPragmaFound       : Boolean := False;
   Unused                 : Boolean;
   ValidAnnotation        : Boolean := False; -- used for task type stubs
   type StubKinds is (BodyStub, Declaration);
   StubKind : StubKinds;
   ValidStubPosition      : Boolean := True;

   ------------------------------------------------------------------

   procedure CheckPosition (Node : in     STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out ValidStubPosition;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table &
   --#         ValidStubPosition         from *,
   --#                                        Node,
   --#                                        STree.Table;
   is
      OuterNode : STree.SyntaxNode;
      NodeType    : SPSymbols.SPSymbol;
   begin
      OuterNode := Node;
      loop
         OuterNode := ParentNode (OuterNode);
         NodeType := SyntaxNodeType (OuterNode);
         exit when NodeType = SPSymbols.subprogram_body or
            NodeType = SPSymbols.package_body or
            NodeType = SPSymbols.task_body;
      end loop;
      OuterNode := ParentNode (ParentNode (OuterNode));
      NodeType := SyntaxNodeType (OuterNode);
      if NodeType = SPSymbols.abody then
         ValidStubPosition := False;
         ErrorHandler.SemanticError (61,
                                     17,
                                     NodePosition (Node),
                                     LexTokenManager.NullString);
      end if;
   end CheckPosition;

   ------------------------------------------------------------------------

   function RequiresSecondAnnotation (SubprogSym : Dictionary.Symbol)
                                     return Boolean
   --# global in Dictionary.Dict;
   is
      GlobalVar : Dictionary.Symbol;
      Required  : Boolean;
      GlobalItem : Dictionary.Iterator;
      EnclosingRegion : Dictionary.Symbol;

   begin
      Required := False;
      if not Dictionary.IsGlobalScope (Dictionary.GetScope (SubprogSym)) then
         EnclosingRegion := Dictionary.GetEnclosingCompilationUnit
            (Dictionary.GetScope (SubprogSym));
         if Dictionary.IsPackage (EnclosingRegion) then
            GlobalItem := Dictionary.FirstGlobalVariable (Dictionary.IsAbstract,
                                                          SubprogSym);
            while GlobalItem /= Dictionary.NullIterator loop
               GlobalVar := Dictionary.CurrentSymbol (GlobalItem);
               if Dictionary.IsAbstractOwnVariable (GlobalVar) and then
                  Dictionary.GetOwner (GlobalVar) = EnclosingRegion
               then
                  Required := True;
                  exit;
               end if;
               GlobalItem := Dictionary.NextSymbol (GlobalItem);
            end loop;
         end if;
      end if;
      return Required;
   end RequiresSecondAnnotation;

   ------------------------------------------------------------------------

   procedure GetAnnoAndConNodes (Node           : in     STree.SyntaxNode;
                                 AnnoNode       :    out STree.SyntaxNode;
                                 ConstraintNode :    out STree.SyntaxNode)
   --# global in STree.Table;
   --# derives AnnoNode,
   --#         ConstraintNode from Node,
   --#                             STree.Table;
   --  pre   SyntaxNodeType (Node) =
   --           SPSymbols.procedure_annotation or
   --        SyntaxNodeType (Node) =
   --           SPSymbols.function_annotation;
   is
      ConstraintNodeLocal : STree.SyntaxNode;
      NodeType : SPSymbols.SPSymbol;
   begin
      ConstraintNodeLocal := Child_Node (Node);
      NodeType := SyntaxNodeType (ConstraintNodeLocal);
      if NodeType = SPSymbols.function_constraint or else
         NodeType = SPSymbols.procedure_constraint
      then
         AnnoNode := STree.NullNode;      --only a constraint found
         ConstraintNode := ConstraintNodeLocal;
      else
         AnnoNode := Node;
         ConstraintNode := LastSiblingOf (ConstraintNodeLocal);
      end if;
   end GetAnnoAndConNodes;

   ------------------------------------------------------------------------
   -- local prpocedures for task stubs

   function EmptyConstraint (Node : STree.SyntaxNode) return Boolean
   --# global in STree.Table;
   is
   begin
      return SyntaxNodeType (Node) = SPSymbols.procedure_constraint and then
        Child_Node (Node) = STree.NullNode;
   end EmptyConstraint;

   ----------------------------------------------------------------------

   function EmptyAnnotation return Boolean
   --# global in AnnoNode;
   --#        in STree.Table;
   is
   begin
      return EmptyConstraint (Child_Node (AnnoNode));
   end EmptyAnnotation;

   ----------------------------------------------------------------------

   procedure ProcessAnnotation
   --# global in     AnnoNode;
   --#        in     CommandLineData.Content;
   --#        in     Scope;
   --#        in     TaskSym;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --#        in out ValidAnnotation;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.StringTable,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                     from *,
   --#                                          AnnoNode,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          LexTokenManager.StringTable,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TaskSym,
   --#                                          TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS           from AnnoNode,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          LexTokenManager.StringTable,
   --#                                          Scope,
   --#                                          SPARK_IO.FILE_SYS,
   --#                                          STree.Table,
   --#                                          TaskSym,
   --#                                          TheHeap &
   --#         ValidAnnotation             from *,
   --#                                          AnnoNode,
   --#                                          STree.Table;
   is
      CurrentNode : STree.SyntaxNode;

      procedure RaiseError (Node : in STree.SyntaxNode)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.StringTable;
      --#        in     STree.Table;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out SPARK_IO.FILE_SYS;
      --# derives ErrorHandler.ErrorContext,
      --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
      --#                                        Dictionary.Dict,
      --#                                        ErrorHandler.ErrorContext,
      --#                                        LexTokenManager.StringTable,
      --#                                        Node,
      --#                                        SPARK_IO.FILE_SYS,
      --#                                        STree.Table;
      is
      begin
         ErrorHandler.SemanticError (990, ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     LexTokenManager.NullString);
      end RaiseError;


   begin
      CurrentNode := Child_Node (AnnoNode);
      -- to be legal, CurrentNode must be a moded_global_definition
      if SyntaxNodeType (CurrentNode) = SPSymbols.moded_global_definition then
         CurrentNode := Next_Sibling (CurrentNode);
         if SyntaxNodeType (CurrentNode) = SPSymbols.dependency_relation then
            CurrentNode := Next_Sibling (CurrentNode);
         end if;
         if EmptyConstraint (CurrentNode) then
            ValidAnnotation := True;
            wf_procedure_annotation (Node         => AnnoNode,
                                     CurrentScope => Scope,
                                     SubprogSym   => TaskSym,
                                     FirstSeen    => False);
         else
            RaiseError (CurrentNode);
         end if;
      else
         RaiseError (CurrentNode);
      end if;
   end ProcessAnnotation;

   ------------------------------------------------------------------------

begin --wf_body_stub
      -- ASSUME Node = body_stub
   --# accept F, 10, Unused, "Export not required here" &
   --#        F, 33, Unused, "Export not required here";

   PragmaNode := LastSiblingOf (Child_Node (Node));
   if SyntaxNodeType (PragmaNode) = SPSymbols.apragma then
      StubKind := Declaration;
      if IsExternalInterface (PragmaNode) then
         -- either Interface of Import correctly used for language variant
         InterfacingPragmaFound := True;
      else
         -- some other pragma found
         OtherPragmaFound := True;
         --ErrorHandler.APragma (NodeLexString (Child_Node (PragmaNode)),
         --                      NodePosition (PragmaNode));
      end if;
   else
      -- no pragma, which means source is of the form "is separate"
      StubKind := BodyStub;
      CheckPosition (Node); -- also sets global ValidStubPosition
   end if;
   --# assert True;
   NodeType := SyntaxNodeType (Child_Node (Node));
   case NodeType is
      when SPSymbols.dotted_simple_name =>   -- package
         IdentNode := Child_Node (Child_Node (Node));
         PackSym := Dictionary.LookupImmediateScope (NodeLexString (IdentNode),
                                                     Scope,
                                                     Dictionary.ProgramContext);
         if SyntaxNodeType (IdentNode) /= SPSymbols.identifier then
            ErrorHandler.SemanticError (613,
                                        ErrorHandler.NoReference,
                                        NodePosition (IdentNode),
                                        LexTokenManager.NullString);
         elsif PackSym = Dictionary.NullSymbol or else
            not Dictionary.IsPackage (PackSym)
         then
            ErrorHandler.SemanticError (11,
                                        ErrorHandler.NoReference,
                                        NodePosition (IdentNode),
                                        NodeLexString (IdentNode));
         elsif Dictionary.HasBody (PackSym) or else
            Dictionary.HasBodyStub (PackSym)
         then
            ErrorHandler.SemanticError (16,
                                        ErrorHandler.NoReference,
                                        NodePosition (IdentNode),
                                        NodeLexString (IdentNode));
         elsif Dictionary.IsPackage (Dictionary.GetRegion (Scope)) and then
            not Dictionary.IsEmbeddedPackage (Dictionary.GetRegion (Scope))
            and then
            Dictionary.LookupSelectedItem (Dictionary.GetRegion (Scope),
                                           NodeLexString (IdentNode),
                                           Dictionary.GlobalScope,
                                           Dictionary.ProofContext)
            /= Dictionary.NullSymbol
         then -- name exists as child
            ErrorHandler.SemanticError (10,
                                        ErrorHandler.NoReference,
                                        NodePosition (IdentNode),
                                        NodeLexString (IdentNode));
         else
            if ValidStubPosition then
               Dictionary.AddBodyStub (PackSym,
                                       Dictionary.Location'(NodePosition (Node),
                                                            NodePosition (Node)));
            end if;
         end if;

      when SPSymbols.procedure_specification =>
         if ValidStubPosition then
            ScopeLocal := Scope;
            SpecNode := Child_Node (Node);
            GetAnnoAndConNodes (Next_Sibling (SpecNode),
                                 --to get
                                AnnoNode,
                                ConstraintNode);
            FormalNode := Next_Sibling (Child_Node (Child_Node (Node)));
            --# assert True;
            if SyntaxNodeType (FormalNode) /= SPSymbols.formal_part then
               FormalNode := STree.NullNode;
            end if;
            wf_procedure_specification (SpecNode,
                                        False, --not hidden
                                        ScopeLocal,
                                          --to get
                                        SubprogSym,
                                        FirstSeen);
            --# assert True;
            if SubprogSym /= Dictionary.NullSymbol then
               if StubKind = Declaration and then not FirstSeen then --illegal redeclaration
                  ErrorHandler.APragma (NodeLexString (Child_Node (PragmaNode)),
                                        NodePosition (PragmaNode));
                  ErrorHandler.SemanticError (10,    ErrorHandler.NoReference,
                                              NodePosition (Node),
                                              Dictionary.GetSimpleName (SubprogSym));

               else -- this else means we don't process illegal redeclarations further
                  if FirstSeen and OtherPragmaFound then  -- only an interfacing pragma will do
                     if CommandLineData.IsSpark83 then
                        ErrorHandler.SemanticError (70,
                                                    18,
                                                    NodePosition (PragmaNode),
                                                    LexTokenManager.InterfaceToken);
                     else
                        ErrorHandler.SemanticError (70,
                                                    18,
                                                    NodePosition (PragmaNode),
                                                    LexTokenManager.ImportToken);
                     end if;
                  end if; -- wrong pragma

                  -- If we are here then we have either:
                  --               a legal declaration using a correct interfacing pragma; or
                  --               a declaration with the wrong pragma that we have reported; or
                  --               a legal "is separate"
                  -- In each case we can go on to check formal parts and annotations

                  if FormalNode /= STree.NullNode then
                     wf_formal_part (FormalNode,
                                     ScopeLocal,
                                     SubprogSym,
                                     FirstSeen,
                                     Dictionary.ProgramContext);

                  elsif Dictionary.GetNumberOfSubprogramParameters (SubprogSym) /= 0 then
                     ErrorHandler.SemanticError (152,
                                                 ErrorHandler.NoReference,
                                                 NodePosition (Node),
                                                 Dictionary.GetSimpleName (SubprogSym));

                  end if; -- formal part to check

                  --# assert True;
                  if AnnoNode /= STree.NullNode then --Annotation found
                     if (FirstSeen or RequiresSecondAnnotation (SubprogSym)) then
                        wf_procedure_annotation (AnnoNode,
                                                 ScopeLocal,
                                                 SubprogSym,
                                                 FirstSeen);
                     else --unexpected anno
                        ErrorHandler.SemanticError (155,
                                                    16,
                                                    NodePosition (SpecNode),
                                                    Dictionary.GetSimpleName (SubprogSym));
                     end if;

                  elsif not FirstSeen and then RequiresSecondAnnotation (SubprogSym) then
                     ErrorHandler.SemanticError (87,
                                                 16,
                                                 NodePosition (SpecNode),
                                                 Dictionary.GetSimpleName (SubprogSym));
                     Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsRefined,
                                                                     SubprogSym);

                  elsif FirstSeen and then
                    (CommandLineData.IsSpark83 or CommandLineData.Content.DoInformationFlow)
                  then
                     ErrorHandler.SemanticError (154,
                                                 19,
                                                 NodePosition (SpecNode),
                                                 Dictionary.GetSimpleName (SubprogSym));
                     Dictionary.SetSubprogramSignatureNotWellformed (Dictionary.IsAbstract,
                                                                     SubprogSym);
                  end if;
                  if InterfacingPragmaFound then
                     wf_external_interface (PragmaNode,
                                            Dictionary.GetSimpleName (SubprogSym),
                                            Unused);
                  end if;

                  --# assert True;
                  --clause to create full derives from modes
                  if CommandLineData.IsSpark95 and then
                    not CommandLineData.Content.DoInformationFlow
                  then
                     if FirstSeen then
                        CreateFullSubProgDependency (Node,
                                                     SubprogSym,
                                                     Dictionary.IsAbstract);

                     elsif RequiresSecondAnnotation (SubprogSym) then
                        CreateFullSubProgDependency (Node,
                                                     SubprogSym,
                                                     Dictionary.IsRefined);
                     end if;
                  end if;

                  --# assert True;
                  if Child_Node (ConstraintNode) /= STree.NullNode then
                     -- a pre/post exists.  Should it?
                     -- checks to see if constraint found is allowed
                     if not (FirstSeen or else
                               RequiresSecondAnnotation (SubprogSym) or else
                               HasParameterOrGlobalOfLocalPrivateType (SubprogSym)) then
                        --annotation not required
                        --two possible errors: misplaced anno or duplicate anno
                        if Dictionary.HasPrecondition (Dictionary.IsAbstract, SubprogSym) or else
                          Dictionary.HasPostcondition (Dictionary.IsAbstract, SubprogSym)
                        then --illegal duplicate anno
                           ErrorHandler.SemanticError (343,
                                                       ErrorHandler.NoReference,
                                                       NodePosition (ConstraintNode),
                                                       Dictionary.GetSimpleName (SubprogSym));
                        else --misplaced anno
                           ErrorHandler.SemanticError (342,
                                                       ErrorHandler.NoReference,
                                                       NodePosition (ConstraintNode),
                                                       Dictionary.GetSimpleName (SubprogSym));
                        end if;
                     else --annotation is required so continue
                        wf_procedure_constraint (ConstraintNode,
                                                 Dictionary.LocalScope (SubprogSym),
                                                 FirstSeen);
                     end if;
                  end if;
               end if; -- not an illegal redclaration
            end if; -- subprogsym not null

            if FirstSeen then
               CheckNoOverloadingFromTaggedOps (SpecNode,
                                                SubprogSym,
                                                Scope,
                                                Dictionary.IsRefined);
            end if;
         end if; -- don't add stub if position illegal

      when SPSymbols.function_specification =>
         if ValidStubPosition then
            ScopeLocal := Scope;
            SpecNode := Child_Node (Node);
            GetAnnoAndConNodes (Next_Sibling (SpecNode),
                                 --to get
                                AnnoNode,
                                ConstraintNode);
            FormalNode := Next_Sibling (Child_Node (Child_Node (Node)));
            --# assert True;
            if SyntaxNodeType (FormalNode) /= SPSymbols.formal_part then
               FormalNode := STree.NullNode;
            end if;
            wf_function_specification (SpecNode,
                                       False, --not hidden
                                       ScopeLocal,
                                       --to get
                                       SubprogSym,
                                       FirstSeen);
            --# assert True;
            if SubprogSym /= Dictionary.NullSymbol then
               if StubKind = Declaration and then not FirstSeen then --illegal redeclaration
                  ErrorHandler.APragma (NodeLexString (Child_Node (PragmaNode)),
                                        NodePosition (PragmaNode));
                  ErrorHandler.SemanticError (10,    ErrorHandler.NoReference,
                                              NodePosition (Node),
                                              Dictionary.GetSimpleName (SubprogSym));

               else -- this else means we don't process illegal redeclarations further
                  if FirstSeen and OtherPragmaFound then  -- only an interfacing pragma will do
                     if CommandLineData.IsSpark83 then
                        ErrorHandler.SemanticError (70,
                                                    18,
                                                    NodePosition (PragmaNode),
                                                    LexTokenManager.InterfaceToken);
                     else
                        ErrorHandler.SemanticError (70,
                                                    18,
                                                    NodePosition (PragmaNode),
                                                    LexTokenManager.ImportToken);
                     end if;
                  end if; -- wrong pragma

                  -- If we are here then we have either:
                  --               a legal declaration using a correct interfacing pragma; or
                  --               a declaration with the wrong pragma that we have reported; or
                  --               a legal "is separate"
                  -- In each case we can go on to check formal parts and annotations

                  if FormalNode /= STree.NullNode then
                     wf_formal_part (FormalNode,
                                     ScopeLocal,
                                     SubprogSym,
                                     FirstSeen,
                                     Dictionary.ProgramContext);

                  elsif Dictionary.GetNumberOfSubprogramParameters (SubprogSym) /= 0 then
                     ErrorHandler.SemanticError (152,
                                                 ErrorHandler.NoReference,
                                                 NodePosition (Node),
                                                 Dictionary.GetSimpleName (SubprogSym));

                  end if;

                  --# assert True;
                  if FirstSeen or else
                    RequiresSecondAnnotation (SubprogSym)
                  then
                     if AnnoNode /= STree.NullNode then
                        wf_function_annotation (AnnoNode,
                                                ScopeLocal,
                                                SubprogSym,
                                                FirstSeen);
                     elsif RequiresSecondAnnotation (SubprogSym) then
                        ErrorHandler.SemanticError (87,
                                                    16,
                                                    NodePosition (SpecNode),
                                                    Dictionary.GetSimpleName (SubprogSym));
                     end if;
                  else
                     if AnnoNode /= STree.NullNode then
                        ErrorHandler.SemanticError (336,
                                                    16,
                                                    NodePosition (SpecNode),
                                                    Dictionary.GetSimpleName (SubprogSym));
                     end if;
                  end if;
                  --# assert True;
                  if InterfacingPragmaFound then
                     wf_external_interface (PragmaNode,
                                            Dictionary.GetSimpleName (SubprogSym),
                                            Unused);
                  end if;
                  --# assert True;
                  if Child_Node (ConstraintNode) /= STree.NullNode then
                     --checks to see if constraint found is allowed
                     if not (FirstSeen or RequiresSecondAnnotation (SubprogSym)) then
                        --annotation not required
                        --two possible errors: misplaced anno or duplicate anno
                        if Dictionary.HasPrecondition (Dictionary.IsAbstract, SubprogSym) or else
                          Dictionary.HasPostcondition (Dictionary.IsAbstract, SubprogSym)
                        then --illegal duplicate anno
                           ErrorHandler.SemanticError (343,
                                                       ErrorHandler.NoReference,
                                                       NodePosition (ConstraintNode),
                                                       Dictionary.GetSimpleName (SubprogSym));
                        else --misplaced anno
                           ErrorHandler.SemanticError (342,
                                                       ErrorHandler.NoReference,
                                                       NodePosition (ConstraintNode),
                                                       Dictionary.GetSimpleName (SubprogSym));
                        end if;
                     else --annotation is required so continue
                        wf_function_constraint (ConstraintNode,
                                                Dictionary.LocalScope (SubprogSym),
                                                FirstSeen);
                     end if;
                  end if;
               end if;  -- not an illegal redeclaration
            end if; -- not null symbol for function

            if FirstSeen then
               CheckNoOverloadingFromTaggedOps (SpecNode,
                                                SubprogSym,
                                                Scope,
                                                Dictionary.IsRefined);
            end if;
         end if; -- don't process at all if stub position illegal

      when SPSymbols.task_stub =>
         if CommandLineData.RavenscarSelected then
            if ValidStubPosition then
               IdentNode := Child_Node (Child_Node (Node));
               IdentStr := NodeLexString (IdentNode);
               AnnoNode := Next_Sibling (IdentNode);
               TaskSym := Dictionary.LookupItem (NodeLexString (IdentNode),
                                                 Scope,
                                                 Dictionary.ProgramContext);
               if TaskSym = Dictionary.NullSymbol or else
                 not Dictionary.IsTaskType (TaskSym) then
                  ErrorHandler.SemanticError (898,
                                              ErrorHandler.NoReference,
                                              NodePosition (IdentNode),
                                              NodeLexString (IdentNode));

               elsif Dictionary.HasBody (TaskSym) or else
                 Dictionary.HasBodyStub (TaskSym)
               then
                  ErrorHandler.SemanticError (899,
                                              ErrorHandler.NoReference,
                                              NodePosition (IdentNode),
                                              NodeLexString (IdentNode));
               else
                  -- valid so far
                  Dictionary.AddBodyStub (TaskSym,
                                          Dictionary.Location'(NodePosition (Node),
                                                               NodePosition (Node)));

                  -- check annotation
                  if RequiresSecondAnnotation (TaskSym) then
                     if EmptyAnnotation then
                        ErrorHandler.SemanticError (154, ErrorHandler.NoReference,
                                                    NodePosition (Node),
                                                    IdentStr);
                     else -- anno present and required
                        ProcessAnnotation;
                     end if;
                  else -- second anno not required
                     if not EmptyAnnotation then
                        ErrorHandler.SemanticError (155, ErrorHandler.NoReference,
                                                    NodePosition (AnnoNode),
                                                    IdentStr);
                     end if;
                  end if;

                  -- Create "full" derives if annotation is present, valid and DFA is selected.
                  -- We know we are in SPARK 95.
                  if ValidAnnotation and then
                    not CommandLineData.Content.DoInformationFlow then
                     CreateFullSubProgDependency (Node,
                                                  TaskSym,
                                                  Dictionary.IsRefined);
                  end if;
               end if;
            end if; -- don't process stub if illegally positioned
         else -- illegal
            ErrorHandler.SemanticError (850,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        LexTokenManager.NullString);
         end if;

      when SPSymbols.protected_stub =>
         if CommandLineData.RavenscarSelected then
            if ValidStubPosition then
               IdentNode := Child_Node (Child_Node (Node));
               ProtectedSym := Dictionary.LookupItem (NodeLexString (IdentNode),
                                                      Scope,
                                                      Dictionary.ProgramContext);
               if ProtectedSym = Dictionary.NullSymbol or else
                 not (Dictionary.IsType (ProtectedSym) and then Dictionary.TypeIsProtected (ProtectedSym)) then
                  ErrorHandler.SemanticError (898,
                                              ErrorHandler.NoReference,
                                              NodePosition (IdentNode),
                                              NodeLexString (IdentNode));

               elsif Dictionary.HasBody (ProtectedSym) or else
                 Dictionary.HasBodyStub (ProtectedSym)
               then
                  ErrorHandler.SemanticError (899,
                                              ErrorHandler.NoReference,
                                              NodePosition (IdentNode),
                                              NodeLexString (IdentNode));
               else
                  Dictionary.AddBodyStub (ProtectedSym,
                                          Dictionary.Location'(NodePosition (Node),
                                                               NodePosition (Node)));
               end if;
            end if; -- don't process stub if illegally positioned

         else -- illegal
            ErrorHandler.SemanticError (850,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        LexTokenManager.NullString);
         end if;

      when others =>
         SystemErrors.FatalError (SystemErrors.InvalidSyntaxTree, "in wf_body_stub");
   end case;
end wf_body_stub;
