-- $Id: sem-compunit-wf_attribute_designator.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.CompUnit)
procedure wf_attribute_designator (Node         : in     STree.SyntaxNode;
                                   Scope        : in     Dictionary.Scopes;
                                   EStack       : in out ExpStack.ExpStackType;
                                   IsAnnotation : in     Boolean;
                                   RefVar       : in     SeqAlgebra.Seq)
is
   ATT_LOOKUP : constant Annotation_Symbol_Table :=
      Annotation_Symbol_Table'(False => SPSymbols.attribute_designator,
                               True  => SPSymbols.annotation_attribute_designator);
   EXP_LOOKUP : constant Annotation_Symbol_Table :=
      Annotation_Symbol_Table'(False => SPSymbols.expression,
                               True  => SPSymbols.annotation_expression);

   IdentNode,
   ArgExpNode          : STree.SyntaxNode;
   SecondArgExpNode    : STree.SyntaxNode;
   TypeSoFar,
   ArgumentExpression  : ExpRecord;
   SecondArgumentExpression : ExpRecord;
   ArgumentFound       : Boolean;
   SecondArgumentFound : Boolean;
   IdentStr            : LexTokenManager.LexString;
   Val,
   UnusedVal,
   RHSval              : Maths.Value;
   BaseFound           : Boolean;
   OkSoFar             : Boolean;
   PrefixKind          : Dictionary.PrefixSort;
   PrefixType          : Dictionary.Symbol;
   VCGtype             : Dictionary.Symbol;

   ----------------------------------------------------------------------

   procedure GetPrefix (Prefix :    out ExpRecord;
                        Kind   :    out Dictionary.PrefixSort;
                        EStack : in out ExpStack.ExpStackType)
   --# global in     BaseFound;
   --#        in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     IdentNode;
   --#        in     LexTokenManager.StringTable;
   --#        in     Node;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from BaseFound,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        EStack,
   --#                                        IdentNode,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table &
   --#         EStack                    from * &
   --#         Kind                      from BaseFound,
   --#                                        Dictionary.Dict,
   --#                                        EStack &
   --#         Prefix                    from BaseFound,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        EStack,
   --#                                        Scope;
   is
      Result : ExpRecord;

   begin
      Kind := Dictionary.AType;
      ExpStack.Pop (Result, EStack); --this is type of prefix expression
      if Result.Sort = IsUnknown then
         Result := UnknownTypeRecord;
      elsif Result.Sort = IsTypeMark then
         Result.IsStatic := Dictionary.IsStatic (Result.TypeSymbol, Scope);
         Result.IsConstant := True;
         Result.IsARange   := False;
         if BaseFound then
            if CommandLineData.IsSpark95 and then
               not Dictionary.TypeIsScalar (Result.TypeSymbol)
            then
               Result := UnknownTypeRecord;
               ErrorHandler.SemanticError (96,
                                           ErrorHandler.NoReference,
                                           NodePosition (Node),
                                           LexTokenManager.BaseToken);
            end if;
            Kind := Dictionary.ABaseType;
         end if;
      elsif Dictionary.IsObject (Result.OtherSymbol) or else
         Dictionary.IsRecordSubcomponent (Result.OtherSymbol)
      then
         if Dictionary.IsUniversalIntegerType (Result.TypeSymbol) or else
            Dictionary.IsUniversalRealType (Result.TypeSymbol)
         then  --its a named number and not a proper object
            Result := UnknownTypeRecord;
            ErrorHandler.SemanticError (31,
                                        ErrorHandler.NoReference,
                                        NodePosition (IdentNode),
                                        LexTokenManager.NullString);
         else
            Result.IsStatic := Dictionary.IsStatic (Result.TypeSymbol, Scope);
            Result.IsConstant := Dictionary.IsConstant (Result.TypeSymbol);
            Result.IsARange   := False;
            Result.VariableSymbol := Dictionary.NullSymbol;
            Result.IsAVariable := False;
            Result.IsAnEntireVariable := False;
            Kind := Dictionary.AnObject;
         end if;
      elsif Result = UnknownTypeRecord then
         null;
      else
         Result := UnknownTypeRecord;
         ErrorHandler.SemanticError (31,
                                     ErrorHandler.NoReference,
                                     NodePosition (IdentNode),
                                     LexTokenManager.NullString);
      end if;
      Prefix := Result;
   end GetPrefix;

   ----------------------------------------------------------------------

   -- this function should gradually wither away as attributes are implemented
   function NotYetImplemented (Str : LexTokenManager.LexString) return Boolean
   is
   begin
      return  (Str = LexTokenManager.AdjacentToken     or else
               Str = LexTokenManager.ComposeToken      or else
               Str = LexTokenManager.Copy_SignToken    or else
               Str = LexTokenManager.Leading_PartToken or else
               Str = LexTokenManager.RemainderToken    or else
               Str = LexTokenManager.ScalingToken      or else
               Str = LexTokenManager.ExponentToken     or else
               Str = LexTokenManager.FractionToken     or else
               Str = LexTokenManager.MachineToken      or else
               Str = LexTokenManager.ModelToken        or else
               Str = LexTokenManager.RoundingToken     or else
               Str = LexTokenManager.TruncationToken   or else
               Str = LexTokenManager.Unbiased_RoundingToken);
   end NotYetImplemented;

   ----------------------------------------------------------------------

   -- identifies special attributes that are part of the pre/post annotation
   -- language only
   function IsProofAttribute (Str : LexTokenManager.LexString) return Boolean
   is
   begin -- IsProofAttribute
      return  (Str = LexTokenManager.TailToken     or else
                 Str = LexTokenManager.AppendToken);
   end IsProofAttribute;

   ----------------------------------------------------------------------

   function ProofAttributeIsVisible (IdentStr   : LexTokenManager.LexString;
                                     PrefixKind : Dictionary.PrefixSort;
                                     PrefixSym  : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   is
   begin
      return PrefixKind = Dictionary.AnObject and then
                ((IdentStr = LexTokenManager.TailToken and then
                    Dictionary.GetOwnVariableOrConstituentMode (PrefixSym) = Dictionary.InMode) or else
                   (IdentStr = LexTokenManager.AppendToken and then
                      Dictionary.GetOwnVariableOrConstituentMode (PrefixSym) = Dictionary.OutMode));
   end ProofAttributeIsVisible;

   ----------------------------------------------------------------------

   function NeverTakesArguments (Str : LexTokenManager.LexString) return Boolean
   is
   begin
      return not (Str = LexTokenManager.FirstToken   or
                    Str = LexTokenManager.LastToken    or
                    Str = LexTokenManager.LengthToken  or
                    Str = LexTokenManager.PosToken     or
                    Str = LexTokenManager.PredToken    or
                    Str = LexTokenManager.RangeToken   or
                    Str = LexTokenManager.SuccToken    or
                    Str = LexTokenManager.ValToken     or
                    Str = LexTokenManager.MinToken     or
                    Str = LexTokenManager.MaxToken     or
                    Str = LexTokenManager.TailToken    or
                    Str = LexTokenManager.CeilingToken or
                    Str = LexTokenManager.FloorToken   or
                    Str = LexTokenManager.AppendToken);
   end NeverTakesArguments;

   ----------------------------------------------------------------------

   function AlwaysTakesOneArgument (Str : LexTokenManager.LexString) return Boolean
   is
   begin
      return (Str = LexTokenManager.PosToken     or else
              Str = LexTokenManager.PredToken    or else
              Str = LexTokenManager.SuccToken    or else
              Str = LexTokenManager.ValToken     or else
              Str = LexTokenManager.FloorToken   or else
              Str = LexTokenManager.CeilingToken or else
              Str = LexTokenManager.TailToken);
   end AlwaysTakesOneArgument;

   ----------------------------------------------------------------------

   function AlwaysTakesTwoArguments (Str : LexTokenManager.LexString) return Boolean
   is
   begin
      return (Str = LexTokenManager.MinToken    or else
              Str = LexTokenManager.MaxToken    or else
              Str = LexTokenManager.AppendToken);
   end AlwaysTakesTwoArguments;

   ----------------------------------------------------------------------

   function ArgumentTypeCorrect (Str         : LexTokenManager.LexString;
                                 PrefixType,
                                 ArgType     : Dictionary.Symbol) return Boolean
   --# global in Dictionary.Dict;
   --#        in Scope;
   is
      Result : Boolean;
   begin
      if Str = LexTokenManager.ValToken then
         Result := Dictionary.IsIntegerTypeMark (ArgType, Scope) or else
                   Dictionary.IsModularTypeMark (ArgType, Scope) or else
            Dictionary.IsUnknownTypeMark (ArgType);
      else
         Result := Dictionary.CompatibleTypes (Scope, PrefixType, ArgType);
      end if;
      return Result;
   end ArgumentTypeCorrect;

   ----------------------------------------------------------------------

   function AttributeConsideredStatic (Str : LexTokenManager.LexString)
                                      return Boolean
   --# global in Dictionary.Dict;
   --#        in Scope;
   --#        in TypeSoFar;
   is
      Result : Boolean;
   begin
      if Str = LexTokenManager.AftToken             or else
        Str = LexTokenManager.BaseToken             or else
        Str = LexTokenManager.DeltaToken            or else
        Str = LexTokenManager.DigitsToken           or else
        Str = LexTokenManager.EmaxToken             or else
        Str = LexTokenManager.EpsilonToken          or else
        Str = LexTokenManager.FirstToken            or else
        Str = LexTokenManager.ForeToken             or else
        Str = LexTokenManager.LargeToken            or else
        Str = LexTokenManager.LastToken             or else
        Str = LexTokenManager.MachineEmaxToken      or else
        Str = LexTokenManager.MachineEminToken      or else
        Str = LexTokenManager.MachineMantissaToken  or else
        Str = LexTokenManager.MachineOverflowsToken or else
        Str = LexTokenManager.MachineRadixToken     or else
        Str = LexTokenManager.MachineRoundsToken    or else
        Str = LexTokenManager.MantissaToken         or else
        Str = LexTokenManager.PredToken             or else
        Str = LexTokenManager.PosToken              or else
        Str = LexTokenManager.SafeEmaxToken         or else
        Str = LexTokenManager.SafeLargeToken        or else
        Str = LexTokenManager.SafeSmallToken        or else
        Str = LexTokenManager.SmallToken            or else
        Str = LexTokenManager.SuccToken             or else
        Str = LexTokenManager.ValToken              or else
        Str = LexTokenManager.Component_SizeToken   or else
        Str = LexTokenManager.DenormToken           or else
        Str = LexTokenManager.Model_EminToken       or else
        Str = LexTokenManager.Model_EpsilonToken    or else
        Str = LexTokenManager.Model_MantissaToken   or else
        Str = LexTokenManager.Model_SmallToken      or else
        Str = LexTokenManager.Safe_FirstToken       or else
        Str = LexTokenManager.Safe_LastToken        or else
        Str = LexTokenManager.Signed_ZerosToken     or else
        Str = LexTokenManager.MinToken              or else
        Str = LexTokenManager.MaxToken              or else
        Str = LexTokenManager.ModulusToken          or else
        Str = LexTokenManager.FloorToken            or else
        Str = LexTokenManager.CeilingToken then

         Result := True;

      elsif Str = LexTokenManager.SizeToken then
         -- 'Size is static only for a prefix that denotes a static
         -- scalar subtype
         Result := Dictionary.IsScalarType (TypeSoFar.TypeSymbol, Scope);
      else
         Result := False;
      end if;

      return Result;
   end AttributeConsideredStatic;

   ----------------------------------------------------------------------

   function CheckStatic return Boolean
   --# global in ArgumentExpression;
   --#        in ArgumentFound;
   --#        in CommandLineData.Content;
   --#        in Dictionary.Dict;
   --#        in IdentStr;
   --#        in PrefixType;
   --#        in Scope;
   --#        in TypeSoFar;
   is
      Result : Boolean;

      function IsStaticArrayAttribute return Boolean
         --# global in CommandLineData.Content;
         --#        in Dictionary.Dict;
         --#        in IdentStr;
         --#        in PrefixType;
      is
      begin
         return CommandLineData.IsSpark95 and then
            (IdentStr = LexTokenManager.FirstToken or else
             IdentStr = LexTokenManager.LastToken or else
             IdentStr = LexTokenManager.LengthToken) and then
            Dictionary.TypeIsArray (PrefixType) and then
            not Dictionary.IsUnconstrainedArrayType (PrefixType);
      end IsStaticArrayAttribute;

      ----------------------------------------------------------

   begin
      Result := (TypeSoFar.IsStatic and then AttributeConsideredStatic (IdentStr)) or else
         IsStaticArrayAttribute;
      if Result and then ArgumentFound then
         Result := ArgumentExpression.IsStatic;
      end if;
      return Result;
   end CheckStatic;

   ----------------------------------------------------------------------

   function CheckConstant return Boolean
   --# global in ArgumentExpression;
   --#        in ArgumentFound;
   --#        in Dictionary.Dict;
   --#        in IdentStr;
   --#        in Scope;
   --#        in TypeSoFar;
   is
      Result : Boolean;

   begin
      Result := TypeSoFar.IsConstant or else
         Dictionary.IsConstrainedArrayType (TypeSoFar.TypeSymbol,
                                            Scope);
      if Result then
         if IdentStr = LexTokenManager.BaseToken then
            Result := not Dictionary.IsUnconstrainedArrayType
              (Dictionary.GetRootType (TypeSoFar.TypeSymbol));
         elsif IdentStr = LexTokenManager.SizeToken then
            -- S'Size is only considered to be constant/static for
            -- scalar types
            Result := Dictionary.IsScalarType (TypeSoFar.TypeSymbol, Scope);
         end if;
      end if;
      if Result and then ArgumentFound then
         Result := ArgumentExpression.IsConstant;
      end if;

      return Result;
   end CheckConstant;

   ----------------------------------------------------------------------

   function CheckRange return Boolean
   --# global in IdentStr;
   is
   begin
      return IdentStr = LexTokenManager.RangeToken;
   end CheckRange;

   ----------------------------------------------------------------------

   procedure BasicChecks (Ok : out Boolean)
   --# global in     ArgExpNode;
   --#        in     ArgumentFound;
   --#        in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     IdentNode;
   --#        in     IdentStr;
   --#        in     IsAnnotation;
   --#        in     LexTokenManager.StringTable;
   --#        in     Node;
   --#        in     SecondArgExpNode;
   --#        in     SecondArgumentFound;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TypeSoFar;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from ArgExpNode,
   --#                                        ArgumentFound,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        IdentNode,
   --#                                        IdentStr,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        SecondArgExpNode,
   --#                                        SecondArgumentFound,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TypeSoFar &
   --#         Ok                        from ArgumentFound,
   --#                                        CommandLineData.Content,
   --#                                        IdentStr,
   --#                                        IsAnnotation,
   --#                                        SecondArgumentFound,
   --#                                        TypeSoFar &
   --#         TypeSoFar                 from *,
   --#                                        ArgumentFound,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        IdentStr,
   --#                                        IsAnnotation,
   --#                                        SecondArgumentFound;
   is
   begin
      Ok := False;
      if not LexTokenManager.IsAttributeToken (IdentStr,
                                               CommandLineData.IsSpark95)
      then
         TypeSoFar := UnknownTypeRecord;
         ErrorHandler.SemanticError (54,
                                     ErrorHandler.NoReference,
                                     NodePosition (IdentNode),
                                     IdentStr);

      elsif NotYetImplemented (IdentStr) then
         TypeSoFar := UnknownTypeRecord;
         ErrorHandler.SemanticError (30,
                                     ErrorHandler.NoReference,
                                     NodePosition (IdentNode),
                                     IdentStr);

      elsif IsProofAttribute (IdentStr) and then not IsAnnotation then
         TypeSoFar := UnknownTypeRecord;
         ErrorHandler.SemanticError (54,
                                     ErrorHandler.NoReference,
                                     NodePosition (IdentNode),
                                     IdentStr);

      elsif ArgumentFound and then NeverTakesArguments (IdentStr) then
         TypeSoFar := UnknownTypeRecord;
         ErrorHandler.SemanticError (55,
                                     ErrorHandler.NoReference,
                                     NodePosition (ArgExpNode),
                                     IdentStr);

      elsif not ArgumentFound and then (AlwaysTakesOneArgument (IdentStr) or else
                                        (AlwaysTakesTwoArguments (IdentStr))) then
         TypeSoFar := UnknownTypeRecord;
         ErrorHandler.SemanticError (56, ErrorHandler.NoReference,
                                     NodePosition (IdentNode),
                                     LexTokenManager.NullString);

      elsif not SecondArgumentFound and then (AlwaysTakesTwoArguments (IdentStr)) then
         TypeSoFar := UnknownTypeRecord;
         ErrorHandler.SemanticError (56, ErrorHandler.NoReference,
                                     NodePosition (ArgExpNode),
                                     LexTokenManager.NullString);

      elsif SecondArgumentFound and then AlwaysTakesOneArgument (IdentStr) then
         TypeSoFar := UnknownTypeRecord;
         ErrorHandler.SemanticError (49, ErrorHandler.NoReference,
                                     NodePosition (SecondArgExpNode),
                                     IdentStr);

         --check that prefix of Pred, Succ, Pos, Val is typemark
      elsif AlwaysTakesOneArgument (IdentStr) and then
         not IsProofAttribute (IdentStr) and then -- don't want 'Tail to trip this test
         TypeSoFar.Sort = IsObject
      then
         ErrorHandler.SemanticError (63,
                                     ErrorHandler.NoReference,
                                     NodePosition (ParentNode (Node)),
                                     Dictionary.GetSimpleName (TypeSoFar.OtherSymbol));
         TypeSoFar := UnknownTypeRecord;

      else
         Ok := True;
      end if;
   end BasicChecks;

   ----------------------------------------------------------------------

   procedure BaseChecks (Continue : out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     IdentNode;
   --#        in     IdentStr;
   --#        in     IsAnnotation;
   --#        in     LexTokenManager.StringTable;
   --#        in     Node;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TypeSoFar;
   --# derives Continue                  from IdentStr &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        IdentNode,
   --#                                        IdentStr,
   --#                                        IsAnnotation,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TypeSoFar &
   --#         TypeSoFar                 from *,
   --#                                        Dictionary.Dict,
   --#                                        IdentStr,
   --#                                        IsAnnotation,
   --#                                        Node,
   --#                                        STree.Table;
   is
   begin
      if IdentStr = LexTokenManager.BaseToken then
         Continue := False; --whatever happens we don't want to do any more checks
         if SyntaxNodeType (ParentNode (Node)) /= ATT_LOOKUP (IsAnnotation) or else
            SyntaxNodeType (Child_Node (Node)) /= SPSymbols.attribute_ident
         then
            TypeSoFar := UnknownTypeRecord;
            if IsAnnotation then
               ErrorHandler.SemanticError (97,
                                           2,
                                           NodePosition (Node),
                                           LexTokenManager.NullString);
            else
               ErrorHandler.SemanticError (97,
                                           1,
                                           NodePosition (Node),
                                           LexTokenManager.NullString);
            end if;
         elsif TypeSoFar.Sort /= IsTypeMark then
            TypeSoFar := UnknownTypeRecord;
            ErrorHandler.SemanticError (96,
                                        ErrorHandler.NoReference,
                                        NodePosition (IdentNode),
                                        IdentStr);
         end if;
      else
         Continue := True;
      end if;
   end BaseChecks;

   -----------------------------------------------------------------------------

   procedure ProcessArrayAttribute
      --430 annotation completed
   --# global in     ArgExpNode;
   --#        in     ArgumentExpression;
   --#        in     ArgumentFound;
   --#        in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     IdentStr;
   --#        in     LexTokenManager.StringTable;
   --#        in     PrefixType;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out OkSoFar;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TypeSoFar;
   --#        in out VCGtype;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from ArgExpNode,
   --#                                        ArgumentExpression,
   --#                                        ArgumentFound,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TypeSoFar &
   --#         OkSoFar                   from *,
   --#                                        ArgumentExpression,
   --#                                        ArgumentFound,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        TypeSoFar &
   --#         TypeSoFar                 from *,
   --#                                        ArgumentExpression,
   --#                                        ArgumentFound,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        IdentStr,
   --#                                        PrefixType,
   --#                                        Scope &
   --#         VCGtype                   from *,
   --#                                        ArgumentExpression,
   --#                                        ArgumentFound,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        IdentStr,
   --#                                        TypeSoFar;
   --#
   is
      IndexNumber : Integer;
      Continue    : Boolean;

      procedure ProcessArgument (Ok : out Boolean)
         -- 430 annotation completed
         --# global in     ArgExpNode;
         --#        in     ArgumentExpression;
         --#        in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     IdentStr;
         --#        in     LexTokenManager.StringTable;
         --#        in     Scope;
         --#        in     STree.Table;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out IndexNumber;
         --#        in out OkSoFar;
         --#        in out SPARK_IO.File_Sys;
         --#        in out TypeSoFar;
         --# derives ErrorHandler.ErrorContext,
         --#         SPARK_IO.File_Sys         from ArgExpNode,
         --#                                        ArgumentExpression,
         --#                                        CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        ErrorHandler.ErrorContext,
         --#                                        LexTokenManager.StringTable,
         --#                                        Scope,
         --#                                        SPARK_IO.File_Sys,
         --#                                        STree.Table,
         --#                                        TypeSoFar &
         --#         IndexNumber               from *,
         --#                                        ArgumentExpression,
         --#                                        CommandLineData.Content,
         --#                                        Dictionary.Dict &
         --#         Ok                        from ArgumentExpression,
         --#                                        CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        TypeSoFar &
         --#         OkSoFar                   from *,
         --#                                        ArgumentExpression,
         --#                                        CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        TypeSoFar &
         --#         TypeSoFar                 from *,
         --#                                        ArgumentExpression,
         --#                                        CommandLineData.Content,
         --#                                        Dictionary.Dict,
         --#                                        IdentStr;
      is
         Err : Maths.ErrorCode;

         ---------------------------------------------------------
         procedure SetIllegalResult
         --# global in     Dictionary.Dict;
         --#        in     IdentStr;
         --#           out OkSoFar;
         --#           out TypeSoFar;
         --# derives OkSoFar   from  &
         --#         TypeSoFar from Dictionary.Dict,
         --#                        IdentStr;
         is
         begin
            OkSoFar := False;
            TypeSoFar := UnknownTypeRecord;
            TypeSoFar.IsARange := CheckRange;
         end SetIllegalResult;

         ----------------------------------------------------------
      begin --ProcessArgument
         Ok := False;
         -- Ada83 LRM says arg N must be Universal Integer.
         -- Ada95 LRM says arg N must be Universal Integer or any integer
         --   (signed or modular) type.
         if not (Dictionary.IsUniversalIntegerType (ArgumentExpression.TypeSymbol) or
                 Dictionary.IsUnknownTypeMark (ArgumentExpression.TypeSymbol)      or
                 ((Dictionary.TypeIsInteger (ArgumentExpression.TypeSymbol) or
                   Dictionary.TypeIsModular (ArgumentExpression.TypeSymbol)) and
                  CommandLineData.IsSpark95))
         then
            SetIllegalResult;
            ErrorHandler.SemanticError (38,
                                        ErrorHandler.NoReference,
                                        NodePosition (ArgExpNode),
                                        LexTokenManager.NullString);
         elsif not ArgumentExpression.IsStatic then
            SetIllegalResult;
            ErrorHandler.SemanticError (36,
                                        1,
                                        NodePosition (ArgExpNode),
                                        LexTokenManager.NullString);
         else --we have a static expression of the correct type
            Maths.ValueToInteger (ArgumentExpression.Value,
                                    --to get
                                  IndexNumber,
                                  Err);
            if Err = Maths.NoError then
               if IndexNumber > 0 and then
                  IndexNumber <= Dictionary.GetNumberOfDimensions (TypeSoFar.TypeSymbol)
               then
                  Ok := True;
               else  --number out of range
                  ErrorHandler.SemanticErrorSym (403,
                                                 ErrorHandler.NoReference,
                                                 NodePosition (ArgExpNode),
                                                 TypeSoFar.TypeSymbol,
                                                 Scope);
                  SetIllegalResult;
               end if;
            else    --maths conversion error
               SetIllegalResult;
            end if;
         end if;
      end ProcessArgument;

   begin --ProcessArrayAttribute
      Continue := True;
      IndexNumber := 1; --default value if no argument found
      if ArgumentFound then
         ProcessArgument (Continue);
      end if;
      if Continue then
         -- Set suitable symbol to be planted in the syntax tree for use by VCG
         if IdentStr = LexTokenManager.Component_SizeToken then
            -- For component_size we want the type of the array for use by the VCG
            VCGtype := Dictionary.GetRootType (TypeSoFar.TypeSymbol);
         elsif Dictionary.IsUnconstrainedArrayType (TypeSoFar.TypeSymbol) then
            -- For unconstrained arrays, obtain the implcitly declared constraint symbol for the array object
            VCGtype := Dictionary.GetSubprogramParameterConstraint (TypeSoFar.OtherSymbol, IndexNumber);
         else
            -- For constrained arrays then obtain appropriate index for the array type; this is what the VCG needs
            VCGtype := Dictionary.GetArrayIndex (TypeSoFar.TypeSymbol, IndexNumber);
         end if;

         TypeSoFar.IsStatic    := CheckStatic;
         TypeSoFar.IsConstant  := CheckConstant;
         TypeSoFar.IsARange    := CheckRange;
         TypeSoFar.TypeSymbol  := Dictionary.GetArrayAttributeType (IdentStr,
                                                                    TypeSoFar.TypeSymbol,
                                                                    IndexNumber);
         TypeSoFar.OtherSymbol := Dictionary.NullSymbol;
         TypeSoFar.Sort        := TypeResult;
      end if;

   end ProcessArrayAttribute;


   -----------------------------------------------------------------------------

begin -- wf_attribute_designator
      --node = attribute_designator

   ArgumentExpression       := NullExpRecord;
   SecondArgumentExpression := NullExpRecord;

   BaseFound := False;
   IdentNode := Child_Node (Node);
   if SyntaxNodeType (IdentNode) =
      ATT_LOOKUP (IsAnnotation)
   then
      IdentNode := Next_Sibling (IdentNode);
      BaseFound := True;
   end if;
   IdentStr := NodeLexString (IdentNode);
   ArgExpNode := Child_Node (Next_Sibling (IdentNode));
   SecondArgExpNode := Next_Sibling (ArgExpNode);

   --# assert True;

   -- look for second argument
   if SyntaxNodeType (SecondArgExpNode) = EXP_LOOKUP (IsAnnotation) then
      SecondArgumentFound := True;
      ExpStack.Pop (SecondArgumentExpression, EStack);
   else
      SecondArgumentFound := False;
   end if;

   --# assert True;

   -- look for first argument
   if SyntaxNodeType (ArgExpNode) = EXP_LOOKUP (IsAnnotation) then
      ArgumentFound := True;
      ExpStack.Pop (ArgumentExpression, EStack);
   else
      ArgumentFound := False;
   end if;

   GetPrefix (TypeSoFar, PrefixKind, EStack);
   PrefixType := TypeSoFar.TypeSymbol;
   VCGtype := PrefixType;

   -- if clause to add prefix variable to reference list

   --# assert True;

   -- X'Valid _IS_ considered a read of X for flow analysis.
   if (not IsAnnotation) and then
      Dictionary.IsVariableOrSubcomponent (TypeSoFar.OtherSymbol) and then
      (Dictionary.IsUnconstrainedArrayType (TypeSoFar.TypeSymbol) or
       IdentStr = LexTokenManager.ValidToken)

   then
      SeqAlgebra.AddMember
         (TheHeap,
          RefVar,
          Natural (Dictionary.SymbolRef (TypeSoFar.OtherSymbol)));
   end if;

   --# assert True;

   BasicChecks (OkSoFar);
   if OkSoFar then
      BaseChecks (OkSoFar);
   end if;

   --# assert True;

   -- any attempt to use proof attributes 'Tail and 'Append in program context
   -- will have been trapped by basic checks which will have set OkSoFar to False.
   -- We can therefore treat type and argument checking of these proof attributes
   -- normally from here on because if they are being checked then they must be
   -- being used in a valid context

   if OkSoFar then
      if Dictionary.AttributeIsVisibleButObselete (IdentStr,
                                                   PrefixKind,
                                                   TypeSoFar.TypeSymbol,
                                                   Scope)
      then
         ErrorHandler.SemanticWarning (310, NodePosition (IdentNode), IdentStr);
      end if;

      if not (Dictionary.AttributeIsVisible (IdentStr,
                                            PrefixKind,
                                            TypeSoFar.TypeSymbol,
                                             Scope) or else
              ProofAttributeIsVisible (IdentStr,
                                       PrefixKind,
                                       TypeSoFar.OtherSymbol))
      then
         TypeSoFar := UnknownTypeRecord;
         OkSoFar := False;
         ErrorHandler.SemanticError (96,
                                     ErrorHandler.NoReference,
                                     NodePosition (IdentNode),
                                     IdentStr);
      elsif Dictionary.IsArrayAttribute (IdentStr,
                                         TypeSoFar.TypeSymbol)
      then
         if SecondArgumentFound then
            -- must be error, array attributes take a maximum of one argument
            TypeSoFar := UnknownTypeRecord;
            OkSoFar := False;
            ErrorHandler.SemanticError (49,
                                        ErrorHandler.NoReference,
                                        NodePosition (SecondArgExpNode),
                                        IdentStr);
         else -- zero or one expression provided
            ProcessArrayAttribute;
         end if;

      elsif IdentStr = LexTokenManager.AccessToken then
         -- 'Access only allowed if subject is aliased.  We could roll this into AttributeIsVisible
         -- but this would make the error unclear, so we do a special check here
         if Dictionary.VariableIsAliased (TypeSoFar.OtherSymbol) then
            -- valid application
            TypeSoFar.IsStatic    := False;
            TypeSoFar.IsConstant  := True;
            TypeSoFar.IsARange    := False;
            TypeSoFar.TypeSymbol  := Dictionary.GetScalarAttributeType (IdentStr,
                                                                        TypeSoFar.TypeSymbol);
            TypeSoFar.VariableSymbol := TypeSoFar.OtherSymbol;
            TypeSoFar.OtherSymbol := Dictionary.NullSymbol;
            TypeSoFar.Sort        := TypeResult;
            -- note we preserve OtherSymbol in VariableSymbol for use in wf_discriminant_constraint
         else
            OkSoFar := False;
            ErrorHandler.SemanticErrorSym (895,
                                           ErrorHandler.NoReference,
                                           NodePosition (Node),
                                           TypeSoFar.OtherSymbol,
                                           Scope);
         end if;

      else --an "ordinary" attribute
         if ArgumentFound then
            if not (AlwaysTakesOneArgument (IdentStr) or else
                    AlwaysTakesTwoArguments (IdentStr)) then
               TypeSoFar := UnknownTypeRecord;
               OkSoFar := False;
               ErrorHandler.SemanticError (55, ErrorHandler.NoReference,
                                           NodePosition (ArgExpNode),
                                           IdentStr);




            elsif ArgumentTypeCorrect (IdentStr,
                                       TypeSoFar.TypeSymbol,
                                       ArgumentExpression.TypeSymbol)
            then -- first argument type is ok, is there a second
               if not SecondArgumentFound or else
                  ArgumentTypeCorrect (IdentStr,
                                       TypeSoFar.TypeSymbol,
                                       SecondArgumentExpression.TypeSymbol)
               then --either no second arg or it type checks ok
                  TypeSoFar.IsStatic    := CheckStatic;
                  TypeSoFar.IsConstant  := CheckConstant;
                  TypeSoFar.IsARange    := CheckRange;
                  TypeSoFar.TypeSymbol  := Dictionary.GetScalarAttributeType
                     (IdentStr,
                      TypeSoFar.TypeSymbol);
                  TypeSoFar.OtherSymbol := Dictionary.NullSymbol;
                  TypeSoFar.Sort        := TypeResult;

               else -- second argument type wrong
                  TypeSoFar := UnknownTypeRecord;
                  OkSoFar := False;
                  ErrorHandler.SemanticError (38,
                                              ErrorHandler.NoReference,
                                              NodePosition (SecondArgExpNode),
                                              LexTokenManager.NullString);
               end if;

            else -- first argument type wrong
               TypeSoFar := UnknownTypeRecord;
               OkSoFar := False;
               ErrorHandler.SemanticError (38,
                                           ErrorHandler.NoReference,
                                           NodePosition (ArgExpNode),
                                           LexTokenManager.NullString);
            end if;
         else --no argument found so just set up result
            TypeSoFar.IsStatic    := CheckStatic;
            TypeSoFar.IsConstant  := CheckConstant;
            TypeSoFar.IsARange    := CheckRange;
            TypeSoFar.TypeSymbol  := Dictionary.GetScalarAttributeType (IdentStr,
                                                                        TypeSoFar.TypeSymbol);
            TypeSoFar.OtherSymbol := Dictionary.NullSymbol;
            TypeSoFar.Sort        := TypeResult;
         end if;
      end if;
   end if;

   --# assert True;
   --if Dict has returned a null symbol for the attribute type then convert it to
   --the unknown type symbol
   if TypeSoFar.TypeSymbol = Dictionary.NullSymbol then
      TypeSoFar.TypeSymbol := Dictionary.GetUnknownTypeMark;
   end if;

   --# assert True;

   --# accept Flow, 10, UnusedVal, "Expected ineffective assignment" &
   --#        Flow, 33, UnusedVal, "Expected ineffective assignment";
   if OkSoFar then
      if ArgumentFound then
         if SecondArgumentFound then
            -- we could statically evaluate Max and Min here but since use of these functions
            -- with two static arguments seems unlikely it has been left for now.  However,
            -- we must check that any static argument is in type range.
            Val := ArgumentExpression.Value;
            ConstraintCheck (Val,
                             UnusedVal,
                             IsAnnotation,
                             Dictionary.GetRootType (PrefixType),
                             NodePosition (ArgExpNode));

            Val := SecondArgumentExpression.Value;
            ConstraintCheck (Val,
                             UnusedVal,
                             IsAnnotation,
                             Dictionary.GetRootType (PrefixType),
                             NodePosition (SecondArgExpNode));

         else -- just one argument found
            Val := ArgumentExpression.Value;
         end if;

      else -- no arguments found
         Val := Maths.NoValue;
      end if;
      -- constraint checking of arguments to attributes other than Min/Max done as part
      -- of CalcAttribute
      CalcAttribute (Node,
                     IdentStr,              --the attribute name
                     PrefixKind,            --object, type or basetype
                     PrefixType,
                     BaseFound,
                     IsAnnotation,
                     -- using and to get
                     Val,
                     RHSval);
      TypeSoFar.Value    := Val;
      TypeSoFar.RangeRHS := RHSval;
   end if;

   ExpStack.Push (TypeSoFar, EStack);

   --# assert True;
   STree.AddNodeSymbol (Node, VCGtype);

end wf_attribute_designator;
