-- $Id: sem-compunit-wf_arange.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

-- Overview: Called to check validity of a
-- arange node.  Replaces calls to StaticARange, BaseTypeARange and
-- CheckTypeARange
----------------------------------------------------------------------------

separate (Sem.CompUnit)

procedure wf_arange (Node         : in     STree.SyntaxNode;
                     Scope        : in     Dictionary.Scopes;
                     EStack       : in out ExpStack.ExpStackType;
                     IsAnnotation : in     Boolean)
is
   ATTRIB_LOOKUP : constant Annotation_Symbol_Table :=
      Annotation_Symbol_Table'(False => SPSymbols.attribute,
                               True  => SPSymbols.annotation_attribute);

   NextNode  : STree.SyntaxNode;
   Left,
   Right,
   Result    : ExpRecord;
   LeftType,
   RightType : Dictionary.Symbol;

   ---------------------------------------------------------------

   procedure IntegerImplicitTypeConversion (LeftType,
                                            RightType : in out Dictionary.Symbol;
                                            Scope     : in     Dictionary.Scopes)
   --# global in Dictionary.Dict;
   --# derives LeftType,
   --#         RightType from Dictionary.Dict,
   --#                        LeftType,
   --#                        RightType,
   --#                        Scope;
   is

   begin
      if Dictionary.IsUniversalIntegerType (LeftType) then
         if Dictionary.IsIntegerTypeMark (RightType, Scope) or
            Dictionary.IsModularTypeMark (RightType, Scope) then
            LeftType := RightType;
         end if;

      elsif Dictionary.IsUniversalIntegerType (RightType) then
         if Dictionary.IsIntegerTypeMark (LeftType, Scope) or
            Dictionary.IsModularTypeMark (LeftType, Scope) then
            RightType := LeftType;
         end if;

      end if;
   end IntegerImplicitTypeConversion;

   ---------------------------------------------------------------

   procedure RealImplicitTypeConversion (LeftType,
                                         RightType : in out Dictionary.Symbol;
                                         Scope     : in     Dictionary.Scopes)
   --# global in Dictionary.Dict;
   --# derives LeftType,
   --#         RightType from Dictionary.Dict,
   --#                        LeftType,
   --#                        RightType,
   --#                        Scope;
   is

   begin
      if Dictionary.IsUniversalRealType (LeftType) then
         if Dictionary.IsRealTypeMark (RightType, Scope) then
            LeftType := RightType;
         end if;

      elsif Dictionary.IsUniversalRealType (RightType) then
         if Dictionary.IsRealTypeMark (LeftType, Scope) then
            RightType := LeftType;
         end if;

      end if;
   end RealImplicitTypeConversion;

   ---------------------------------------------------------------

   function RangeIsEmpty (Left, Right : Maths.Value) return Boolean
   is
      Unused      : Maths.ErrorCode;
      MathsResult : Maths.Value;
      FuncResult  : Boolean;
   begin
      --# accept Flow, 10, Unused, "Expected ineffective assignment" &
      --#        Flow, 33, Unused, "Expected to be neither referenced nor exported";
      Maths.Lesser (Right,
                    Left,
                     --to get
                    MathsResult,
                    Unused);  --not used because it can only be ok or type mismatch
      Maths.ValueToBool (MathsResult,
                           --to get
                         FuncResult,
                         Unused);
      return FuncResult;
   end RangeIsEmpty;

   ---------------------------------------------------------------

begin --wf_arange
   NextNode := Child_Node (Node);
   if SyntaxNodeType (NextNode) =
      ATTRIB_LOOKUP (IsAnnotation) then
      ExpStack.Pop (Result, EStack);
      if not Result.IsARange then
         Result.IsARange := True;
         Result.ErrorsInExpression := True;
         ErrorHandler.SemanticError (98,
                                     ErrorHandler.NoReference,
                                     NodePosition (NextNode),
                                     LexTokenManager.NullString);
      end if;
   else -- explicit range of the form "Left .. Right"
      ExpStack.Pop (Right, EStack);
      ExpStack.Pop (Left, EStack);

      Result := NullTypeRecord; --safety: we may not set all fields below

      -- In this case neither "Left" nor "Right" can themselves denote a Range.
      -- The following two checks prevent cases such as
      --   S'First .. S'Range
      --   S'Range .. S'Last
      --   S'Range .. S'Range
      -- which are all illegal.  We check both Left and Right separately so
      -- that two errors are issued for the latter case.

      if Left.IsARange then
         Result := UnknownTypeRecord;
         ErrorHandler.SemanticError (114,
                                     ErrorHandler.NoReference,
                                     NodePosition (NextNode),
                                     LexTokenManager.NullString);
      end if;

      if Right.IsARange then
         Result := UnknownTypeRecord;
         ErrorHandler.SemanticError (114,
                                     ErrorHandler.NoReference,
                                     NodePosition (Next_Sibling (NextNode)),
                                     LexTokenManager.NullString);
      end if;

      if not Left.IsARange and not Right.IsARange then

         -- Neither Left nor Right is a Range, so we can proceed...
         Result.IsConstant := Left.IsConstant and Right.IsConstant;
         Result.IsStatic   := Left.IsStatic and Right.IsStatic;
         Result.IsARange := True;
         LeftType  := Dictionary.GetRootType (Left.TypeSymbol);
         RightType := Dictionary.GetRootType (Right.TypeSymbol);
         IntegerImplicitTypeConversion (LeftType,
                                        RightType,
                                        Scope);
         RealImplicitTypeConversion (LeftType,
                                     RightType,
                                     Scope);
         if LeftType /= RightType then
            Result := UnknownTypeRecord;
            ErrorHandler.SemanticError (42,
                                        ErrorHandler.NoReference,
                                        NodePosition (Next_Sibling (NextNode)),
                                        LexTokenManager.NullString);
         elsif not (Dictionary.IsScalarType (LeftType, Scope) or else
                      Dictionary.IsUnknownTypeMark (LeftType))
         then
            Result := UnknownTypeRecord;
            ErrorHandler.SemanticError (44,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        LexTokenManager.NullString);
         else
            Result.TypeSymbol := LeftType;
            Result.Value := Left.Value;
            Result.RangeRHS := Right.Value;

            -- check that static range is non empty
            if (not IsAnnotation) and then
              RangeIsEmpty (Left.Value, Right.Value)
            then
               Result.Value    := Maths.NoValue;
               Result.RangeRHS := Maths.NoValue;
               ErrorHandler.SemanticError (409,
                                           ErrorHandler.NoReference,
                                           NodePosition (Node),
                                           LexTokenManager.NullString);
            end if;
         end if;
      end if;

      Result.ErrorsInExpression := Result.ErrorsInExpression or
         Left.ErrorsInExpression or
         Right.ErrorsInExpression;
   end if;

   ExpStack.Push (Result, EStack);

end wf_arange;
