-- $Id: sem-compunit-walkstatements.adb 12351 2009-02-02 15:03:51Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

separate (Sem.CompUnit)

procedure WalkStatements (SeqNode : in     STree.SyntaxNode;
                          Scope   : in     Dictionary.Scopes;
                          Table   : in out RefList.HashTable)
is

   --# inherit CompleteCheck,
   --#         Dictionary,
   --#         ExaminerConstants,
   --#         Sem,
   --#         SPARK_IO,
   --#         SystemErrors;
   package CaseStack
   --# own State;
   is
      procedure Init;
      --# global out State;
      --# derives State from ;

      procedure Push (CaseFlags   : in Sem.TypCaseFlags;
                      CompleteADT : in CompleteCheck.T;
                      Sym         : in Dictionary.Symbol;
                      LowerBound  : in Sem.TypTypeBound;
                      UpperBound  : in Sem.TypTypeBound);
      --# global in out State;
      --# derives State from *,
      --#                    CaseFlags,
      --#                    CompleteADT,
      --#                    LowerBound,
      --#                    Sym,
      --#                    UpperBound;

      procedure Pop (CaseFlags   : out Sem.TypCaseFlags;
                     CompleteADT : out CompleteCheck.T;
                     Sym         : out Dictionary.Symbol;
                     LowerBound  : out Sem.TypTypeBound;
                     UpperBound  : out Sem.TypTypeBound);
      --# global in out State;
      --# derives CaseFlags,
      --#         CompleteADT,
      --#         LowerBound,
      --#         State,
      --#         Sym,
      --#         UpperBound  from State;

   end CaseStack;

   -----------------------------------------------------------------
   --  VariableUpdateHistory
   --
   --  Description:
   --    An object of the ADT, VariableUpdateHistory.History_T, is used to
   --    maintain a list of variables (represented by natural numbers) and
   --    each variable has a single associated STree.SyntaxNode
   --    The node is used to record the last occurrence of an update
   --    of each variable as the syntax tree is traversed.
   --    To use an object of History_T it must fist be
   --    created using CreateHistory and must disposed of when its no
   --    longer required (and certainly before leaving the scope in
   --    which the History_T object is declared) using DisposeOfHistory
   --    If the specified Heap.HeapRecord object becomes exhausted
   --    the Examiner will fail with a fatal error.
   ------------------------------------------------------------------
   --# inherit ExaminerConstants,
   --#         Heap,
   --#         Statistics,
   --#         STree;
   package VariableUpdateHistory
   is
      type History_T is private;

      -----------------------------------------------------------------
      --  CreateHistory
      --
      --  Description:
      --    Initialises an object of type History_T.  This subprogram
      --    must be called prior to using the object.
      --    TheHeap object must be an initialised Heap.HeapRecord Object.
      ------------------------------------------------------------------
      procedure CreateHistory (TheHeap : in out Heap.HeapRecord;
                               History :    out History_T);
      --# global in out Statistics.TableUsage;
      --# derives History,
      --#         TheHeap               from TheHeap &
      --#         Statistics.TableUsage from *,
      --#                                    TheHeap;

      -----------------------------------------------------------------
      --  DisposeOfHistory
      --
      --  Description:
      --    Disposes of an object of type History_T.  This subprogram
      --    must be called when object is no longer required and
      --    certainly before leaving the scope in which the History_T
      --    object is declared.  TheHeap object must be the same object as
      --    was used in the call to CreateHistory for the History_T object.
      ------------------------------------------------------------------
      procedure DisposeOfHistory (TheHeap : in out Heap.HeapRecord;
                                  History : in     History_T);
      --# derives TheHeap from *,
      --#                      History;

      -----------------------------------------------------------------
      --  AddUpdate
      --
      --  Description:
      --    Adds a variable - node pair to the History_T object (History).
      --    If the Variable is not present in History, it is added to
      --    History along with its associated node (Node).
      --    Otherwise, if the Variable is present in the History, the
      --    node associated with the Variable is updated to the value
      --    of the given Node.
      --    TheHeap must be the same object as used in the preceding
      --    call to CreateHistory.
      ------------------------------------------------------------------
      procedure AddUpdate (TheHeap  : in out Heap.HeapRecord;
                           History  : in out History_T;
                           Variable : in     Natural;
                           Node     : in     STree.SyntaxNode);
      --# global in out Statistics.TableUsage;
      --# derives History,
      --#         TheHeap               from History,
      --#                                    Node,
      --#                                    TheHeap,
      --#                                    Variable &
      --#         Statistics.TableUsage from *,
      --#                                    History,
      --#                                    TheHeap,
      --#                                    Variable;

      -----------------------------------------------------------------
      --  GetLastUpdate
      --
      --  Description:
      --    Obtains the value of the node currently associated with the
      --    given variable in the History.  If the given Variable does
      --    not exist in the History a STree.NullNode will be
      --    returned.
      --    TheHeap must be the same object as used in the preceding
      --    call to CreateHistory.
      ------------------------------------------------------------------
      function GetLastUpdate (TheHeap  : Heap.HeapRecord;
                              History  : History_T;
                              Variable : Natural)
                              return STree.SyntaxNode;

   private
      type History_T is range 0 .. ExaminerConstants.HeapListLength;
   end VariableUpdateHistory;

   --------------------------------------------

   LScope    : Dictionary.Scopes;
   NextNode,
   LastNode  : STree.SyntaxNode;
   NodeType  : SPSymbols.SPSymbol;
   PureProtectedExportList : VariableUpdateHistory.History_T;
   --------------------------------------------

   package body CaseStack
   --# own State is S, TopPtr;
   is
      type StackRecord is record
         CaseFlags   : Sem.TypCaseFlags;
         CompleteADT : CompleteCheck.T;
         Sym         : Dictionary.Symbol;
         UpperBound  : Sem.TypTypeBound;
         LowerBound  : Sem.TypTypeBound;
      end record;

      NullRecord : constant StackRecord :=
        StackRecord'(CaseFlags   => Sem.NullCaseFlags,
                     CompleteADT => CompleteCheck.NullT,
                     Sym         => Dictionary.NullSymbol,
                     UpperBound  => Sem.UnknownTypeBound,
                     LowerBound  => Sem.UnknownTypeBound);

      subtype IndexRange is Integer range
         1 .. ExaminerConstants.WalkStmtStackMax;
      type StackArray is array (IndexRange) of StackRecord;
      subtype TopRange is Integer range
         0 .. ExaminerConstants.WalkStmtStackMax;

      S      : StackArray;
      TopPtr : TopRange;

      procedure Init
      --# global out S;
      --#        out TopPtr;
      --# derives S,
      --#         TopPtr from ;
      is
      begin
         TopPtr := 0;
         S      := StackArray'(others => NullRecord);
      end Init;

      procedure Push (CaseFlags   : in Sem.TypCaseFlags;
                      CompleteADT : in CompleteCheck.T;
                      Sym         : in Dictionary.Symbol;
                      LowerBound  : in Sem.TypTypeBound;
                      UpperBound  : in Sem.TypTypeBound)
      --# global in out S;
      --#        in out TopPtr;
      --# derives S      from *,
      --#                     CaseFlags,
      --#                     CompleteADT,
      --#                     LowerBound,
      --#                     Sym,
      --#                     TopPtr,
      --#                     UpperBound &
      --#         TopPtr from *;
      is
      begin
         if TopPtr = ExaminerConstants.WalkStmtStackMax then
            SystemErrors.FatalError (SystemErrors.CaseStackOverflow, "");
         end if;

         --# check TopPtr < ExaminerConstants.WalkStmtStackMax;

         TopPtr := TopPtr + 1;
         S (TopPtr) := StackRecord'(CaseFlags    => CaseFlags,
                                    CompleteADT => CompleteADT,
                                    Sym         => Sym,
                                    LowerBound  => LowerBound,
                                    UpperBound  => UpperBound);
      end Push;


      procedure Pop (CaseFlags   : out Sem.TypCaseFlags;
                     CompleteADT : out CompleteCheck.T;
                     Sym         : out Dictionary.Symbol;
                     LowerBound  : out Sem.TypTypeBound;
                     UpperBound  : out Sem.TypTypeBound)
      --# global in     S;
      --#        in out TopPtr;
      --# derives CaseFlags,
      --#         CompleteADT,
      --#         LowerBound,
      --#         Sym,
      --#         UpperBound  from S,
      --#                          TopPtr &
      --#         TopPtr      from *;
      is
      begin
         if TopPtr = 0 then
            SystemErrors.FatalError (SystemErrors.CaseStackUnderflow,
                                     "in WalkStatements.CaseStack.Pop");
         end if;

         --# check TopPtr > 0;

         CaseFlags := S (TopPtr).CaseFlags;
         CompleteADT := S (TopPtr).CompleteADT;
         Sym := S (TopPtr).Sym;
         LowerBound := S (TopPtr).LowerBound;
         UpperBound := S (TopPtr).UpperBound;
         TopPtr := TopPtr - 1;
      end Pop;

   end CaseStack;

   --------------------------------------------------------------------

   package body VariableUpdateHistory is separate;

   --------------------------------------------------------------------

   function IsLastInSequence (Node : STree.SyntaxNode)
                             return Boolean
   --# global in STree.Table;
      -- Node is an instance of a simple or compound statement;
   is
      LocalNode : STree.SyntaxNode;
      Result    : Boolean;

      function NoNodeOrOnlyPermittedNodesPresent (SeqNode : STree.SyntaxNode) return Boolean
      --# global in STree.Table;
      is
         Result : Boolean;
         SeqNodeToCheck : STree.SyntaxNode;

         function PermittedStatement (SeqNode : STree.SyntaxNode) return Boolean
         --# global in STree.Table;
         is
            Result : Boolean;
         begin
            -- First implementation of this function only allows justification_statements.
            -- It could be extended to allow pragmas and proof statements if desired.
            Result := SyntaxNodeType
              (Child_Node                           -- simple, compound, justification, proof statement or apragma
                 (Next_Sibling (SeqNode))) =      -- statement
              SPSymbols.justification_statement;

            return Result;
         end PermittedStatement;

      begin --NoNodeOrOnlyPermittedNodesPresent

         -- SeqNode is sequence_of_statements on entry.
         SeqNodeToCheck := SeqNode;
         loop
            -- exit when we get to the top of the sequence of statements and there are
            -- no more statements to check; this happens immediately on first pass through
            -- loop if there no statements of any kind after the one we are checking on entry
            -- to IsLastInSequence
            if SyntaxNodeType (ParentNode (SeqNodeToCheck)) /= SPSymbols.sequence_of_statements then
               -- it must be subprogram_implementation or something else but there definitely no
               -- more statements
               Result := True;
               exit;
            end if;

            -- failure case, a non-permitted statement
            if not PermittedStatement (SeqNodeToCheck) then
               Result := False;
               exit;
            end if;

            --move up chain of sequence_of_statements
            SeqNodeToCheck := ParentNode (SeqNodeToCheck);
         end loop;
         return Result;
      end NoNodeOrOnlyPermittedNodesPresent;

   begin -- IsLastInSequence

      -- On entry, node is one of: exit_statement, loop_statement or return_statement.
      -- These nodes, under certain circumstances are required to be thelast executable
      -- statement in a sequence of statements.
      --
      -- Grammar:
      --
      -- e.g. subprogram_implementation
      --                 |
      --        sequence_of_statements --- (designator, hidden part etc.)
      --                 |
      --        sequence_of_statements --- statement (last one in seq)
      --                 |
      --        sequence_of_statements --- statement
      --                 |
      --             statement  (first one in seq)
      --

      -- climb to simple_statement or compound_statement node;
      LocalNode := ParentNode (Node);
      -- climb to statement node;
      LocalNode := ParentNode (LocalNode);
      -- climb to sequence_of_statements;
      LocalNode := ParentNode (LocalNode);
      --
      -- If the Seq we have reached has no Next_Sibling then there are no statements of any kind
      -- after the node we are checking and which was passed in a parameter Node.
      -- If there is a Next_Sibling we need to check that the associated statements are
      -- permitted ones, i.e. they are not unreachable executable statements.
      --
      -- For now, we only do the permitted node check for the loop case; this is where we have a customer report
      -- and the other two cases have separate problems: allowing something after exit confuses the flow analyser
      -- and there is a separate redundant check in wf_subprogram_body that still traps return not being last
      -- statement even if we allow it here

      if SyntaxNodeType (Node) = SPSymbols.loop_statement then -- Node is what we entered with
         -- allow "permitted statements" after loop
         Result := NoNodeOrOnlyPermittedNodesPresent (LocalNode);
      else
         -- don't allow statements after exit or return
         Result := SyntaxNodeType (ParentNode (LocalNode)) /= SPSymbols.sequence_of_statements;
      end if;

      return Result;
   end IsLastInSequence;

   --------------------------------------------------------------------

   function ParentOfSequence (Node : STree.SyntaxNode)
                             return STree.SyntaxNode
   --# global in STree.Table;
      -- Node is an instance of a simple or compound statement;
      -- ParentOfSequence is the parent of the sequence_of_statements to which Node
      -- belongs.
   is
      LocalNode        : STree.SyntaxNode;
      PredecessorFound : Boolean;
   begin
      -- climb to simple_statement or compound_statement node;
      LocalNode := ParentNode (Node);
      -- climb to statement node;
      LocalNode := ParentNode (LocalNode);
      -- climb to sequence_of_statements node;
      LocalNode := ParentNode (LocalNode);
      PredecessorFound := False;
      while not PredecessorFound loop
         case SyntaxNodeType (LocalNode) is
            when SPSymbols.sequence_of_statements =>
               LocalNode := ParentNode (LocalNode);
            when others =>
               PredecessorFound := True;
         end case;
      end loop;
      return LocalNode;
   end ParentOfSequence;

   --------------------------------------------------------------------

   -- patch a relation so that implicit side-effects are included.  For
   -- stream exports this means adding a self-reference to the import list
   -- for stream imports it means adding a complete new relation deriving
   -- it from itself
   procedure AddStreamEffects (Table   : in out RefList.HashTable;
                               TheHeap : in out Heap.HeapRecord;
                               Node    : in     STree.SyntaxNode;
                               Export  : in     Dictionary.Symbol;
                               Imports : in     SeqAlgebra.Seq)
   --# global in     Dictionary.Dict;
   --#        in out Statistics.TableUsage;
   --# derives Statistics.TableUsage,
   --#         TheHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    Export,
   --#                                    Imports,
   --#                                    Node,
   --#                                    Table,
   --#                                    TheHeap &
   --#         Table                 from *,
   --#                                    Dictionary.Dict,
   --#                                    Imports,
   --#                                    Node,
   --#                                    TheHeap;
   is
      CurrentMember : SeqAlgebra.MemberOfSeq;
      CurrentImport : Dictionary.Symbol;

      -- add relation of the form Import <- {Import} to Node
      procedure AddSelfDependency (Table   : in out RefList.HashTable;
                                   TheHeap : in out Heap.HeapRecord;
                                   Node    : in     STree.SyntaxNode;
                                   Import  : in     Dictionary.Symbol)
      --# global in out Statistics.TableUsage;
      --# derives Statistics.TableUsage,
      --#         Table,
      --#         TheHeap               from *,
      --#                                    Import,
      --#                                    Node,
      --#                                    Table,
      --#                                    TheHeap;
      is
         ImportSeq : SeqAlgebra.Seq;
      begin --AddSelfDependency
         SeqAlgebra.CreateSeq (TheHeap, ImportSeq);
         SeqAlgebra.AddMember (TheHeap,
                               ImportSeq,
                               Natural (Dictionary.SymbolRef (Import)));
         RefList.AddRelation (Table,
                              TheHeap,
                              Node,
                              Import,
                              ImportSeq);
      end AddSelfDependency;

   begin -- AddStreamEffects
      -- traverse Import list adding self references for any IN streams found
      CurrentMember := SeqAlgebra.FirstMember (TheHeap, Imports);
      while not SeqAlgebra.IsNullMember (CurrentMember)
      loop
         CurrentImport := Dictionary.ConvertSymbolRef
            (ExaminerConstants.RefType
             (SeqAlgebra.ValueOfMember (TheHeap, CurrentMember)));

         -- if the CurrentImport is a stream then add a new identity relation
         --890--added GetMostEnclosingObject call
         if Dictionary.GetOwnVariableOrConstituentMode
            (Dictionary.GetMostEnclosingObject (CurrentImport)) /=
            Dictionary.DefaultMode then
            -- we know it is mode in because wffs prevent reading of mode outs
            AddSelfDependency (Table,
                               TheHeap,
                               Node,
                               CurrentImport);
         end if;

         CurrentMember := SeqAlgebra.NextMember (TheHeap, CurrentMember);
      end loop;

      -- finally, see if the Export is an OutStream and if so add it to the Import list
      --890--added GetMostEnclosingObject call
      if Dictionary.GetOwnVariableOrConstituentMode (Dictionary.GetMostEnclosingObject (Export)) /=
         Dictionary.DefaultMode then
         -- we know it is mode out because wffs prevent writing of mode ins
         SeqAlgebra.AddMember (TheHeap,
                               Imports,
                               Natural (Dictionary.SymbolRef (Export)));
      end if;
   end AddStreamEffects;

   --------------------------------------------------------------------

   procedure wf_assign (Node  : in STree.SyntaxNode;
                        Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out Table;
   --#        in out TheHeap;
   --#           out AggregateStack.State;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.StringTable,
   --#         STree.Table                 from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          SPARK_IO.File_Sys,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         Statistics.TableUsage,
   --#         Table,
   --#         TheHeap                     from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          Table,
   --#                                          TheHeap;
      is separate;

   --------------------------------------------------------------------

   procedure wf_condition (Node  : in STree.SyntaxNode;
                           Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out Table;
   --#        in out TheHeap;
   --#           out AggregateStack.State;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.StringTable,
   --#         STree.Table                 from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          SPARK_IO.File_Sys,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         Statistics.TableUsage,
   --#         Table,
   --#         TheHeap                     from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          Table,
   --#                                          TheHeap;
   is separate;

   --------------------------------------------------------------------
   --898
   -- This procedure prepares the flow analyser to deal with ininite loops.  If the loop
   -- has some means of exiting (iteration scheme or exit statements) then nothing is
   -- done.  Otherwise an empty referenced variable list is associated with the
   -- end_of_loop node to act as a stable exit expression in the manner of "exit when false".
   -- A Boolean type symbol is planted in the syntax tree at this point as a signal to the
   -- flow analyser that it should model the default exit as a way of providing a syntactic
   -- exit from the loop
   procedure SetUpDefaultLoopExit (Node  : in STree.SyntaxNode;
                                   Scope : in Dictionary.Scopes)
   --# global in     Dictionary.Dict;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out Table;
   --#        in out TheHeap;
   --# derives Statistics.TableUsage,
   --#         Table,
   --#         TheHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    Node,
   --#                                    Scope,
   --#                                    Table,
   --#                                    TheHeap &
   --#         STree.Table           from *,
   --#                                    Dictionary.Dict,
   --#                                    Node,
   --#                                    Scope;
   is
      RefVar          : SeqAlgebra.Seq;

   begin -- SetUpDefaultLoopExit

      -- and end_of_loop node is placed in the syntax tree after the sequence of statements
      -- it controls.  If the loop has no exits we attach and empty referenced variable
      -- list to this node so that that the flow analyser can pretend that there is an
      -- "exit when false" at this point.  We plant type Boolean in the syntax tree to signal
      --to the flow analyser that this default exit point is active.  If the loop has an
      -- iteration scheme or already has exits then we do nothing here.
      if not Dictionary.GetLoopHasExits (Dictionary.GetRegion (Scope)) then
         SeqAlgebra.CreateSeq (TheHeap, RefVar);
         RefList.AddRelation (Table,
                              TheHeap,
                              Node,
                              Dictionary.NullSymbol,
                              RefVar);
         STree.AddNodeSymbol (Node,
                                          Dictionary.GetPredefinedBooleanType);
      end if;
   end SetUpDefaultLoopExit;
   --898
   --------------------------------------------------------------------

   procedure wf_return (Node  : in STree.SyntaxNode;
                        Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.StringTable,
   --#         STree.Table                 from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          SPARK_IO.File_Sys,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         Statistics.TableUsage,
   --#         Table,
   --#         TheHeap                     from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          Table,
   --#                                          TheHeap;
   is separate;

   --------------------------------------------------------------------

   procedure wf_case (Node  : in STree.SyntaxNode;
                      Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in out CaseStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out Table;
   --#        in out TheHeap;
   --#           out AggregateStack.State;
   --# derives AggregateStack.State        from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         CaseStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.StringTable,
   --#         STree.Table                 from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          SPARK_IO.File_Sys,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         Statistics.TableUsage,
   --#         Table,
   --#         TheHeap                     from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          Table,
   --#                                          TheHeap;
   is separate;

   --------------------------------------------------------------------

   procedure wf_case_choice (Node  : in STree.SyntaxNode;
                             Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in out CaseStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out Table;
   --#        in out TheHeap;
   --#           out AggregateStack.State;
   --# derives AggregateStack.State,
   --#         CaseStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.StringTable,
   --#         STree.Table                 from CaseStack.State,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys           from CaseStack.State,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          SPARK_IO.File_Sys,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         Statistics.TableUsage,
   --#         Table,
   --#         TheHeap                     from *,
   --#                                          CaseStack.State,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          Table,
   --#                                          TheHeap;
      is separate;

   --------------------------------------------------------------------

   procedure wf_exit (Node          : in     STree.SyntaxNode;
                      TheLoop       : in     Dictionary.Symbol;
                      ConditionNode :    out STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ConditionNode             from Node,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        TheLoop;
      is separate;

   --------------------------------------------------------------------

   procedure wf_delay_until (Node  : in STree.SyntaxNode;
                             Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out Table;
   --#        in out TheHeap;
   --#           out AggregateStack.State;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.StringTable,
   --#         STree.Table                 from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          SPARK_IO.File_Sys,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         Statistics.TableUsage,
   --#         Table,
   --#         TheHeap                     from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          Table,
   --#                                          TheHeap;
      is separate;

   --------------------------------------------------------------------

   procedure down_loop (Node  : in     STree.SyntaxNode;
                        Scope : in out Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict           from *,
   --#                                        Node,
   --#                                        Scope,
   --#                                        STree.Table &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table &
   --#         Scope                     from Dictionary.Dict;
      is separate;

   --------------------------------------------------------------------

   procedure up_loop (Node  : in     STree.SyntaxNode;
                      Scope : in out Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table &
   --#         Scope                     from *,
   --#                                        Dictionary.Dict;
      is separate;

   --------------------------------------------------------------------

   procedure wf_loop_param (Node  : in STree.SyntaxNode;
                            Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.StringTable,
   --#         STree.Table                 from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          SPARK_IO.File_Sys,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         Statistics.TableUsage,
   --#         Table,
   --#         TheHeap                     from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          Table,
   --#                                          TheHeap;
      is separate;

   --------------------------------------------------------------------

   procedure wf_proc_call (Node  : in STree.SyntaxNode;
                           Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         LexTokenManager.StringTable from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         Table,
   --#         TheHeap                     from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          Table,
   --#                                          TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          SPARK_IO.File_Sys,
   --#                                          STree.Table,
   --#                                          Table,
   --#                                          TheHeap;
      is separate;

   --------------------------------------------------------------------

   procedure up_case (Node : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out CaseStack.State;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives CaseStack.State           from * &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CaseStack.State,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table;
      is separate;

   --------------------------------------------------------------------

   procedure Wf_Proof_Statement_Or_Loop_Invariant
     (Node  : in STree.SyntaxNode;
      Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --#           out AggregateStack.State;
   --# derives AggregateStack.State        from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.StringTable,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                     from *,
   --#                                          CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          STree.Table,
   --#                                          TheHeap &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          GlobalComponentData,
   --#                                          LexTokenManager.StringTable,
   --#                                          Node,
   --#                                          Scope,
   --#                                          SPARK_IO.File_Sys,
   --#                                          STree.Table,
   --#                                          TheHeap;
   is
      Unused : Boolean;
      pragma Unreferenced (Unused);
   begin
      -- Proof_Statement and Loop_Invariant have the same shape in the
      -- grammar, so this procedure WFFs both.
      --# accept Flow, 10, Unused, "Expected ineffective assignment to Unused";
      wf_predicate (Node           =>  Child_Node (Child_Node (Node)),
                    Scope          =>  Scope,
                    Context        =>  Postcondition, -- ~ may be allowed
                     --to get
                    ErrorsFound    =>  Unused);
      --# end accept;
      --# accept Flow, 33, Unused, "Expected to be neither referenced nor exported";
   end Wf_Proof_Statement_Or_Loop_Invariant;

   --------------------------------------------------------------------

   procedure InitComponentData (Scope         : in     Dictionary.Scopes;
                                ComponentData :    out ComponentManager.ComponentData;
                                TheHeap       : in out Heap.HeapRecord)
   --# global in     Dictionary.Dict;
   --#        in out Statistics.TableUsage;
   --# derives ComponentData         from Dictionary.Dict,
   --#                                    Scope,
   --#                                    TheHeap &
   --#         Statistics.TableUsage,
   --#         TheHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    Scope,
   --#                                    TheHeap;
   is
      Temp_ComponentData : ComponentManager.ComponentData;
      SubprogSym,
      OwnVarSym,
      ConstituentSym : Dictionary.Symbol;
      It,
      ConstituentIt  : Dictionary.Iterator;

      -----------------

      procedure MaybeAdd (Sym           : in     Dictionary.Symbol;
                          ComponentData : in out ComponentManager.ComponentData;
                          TheHeap       : in out Heap.HeapRecord)
      --# global in     Dictionary.Dict;
      --#        in out Statistics.TableUsage;
      --# derives ComponentData,
      --#         Statistics.TableUsage,
      --#         TheHeap               from *,
      --#                                    ComponentData,
      --#                                    Dictionary.Dict,
      --#                                    Sym,
      --#                                    TheHeap;
      is
      begin --MaybeAdd
         if Dictionary.TypeIsRecord (Dictionary.GetType (Sym)) then
            ComponentManager.AddRoot (ComponentData,
                                      TheHeap,
                                      Sym);
         end if;
      end MaybeAdd;

      -------------------

   begin --InitComponentData
         --ensure we start with a completely fresh copy of ComponentData
      ComponentManager.Initialise (Temp_ComponentData);

      --add roots of all record types as specified in S.P0468.53.9 para 6.1
      SubprogSym := Dictionary.GetRegion (Scope);
      if Dictionary.IsSubprogram (SubprogSym) or else
        (Dictionary.IsType (SubprogSym) and then
           Dictionary.TypeIsTask (SubprogSym)) then
         --initialize using parameters, globals and local variables
         --first do local variables
         It := Dictionary.FirstLocalVariable (SubprogSym);
         while not Dictionary.IsNullIterator (It)
         loop
            MaybeAdd (Dictionary.CurrentSymbol (It),
                      Temp_ComponentData,
                      TheHeap);
            It := Dictionary.NextSymbol (It);
         end loop;

         --then parameters
         It := Dictionary.FirstGlobalVariable (Dictionary.GetAbstraction (SubprogSym,
                                                                          Scope),
                                               SubprogSym);
         while not Dictionary.IsNullIterator (It)
         loop
            MaybeAdd (Dictionary.CurrentSymbol (It),
                      Temp_ComponentData,
                      TheHeap);
            It := Dictionary.NextSymbol (It);
         end loop;

         --then globals
         It := Dictionary.FirstSubprogramParameter (SubprogSym);
         while not Dictionary.IsNullIterator (It)
         loop
            MaybeAdd (Dictionary.CurrentSymbol (It),
                      Temp_ComponentData,
                      TheHeap);
            It := Dictionary.NextSymbol (It);
         end loop;

      else --package init part, initialize using own variable list, looking at constituents which
         --are not own variables of embedded packages where they are found.
         It := Dictionary.FirstOwnVariable (SubprogSym);
         while not Dictionary.IsNullIterator (It)
         loop
            OwnVarSym := Dictionary.CurrentSymbol (It);
            if Dictionary.IsConcreteOwnVariable (OwnVarSym) then
               MaybeAdd (OwnVarSym,
                         Temp_ComponentData,
                         TheHeap);
            else -- must be abstract
               ConstituentIt := Dictionary.FirstConstituent (OwnVarSym);
               while not Dictionary.IsNullIterator (ConstituentIt)
               loop
                  ConstituentSym := Dictionary.CurrentSymbol (ConstituentIt);
                  if not Dictionary.IsOwnVariable (ConstituentSym) then
                     MaybeAdd (ConstituentSym,
                               Temp_ComponentData,
                               TheHeap);
                  end if;
                  ConstituentIt := Dictionary.NextSymbol (ConstituentIt);
               end loop;
            end if;
            It := Dictionary.NextSymbol (It);
         end loop;
      end if;
      ComponentData := Temp_ComponentData;
   end InitComponentData;

   -----------------------------------------------------------------
   --  CheckForMutuallyExclusiveBranches
   --
   --  Description:
   --    Given two STree.SyntaxNodes, GivenNode and
   --    PrecedingNode, this subprogram checks that the GivenNode
   --    and the PrecedingNode are on mutually exclusive branches.
   --    For checking the updates of pure exported protected variables
   --    the PrecedingNode is the last update node as the syntax tree is
   --    traversed and the GivenNode is the update node just encountered.
   --    If the subprogram is repeatedly applied for each update node
   --    encountered (with its previous node) then this is sufficient
   --    to ensure that all updates are on mutually exclusive paths
   --    provided that AreMutuallyExclusive always has a return value
   --    of True.  See S.P0468.53.49.
   ------------------------------------------------------------------
   procedure CheckForMutuallyExclusiveBranches
      (GivenNode,
       PrecedingNode        : in     STree.SyntaxNode;
       TheHeap              : in out Heap.HeapRecord;
       AreMutuallyExclusive :    out Boolean
      )
   --# global in     STree.Table;
   --#        in out Statistics.TableUsage;
   --# derives AreMutuallyExclusive  from GivenNode,
   --#                                    PrecedingNode,
   --#                                    STree.Table,
   --#                                    TheHeap &
   --#         Statistics.TableUsage,
   --#         TheHeap               from *,
   --#                                    GivenNode,
   --#                                    PrecedingNode,
   --#                                    STree.Table,
   --#                                    TheHeap;
   is separate;

   --------------------------------------------------------------------

   procedure CheckPureProtectedExportsUpdatedOnceOnly
     (Node                    : in     STree.SyntaxNode;
      Scope                   : in     Dictionary.Scopes;
      PureProtectedExportList : in out VariableUpdateHistory.History_T)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in     Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        PureProtectedExportList,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        STree.Table,
   --#                                        Table,
   --#                                        TheHeap &
   --#         PureProtectedExportList,
   --#         Statistics.TableUsage,
   --#         TheHeap                   from *,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        Node,
   --#                                        PureProtectedExportList,
   --#                                        STree.Table,
   --#                                        Table,
   --#                                        TheHeap;
   is
      TheExport                   : Natural;
      TheExportAtom               : Heap.Atom;
      TheExportSym                : Dictionary.Symbol;
      TheImports                  : SeqAlgebra.Seq;
      PreviousNode                : STree.SyntaxNode;
      OnMutuallyExclusiveBranches : Boolean;
   begin
      -- does this node has exports
      if RefList.NodeHasExportList (Table   => Table,
                                    TheHeap => TheHeap,
                                    Node    => Node) then

         -- Get the first export for this node.
         RefList.FirstExport (Table     => Table,
                              TheHeap   => TheHeap,
                              Node      => Node,
                              TheExport => TheExportAtom);

         while not Heap.IsNullPointer (TheExportAtom) loop

            -- Get the symbol for this export.
            TheExportSym := Dictionary.ConvertSymbolRef
              (ExaminerConstants.RefType
                 (Heap.AValue (TheHeap => TheHeap,
                               A       => TheExportAtom)));

            -- Is it protected?
            if Dictionary.IsOwnVariable (TheExportSym) and then

               -- do not report error if the thing being updates is a suspension object
               not Dictionary.IsPredefinedSuspensionObjectType (Dictionary.GetType (TheExportSym)) and then

               Dictionary.GetOwnVariableProtected (TheExportSym) and then

               -- does it have a wellformed set of imports?
               -- Note: the imports list will be empty if there are none.
               RefList.ExportHasDependencies (TheExport => TheExportAtom,
                                             TheHeap   => TheHeap) then

               -- Get the imports for this export.
               TheImports := RefList.DependencyList (TheHeap   => TheHeap,
                                                     TheExport => TheExportAtom);

               TheExport := Heap.AValue (TheHeap => TheHeap,
                                         A       => TheExportAtom);

               -- Is it a pure export? (pure exports don't depend on themselves)
               if not SeqAlgebra.IsMember
                 (TheHeap    => TheHeap,
                  S          => TheImports,
                  GivenValue => TheExport) then

                  -- It's a pure export.
                  -- Has this export already been updated?
                  PreviousNode :=
                     VariableUpdateHistory.GetLastUpdate (
                        TheHeap,
                        PureProtectedExportList,
                        TheExport);
                  if PreviousNode = STree.NullNode then
                     -- The export has not been previously encountered
                     -- Add the pure export to the update history
                     VariableUpdateHistory.AddUpdate (TheHeap,
                                                      PureProtectedExportList,
                                                      TheExport,
                                                      Node);
                  else
                     -- The export has previously been updated, check
                     -- that the previous update was on a mutually
                     -- exlusive path
                     CheckForMutuallyExclusiveBranches (Node,
                                                        PreviousNode,
                                                        TheHeap,
                                                        OnMutuallyExclusiveBranches);
                     if OnMutuallyExclusiveBranches then
                        -- The update is valid, add the new pair to the export list
                        -- replacing the current pair involving TheExport.
                        VariableUpdateHistory.AddUpdate (TheHeap,
                                                         PureProtectedExportList,
                                                         TheExport,
                                                         Node);
                     else
                        -- semantic error.
                        ErrorHandler.SemanticErrorSym (957,
                                                       ErrorHandler.NoReference,
                                                       NodePosition (Node),
                                                       TheExportSym,
                                                       Scope);
                     end if;
                  end if;
               end if;
            end if;

            TheExportAtom := RefList.NextExport (TheHeap   => TheHeap,
                                                 TheExport => TheExportAtom);
         end loop;
      end if;
   end CheckPureProtectedExportsUpdatedOnceOnly;

   --------------------------------------------------------------------

begin -- WalkStatements
   CaseStack.Init;
   InitComponentData (Scope, GlobalComponentData, TheHeap);

   -- Create a VariablUpdateHistory to retain the immediately previous
   -- node at which the update of each pure export protected varible occurred.
   VariableUpdateHistory.CreateHistory (TheHeap,
                                        PureProtectedExportList);

   -- scope may change locally in loops but will always
   -- be back to original scope on exit from procedure
   LScope := Scope;

   NextNode := SeqNode;
   loop -- Down loop --------------------------------------------------
      LastNode := NextNode;
      NodeType := SyntaxNodeType (LastNode);
      case NodeType is
         when SPSymbols.assignment_statement =>
            wf_assign (LastNode, LScope);

            CheckPureProtectedExportsUpdatedOnceOnly
              (Node                    => LastNode,
               Scope                   => LScope,
               PureProtectedExportList => PureProtectedExportList);

            NextNode := STree.NullNode;

         when SPSymbols.condition =>
            wf_condition (LastNode, LScope);
            NextNode := STree.NullNode;

         when SPSymbols.exit_statement =>
            Dictionary.MarkLoopHasExits (Dictionary.GetRegion (LScope)); -- 898
            wf_exit (LastNode, Dictionary.GetRegion (LScope), NextNode);

            -- 898
         when SPSymbols.end_of_loop =>
            SetUpDefaultLoopExit (LastNode, LScope);
            NextNode := Child_Node (LastNode);
            -- 898

         when SPSymbols.case_statement =>
            wf_case (LastNode, LScope);
            NextNode := Child_Node (LastNode);

         when SPSymbols.case_choice =>
            wf_case_choice (LastNode, LScope);
            NextNode := STree.NullNode;

         when SPSymbols.return_statement =>
            wf_return (LastNode, LScope);
            NextNode := STree.NullNode;

         when SPSymbols.loop_statement =>
            down_loop (LastNode,
                        --using and to get
                       LScope);
            NextNode := Child_Node (LastNode);

         when SPSymbols.loop_invariant =>
            -- Loop invariant which preceeds the "loop" keyword is part of the
            -- loop iteration scheme, so doesn't appear as a statement, so we
            -- must WFF is here.
            Wf_Proof_Statement_Or_Loop_Invariant (LastNode, LScope);
            NextNode := STree.NullNode;

         when SPSymbols.loop_parameter_specification =>
            wf_loop_param (LastNode, LScope);
            NextNode := Child_Node (LastNode);

         when SPSymbols.apragma =>
            wf_pragma (LastNode, LScope);
            NextNode := STree.NullNode;

         when SPSymbols.procedure_call_statement =>
            wf_proc_call (LastNode, LScope);

            CheckPureProtectedExportsUpdatedOnceOnly
              (Node                    => LastNode,
               Scope                   => LScope,
               PureProtectedExportList => PureProtectedExportList);

            NextNode := STree.NullNode;

         when SPSymbols.proof_statement =>
            Wf_Proof_Statement_Or_Loop_Invariant (LastNode, LScope);
            NextNode := STree.NullNode;

         when SPSymbols.expression =>
            NextNode := STree.NullNode;

         when SPSymbols.delay_statement =>
            --# accept Flow, 41, "Expected stable expression";
            if CommandLineData.RavenscarSelected then -- expect stable
            --# end accept;
               wf_delay_until (LastNode, LScope);
            else -- illegal
               ErrorHandler.SemanticError (850,
                                           ErrorHandler.NoReference,
                                           NodePosition (LastNode),
                                           LexTokenManager.NullString);
            end if;
            NextNode := STree.NullNode;

            --PNA---------------------
         when SPSymbols.justification_statement =>
            wf_justification_statement (LastNode,
                                        LScope);
            NextNode := STree.NullNode;

         when others =>
            if NodeType in SPSymbols.SPNonTerminal then
               NextNode := Child_Node (LastNode);
            else
               NextNode := STree.NullNode;
            end if;
      end case;

      -- Up Loop -------------------------------------------------------
      if NextNode = STree.NullNode then
         loop
            NextNode := Next_Sibling (LastNode);
            exit when NextNode /= STree.NullNode; -- new branch to right

            NextNode := ParentNode (LastNode);
            LastNode := NextNode;
            NodeType := SyntaxNodeType (LastNode);
            case NodeType is
               when SPSymbols.loop_statement =>
                  up_loop (LastNode,
                           -- using and to get
                           LScope);

               when SPSymbols.case_statement =>
                  up_case (LastNode);
               when others =>
                  null;
            end case;

            exit when NextNode = SeqNode;  -- got back to top
         end loop; -- up
      end if;

      exit when NextNode = SeqNode; -- met start point on way up

   end loop; -- down

   -- Dispose of the update history
   VariableUpdateHistory.DisposeOfHistory (TheHeap,
                                          PureProtectedExportList);
end WalkStatements;
