-- $Id: sem-compunit-walkstatements-wf_exit.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.CompUnit.WalkStatements)

procedure wf_exit (Node          : in     STree.SyntaxNode;
                   TheLoop       : in     Dictionary.Symbol;
                   ConditionNode :    out STree.SyntaxNode)
is
   IfNode,
   LocalNode : STree.SyntaxNode;
   ExitLabel : STree.SyntaxNode;
   Condition : STree.SyntaxNode;
begin
   -- ASSUME Node = exit_statement
   -- The procedure checks that the conditions
   -- of Section 5.7 of the SPARK Definition apply to the exit statement.

   LocalNode := Next_Sibling (Child_Node (Node));

   if LocalNode = STree.NullNode then
      -- A simple exit statement - no label identifier and no exit condition
      Condition := STree.NullNode;
      ExitLabel := STree.NullNode;

   elsif SyntaxNodeType (LocalNode) = SPSymbols.simple_name then
      -- Exit has a label name
      Condition := Next_Sibling (LocalNode); -- get the exit condition
      ExitLabel := Child_Node (LocalNode); -- get the label identifier

   else
      -- Must be an exit with a condition but no label
      Condition := LocalNode;
      ExitLabel := STree.NullNode;

   end if;

   ConditionNode := Condition;

   if ExitLabel /= STree.NullNode then
      -- Exit names a loop label.  It must match the label attached to the
      -- most closely enclosing loop statement.
      if (not Dictionary.LoopHasName (TheLoop)) or else
         Dictionary.GetSimpleName (TheLoop) /= NodeLexString (ExitLabel) then
         -- Enclosing loop does not have a label, or labels
         -- are present, but do not match
         ErrorHandler.SemanticError (724,
                                     ErrorHandler.NoReference,
                                     NodePosition (ExitLabel),
                                     LexTokenManager.NullString);
      end if;
   end if;

   -- determine whether exit statement contains a when clause;
   if Condition /= STree.NullNode then
      -- exit statement contains a when clause, therefore check that condition (2)
      -- of SPARK Definition Section 5.7 applies, i.e. check that closest-
      -- containing compound statement is a loop statement;
      LocalNode := ParentOfSequence (Node);
      case SyntaxNodeType (LocalNode) is
         when SPSymbols.loop_statement =>
            null;
         when others =>
            ErrorHandler.ControlFlowError (ErrorHandler.MisplacedExit,
                                           NodePosition (Node));
      end case;
   else
      -- exit statement is in an if_statement, therefore check that condition (3)
      -- of SPARK Definition Section 5.7 applies:
      -- check that exit-statement is last in its sequence of statements;
      if IsLastInSequence (Node) then
         -- check that closest containing compound statement is an if_statement;
         LocalNode := ParentOfSequence (Node);
         case SyntaxNodeType (LocalNode) is
            when SPSymbols.if_statement =>
               -- check remainder of condition (3);
               IfNode := LocalNode;
               -- advance to condition node;
               LocalNode := Child_Node (LocalNode);
               -- advance to sequence_of_statements node;
               LocalNode := Next_Sibling (LocalNode);
               -- advance to elsif node;
               LocalNode := Next_Sibling (LocalNode);
               -- check that elsif_part is null;
               if Child_Node (LocalNode) = STree.NullNode then
                  -- advance to else node;
                  LocalNode := Next_Sibling (LocalNode);
                  -- check that else_part is null;
                  if Child_Node (LocalNode) = STree.NullNode then
                     -- check that closest-containing compound statement is a loop statement;
                     LocalNode := ParentOfSequence (IfNode);
                     case SyntaxNodeType (LocalNode) is
                        when SPSymbols.loop_statement =>
                           null;
                        when others =>
                           ErrorHandler.ControlFlowError (ErrorHandler.MisplacedExit,
                                                          NodePosition (Node));
                     end case;
                  else
                     ErrorHandler.ControlFlowError (ErrorHandler.MisplacedExit,
                                                    NodePosition (Node));
                  end if;
               else
                  ErrorHandler.ControlFlowError (ErrorHandler.MisplacedExit,
                                                 NodePosition (Node));
               end if;
            when others =>
               ErrorHandler.ControlFlowError (ErrorHandler.MisplacedExit,
                                              NodePosition (Node));
         end case;
      else
         ErrorHandler.ControlFlowError (ErrorHandler.MisplacedExit,
                                        NodePosition (Node));
      end if;
   end if;
end wf_exit;
