-- $Id: sem-compunit-walkstatements-wf_case_choice.adb 11946 2008-12-18 16:11:11Z rod chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.CompUnit.WalkStatements)

procedure wf_case_choice (Node  : in STree.SyntaxNode;
                          Scope : in Dictionary.Scopes)
is
   type CaseChoiceSorts is (SingleExpression, ExplicitRange, RangeConstraint);

   CaseChoiceSort : CaseChoiceSorts;
   FirstNode,
   SecondNode     : STree.SyntaxNode;
   FirstResult,
   SecondResult   : ExpRecord;
   RefVar         : SeqAlgebra.Seq;

   CaseFlags          : TypCaseFlags;
   CompleteADT        : CompleteCheck.T;
   CaseTypeSymbol     : Dictionary.Symbol;
   CaseTypeLowerBound : TypTypeBound;
   CaseTypeUpperBound : TypTypeBound;

   IsARange              : Boolean;     -- these refer to the value/range
   ChoiceLowerMathsValue : Maths.Value; -- specified in the case choice
   ChoiceUpperMathsValue : Maths.Value
      := Maths.NoValue;  -- init to remove spurious flowerrs
   ChoiceLowerBound,
   ChoiceUpperBound      : TypTypeBound
      := UnknownTypeBound;  -- init to remove spurious flowerrs

   LowerBoundUnknown     : Boolean;
   UpperBoundUnknown     : Boolean := False;
   LowerBoundOutOfRange  : Boolean;
   UpperBoundOutOfRange  : Boolean := False;

   SemanticErrorsFound : Boolean := False;

   OutOfRangeSeen      : Boolean;
   OverlapSeen         : CompleteCheck.TypOverlapState;
   BothChoiceBoundsKnown : Boolean := False;
   RangeConstraintLowerBound     : TypTypeBound;
   RangeConstraintUpperBound     : TypTypeBound;

   ------------------------------------------------------------------------

   procedure ConvertChoiceBound (MathsValue      : in     Maths.Value;
                                 Bound           :    out TypTypeBound;
                                 UnknownBound    :    out Boolean;
                                 BoundOutOfRange :    out Boolean)
   --# derives Bound,
   --#         BoundOutOfRange,
   --#         UnknownBound    from MathsValue;
   is
      Int        : Integer;
      MathsError : Maths.ErrorCode;
   begin
      if Maths.HasNoValue (MathsValue) then
         Bound := TypTypeBound'(Value     => 0,
                                IsDefined => False);
         UnknownBound := True;
         BoundOutOfRange := False;
      else
         Maths.ValueToInteger (MathsValue, Int, MathsError);
         if MathsError = Maths.NoError then
            Bound := TypTypeBound'(Value     => Int,
                                   IsDefined => True);
            UnknownBound := False;
            BoundOutOfRange := False;
         else
            Bound := TypTypeBound'(Value     => 0,
                                   IsDefined => False);
            UnknownBound := False;
            BoundOutOfRange := True;
         end if;
      end if;
   end ConvertChoiceBound;

   ------------------------------------------------------------------------
   -- note: returns True if any of the bounds is undefined, unless the
   -- choice is not a range, in which case, ChoiceUpper is unused
   function IsChoiceInRange (ChoiceLower   : TypTypeBound;
                             ChoiceUpper   : TypTypeBound;
                             ChoiceIsRange : Boolean;
                             RangeLower    : TypTypeBound;
                             RangeUpper    : TypTypeBound) return Boolean
   is
      Result : Boolean;
   begin
      if (ChoiceLower.IsDefined and RangeLower.IsDefined
          and ChoiceLower.Value < RangeLower.Value)
         or
         (ChoiceLower.IsDefined and RangeUpper.IsDefined
          and ChoiceLower.Value > RangeUpper.Value)
         or
         (ChoiceIsRange and ChoiceUpper.IsDefined and RangeUpper.IsDefined
          and ChoiceUpper.Value > RangeUpper.Value)
      then
         Result := False;
      else
         Result := True;
      end if;

      return Result;
   end IsChoiceInRange;

   ------------------------------------------------------------------------

   function IsRangeEmpty (RangeLower : TypTypeBound;
                          RangeUpper : TypTypeBound) return Boolean
   is
      Result : Boolean;
   begin
      if RangeLower.IsDefined and RangeUpper.IsDefined and
         RangeLower.Value > RangeUpper.Value
      then
         Result := True;
      else
         Result := False;
      end if;

      return Result;
   end IsRangeEmpty;

   -----------------------------------------------------------------------

   procedure ConvertBooleanMathsValue (Value : in out Maths.Value)
   --# derives Value from *;
   is
   begin
      if Value = Maths.FalseValue then
         Value := Maths.ZeroInteger;
      elsif Value = Maths.TrueValue then
         Value := Maths.OneInteger;
      end if;
   end ConvertBooleanMathsValue;

   -----------------------------------------------------------------------

begin -- wf_case_choice
      -- ASSUME Node = case_choice

   SecondResult := UnknownTypeRecord;
   CaseStack.Pop (CaseFlags,
                  CompleteADT,
                  CaseTypeSymbol,
                  CaseTypeLowerBound,
                  CaseTypeUpperBound);


   FirstNode := Child_Node (Node);
   SeqAlgebra.CreateSeq (TheHeap, RefVar);
   WalkExpression (ExpNode               => FirstNode,
                   Scope                 => Scope,
                   TypeContext           => CaseTypeSymbol,
                   ContextRequiresStatic => False,
                     --to get
                   Result  => FirstResult,
                   RefVar  => RefVar,
                   ComponentData => GlobalComponentData);
   SecondNode := Next_Sibling (FirstNode);

   --# assert True;

   if SecondNode = STree.NullNode then
      CaseChoiceSort := SingleExpression;
   else
      WalkExpression (ExpNode               => SecondNode,
                      Scope                 => Scope,
                      TypeContext           => CaseTypeSymbol,
                      ContextRequiresStatic => False,
                        --to get
                      Result  => SecondResult,
                      RefVar  => RefVar,
                      ComponentData => GlobalComponentData);
      if SyntaxNodeType (SecondNode) =
         SPSymbols.simple_expression
      then
         CaseChoiceSort := ExplicitRange;
      else
         CaseChoiceSort := RangeConstraint;
      end if;
   end if;

   --# assert True;

   case CaseChoiceSort is
      when SingleExpression =>
         if not FirstResult.IsStatic then
            ErrorHandler.SemanticError (36,
                                        1,
                                        NodePosition (FirstNode),
                                        LexTokenManager.NullString);
            SemanticErrorsFound := True;
         end if;
         if not Dictionary.CompatibleTypes (Scope,
                                            FirstResult.TypeSymbol,
                                            CaseTypeSymbol)
         then
            ErrorHandler.SemanticError (38,
                                        ErrorHandler.NoReference,
                                        NodePosition (FirstNode),
                                        LexTokenManager.NullString);
            SemanticErrorsFound := True;
         end if;

         -- code to work out whether we have a single choice or a
         -- range and to collect the appropriate values
         -- note that these will be nonsense if semantic errors have been found
         ChoiceLowerMathsValue := FirstResult.Value;
         if FirstResult.IsARange then
            IsARange := True;
            ChoiceUpperMathsValue := FirstResult.RangeRHS;
         else
            IsARange := False;
         end if;

      when ExplicitRange =>
         if not Dictionary.CompatibleTypes (Scope,
                                            FirstResult.TypeSymbol,
                                            SecondResult.TypeSymbol)
         then
            ErrorHandler.SemanticError (42,
                                        ErrorHandler.NoReference,
                                        NodePosition (SecondNode),
                                        LexTokenManager.NullString);
            SemanticErrorsFound := True;
         elsif not Dictionary.CompatibleTypes (Scope,
                                               FirstResult.TypeSymbol,
                                               CaseTypeSymbol)
         then
            ErrorHandler.SemanticError (106,
                                        ErrorHandler.NoReference,
                                        NodePosition (FirstNode),
                                        LexTokenManager.NullString);
            SemanticErrorsFound := True;
         end if;
         if not (FirstResult.IsStatic and SecondResult.IsStatic) then
            ErrorHandler.SemanticError (45,
                                        1,
                                        NodePosition (FirstNode),
                                        LexTokenManager.NullString);
            SemanticErrorsFound := True;
         end if;

         -- code to collect the appropriate values for the extent of the range
         -- note that these will be nonsense if semantic errors have been found
         ChoiceLowerMathsValue := FirstResult.Value;
         ChoiceUpperMathsValue := SecondResult.Value;
         IsARange := True;

      when RangeConstraint =>
         if not Dictionary.CompatibleTypes (Scope,
                                            FirstResult.TypeSymbol,
                                            SecondResult.TypeSymbol)
         then
            ErrorHandler.SemanticError (42,
                                        ErrorHandler.NoReference,
                                        NodePosition (SecondNode),
                                        LexTokenManager.NullString);
            SemanticErrorsFound := True;
         elsif not Dictionary.CompatibleTypes (Scope,
                                               FirstResult.TypeSymbol,
                                               CaseTypeSymbol)
         then
            ErrorHandler.SemanticError (106,
                                        ErrorHandler.NoReference,
                                        NodePosition (FirstNode),
                                        LexTokenManager.NullString);
            SemanticErrorsFound := True;
         end if;
         if not (FirstResult.IsConstant and FirstResult.IsARange) then
            ErrorHandler.SemanticError (95,
                                        ErrorHandler.NoReference,
                                        NodePosition (FirstNode),
                                        LexTokenManager.NullString);
            SemanticErrorsFound := True;
         end if;
         if not SecondResult.IsStatic then
            ErrorHandler.SemanticError (45,
                                        1,
                                        NodePosition (SecondNode),
                                        LexTokenManager.NullString);
            SemanticErrorsFound := True;
         end if;

         -- code to collect the appropriate values for the extent of the range
         -- note that these will be nonsense if semantic errors have been found
         ChoiceLowerMathsValue := SecondResult.Value;
         ChoiceUpperMathsValue := SecondResult.RangeRHS;
         IsARange := True;

         -- somewhere need to check that SecondResult range is within the type
         -- given by FirstResult
   end case;

   --# assert True;

   if not SemanticErrorsFound then
      ConvertBooleanMathsValue (ChoiceLowerMathsValue);
      ConvertChoiceBound (ChoiceLowerMathsValue,
                          ChoiceLowerBound,
                          LowerBoundUnknown,
                          LowerBoundOutOfRange);

      if IsARange then
         ConvertBooleanMathsValue (ChoiceUpperMathsValue);
         ConvertChoiceBound (ChoiceUpperMathsValue,
                             ChoiceUpperBound,
                             UpperBoundUnknown,
                             UpperBoundOutOfRange);
      else
         ChoiceUpperBound := UnknownTypeBound;
      end if;

      if LowerBoundOutOfRange or (IsARange and then UpperBoundOutOfRange) then
         BothChoiceBoundsKnown := False;
         ErrorHandler.SemanticWarning (305,
                                       NodePosition (FirstNode),
                                       LexTokenManager.NullString);
      elsif LowerBoundUnknown or (IsARange and UpperBoundUnknown) then
         BothChoiceBoundsKnown := False;
         CompleteADT.Undeterminable := True;
         ErrorHandler.SemanticWarning (200,
                                       NodePosition (FirstNode),
                                       LexTokenManager.NullString);
      else
         BothChoiceBoundsKnown := True;
      end if;

      --# assert True;

      if BothChoiceBoundsKnown then
         -- check the case choice lies within controlling type
         if not IsChoiceInRange (ChoiceLowerBound,
                                 ChoiceUpperBound,
                                 IsARange,
                                 CaseTypeLowerBound,
                                 CaseTypeUpperBound)
         then
            if CaseChoiceSort = RangeConstraint then
               ErrorHandler.SemanticError (410,
                                           ErrorHandler.NoReference,
                                           NodePosition (SecondNode),
                                           LexTokenManager.NullString);
            else
               ErrorHandler.SemanticError (410,
                                           ErrorHandler.NoReference,
                                           NodePosition (FirstNode),
                                           LexTokenManager.NullString);
            end if;
            SemanticErrorsFound := True;
         elsif IsARange and IsRangeEmpty (ChoiceLowerBound,
                                          ChoiceUpperBound)
         then
            if CaseChoiceSort = RangeConstraint then
               ErrorHandler.SemanticError (409,
                                           ErrorHandler.NoReference,
                                           NodePosition (SecondNode),
                                           LexTokenManager.NullString);
            else
               ErrorHandler.SemanticError (409,
                                           ErrorHandler.NoReference,
                                           NodePosition (FirstNode),
                                           LexTokenManager.NullString);
            end if;
            SemanticErrorsFound := True;
         end if;

         --# assert True;

         -- check the case choice lies within RangeConstraint type
         if CaseChoiceSort = RangeConstraint then
            GetTypeBounds (FirstResult.TypeSymbol,
                           RangeConstraintLowerBound,
                           RangeConstraintUpperBound);

            if not IsChoiceInRange (ChoiceLowerBound,
                                    ChoiceUpperBound,
                                    IsARange,
                                    RangeConstraintLowerBound,
                                    RangeConstraintUpperBound)
            then
               ErrorHandler.SemanticError (413,
                                           ErrorHandler.NoReference,
                                           NodePosition (SecondNode),
                                           LexTokenManager.NullString);
               SemanticErrorsFound := True;
            end if;
         end if;
      end if;
   end if;

   --# assert True;

   if (not SemanticErrorsFound)
      and then BothChoiceBoundsKnown
      and then (CaseFlags.CheckCompleteness or CaseFlags.CheckOverlap)
   then
      if IsARange then
         CompleteCheck.SeenRange (CompleteADT,
                                  ChoiceLowerBound.Value,
                                  ChoiceUpperBound.Value,
                                  OutOfRangeSeen,
                                  OverlapSeen);
      else
         CompleteCheck.SeenElement (CompleteADT,
                                    ChoiceLowerBound.Value,
                                    OutOfRangeSeen,
                                    OverlapSeen);
      end if;
      if OutOfRangeSeen then
         CaseFlags.OutOfRangeSeen := True;
      end if;
      if CaseFlags.CheckOverlap and OverlapSeen = CompleteCheck.Overlap then
         ErrorHandler.SemanticError (407,
                                     ErrorHandler.NoReference,
                                     NodePosition (FirstNode),
                                     LexTokenManager.NullString);
         SemanticErrorsFound := True;
      end if;
   end if;

   -- add reference variable list to RefList hash table
   RefList.AddRelation (Table,
                        TheHeap,
                        Node,
                        Dictionary.NullSymbol,
                        RefVar);

   --# assert True;

   if SemanticErrorsFound then
      CaseFlags.CheckCompleteness := False;
   end if;

   CaseStack.Push (CaseFlags,
                   CompleteADT,
                   CaseTypeSymbol,
                   CaseTypeLowerBound,
                   CaseTypeUpperBound);
end wf_case_choice;
