-- $Id: sem-compunit-walkstatements-wf_assign.adb 11946 2008-12-18 16:11:11Z rod chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.CompUnit.WalkStatements)
procedure wf_assign (Node  : in STree.SyntaxNode;
                     Scope : in Dictionary.Scopes)
is
   NameResult,
   ExpResult    : ExpRecord;
   ExpectedType,
   NameSym      : Dictionary.Symbol;
   MostEnclosingNameSym : Dictionary.Symbol;
   NameNode,
   ExpNode      : STree.SyntaxNode;
   RefVar       : SeqAlgebra.Seq;

   ---------------------------------------------------------------------

   procedure CheckWriteToStructuredVar (NameSym,
                                        ExpectedType : in Dictionary.Symbol)
   --# global in     Dictionary.Dict;
   --#        in     RefVar;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives Statistics.TableUsage,
   --#         TheHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    ExpectedType,
   --#                                    NameSym,
   --#                                    RefVar,
   --#                                    TheHeap;
   is
   begin
      if ExpectedType /= Dictionary.GetType (NameSym) then
         --we must be writing to a component of a structured variable
         --so must add structure variable to list of referenced variables
         SeqAlgebra.AddMember (TheHeap,
                               RefVar,
                               Natural (Dictionary.SymbolRef (NameSym)));
      end if;
   end CheckWriteToStructuredVar;

   ---------------------------------------------------------------------

begin -- wf_assignment_statement
      -- ASSUME Node = assignment_statement
      -- This procedure checks the following:
      -- (1) the assigned identifier is declared and visible, and
      -- (2) it is a variable, and
      -- (3) this variable is not a loop parameter, and
      -- (4) it is not a formal parameter of mode in, and
      -- (5) it may be a package own var declared in a non-enclosing scope but
      --     a warning is given.
      -- additions for streams
      -- (6) check that the assigned variable is not of mode in
      -- (7) check that assigning expression is not a mode out variable
   NameNode := Child_Node (Node);
   ExpNode  := Next_Sibling (NameNode);
   SeqAlgebra.CreateSeq (TheHeap, RefVar);
   --# accept Flow, 10, AggregateStack.State, "Expected ineffective assignment";
   WalkExpression (  -- ineffective AS.S, ES.S, LH all OK
                     ExpNode               => NameNode,
                     Scope                 => Scope,
                     TypeContext           => Dictionary.GetUnknownTypeMark,
                     ContextRequiresStatic => False,
                     --to get
                     Result  => NameResult,
                     RefVar  => RefVar,
                     ComponentData => GlobalComponentData);
   --# end accept;
   NameSym := NameResult.OtherSymbol;
   MostEnclosingNameSym := Dictionary.GetMostEnclosingObject (NameSym);
   if not NameResult.IsAVariable and then
      not Dictionary.IsUnknownTypeMark (NameResult.TypeSymbol)
   then
      ExpectedType := Dictionary.GetUnknownTypeMark;
      ErrorHandler.SemanticError (609,
                                  14,
                                  NodePosition (NameNode),
                                  LexTokenManager.NullString);
   elsif NameSym = Dictionary.NullSymbol then
      ExpectedType := Dictionary.GetUnknownTypeMark;
   else
      ExpectedType := NameResult.TypeSymbol;

      -- seed syntax tree with expected type for run-time check
      STree.AddNodeSymbol (Node, ExpectedType); --353
      if Dictionary.IsVariableOrSubcomponent (NameSym) then
         -- Check for assignment
         --     to structured variables so that they generate a reference to the
         --     variable as well; A (I) = 3; is a reference of A as well as write
         --     to it.  Call moved here because if A is not a variable in the
         --     first place then the check is meaningless.
         CheckWriteToStructuredVar (NameSym,
                                    ExpectedType);

         --  check that target is not unconstrained array
         if Dictionary.IsUnconstrainedArrayType (ExpectedType) then
            ErrorHandler.SemanticError (39,
                                        ErrorHandler.NoReference,
                                        NodePosition (NameNode),
                                        LexTokenManager.NullString);
         elsif Dictionary.IsLoopParameter (NameSym) then
            ErrorHandler.SemanticError (168,
                                        ErrorHandler.NoReference,
                                        NodePosition (NameNode),
                                        Dictionary.GetSimpleName (NameSym));
         elsif not InPackageInitialization (Scope) and then
            Dictionary.IsOwnVariable (MostEnclosingNameSym) and then
            not IsEnclosingPackage (Dictionary.GetOwner (MostEnclosingNameSym),
                                    Scope)
         then
            ErrorHandler.SemanticWarningSym (169,
                                             NodePosition (NameNode),
                                             NameSym,
                                             Scope);
         elsif InPackageInitialization (Scope) and then
            UnexpectedInitialization (MostEnclosingNameSym)
         then
            ErrorHandler.SemanticError (333,
                                        ErrorHandler.NoReference,
                                        NodePosition (NameNode),
                                        Dictionary.GetSimpleName (MostEnclosingNameSym));
         elsif InPackageInitialization (Scope) and then
           Dictionary.PartitionElaborationPolicyIsConcurrent and then
           Dictionary.IsOwnVariable (NameSym) and then
           Dictionary.GetOwnVariableMode (NameSym) = Dictionary.DefaultMode and then
           (Dictionary.GetOwnVariableProtected (NameSym) or else
              Dictionary.IsVirtualElement (NameSym))
         then
            ErrorHandler.SemanticError (874,
                                        ErrorHandler.NoReference,
                                        NodePosition (NameNode),
                                        Dictionary.GetSimpleName (MostEnclosingNameSym));
         elsif Dictionary.IsSubprogramParameter (MostEnclosingNameSym) then
            if Dictionary.GetSubprogramParameterMode (MostEnclosingNameSym) =
               Dictionary.InMode or
               Dictionary.GetSubprogramParameterMode (MostEnclosingNameSym) =
               Dictionary.DefaultMode
            then
               ErrorHandler.SemanticError (170,
                                           ErrorHandler.NoReference,
                                           NodePosition (NameNode),
                                           Dictionary.GetSimpleName (MostEnclosingNameSym));
            end if;

            -- check for direct update of global by function
         elsif Dictionary.IsFunction (Dictionary.GetEnclosingCompilationUnit (Scope)) and then
            Dictionary.IsGlobalVariable (Dictionary.GetAbstraction
                                         (Dictionary.GetEnclosingCompilationUnit (Scope),
                                          Scope),
                                         Dictionary.GetEnclosingCompilationUnit (Scope),
                                         MostEnclosingNameSym)
         then
            ErrorHandler.SemanticError (327,
                                        ErrorHandler.NoReference,
                                        NodePosition (NameNode),
                                        Dictionary.GetSimpleName (MostEnclosingNameSym));
         elsif Dictionary.GetOwnVariableOrConstituentMode (MostEnclosingNameSym) =
            Dictionary.InMode then
            ErrorHandler.SemanticError (717,
                                        ErrorHandler.NoReference,
                                        NodePosition (NameNode),
                                        Dictionary.GetSimpleName (MostEnclosingNameSym));
         end if;
      else
         ExpectedType := Dictionary.GetUnknownTypeMark;
         ErrorHandler.SemanticError (6,
                                     ErrorHandler.NoReference,
                                     NodePosition (NameNode),
                                     Dictionary.GetSimpleName (NameSym));
      end if;
   end if;

   WalkExpression (ExpNode,
                   Scope,
                   ExpectedType, -- Should contextual type analysis be needed, the
                                 -- expression on the RHS
                                 -- is expected to have the type of the var. on the LHS,
                   False,
                     --to get
                   ExpResult,
                   RefVar,
                   GlobalComponentData);

   AssignmentCheck (NodePosition (ExpNode),
                    Scope,
                    ExpectedType,
                    ExpResult);

   -- check that we are not trying to read an out stream
   if ExpResult.IsAVariable and then
      Dictionary.GetOwnVariableOrConstituentMode (ExpResult.VariableSymbol) =
      Dictionary.OutMode then
      ErrorHandler.SemanticErrorSym (718,
                                     ErrorHandler.NoReference,
                                     NodePosition (ExpNode),
                                     ExpResult.VariableSymbol,
                                     Scope);
   end if;

   -- if expression represents an IN stream variable then put type of expression
   -- into syntax tree for the benefit of the RTC procedure ModelAssignmentStatement
   if ExpResult.IsAVariable and then
      Dictionary.GetOwnVariableOrConstituentMode (ExpResult.VariableSymbol) =
      Dictionary.InMode then
      -- Mark the enclosing compilation unit as assigning an external variable
      -- This may be too coarse; may be we should just mark enclosing subprog?
      Dictionary.AddAssignsFromExternal (
         Dictionary.GetEnclosingCompilationUnit (Scope));

      STree.AddNodeSymbol (ExpNode, ExpResult.TypeSymbol);
      -- Check to see if the variable has been marked as always valid.
      -- Note that the OtherSymbol is checked,not the variableSymbol,
      -- since this will be the Subcomponent symbol if we are referring to
      -- a record component
      if Dictionary.VariableOrSubcomponentIsMarkedValid (ExpResult.OtherSymbol) then
         --MCA & TJJ: do we also need to add a use of 'Always_Valid to the summary?
         --Debug.PrintSym ("Access is Always_Valid =", ExpResult.OtherSymbol);
         null;
      else
         -- and issue warning about possible validity problems.
         -- The warning is stronger when the external variable is a type that doesn't
         -- generate run-time checks
         if Dictionary.TypeIsScalar (ExpResult.TypeSymbol) and then
           not Dictionary.TypeIsBoolean (ExpResult.TypeSymbol) then
            --weaker warning
            ErrorHandler.SemanticWarningSym (392,
                                             NodePosition (ExpNode),
                                             ExpResult.OtherSymbol,
                                             Scope);
         else
            --stronger warning
            ErrorHandler.SemanticWarningSym (393,
                                             NodePosition (ExpNode),
                                             ExpResult.OtherSymbol,
                                             Scope);
         end if;
      end if;
   end if;

   -- if the expression represents a use of unchecked conversion then plant the return
   -- type in the syntax tree for the benefit of the RTC procedure ModelAssignmentStatement
   -- TJJ: Note a more explicit way of designating and checking for this would
   -- be better so that it is easier to determine the extent of use of this idiom.
   if Dictionary.IsAnUncheckedConversion (ExpResult.OtherSymbol) then
      STree.AddNodeSymbol (ExpNode,
                                       Dictionary.GetType (ExpResult.OtherSymbol));
   end if;

   if Dictionary.TypeIsLimited (ExpResult.TypeSymbol, Scope) then
      ErrorHandler.SemanticError (308,
                                  ErrorHandler.NoReference,
                                  NodePosition (ExpNode),
                                  LexTokenManager.NullString);
   end if;
   -- patch flow relations to take into account stream volatility
   AddStreamEffects (Table,
                     TheHeap,
                     Node,
                     NameSym,
                     RefVar);
   -- add export and list of imports to RefList hash table
   RefList.AddRelation (Table,
                        TheHeap,
                        Node,
                        NameSym,
                        RefVar);
end wf_assign;
