-- $Id: sem-compunit-walkannotationexpression.adb 11946 2008-12-18 16:11:11Z rod chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with SPrint;
with ELStrings;

separate (Sem.CompUnit)

procedure WalkAnnotationExpression
   (ExpNode        : in     STree.SyntaxNode;
    Scope          : in     Dictionary.Scopes;
    TypeContext    : in     Dictionary.Symbol;
    Context        : in     Anno_Tilde_Context;
    Result         :    out ExpRecord;
    ComponentData  : in out ComponentManager.ComponentData)
is

   LocalNode,
   NextNode,
   LastNode    : STree.SyntaxNode;
   NodeType    : SPSymbols.SPSymbol;
   LHeap       : Lists.ListHeap;
   Val         : Maths.Value;
   EStack      : ExpStack.ExpStackType;
   TStack      : TypeContextStack.TStackType;
   RefVar      : SeqAlgebra.Seq;
   LScope      : Dictionary.Scopes;
   TempResult  : ExpRecord;

   --------------------------------------------------------------
   -- Procedures for debugging Expression syntax and tree walking
   -- These are hidden with "derives ;" so as not to pollute the
   -- annotations
   --------------------------------------------------------------

   procedure Dump_Syntax_Tree
   --# derives ;
   is
      --# hide Dump_Syntax_Tree;
   begin
      if CommandLineData.Content.Debug.Expressions then
         SPrint.Dump_Syntax_Tree (ExpNode, 0);
      end if;
   end Dump_Syntax_Tree;

   procedure Dump_Down_Node
   --# derives ;
   is
      --# hide Dump_Down_Node;
   begin
      if CommandLineData.Content.Debug.Expressions then
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                            "WalkAnnotationExpression DOWN encounters node" &
                            ExaminerConstants.RefType'Image (STree.NodeToRef (LastNode)) &
                            ' ' &
                            SPSymbols.SPSymbol'Image (SyntaxNodeType (LastNode)),
                            0);
      end if;
   end Dump_Down_Node;

   procedure Dump_Up_Node
   --# derives ;
   is
      --# hide Dump_Up_Node;
   begin
      if CommandLineData.Content.Debug.Expressions then
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                            "WalkAnnotationExpression UP   encounters node" &
                            ExaminerConstants.RefType'Image (STree.NodeToRef (LastNode)) &
                            ' ' &
                            SPSymbols.SPSymbol'Image (SyntaxNodeType (LastNode)),
                            0);
      end if;
   end Dump_Up_Node;

   procedure Dump_Result
   --# derives ;
   is
      --# hide Dump_Result;
   begin
      if CommandLineData.Content.Debug.Expressions then
         if TempResult.IsConstant then
            SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                                 "WalkAnnotationExpression constant result is ", 0);
            ELStrings.PutLine (SPARK_IO.Standard_Output,
                                         Maths.ValueToString (TempResult.Value));
         else
            SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                                 "WalkAnnotationExpression result is not constant", 0);
         end if;

      end if;
   end Dump_Result;

begin --WalkAnnotationExpression

   Dump_Syntax_Tree;

   ExpStack.Init (EStack);
   AggregateStack.Init;
   Lists.Init (LHeap);
   TypeContextStack.Init (TStack);
   TypeContextStack.Push (TypeContext, TStack);

   LScope := Scope; -- scope may change locally in loops but will always
   -- be back to original scope on exit from procedure.
   -- In all calls below LScope replaces Scope.

   SeqAlgebra.CreateSeq (TheHeap  => TheHeap,
                         S        => RefVar);

   NextNode := ExpNode;
   loop -----------------------------------------------------down loop----
      LastNode := NextNode;
      NodeType := SyntaxNodeType (LastNode);

      Dump_Down_Node;

      case NodeType is
         when SPSymbols.character_literal =>
            GetCharacterLiteral (LastNode, Val);
            ExpStack.Push (ExpRecord'(TypeSymbol            => Dictionary.GetPredefinedCharacterType,
                                      OtherSymbol           => Dictionary.NullSymbol,
                                      StreamSymbol          => Dictionary.NullSymbol,
                                      TaggedParameterSymbol => Dictionary.NullSymbol,
                                      VariableSymbol        => Dictionary.NullSymbol,
                                      ParamCount            => 0,
                                      ParamList             => Lists.NullList,
                                      Sort                  => TypeResult,
                                      ArgListFound          => False,
                                      IsAVariable           => False,
                                      IsAnEntireVariable    => False,
                                      ErrorsInExpression    => False,
                                      HasOperators          => False,
                                      IsConstant            => True,
                                      IsStatic              => True,
                                      IsARange              => False,
                                      Value                 => Val,
                                      RangeRHS              => Maths.NoValue),
                           EStack);
            NextNode := STree.NullNode;

         when SPSymbols.string_literal =>
            GetStringLiteralLength (LastNode,
                                    --to get
                                    Val);

            ExpStack.Push (ExpRecord'(TypeSymbol            => Dictionary.GetPredefinedStringType,
                                      OtherSymbol           => Dictionary.NullSymbol,
                                      StreamSymbol          => Dictionary.NullSymbol,
                                      TaggedParameterSymbol => Dictionary.NullSymbol,
                                      VariableSymbol        => Dictionary.NullSymbol,
                                      ParamCount            => 0,
                                      ParamList             => Lists.NullList,
                                      Sort                  => TypeResult,
                                      ArgListFound          => False,
                                      IsAVariable           => False,
                                      IsAnEntireVariable    => False,
                                      ErrorsInExpression    => False,
                                      HasOperators          => False,
                                      IsConstant            => True,
                                      IsStatic              => CommandLineData.IsSpark95,
                                      IsARange              => False,
                                      Value                 => Maths.NoValue,
                                      RangeRHS              => Val),
                           EStack);
            NextNode := STree.NullNode;

         when SPSymbols.numeric_literal =>
            LocalNode := Child_Node (Child_Node (LastNode));
            GetLiteralValue (LocalNode, Val);
            if SyntaxNodeType (LocalNode) = SPSymbols.real_number then
               ExpStack.Push (ExpRecord'(TypeSymbol            => Dictionary.GetUniversalRealType,
                                         OtherSymbol           => Dictionary.NullSymbol,
                                         StreamSymbol          => Dictionary.NullSymbol,
                                         TaggedParameterSymbol => Dictionary.NullSymbol,
                                         VariableSymbol        => Dictionary.NullSymbol,
                                         ParamCount            => 0,
                                         ParamList             => Lists.NullList,
                                         Sort                  => TypeResult,
                                         ArgListFound          => False,
                                         IsAVariable           => False,
                                         IsAnEntireVariable    => False,
                                         ErrorsInExpression    => False,
                                         HasOperators          => False,
                                         IsConstant            => True,
                                         IsStatic              => True,
                                         IsARange              => False,
                                         Value                 => Val,
                                         RangeRHS              => Maths.NoValue),
                              EStack);
            else
               ExpStack.Push (ExpRecord'(TypeSymbol  => Dictionary.GetUniversalIntegerType,
                                         OtherSymbol           => Dictionary.NullSymbol,
                                         StreamSymbol          => Dictionary.NullSymbol,
                                         TaggedParameterSymbol => Dictionary.NullSymbol,
                                         VariableSymbol        => Dictionary.NullSymbol,
                                         ParamCount            => 0,
                                         ParamList             => Lists.NullList,
                                         Sort                  => TypeResult,
                                         ArgListFound          => False,
                                         IsAVariable           => False,
                                         IsAnEntireVariable    => False,
                                         ErrorsInExpression    => False,
                                         HasOperators          => False,
                                         IsConstant            => True,
                                         IsStatic              => True,
                                         IsARange              => False,
                                         Value                 => Val,
                                         RangeRHS              => Maths.NoValue),
                              EStack);
            end if;
            NextNode := STree.NullNode;

         when SPSymbols.annotation_selector =>
            NextNode := STree.NullNode;

         when SPSymbols.annotation_simple_name =>
            if SyntaxNodeType (ParentNode (LastNode)) =
               SPSymbols.annotation_named_argument_association
            then
               --do not look at identifier in this case
               NextNode := STree.NullNode;
            else
               NextNode := Child_Node (LastNode);
            end if;

         when SPSymbols.identifier =>
            wf_identifier (Node => LastNode,
                           Scope => LScope,
                           EStack => EStack,
                           RefVar => RefVar,
                           Context => Context,
                           IsAnnotation   => True);
            NextNode := STree.NullNode;

         when SPSymbols.annotation_name_argument_list =>
            down_wf_name_argument_list (Node         => LastNode,
                                        Scope        => LScope,
                                        EStack       => EStack,
                                        HeapParam    => LHeap,
                                        NextNode     => NextNode,
                                        IsAnnotation => True);

         when SPSymbols.annotation_aggregate =>
            down_wf_aggregate (Node         => LastNode,
                               Scope        => LScope,
                               NextNode     => NextNode,
                               EStack       => EStack,
                               HeapParam    => LHeap,
                               IsAnnotation => True);


         when SPSymbols.annotation_aggregate_choice_rep =>
            down_wf_aggregate_choice_rep (Node         => LastNode,
                                          Scope        => LScope,
                                          EStack       => EStack,
                                          HeapParam    => LHeap,
                                          NextNode     => NextNode,
                                          IsAnnotation => True);

         when SPSymbols.record_component_selector_name =>
            wf_record_component_selector_name (Node         => LastNode,
                                               Scope        => Scope,
                                               EStack       => EStack,
                                               HeapParam    => LHeap,
                                               NextNode     => NextNode);

         when SPSymbols.annotation_aggregate_or_expression =>
            down_wf_aggregate_or_expression (Node         => LastNode,
                                             EStack       => EStack,
                                             IsAnnotation => True,
                                             NextNode     => NextNode);

         when SPSymbols.store =>
            down_wf_store (LastNode,
                           LScope,
                           EStack,
                           --to get
                           NextNode);

         when SPSymbols.store_list =>
            down_wf_store_list (LastNode,
                                LScope,
                                EStack,
                                 --to get
                                NextNode);

         when SPSymbols.quantified_expression =>
            down_wf_quantifier (LastNode,
                                 --using and to get
                                EStack,
                                LScope,
                                 --to get
                                NextNode);



         when SPSymbols.annotation_attribute_designator =>

            TypeContextStack.Push
              (AttributeDesignatorTypeFromContext (LastNode, False, EStack, TStack), TStack);

            NextNode := Child_Node (LastNode);

         when SPSymbols.annotation_range_constraint =>

            TypeContextStack.Push
              (RangeConstraintTypeFromContext (LastNode, EStack, TStack), TStack);

            NextNode := Child_Node (LastNode);

         when SPSymbols.annotation_simple_expression =>

            TypeContextStack.Push
              (SimpleExpressionTypeFromContext (LastNode, TStack), TStack);

            NextNode := Child_Node (LastNode);

         when SPSymbols.annotation_expression =>

            TypeContextStack.Push
              (ExpressionTypeFromContext (LastNode, EStack, TStack), TStack);

            NextNode := Child_Node (LastNode);

         when SPSymbols.annotation_primary =>

            TypeContextStack.Push
              (PrimaryTypeFromContext (LastNode, TStack), TStack);

            NextNode := Child_Node (LastNode);


         when others =>
            if NodeType in SPSymbols.SPNonTerminal then
               NextNode := Child_Node (LastNode);
            else
               NextNode := STree.NullNode;
            end if;
      end case;

      -------------------------------------------------up loop----------
      if NextNode = STree.NullNode then
         loop
            NextNode := Next_Sibling (LastNode);
            exit when NextNode /= STree.NullNode; --new branch to right

            NextNode := ParentNode (LastNode);
            LastNode := NextNode;
            NodeType := SyntaxNodeType (LastNode);

            Dump_Up_Node;

            case NodeType is
               when SPSymbols.annotation_expression =>
                  wf_expression (Node         => LastNode,
                                 Scope        => LScope,
                                 EStack       => EStack,
                                 TStack       => TStack,
                                 IsAnnotation => True);

                  TypeContextStack.Pop (TStack);

               when  SPSymbols.annotation_expression_rep1 |
                  SPSymbols.annotation_expression_rep2 |
                  SPSymbols.annotation_expression_rep3 |
                  SPSymbols.annotation_expression_rep4 |
                  SPSymbols.annotation_expression_rep5 |
                  SPSymbols.annotation_expression_rep6 |
                  SPSymbols.annotation_expression_rep7 =>

                  wf_expression (Node         => LastNode,
                                 Scope        => LScope,
                                 EStack       => EStack,
                                 TStack       => TStack,
                                 IsAnnotation => True);

               when SPSymbols.annotation_simple_expression =>
                  wf_simple_expression (Node                  => LastNode,
                                        Scope                 => LScope,
                                        EStack                => EStack,
                                        TStack                => TStack,
                                        ContextRequiresStatic => False,
                                        IsAnnotation          => True);

                  TypeContextStack.Pop (TStack);

               when SPSymbols.annotation_simple_expression_opt =>
                  wf_simple_expression_opt (Node         => LastNode,
                                            Scope        => LScope,
                                            EStack       => EStack,
                                            TStack       => TStack,
                                            IsAnnotation => True);

               when SPSymbols.annotation_term =>
                  wf_term (Node                  => LastNode,
                           Scope                 => LScope,
                           EStack                => EStack,
                           TStack                => TStack,
                           ContextRequiresStatic => False,
                           IsAnnotation          => True);

               when SPSymbols.annotation_factor =>
                  wf_factor (Node         => LastNode,
                             Scope        => LScope,
                             EStack       => EStack,
                             TStack       => TStack,
                             IsAnnotation => True);

               when SPSymbols.annotation_relation =>
                  wf_relation (Node         => LastNode,
                               Scope        => LScope,
                               EStack       => EStack,
                               TStack       => TStack,
                               IsAnnotation => True);

               when SPSymbols.annotation_range_constraint =>

                  TypeContextStack.Pop (TStack);

               when SPSymbols.annotation_arange =>
                  wf_arange (Node         => LastNode,
                             Scope        => LScope,
                             EStack       => EStack,
                             IsAnnotation => True);

               when SPSymbols.annotation_selected_component =>
                  wf_selected_component (Node           => LastNode,
                                         Scope          => LScope,
                                         EStack         => EStack,
                                         RefVar         => RefVar,
                                         ComponentData  => ComponentData,
                                         Context        => Context,
                                         IsAnnotation   => True);

               when SPSymbols.annotation_attribute =>
                  wf_attribute (EStack);

               when SPSymbols.annotation_attribute_designator =>
                  wf_attribute_designator (Node         => LastNode,
                                           Scope        => LScope,
                                           EStack       => EStack,
                                           RefVar       => RefVar,
                                           IsAnnotation => True);

                  TypeContextStack.Pop (TStack);

               when SPSymbols.annotation_primary =>
                  wf_primary (Node          => LastNode,
                              Scope         => LScope,
                              EStack        => EStack,
                              RefVar        => RefVar,
                              IsAnnotation  => True,
                              ComponentData => ComponentData);

                  TypeContextStack.Pop (TStack);

               when SPSymbols.annotation_positional_argument_association =>

                  wf_positional_argument_association (Node     => LastNode, --may be changed
                                                      Scope        => LScope,
                                                      EStack       => EStack,
                                                      IsAnnotation => True,
                                                      RefVar        => RefVar,
                                                      ComponentData => ComponentData);

               when SPSymbols.annotation_named_argument_association =>
                  wf_named_argument_association (Node         => LastNode,
                                                 Scope        => LScope,
                                                 EStack       => EStack,
                                                 HeapParam    => LHeap,
                                                 IsAnnotation => True);

               when SPSymbols.annotation_name_argument_list =>
                  up_wf_name_argument_list (Node         => LastNode,
                                            EStack       => EStack,
                                            HeapParam    => LHeap,
                                            IsAnnotation => True);

               when SPSymbols.annotation_ancestor_part =>
                  wf_ancestor_part (Node         => LastNode, -- may be changed
                                    Scope        => Scope,
                                    EStack       => EStack,
                                    HeapParam    => LHeap,
                                    IsAnnotation => True);

               when SPSymbols.annotation_aggregate_choice =>
                  up_wf_aggregate_choice (Node         => LastNode,
                                          Scope        => LScope,
                                          EStack       => EStack);

               when SPSymbols.annotation_named_association_rep =>
                  up_wf_named_association_rep (Node         => LastNode,
                                               Scope        => LScope,
                                               EStack       => EStack,
                                               HeapParam    => LHeap,
                                               IsAnnotation => True);

               when SPSymbols.annotation_named_record_component_association =>
                  up_wf_named_record_component_association (Node         => LastNode,
                                                            Scope        => Scope,
                                                            EStack       => EStack,
                                                            HeapParam    => LHeap);

               when SPSymbols.annotation_positional_association |
                 SPSymbols.annotation_record_component_association =>
                  up_wf_positional_association (Node         => LastNode,
                                                EStack       => EStack);

               when SPSymbols.annotation_aggregate_or_expression =>
                  up_wf_aggregate_or_expression (Node         => LastNode, -- may be changed
                                                 Scope        => LScope,
                                                 EStack       => EStack,
                                                 IsAnnotation => True);

               when SPSymbols.annotation_positional_record_component_association =>
                  wf_positional_record_component_association
                    (Node         => LastNode,  -- may be changed by call
                     Scope        => Scope,
                     EStack       => EStack,
                     IsAnnotation => True);

               when SPSymbols.annotation_component_association =>
                  up_wf_component_association (Node         => LastNode,
                                               Scope        => LScope,
                                               EStack       => EStack,
                                               IsAnnotation => True);

               when SPSymbols.annotation_aggregate        |
                 SPSymbols.annotation_extension_aggregate =>
                  up_wf_aggregate (Node         => LastNode,
                                   Scope        => LScope,
                                   EStack       => EStack,
                                   IsAnnotation => True);

               when SPSymbols.annotation_qualified_expression =>
                  wf_qualified_expression (Node         => LastNode,
                                           Scope        => LScope,
                                           EStack       => EStack,
                                           IsAnnotation => True);

               when SPSymbols.store_list =>
                  wf_store_list (LastNode,  --LastNode may be changed by call
                                 LScope,
                                 EStack);

               when SPSymbols.store =>
                  up_wf_store (LastNode,
                               LScope,
                               EStack);

               when SPSymbols.quantified_expression =>
                  up_wf_quantifier (LastNode,
                                    EStack,
                                    --to get
                                    LScope);

               when others =>
                  null;

            end case;

            exit when NextNode = ExpNode;  --got back to top
         end loop; --up
      end if;

      exit when NextNode = ExpNode; --met start point on way up

   end loop; --down

   if not ExpStack.HasOneEntry (EStack) then
      SystemErrors.FatalError (SystemErrors.ExpressionStackCorrupt, "in WalkAnnotationExpression");
   end if;

   if not (TypeContextStack.HasOneEntry (TStack) and then
           TypeContextStack.Top (TStack) = TypeContext) then
      SystemErrors.FatalError (SystemErrors.TypeContextStackCorrupt, "in WalkAnnotationExpression");
   end if;

   --# accept Flow, 10, EStack, "Expected ineffective assignment";
   ExpStack.Pop (TempResult, EStack); -- expect ineffective assignment
   --# end accept;

   Dump_Result;

   Result := TempResult;

   SeqAlgebra.DisposeOfSeq (TheHeap => TheHeap,
                            S       => RefVar);


end WalkAnnotationExpression;  -- AggregateStack will appear later
