-- $Id: sem-compunit-up_wf_subprogram_body.adb 12351 2009-02-02 15:03:51Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.CompUnit)

procedure up_wf_subprogram_body (Node  : in     STree.SyntaxNode;
                                 Scope : in out Dictionary.Scopes)
is
   IdentNode,
   EndDesigNode : STree.SyntaxNode;
   IdentStr     : LexTokenManager.LexString;
   SubProgSym   : Dictionary.Symbol;
   Abstraction  : Dictionary.Abstractions;

   ---------------------------------------------------------

   procedure CheckGlobalImportsAreInitialized (Sym  : in Dictionary.Symbol;
                                               Node : in STree.SyntaxNode)
   --# global in     Abstraction;
   --#        in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        Sym;
   is
      It     : Dictionary.Iterator;
      VarSym : Dictionary.Symbol;

   begin
      It := Dictionary.FirstGlobalVariable (Abstraction, Sym);
      while not Dictionary.IsNullIterator (It) loop
         VarSym := Dictionary.CurrentSymbol (It);
         if Dictionary.IsImport (Abstraction,
                                 Sym,
                                 VarSym) and then
            not Dictionary.OwnVariableIsInitialized (VarSym) and then
            Dictionary.GetOwnVariableOrConstituentMode (VarSym) = Dictionary.DefaultMode
         then
            ErrorHandler.SemanticError (167,
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        Dictionary.GetSimpleName (VarSym));
         end if;
         It := Dictionary.NextSymbol (It);
      end loop;
   end CheckGlobalImportsAreInitialized;

   ---------------------------------------------------------

   procedure MainProgramCeilingPriorityCheck (Sym  : in Dictionary.Symbol;
                                              Node : in STree.SyntaxNode)
   --# global in     Abstraction;
   --#        in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from Abstraction,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        Scope,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table,
   --#                                        Sym;
   is
      It                     : Dictionary.Iterator;
      MainProgramPriorityLex : LexTokenManager.LexString;
   begin
      if Dictionary.MainProgramPrioritySupplied then
         MainProgramPriorityLex := Dictionary.GetMainProgramPriority;
         if MainProgramPriorityLex /= LexTokenManager.NullString then
            -- We have a valid value for the priority. This will have been range-checked if
            -- Priority has been supplied in the Config file. We can do the ceiling check
            -- irrespective of whether the range check was performed or not, as long as the
            -- priority values are known not to be out of any supplied range.
            -- The Lex value was created using StorageRep in CheckPriorityPragma, so we can
            -- convert it back to a Value using ValueRep.
            It := Dictionary.FirstGlobalVariable (Abstraction, Sym);
            CheckCeilingPriority (Sym              => Sym,
                                  Scope            => Scope,
                                  CheckList        => It,
                                  PriorityLexValue => MainProgramPriorityLex,
                                  ErrorNode        => Node);
         else
            -- An out of range Priority value was supplied for Main. This will have already
            -- been reported as a semantic error, so we don't need any further errors or
            -- warnings here, but of course we can't do the ceiling check.
            null;
         end if;
      elsif Dictionary.BodyIsHidden (Sym) then
         -- Pragma priority may be there but is unavailable.
         ErrorHandler.SemanticWarning (311,
                                       NodePosition (Node),
                                       Dictionary.GetSimpleName (Sym));
      else
         -- "A pragma Priority is required for the main program"
         ErrorHandler.SemanticError (933,
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     LexTokenManager.NullString);
      end if;
   end MainProgramCeilingPriorityCheck;  -- Expect Unused is not used

   ---------------------------------------------------------

   procedure CheckDelayPropertyAccountedFor (ProcOrTask : in     Dictionary.Symbol;
                                             Node       : in     STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     STree.Table;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Node,
   --#                                        ProcOrTask,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        STree.Table;
   is
   begin
      if Dictionary.HasDelayProperty (ProcOrTask) and then
        (not Dictionary.DelayPropertyIsAccountedFor (ProcOrTask)) and then
        (not Dictionary.BodyIsHidden (ProcOrTask)) then
         ErrorHandler.SemanticError (915,
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     Dictionary.GetSimpleName (ProcOrTask));
      end if;
   end CheckDelayPropertyAccountedFor;

   ---------------------------------------------------------

begin --up_wf_subprogram_body
   SubProgSym := Dictionary.GetRegion (Scope);
   --determine which annotation to use
   Abstraction := Dictionary.GetAbstraction (SubProgSym, Scope);
   IdentNode := LastChildOf (Node);
   IdentStr := NodeLexString (IdentNode);
   EndDesigNode := LastSiblingOf (Child_Node (LastSiblingOf (Child_Node (Node))));

   if SyntaxNodeType (EndDesigNode) = SPSymbols.designator then
      if IdentStr /= NodeLexString (Child_Node (EndDesigNode)) then
         ErrorHandler.SemanticError (58,
                                     ErrorHandler.NoReference,
                                     NodePosition (EndDesigNode),
                                     IdentStr);
      end if;
   end if;

   if Dictionary.IsMainProgram (SubProgSym) then
      --check that global imports are initialized has been done in
      --wf_dependency_clause for procedure main programs but a check is
      --needed here for the (very unlikely) case of a function main prog
      if Dictionary.IsFunction (SubProgSym) then
         CheckGlobalImportsAreInitialized (SubProgSym,
                                           EndDesigNode);
      end if;

      if CommandLineData.RavenscarSelected then
         -- For Ravenscar, perform the ceiling priority check for the main program PO calls.
         MainProgramCeilingPriorityCheck (SubProgSym, Node);
      end if;
   end if;

   CheckEmbedBodies (SubProgSym,
                     EndDesigNode);

   CheckDelayPropertyAccountedFor (SubProgSym, Node);
   CheckSuspendsListAccountedFor (SubProgSym, Node);

   Scope := Dictionary.GetEnclosingScope (Scope);
end up_wf_subprogram_body;
