-- $Id: sem-compunit-stackidentifier.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.CompUnit)
procedure StackIdentifier (Sym            : in     Dictionary.Symbol;
                           IdStr          : in     LexTokenManager.LexString;
                           Node           : in     STree.SyntaxNode;
                           Prefix         : in     Dictionary.Symbol;
                           Scope          : in     Dictionary.Scopes;
                           EStack         : in out ExpStack.ExpStackType;
                           RefVar         : in     SeqAlgebra.Seq;
                           Dotted         : in     Boolean;
                           Context        : in     Tilde_Context;
                           IsAnnotation   : in     Boolean)
is
   Result              : ExpRecord;
   SymLocal            : Dictionary.Symbol;
   KindOfOp            : Dictionary.KindsOfOp;
   TaggedParameterType : Dictionary.Symbol;
   Loc                 : LexTokenManager.TokenPosition;

   procedure CheckGlobalsAreVisible (ProcSym : in Dictionary.Symbol;
                                     Loc     : in LexTokenManager.TokenPosition;
                                     Scope   : in Dictionary.Scopes;
                                     RefVar  : in SeqAlgebra.Seq)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     Prefix;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Loc,
   --#                                        Prefix,
   --#                                        ProcSym,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys &
   --#         Statistics.TableUsage,
   --#         TheHeap                   from *,
   --#                                        Dictionary.Dict,
   --#                                        Prefix,
   --#                                        ProcSym,
   --#                                        RefVar,
   --#                                        Scope,
   --#                                        TheHeap;
   is
      It                 : Dictionary.Iterator;
      --CallingSym,
      GlobSym,
      EnclosingUnit : Dictionary.Symbol;
      CallingAbstraction : Dictionary.Abstractions;

      ----------------------------------------------------

      function IsLocalVariable (CallingSym,
                                GlobSym     : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      is
      begin
         return Dictionary.GetRegion (Dictionary.GetScope (GlobSym)) = CallingSym;
      end IsLocalVariable;

      ------------------------------------------------------

      function IsOwnVarOfEmbeddedPackage (CallingSym,
                                          GlobSym : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      is
      begin
         return (Dictionary.IsOwnVariable (GlobSym)) and then
            (Dictionary.GetScope (Dictionary.GetOwner (GlobSym)) =
             Dictionary.LocalScope (CallingSym));
      end IsOwnVarOfEmbeddedPackage;

      -------------------------------------------------------
   begin -- CheckGlobalsAreVisible
         -- first check if we are in package initialization.  If so function
         -- call is illegal and we do not proceed with global checks

      EnclosingUnit := Dictionary.GetEnclosingCompilationUnit (Scope);

      if InPackageInitialization (Scope) then
         if not Dictionary.IsPredefined (ProcSym) then
            -- only predefined function calls are allowed.
            ErrorHandler.SemanticError (329,
                                        ErrorHandler.NoReference,
                                        Loc,
                                        LexTokenManager.NullString);
         end if;

      elsif Dictionary.IsSubprogram (EnclosingUnit) or else
        Dictionary.IsTaskType (EnclosingUnit) then
         -- we need to check that the function globals are local vars, parameters or
         -- globals of enclosing unit
         CallingAbstraction := Dictionary.GetAbstraction (EnclosingUnit, Scope);

         It := Dictionary.FirstGlobalVariable (Dictionary.GetAbstraction (ProcSym,
                                                                          Scope),
                                               ProcSym);
         while not Dictionary.IsNullIterator (It) loop
            GlobSym := SubstituteProtectedTypeSelfReference (Dictionary.CurrentSymbol (It), Prefix);
            SeqAlgebra.AddMember (TheHeap,
                                  RefVar,
                                  Natural (Dictionary.SymbolRef (GlobSym)));

            if (not Dictionary.IsGlobalVariable (CallingAbstraction,
                                                 EnclosingUnit,
                                                 GlobSym)) and then
               (not Dictionary.IsFormalParameter (EnclosingUnit,
                                                  GlobSym)) and then
               (not IsLocalVariable (EnclosingUnit, GlobSym)) and then
               (not IsOwnVarOfEmbeddedPackage (EnclosingUnit, GlobSym))
            then
               ErrorHandler.SemanticErrorSym (25,
                                              ErrorHandler.NoReference,
                                              Loc,
                                              GlobSym,
                                              Scope);
            end if;
            It := Dictionary.NextSymbol (It);
         end loop;
      elsif Dictionary.IsPackage (EnclosingUnit) then
         -- no check required, there is no equivalent of checking the globals or parameters for a package
         -- and simple visibility of local package variables will have been checked when the function's
         -- own global anno was checked
         --
         -- NOTE.  There may be need of check if we relax declarative order rules since an own var
         --        might then be declared after the function that uses it.
         null;
      else
         SystemErrors.FatalError (SystemErrors.OtherInternalError,
                                  "Checking call to function in CheckGlobalsAreVisible where calling scope is " &
                                    "none of package, subprogram or task body");
      end if;
   end CheckGlobalsAreVisible;

   ---------------------------------------

   procedure CheckCalledFunctionIsWellformed (Sym   : in Dictionary.Symbol;
                                              Loc   : in LexTokenManager.TokenPosition;
                                              Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Loc,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        Sym;
   is
   begin
      if not Dictionary.SubprogramSignatureIsWellformed
         (Dictionary.GetAbstraction (Sym, Scope),
          Sym)
      then
         ErrorHandler.SemanticWarning (399,
                                       Loc,
                                       LexTokenManager.NullString);
      end if;
   end CheckCalledFunctionIsWellformed;


   ---------------------------------------

   procedure CheckPackageInitRules (Scope : in Dictionary.Scopes;
                                    Sym   : in Dictionary.Symbol;
                                    Loc   : in LexTokenManager.TokenPosition)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Loc,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        Sym;
   is
   begin
      -- two checks required: (1) illegal tampering with remote own variables
      --                      (2) illegal referencing of moded own variables
      if InPackageInitialization (Scope) then
         --(1)
         if Dictionary.IsOwnVariable (Sym) and then
            Dictionary.GetOwner (Sym) /=
            Dictionary.GetEnclosingCompilationUnit (Scope)
         then
            ErrorHandler.SemanticError (330,
                                        ErrorHandler.NoReference,
                                        Loc,
                                        LexTokenManager.NullString);
         end if;
         --(2)
         if Dictionary.GetOwnVariableOrConstituentMode (Sym) /=
            Dictionary.DefaultMode then
            ErrorHandler.SemanticError (719,
                                        ErrorHandler.NoReference,
                                        Loc,
                                        LexTokenManager.NullString);
         end if;
      end if;
   end CheckPackageInitRules;


   ---------------------------------------

   function IsUnresolvedDeferredConstant (Sym   : Dictionary.Symbol;
                                          Scope : Dictionary.Scopes) return Boolean
   --# global in Dictionary.Dict;
   is
   begin
      return Dictionary.IsConstant (Sym) and then
        not Dictionary.ConstantIsDeferredHere (Sym, Scope) and then
        not Dictionary.IsDeclared (Sym) and then
        not Dictionary.IsLocalScope (Scope);
   end IsUnresolvedDeferredConstant;

   -------------------------------------------

   procedure CheckReferenceAbility (Sym : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     Context;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     Loc;
   --#        in     Scope;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Context,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Loc,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        Sym;
   is
      SubprogSym : Dictionary.Symbol;

      function IsProtectedElementOf (Sym : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      --#        in SubprogSym;
      is
         Result : Boolean := False;
         TheUnit : Dictionary.Symbol;
         It : Dictionary.Iterator;
      begin
         TheUnit := Dictionary.GetRegion (Dictionary.GetScope (SubprogSym));
         if Dictionary.IsProtectedType (TheUnit) then
            It := Dictionary.FirstProtectedElement (TheUnit);
            while It /= Dictionary.NullIterator loop
               Result := Dictionary.CurrentSymbol (It) = Sym;
               exit when Result;
               It := Dictionary.NextSymbol (It);
            end loop;
         end if;
         return Result;
      end IsProtectedElementOf;

   begin -- CheckReferenceAbility
      case Context is
         when Precondition | FunctionReturn =>
            SubprogSym := Dictionary.GetEnclosingCompilationUnit (Scope);
            if not Dictionary.IsFunction (SubprogSym) and then
              Dictionary.IsVariable (Sym) and then
              not Dictionary.IsQuantifiedVariable (Sym) and then
              not Dictionary.IsImport (Dictionary.GetAbstraction (SubprogSym,
                                                                  Scope),
                                       SubprogSym,
                                       Sym) and then
               -- Unconstrained formal parameters can appear in preconditions
              not (Dictionary.IsFormalParameter (SubprogSym, Sym) and then
                     Dictionary.IsUnconstrainedArrayType (Dictionary.GetType (Sym))) and then
               -- in protected bodies, allow freeer access to protected elements so that assertions
               -- can be made about relationship between entry barriers and PO state. This is a temp
               -- solution to the problem described in SEPR 1542
              not (IsProtectedElementOf (Sym))
            then
               ErrorHandler.SemanticError (322,
                                           ErrorHandler.NoReference,
                                           Loc,
                                           LexTokenManager.NullString);
            end if;

         when Postcondition =>
            null;

         when Code =>
            SystemErrors.FatalError (SystemErrors.PreconditionFailure,
                                     "StackIdentifier.CheckReferenceAbility called with Context=Code");
      end case;

   end CheckReferenceAbility;

   -----------------------

   function ContextToUse return Dictionary.Contexts
   --# global in IsAnnotation;
   is
      Result : Dictionary.Contexts;
   begin
      if IsAnnotation then
         Result := Dictionary.ProofContext;
      else
         Result := Dictionary.ProgramContext;
      end if;
      return Result;
   end ContextToUse;

   -----------------------

   procedure CheckUseOfUncheckedConversion (SymLocal : in Dictionary.Symbol;
                                            Loc      : in LexTokenManager.TokenPosition)
   --# global in     CommandLineData.Content;
   --#        in     LexTokenManager.StringTable;
   --#        in     Scope;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict           from *,
   --#                                        Scope,
   --#                                        SymLocal &
   --#         ErrorHandler.ErrorContext,
   --#         SPARK_IO.File_Sys         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Loc,
   --#                                        Scope,
   --#                                        SPARK_IO.File_Sys,
   --#                                        SymLocal;
   is
      function ReturnsTypeWithCheck (FuncSym : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      is
         ReturnType : Dictionary.Symbol;
      begin
         -- Returns true if FuncSym returns a scalar type that is not Boolean.
         -- These are types for which range check VCs are generated; other types
         -- do not have checks.  We want to generate different warnings for use of
         -- unchecked conversions depending on the type returned.
         ReturnType := Dictionary.GetType (FuncSym);
         return Dictionary.TypeIsScalar (ReturnType) and then
           not Dictionary.TypeIsBoolean (ReturnType);
      end ReturnsTypeWithCheck;

   begin -- CheckUseOfUncheckedConversion
      if Dictionary.IsAnUncheckedConversion (SymLocal) then
         -- mark unit as a user of U_C for benefit of VCG BuildGraph
         Dictionary.AddUsesUncheckedConversion (Dictionary.GetEnclosingCompilationUnit (Scope));
         -- warn user
         -- For each warning, use the symbol for the implicit proof function. This is done to
         -- make sure that the symbols will match if there is a justification for the warning.
         if ReturnsTypeWithCheck (SymLocal) then
            -- weaker warning where RTC VCs provide some further protection
            ErrorHandler. SemanticWarningSym (12,
                                              Loc,
                                              Dictionary.GetImplicitProofFunction
                                                 (Dictionary.IsAbstract, SymLocal),
                                              Scope);
         else
            -- stronger warning because no VCs to help us
            ErrorHandler. SemanticWarningSym (13,
                                              Loc,
                                              Dictionary.GetImplicitProofFunction
                                                 (Dictionary.IsAbstract, SymLocal),
                                              Scope);
         end if;
      end if;
   end CheckUseOfUncheckedConversion;

   ------------------------

begin -- StackIdentifier
   Loc := NodePosition (Node);
   TaggedParameterType := Dictionary.NullSymbol;
   -- TaggedParameterType has a value if an inherited operation gets called; in
   -- this case it holds the value of the formal parameter of that operation
   -- that makes it inheritable.  We need this when processing parameters in
   -- wf_positional_ and wf_named_association so that we know when to convert
   -- actual parameters that are of an extended, tagged type.

   SymLocal := Sym;
   -- if we arrive here with a null symbol then a look up in wf_identifier
   -- or wf_selected_component has failed.  Before we can simply report this
   -- we need to see if there are any inherited ops associated with tagged
   -- types that might be visible.
   --
   -- We search only if the prefix we are using is null (i.e. we are stacking a
   -- simple identifier) or if it is a package.  (The alternative, that it is a protected
   -- object means that we do not search for inherited (tagged) ops).
   if SymLocal = Dictionary.NullSymbol and then
     (Prefix = Dictionary.NullSymbol or else Dictionary.IsPackage  (Prefix)) then
      Dictionary.SearchForInheritedOperations (Name             => IdStr,
                                               Scope            => Scope,
                                               Prefix           => Prefix,
                                               Context          => ContextToUse,
                                                -- to get
                                               OpSym            => SymLocal,
                                               KindOfOp         => KindOfOp,
                                               ActualTaggedType => TaggedParameterType);
      if KindOfOp /= Dictionary.AFunction then -- procedure is no good
         SymLocal := Dictionary.NullSymbol;
         -- now check that any function found is an Ada function (or an implicit proof
         -- function associated with an Ada function.  A pure proof function is not
         -- acceptable because proof functions are not primitives in SPARK
      elsif Dictionary.IsProofFunction (SymLocal) and then
        not Dictionary.IsImplicitProofFunction (SymLocal) then
         SymLocal := Dictionary.NullSymbol;
      end if;
   end if;

   -- After looking at inherited ops we carry on as normal.  Either processing
   -- the inherited op found or reporting failure to find anything.

   if SymLocal = Dictionary.NullSymbol then
      Result := UnknownSymbolRecord;
      ErrorHandler.SemanticError2 (1,
                                   ErrorHandler.NoReference,
                                   Loc,
                                   IdStr,
                                   Dictionary.GetSimpleName (Prefix));

   elsif (not IsAnnotation) and then
      IsUnresolvedDeferredConstant (SymLocal, Scope) then
      Result := UnknownSymbolRecord;
      ErrorHandler.SemanticError (611,
                                  ErrorHandler.NoReference,
                                  Loc,
                                  LexTokenManager.NullString);

   elsif Dictionary.IsFunction (SymLocal) then
      -- If we have ended calling an inherited root operation then we need to tell
      -- the VCG which subprogram actually got called.  We know that a root op has
      -- been called if TaggedParameterSym is not null.  In that case we seed the
      -- syntax node with the symbol of the root function called.
      if TaggedParameterType /= Dictionary.NullSymbol then
         STree.AddNodeSymbol (Node, SymLocal);
      end if;
      if IsAnnotation then
         -- provisonal callability checkwhich excludes proof functions, will
         -- need to be sorted out once dict includes implicit proof funcs
         if not (Dictionary.IsProofFunction (SymLocal) or else
                 Dictionary.IsCallable (SymLocal,
                                        Dotted,
                                        Scope))
         then
            ErrorHandler.SemanticError (163,
                                        ErrorHandler.NoReference,
                                        Loc,
                                        Dictionary.GetSimpleName (SymLocal));
         end if;
      else -- not IsAnnotation
         if Dictionary.IsGenericSubprogram (SymLocal) then
            ErrorHandler.SemanticError (654,
                                        ErrorHandler.NoReference,
                                        Loc,
                                        Dictionary.GetSimpleName (SymLocal));
         else
            if Dictionary.IsCallable (SymLocal,
                                      Dotted,
                                      Scope)
            then
               CheckCalledFunctionIsWellformed (SymLocal,
                                                Loc,
                                                Scope);
               CheckGlobalsAreVisible (SymLocal,
                                       Loc,
                                       Scope,
                                       RefVar);
               CheckUseOfUncheckedConversion (SymLocal,
                                              Loc);
            else
               ErrorHandler.SemanticError (163,
                                           ErrorHandler.NoReference,
                                           Loc,
                                           Dictionary.GetSimpleName (SymLocal));
            end if; -- Dictionary.IsCallable
         end if; -- IsGenericSubprogram
      end if; -- IsAnnotation

      Result := ExpRecord'(TypeSymbol            => Dictionary.GetType (SymLocal),
                           OtherSymbol           => SymLocal,
                           StreamSymbol          => StreamReferencesBy (SymLocal, Scope),
                           TaggedParameterSymbol => TaggedParameterType,
                           VariableSymbol        => Dictionary.NullSymbol,
                           ParamCount            => 0,
                           ParamList             => Lists.NullList,
                           Sort                  => IsFunction,
                           ArgListFound          => False,
                           IsStatic              => False,
                           IsAnEntireVariable    => False,
                           ErrorsInExpression    => False,
                           HasOperators          => False,
                           IsConstant            => Dictionary.IsPredefined (SymLocal),
                           IsARange              => False,
                           IsAVariable           => False,
                           Value                 => Maths.NoValue,
                           RangeRHS              => Maths.NoValue);

      if Dictionary.GetNumberOfSubprogramParameters (SymLocal) = 0 then
         Result.Sort := IsObject;
      end if;

   elsif Dictionary.IsObject (SymLocal) then
      if IsAnnotation then
         CheckReferenceAbility (SymLocal);
      else
         CheckPackageInitRules (Scope, SymLocal, Loc);
      end if;

      Result := ExpRecord'(TypeSymbol            => Dictionary.GetType (SymLocal),
                           OtherSymbol           => SymLocal,
                           StreamSymbol          => StreamReferencesBy (SymLocal, Scope),
                           TaggedParameterSymbol => Dictionary.NullSymbol,
                           VariableSymbol        => Dictionary.NullSymbol,
                           ParamCount            => 0,
                           ParamList             => Lists.NullList,
                           Sort                  => IsObject,
                           ArgListFound          => False,
                           IsAVariable           => False,
                           IsAnEntireVariable    => False,
                           ErrorsInExpression    => False,
                           HasOperators          => False,
                           IsStatic              => Dictionary.IsStatic (SymLocal, Scope),
                           IsConstant            => Dictionary.IsConstant (SymLocal),
                           IsARange              => False,
                           Value                 => GetObjectValue (SymLocal),
                           RangeRHS              => Maths.NoValue);

      -- if the constant is of a string subtype then we need to recover the string's length
      -- and place it in RangeRHS because this is what we would get if a string literal was
      -- substituted for the constant and we should get the same ebahviour for both cases
      if Dictionary.IsConstant (SymLocal) and then
        Dictionary.GetRootType (Result.TypeSymbol) = Dictionary.GetPredefinedStringType then
         Result.RangeRHS := Maths.ValueRep (Dictionary.GetArrayAttributeValue (Name      => LexTokenManager.LastToken,
                                                                               TypeMark  => Result.TypeSymbol,
                                                                               Dimension => 1));
      end if;

      if Dictionary.IsVariable (SymLocal) then
         Result.VariableSymbol := SymLocal;
         Result.IsAVariable    := True;
         Result.IsAnEntireVariable := True;
         -- Do not allow protected variables (except streams) to appear in annotation expressions
         if IsAnnotation and then
           Dictionary.IsOwnVariable (SymLocal) and then
           not Dictionary.IsOwnVariableOrConstituentWithMode (SymLocal) and then
           Dictionary.GetOwnVariableProtected (SymLocal) then
            Result.ErrorsInExpression := True;
            ErrorHandler.SemanticErrorSym (940,
                                           ErrorHandler.NoReference,
                                           Loc,
                                           SymLocal,
                                           Scope);
         end if;
      end if;

   elsif Dictionary.IsEnumerationLiteral (SymLocal) then
      Result := ExpRecord'(TypeSymbol            => Dictionary.GetType (SymLocal),
                           OtherSymbol           => SymLocal,
                           StreamSymbol          => Dictionary.NullSymbol,
                           TaggedParameterSymbol => Dictionary.NullSymbol,
                           VariableSymbol        => Dictionary.NullSymbol,
                           ParamCount            => 0,
                           ParamList             => Lists.NullList,
                           Sort                  => TypeResult,
                           ArgListFound          => False,
                           IsAVariable           => False,
                           IsAnEntireVariable    => False,
                           ErrorsInExpression    => False,
                           HasOperators          => False,
                           IsStatic              => True,
                           IsConstant            => True,
                           IsARange              => False,
                           Value                 => GetEnumLitValue (SymLocal),
                           RangeRHS              => Maths.NoValue);

   elsif Dictionary.IsTypeMark (SymLocal) then
      Result := ExpRecord'(TypeSymbol            => SymLocal,
                           OtherSymbol           => SymLocal,
                           StreamSymbol          => Dictionary.NullSymbol,
                           TaggedParameterSymbol => Dictionary.NullSymbol,
                           VariableSymbol        => Dictionary.NullSymbol,
                           ParamCount            => 0,
                           ParamList             => Lists.NullList,
                           Sort                  => IsTypeMark,
                           ArgListFound          => False,
                           IsAVariable           => False,
                           IsAnEntireVariable    => False,
                           ErrorsInExpression    => False,
                           HasOperators          => False,
                           IsStatic              => False,
                           IsConstant            => False,
                           IsARange              => False,
                           Value                 => Maths.NoValue,
                           RangeRHS              => Maths.NoValue);

   elsif Dictionary.IsPackage (SymLocal) then
      Result := ExpRecord'(TypeSymbol            => Dictionary.GetUnknownTypeMark,
                           OtherSymbol           => SymLocal,
                           StreamSymbol          => Dictionary.NullSymbol,
                           TaggedParameterSymbol => Dictionary.NullSymbol,
                           VariableSymbol        => Dictionary.NullSymbol,
                           ParamCount            => 0,
                           ParamList             => Lists.NullList,
                           Sort                  => IsPackage,
                           ArgListFound          => False,
                           IsAVariable           => False,
                           IsAnEntireVariable    => False,
                           ErrorsInExpression    => False,
                           HasOperators          => False,
                           IsStatic              => False,
                           IsConstant            => False,
                           IsARange              => False,
                           Value                 => Maths.NoValue,
                           RangeRHS              => Maths.NoValue);

   elsif Dictionary.IsKnownDiscriminant (SymLocal) then
      Result := ExpRecord'(TypeSymbol            => Dictionary.GetType (SymLocal),
                           OtherSymbol           => SymLocal,
                           StreamSymbol          => Dictionary.NullSymbol,
                           TaggedParameterSymbol => Dictionary.NullSymbol,
                           VariableSymbol        => SymLocal,
                           ParamCount            => 0,
                           ParamList             => Lists.NullList,
                           Sort                  => IsObject,
                           ArgListFound          => False,
                           IsAVariable           => False,
                           IsAnEntireVariable    => False,
                           ErrorsInExpression    => False,
                           HasOperators          => False,
                           IsStatic              => True,
                           IsConstant            => True,
                           IsARange              => False,
                           Value                 => Maths.NoValue,
                           RangeRHS              => Maths.NoValue);

   else
      Result := UnknownSymbolRecord;
      ErrorHandler.SemanticError (5,
                                  ErrorHandler.NoReference,
                                  Loc,
                                  IdStr);
   end if;

   ExpStack.Push (Result, EStack);
end StackIdentifier;
