-- $Id: sem-compunit-expressiontypefromcontext.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.CompUnit)
function ExpressionTypeFromContext (ExpNode       : in     STree.SyntaxNode;
                                    EStack        : in     ExpStack.ExpStackType;
                                    TStack        : in     TypeContextStack.TStackType)
   return Dictionary.Symbol
is
   NewContextType  : Dictionary.Symbol;
   TopOfExpStack   : ExpRecord;
   ParameterNameOK : Boolean;
   ParamSymbol     : Dictionary.Symbol;
   TNode2          : STree.SyntaxNode;
begin
   -- Assume NodeType(ExpNode) = expression or
   --        NodeType(ExpNode) = annotation_expression

   ------------------------------------------------------------------------
   -- Possible parent nodes are:
   --
   -- Group 1 - Expressions appearing in statements or declarations
   --
   -- constant_declaration
   -- variable_declaration
   -- assignment_statement
   -- condition
   -- case_statement
   -- return_statement
   -- delay_statement
   --
   -- Group 2 - Expressions appearing in expressions, common to both code
   --           and annotations
   --
   -- annotation_named_argument_association
   -- named_argument_association
   -- annotation_positional_argument_association
   -- positional_argument_association
   -- annotation_attribute_designator_opt
   -- attribute_designator_opt
   -- annotation_aggregate_or_expression
   -- aggregate_or_expression
   -- annotation_primary
   -- primary
   -- annotation_qualified_expression
   -- qualified_expression
   --
   -- Group 3 - Expressions appearing in annotations only
   --
   -- store
   -- store_list
   -- return_expression
   -- predicate
   ------------------------------------------------------------------------

   case SyntaxNodeType (ParentNode (ExpNode)) is

      when SPSymbols.constant_declaration |
           SPSymbols.variable_declaration |
           SPSymbols.assignment_statement |
           SPSymbols.condition |
           SPSymbols.case_statement |
           SPSymbols.return_statement |
           SPSymbols.delay_statement =>

         -- Context here is passed in from calling environment, and
         -- no change required.
         NewContextType := TypeContextStack.Top (TStack);

      when SPSymbols.annotation_named_argument_association |
        SPSymbols.named_argument_association  =>

         if ExpStack.IsEmpty (EStack) then
            -- Must be a named argument association in a procedure
            -- call.  Wf_Proc_Call processes the parameter name, and
            -- does NOT put it on the expression stack, but it does pass
            -- the expected type into WalkExpression, so this is unchanged
            -- in this case.
            NewContextType := TypeContextStack.Top (TStack);
         else
            -- Must be a named argument association which is the parameter
            -- list of a function call.
            TopOfExpStack := ExpStack.Top (EStack);

            -- We need to find the parameter name, and determine if it is legal:
            FindNamedArgumentAssociationParameter
              (Node                => ParentNode (ExpNode),
               SubprogSym          => TopOfExpStack.OtherSymbol,
               IsAnnotation        => (SyntaxNodeType (ParentNode (ExpNode)) =
                                       SPSymbols.annotation_named_argument_association),

               NameIsParameterName => ParameterNameOK,
               ParamSym            => ParamSymbol);

            if ParameterNameOK then
               -- The parameter name denotes a legal parameter of this subprogram,
               -- so look up its type.
               NewContextType := Dictionary.GetType (ParamSymbol);

            else
               -- The parameter name is illegal.  This will be picked up again later
               -- on in wf_named_argument_association.  The type context simply
               -- becomes unknown.
               NewContextType := Dictionary.GetUnknownTypeMark;
            end if;
         end if;

      when SPSymbols.qualified_expression |
           SPSymbols.annotation_qualified_expression =>

         -- Context changes to the type denoted by the Name preceeding the
         -- expression.  The result of evaluating this Name should be on the
         -- top of the Expression stack.
         TopOfExpStack := ExpStack.Top (EStack);
         NewContextType := TopOfExpStack.TypeSymbol;

      when SPSymbols.positional_argument_association |
           SPSymbols.annotation_positional_argument_association =>

         -- Parent could be name_argument_list or another
         -- positional_argument_association.
         -- This could be part of a
         --   Type conversion
         --   Array index
         --   Function call

         if ExpStack.IsEmpty (EStack) then
            -- EStack might be empty here => we must be processing an
            -- actual param of a procedure call statement.  Context
            -- will have been passed in from wf_proc_call, so no change here.
            NewContextType := TypeContextStack.Top (TStack);
         else
            TopOfExpStack := ExpStack.Top (EStack);

            case TopOfExpStack.Sort is
               when IsObject =>
                  -- Must be an array indexing expression.  The context is therefore
                  -- the subtype corresponding to the N'th dimension of that array type.
                  NewContextType := Dictionary.GetArrayIndex (TopOfExpStack.TypeSymbol,
                                                              TopOfExpStack.ParamCount + 1);

               when IsFunction =>
                  -- Must be an actual parameter of a function call.  The context
                  -- is the subtype indicated by the corresponding formal parameter,
                  -- if there is any such parameter.  If the wrong number of actual
                  -- parameters has been given, then return UnknownTypeMark - this
                  -- error will be picked up later in the UP pass.

                  if TopOfExpStack.ParamCount <
                    Dictionary.GetNumberOfSubprogramParameters (TopOfExpStack.OtherSymbol) then

                     NewContextType := Dictionary.GetType
                       (Dictionary.GetSubprogramParameter (TopOfExpStack.OtherSymbol,
                                                           TopOfExpStack.ParamCount + 1));
                  else
                     NewContextType := Dictionary.GetUnknownTypeMark;
                  end if;

               when IsTypeMark =>
                  -- Must be a type conversion.  The argument of the type conversion
                  -- could be pretty much anything - legal or illegal, so the
                  -- context is unknown.
                  NewContextType := Dictionary.GetUnknownTypeMark;

               when others =>
                  -- Other cases - all errors which will be caught later on...
                  -- We still need to push something onto the TStack to make
                  -- it balance, so simply copy the existing top entry.
                  NewContextType := TypeContextStack.Top (TStack);
            end case;
         end if;

      when SPSymbols.aggregate_or_expression |
           SPSymbols.annotation_aggregate_or_expression =>

         case SyntaxNodeType (ParentNode (ParentNode (ExpNode))) is
            when SPSymbols.component_association |
              SPSymbols.annotation_component_association =>
               -- Must be an array aggregate with a single others clause, so the
               -- new context type is the type of the array element
               TopOfExpStack := ExpStack.Top (EStack);
               NewContextType := Dictionary.GetArrayComponent
                 (TopOfExpStack.TypeSymbol);

            when SPSymbols.named_association |
              SPSymbols.annotation_named_association |
              SPSymbols.named_association_rep |
              SPSymbols.annotation_named_association_rep =>

               TopOfExpStack := ExpStack.Top (EStack);

               case TopOfExpStack.Sort is
                  when IsParameterName =>
                     -- Must be a named field of a record aggregate.  New context is
                     -- the type of that field, unless the field was itself illegal,
                     -- in which case the context is unknown.
                     if TopOfExpStack = NullParameterRecord then
                        NewContextType := Dictionary.GetUnknownTypeMark;
                     else
                        NewContextType := Dictionary.GetType (TopOfExpStack.OtherSymbol);
                     end if;

                  when IsTypeMark =>
                     -- Must be a named element of an array aggregate.  New context is
                     -- the type of the array element.
                     NewContextType := Dictionary.GetArrayComponent
                       (TopOfExpStack.TypeSymbol);

                  when others =>
                     NewContextType := TypeContextStack.Top (TStack);
               end case;

            when SPSymbols.positional_association |
              SPSymbols.positional_association_rep |
              SPSymbols.annotation_positional_association |
              SPSymbols.annotation_positional_association_rep =>

               TopOfExpStack := ExpStack.Top (EStack);

               case TopOfExpStack.Sort is
                  when IsTypeMark =>

                     if Dictionary.TypeIsRecord (TopOfExpStack.TypeSymbol) then
                        -- New context is the type of the N'th field of the record, assuming
                        -- there is such a field.
                        if (TopOfExpStack.ParamCount + 1) <=
                          Dictionary.GetNumberOfComponents (TopOfExpStack.TypeSymbol) then

                           NewContextType := Dictionary.GetType
                             (Dictionary.GetRecordComponent
                              (TopOfExpStack.TypeSymbol, TopOfExpStack.ParamCount + 1));
                        else
                           NewContextType := Dictionary.GetUnknownTypeMark;
                        end if;

                     elsif Dictionary.TypeIsArray (TopOfExpStack.TypeSymbol) then
                        -- New context is the element type of the array
                        NewContextType := Dictionary.GetArrayComponent
                          (TopOfExpStack.TypeSymbol);
                     else
                        -- Must be an error - this will be caught later on in the UP
                        -- pass, but we need to push something so...
                        NewContextType := TypeContextStack.Top (TStack);
                     end if;


                  when others =>
                     -- Must be an error - this will be caught later on in the UP
                     -- pass, but we need to push something so...
                     NewContextType := TypeContextStack.Top (TStack);
               end case;

            when others =>
               NewContextType := TypeContextStack.Top (TStack);
         end case;

      when SPSymbols.annotation_attribute_designator_opt |
           SPSymbols.attribute_designator_opt =>

         -- Context change for attribute arguments is handled in
         -- AttributeDesignatorTypeFromContext, so no change here.
         NewContextType := TypeContextStack.Top (TStack);

      when SPSymbols.annotation_primary |
        SPSymbols.primary =>

         -- Must be a parenthesized expression - context does not change.
         NewContextType := TypeContextStack.Top (TStack);

      when SPSymbols.store =>

         TopOfExpStack := ExpStack.Top (EStack);

         case TopOfExpStack.Sort is

            when IsObject =>

               if Dictionary.TypeIsArray (TopOfExpStack.TypeSymbol) then

                  NewContextType := Dictionary.GetArrayComponent (TopOfExpStack.TypeSymbol);

               elsif Dictionary.TypeIsRecord (TopOfExpStack.TypeSymbol) then

                  -- The record field being updated should be an identifier
                  -- node directly below the store node, so
                  TNode2 := LastChildOf (ParentNode (ExpNode));

                  if SyntaxNodeType (TNode2) = SPSymbols.identifier then
                     ParamSymbol := Dictionary.LookupSelectedItem
                       (TopOfExpStack.TypeSymbol,
                        NodeLexString (TNode2),
                        Dictionary.GetScope (TopOfExpStack.TypeSymbol),
                        Dictionary.ProofContext);

                     if ParamSymbol /= Dictionary.NullSymbol and then
                       Dictionary.IsRecordComponent (ParamSymbol) then

                        NewContextType := Dictionary.GetType (ParamSymbol);
                     else
                        -- error - will be caught in up_wf_store
                        NewContextType := TypeContextStack.Top (TStack);
                     end if;
                  else
                     -- error - will be caught in up_wf_store
                     NewContextType := TypeContextStack.Top (TStack);
                  end if;
               else
                  -- error - will be caught in up_wf_store
                  NewContextType := TypeContextStack.Top (TStack);
               end if;

            when others =>
               -- Must be an error - this will be caught later on in the UP
               -- pass, but we need to push something so...
               NewContextType := TypeContextStack.Top (TStack);
         end case;

      when SPSymbols.store_list =>

         TopOfExpStack := ExpStack.Top (EStack);

         case TopOfExpStack.Sort is

            when IsObject =>

               if Dictionary.TypeIsArray (TopOfExpStack.TypeSymbol) then

                  NewContextType := Dictionary.GetArrayIndex
                    (TopOfExpStack.TypeSymbol, TopOfExpStack.ParamCount + 1);

               else
                  -- error - will be caught in up_wf_store
                  NewContextType := TypeContextStack.Top (TStack);
               end if;

            when others =>
               -- Must be an error - this will be caught later on in the UP
               -- pass, but we need to push something so...
               NewContextType := TypeContextStack.Top (TStack);
         end case;

      when SPSymbols.return_expression |
           SPSymbols.predicate =>
         -- Context for predicate and return_expression is always passed
         -- in from wf_predicate, or wf_function_constraint, so no change needed.
         NewContextType := TypeContextStack.Top (TStack);

      when others =>
         -- In all other cases, the context is unchanged, but we push a copy
         -- of the current context type to keep the stack balanced.
         NewContextType := TypeContextStack.Top (TStack);
   end case;

   return NewContextType;

end ExpressionTypeFromContext;
