-- $Id: sem-compunit-checknamedassociation.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.CompUnit)

procedure CheckNamedAssociation
  (TheFormals             : in Dictionary.Symbol;
   Scope                  : in Dictionary.Scopes;
   NamedArgumentAssocNode : in STree.SyntaxNode)

is

   type Iterator is
      record
         BaseIt   : STree.Iterator;
         SearchIt : STree.Iterator;
         DictIt   : Dictionary.Iterator;
      end record;

   NullIterator : constant Iterator :=
     Iterator'(BaseIt   => STree.NullIterator,
               SearchIt => STree.NullIterator,
               DictIt   => Dictionary.NullIterator);

   It : Iterator;

   function IsNull (It : Iterator) return Boolean
   is
   begin
      return It = NullIterator;
   end IsNull;


   ---------------------------------------------------------------
   -- Gets the first formal parameter for this dictionary entity
   ---------------------------------------------------------------

   function FirstFormal (Sym : Dictionary.Symbol) return Dictionary.Iterator
   --# global in Dictionary.Dict;
   is
      Result : Dictionary.Iterator;
   begin
      if Dictionary.IsGenericSubprogram (Sym) or else
      Dictionary.IsGenericPackage (Sym) then
         -- It's a generic unit.
         Result := Dictionary.FirstGenericFormalParameter (Sym);

      elsif Dictionary.IsSubprogram (Sym) then
         -- It's a subprogram.
         Result := Dictionary.FirstSubprogramParameter (Sym);
      else
         -- It's a task or protected type.
         Result := Dictionary.FirstKnownDiscriminant (Sym);
      end if;
      return Result;
   end FirstFormal;


   ---------------------------------------------------------------
   -- Find duplicate formal parameters
   ---------------------------------------------------------------

   function NextDuplicateFormal (It : Iterator) return Iterator
   --# global in STree.Table;
   is
      MyBaseIt   : STree.Iterator;
      MySearchIt : STree.Iterator;
      Result     : Iterator;
   begin
      if STree.IsNull (It.SearchIt) then
         -- We've not found a duplicate yet
         -- So our base is the first formal parameter
         MyBaseIt := It.BaseIt;
      else
         -- We've found one duplicate and are looking for another.
         -- So our base is the next formal parameter
         MyBaseIt := STree.NextNode (It.BaseIt);
      end if;

      MySearchIt := STree.NullIterator;

      while not STree.IsNull (MyBaseIt) loop
         MySearchIt := STree.NextNode (MyBaseIt);
         while not STree.IsNull (MySearchIt) loop
            -- exit if the identifiers hanging off the base and dup nodes
            -- are the same. i.e. we've found a duplicate.
            exit when NodeLexString (GetNode (MyBaseIt)) =
              NodeLexString (GetNode (MySearchIt));
            MySearchIt := STree.NextNode (MySearchIt);
         end loop;
         -- We found a duplicate
         exit when not STree.IsNull (MySearchIt);
         MyBaseIt := STree.NextNode (MyBaseIt);
      end loop;

      if STree.IsNull (MySearchIt) then
         -- We didn't find a duplicate
         Result := NullIterator;
      else
         Result := Iterator'(BaseIt   => MyBaseIt,
                             SearchIt => MySearchIt,
                             DictIt   => Dictionary.NullIterator);
      end if;
      return Result;
   end NextDuplicateFormal;

   function FirstDuplicateFormal
     (Node : STree.SyntaxNode) return Iterator
   --# global in STree.Table;
   is
      FirstIt : STree.Iterator;
   begin
      FirstIt := STree.FindFirstFormalParameterNode
        (FromRoot => Node);

      return NextDuplicateFormal
        (Iterator'(BaseIt   => FirstIt,
                   SearchIt => STree.NullIterator,
                   DictIt   => Dictionary.NullIterator));
   end FirstDuplicateFormal;


   ---------------------------------------------------------------
   -- Find illegal formal parameters
   ---------------------------------------------------------------

   function NextIllegalFormal (It : Iterator) return Iterator
   --# global in Dictionary.Dict;
   --#        in STree.Table;
   --#        in TheFormals;
   is
      MyBaseIt : STree.Iterator;
      MyDictIt : Dictionary.Iterator;
      Result   : Iterator;
   begin
      if STree.IsNull (It.SearchIt) then
         -- We've not found an illegal name
         -- So our base is the first formal parameter
         MyBaseIt := It.BaseIt;
      else
         -- We've found one illegal and are looking for another.
         -- So our base is the next formal parameter.
         MyBaseIt := STree.NextNode (It.BaseIt);
      end if;

      while not STree.IsNull (MyBaseIt) loop
         MyDictIt := FirstFormal (TheFormals);
         -- Loop through all the formals declared in the type
         while not Dictionary.IsNullIterator (MyDictIt) loop
            -- Looking for a formal to match the one in the tree
            exit when Dictionary.GetSimpleName (Dictionary.CurrentSymbol (MyDictIt)) =
              NodeLexString (GetNode (MyBaseIt));
            MyDictIt := Dictionary.NextSymbol (MyDictIt);
         end loop;
         -- MyDictIt is null if we didn't find it.
         exit when Dictionary.IsNullIterator (MyDictIt);
         MyBaseIt := STree.NextNode (MyBaseIt);
      end loop;

      if STree.IsNull (MyBaseIt) then
         -- We didn't find any more illegal formals
         Result := NullIterator;
      else
         Result := Iterator'(BaseIt   => MyBaseIt,
                             SearchIt => MyBaseIt,
                             DictIt   => Dictionary.NullIterator);
      end if;
      return Result;
   end NextIllegalFormal;

   function FirstIllegalFormal
     (Node : STree.SyntaxNode) return Iterator
   --# global in Dictionary.Dict;
   --#        in STree.Table;
   --#        in TheFormals;
   is
      FirstIt : STree.Iterator;
   begin
      FirstIt := STree.FindFirstFormalParameterNode
        (FromRoot => Node);

      return NextIllegalFormal
        (Iterator'(BaseIt   => FirstIt,
                   SearchIt => STree.NullIterator,
                   DictIt   => Dictionary.NullIterator));
   end FirstIllegalFormal;


   ---------------------------------------------------------------
   -- Find missing formal parameters
   ---------------------------------------------------------------

   function NextMissingFormal (It : Iterator) return Iterator
   --# global in Dictionary.Dict;
   --#        in NamedArgumentAssocNode;
   --#        in STree.Table;
   --#        in TheFormals;
   is
      MyBaseIt : STree.Iterator;
      MyDictIt : Dictionary.Iterator;
      Result   : Iterator;
   begin
      if Dictionary.IsNullIterator (It.DictIt) then
         -- We've not found a missing formal yet
         -- So our base is the first formal
         MyDictIt := FirstFormal (TheFormals);
      else
         -- We've found one missing and are looking for another.
         -- So our base is the next formal parameter
         MyDictIt := Dictionary.NextSymbol (It.DictIt);
      end if;

      while not Dictionary.IsNullIterator (MyDictIt) loop
         MyBaseIt := STree.FindFirstFormalParameterNode
           (FromRoot => NamedArgumentAssocNode);

         -- Loop through all the formals
         while not STree.IsNull (MyBaseIt) loop
            -- Looking for a formal to match the one in the tree
            exit when Dictionary.GetSimpleName (Dictionary.CurrentSymbol (MyDictIt)) =
              NodeLexString (GetNode (MyBaseIt));
            MyBaseIt := STree.NextNode (MyBaseIt);
         end loop;
         -- MyBaseIt is null if we didn't find it.
         exit when STree.IsNull (MyBaseIt);
         MyDictIt := Dictionary.NextSymbol (MyDictIt);
      end loop;

      if Dictionary.IsNullIterator (MyDictIt) then
         Result := NullIterator;
      else
         Result := Iterator'(BaseIt   => STree.NullIterator,
                             SearchIt => STree.NullIterator,
                             DictIt   => MyDictIt);
      end if;
      return Result;
   end NextMissingFormal;

   function FirstMissingFormal return Iterator
   --# global in Dictionary.Dict;
   --#        in NamedArgumentAssocNode;
   --#        in STree.Table;
   --#        in TheFormals;
   is
   begin
      return NextMissingFormal
        (Iterator'(BaseIt   => STree.NullIterator,
                   SearchIt => STree.NullIterator,
                   DictIt   => Dictionary.NullIterator));
   end FirstMissingFormal;

begin

   ------------------------------------------
   -- Report all duplicated formal parameters
   ------------------------------------------

   It := FirstDuplicateFormal (Node => NamedArgumentAssocNode);
   while not IsNull (It) loop
      ErrorHandler.SemanticError (4,
                                  ErrorHandler.NoReference,
                                  NodePosition (GetNode (It.SearchIt)),
                                  NodeLexString (GetNode (It.SearchIt)));
      It := NextDuplicateFormal (It);
   end loop;

   ------------------------------------------
   -- Report all illegal formal parameters
   ------------------------------------------

   It := FirstIllegalFormal (Node => NamedArgumentAssocNode);
   while not IsNull (It) loop
      ErrorHandler.SemanticErrorLex1Sym1 (2,
                                          ErrorHandler.NoReference,
                                          NodePosition (GetNode (It.SearchIt)),
                                          NodeLexString (GetNode (It.SearchIt)),
                                          TheFormals,
                                          Scope);
      It := NextIllegalFormal (It);
   end loop;

   ------------------------------------------
   -- Report all missing formal parameters
   ------------------------------------------

   It := FirstMissingFormal;
   while not IsNull (It) loop
      ErrorHandler.SemanticError
        (23,
         ErrorHandler.NoReference,
         NodePosition (STree.FindLastActualParameterNode
                         (NamedArgumentAssocNode)),
         Dictionary.GetSimpleName
           (Dictionary.CurrentSymbol (It.DictIt)));
      It := NextMissingFormal (It);
   end loop;

end CheckNamedAssociation;
