-- $Id: sem-compunit-checkderivesconsistency.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.CompUnit)

procedure CheckDerivesConsistency (SubprogSym : in Dictionary.Symbol;
                                   Position   : in LexTokenManager.TokenPosition)
is
   R,
   RT         : RelationAlgebra.Relation;
   RDom,
   RTDom,
   AUnchanged : SeqAlgebra.Seq;

   ----------------------------------------------

   function SymToNat (Sym : in Dictionary.Symbol) return Natural
   is
   begin
      return Natural (Dictionary.SymbolRef (Sym));
   end SymToNat;

   ----------------------------------------------

   function MemToSym (Nat : in SeqAlgebra.MemberOfSeq)
                     return Dictionary.Symbol
      --# global in TheHeap;
   is
   begin
      return Dictionary.ConvertSymbolRef
         (ExaminerConstants.RefType (SeqAlgebra.ValueOfMember (TheHeap, Nat)));
   end MemToSym;

   ----------------------------------------------

   function IsExpandableConstituent (Sym, SubProgSym : Dictionary.Symbol)
                                    return Boolean
   --# global in Dictionary.Dict;
   is
      Owner           : Dictionary.Symbol;
      Region          : Dictionary.Symbol;
      EnclosingRegion : Dictionary.Symbol;
      Result          : Boolean := False;
   begin
      if Dictionary.IsConstituent (Sym) then
         Owner := Dictionary.GetOwner (Dictionary.GetSubject (Sym));
         Region := Dictionary.GetRegion (Dictionary.GetScope (SubProgSym));

         Result :=  Owner = Region;
         if not Result and then Dictionary.IsProtectedType (Region) then
            EnclosingRegion := Dictionary.GetRegion (Dictionary.GetScope (Region));
            Result := Owner = EnclosingRegion;
         end if;
      end if;
      return Result;
   end IsExpandableConstituent;

   ----------------------------------------------

   procedure BuildCAExp (CAExp : out SeqAlgebra.Seq)
   --# global in     Dictionary.Dict;
   --#        in     SubProgSym;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives CAExp                 from TheHeap &
   --#         Statistics.TableUsage,
   --#         TheHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    SubProgSym,
   --#                                    TheHeap;
   is
      LCAExp         : SeqAlgebra.Seq;
      ConcreteExport : Dictionary.Symbol;
      ConcreteExportIt,
      ConstituentIt  : Dictionary.Iterator;
      ConstituentSym : Dictionary.Symbol;

   begin
      SeqAlgebra.CreateSeq (TheHeap, LCAExp);
      ConcreteExportIt := Dictionary.FirstExport (Dictionary.IsRefined,
                                                  SubprogSym);
      while not Dictionary.IsNullIterator (ConcreteExportIt) loop
         ConcreteExport := Dictionary.CurrentSymbol (ConcreteExportIt);
         if IsExpandableConstituent (ConcreteExport, SubprogSym) then
            ConstituentIt := Dictionary.FirstConstituent
               (Dictionary.GetSubject (ConcreteExport));
            while not Dictionary.IsNullIterator (ConstituentIt) loop
               ConstituentSym := Dictionary.CurrentSymbol (ConstituentIt);
               -- Previously we only add constituents to LCAExp if they are unmoded; this has been shown to
               -- be incorrect (see SEPR 1844), so now we add all constituents
               SeqAlgebra.AddMember (TheHeap,
                                     LCAExp,
                                     SymToNat (ConstituentSym));

               ConstituentIt := Dictionary.NextSymbol (ConstituentIt);
            end loop;
         end if;
         ConcreteExportIt := Dictionary.NextSymbol (ConcreteExportIt);
      end loop;
      CAExp := LCAExp;
   end BuildCAExp;

   ----------------------------------------------

   procedure BuildCExp (CExp : out SeqAlgebra.Seq)
   --# global in     Dictionary.Dict;
   --#        in     SubProgSym;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives CExp                  from TheHeap &
   --#         Statistics.TableUsage,
   --#         TheHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    SubProgSym,
   --#                                    TheHeap;
   is
      LCExp            : SeqAlgebra.Seq;
      ConcreteExportIt : Dictionary.Iterator;

   begin
      SeqAlgebra.CreateSeq (TheHeap, LCExp);
      ConcreteExportIt := Dictionary.FirstExport (Dictionary.IsRefined,
                                                  SubprogSym);
      while not Dictionary.IsNullIterator (ConcreteExportIt) loop
         SeqAlgebra.AddMember (TheHeap,
                               LCExp,
                               SymToNat (Dictionary.CurrentSymbol (ConcreteExportIt)));
         ConcreteExportIt := Dictionary.NextSymbol (ConcreteExportIt);
      end loop;
      CExp := LCExp;
   end BuildCExp;

   ----------------------------------------------

   procedure AbstractUnchanged (U  : in     SeqAlgebra.Seq;
                                AU :    out SeqAlgebra.Seq)
   --# global in     Dictionary.Dict;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives AU                    from TheHeap &
   --#         Statistics.TableUsage,
   --#         TheHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    TheHeap,
   --#                                    U;
   is
      LAU : SeqAlgebra.Seq;
      Mem : SeqAlgebra.MemberOfSeq;

   begin
      SeqAlgebra.CreateSeq (TheHeap, LAU);
      Mem := SeqAlgebra.FirstMember (TheHeap, U);
      while not SeqAlgebra.IsNullMember (Mem) loop
         SeqAlgebra.AddMember (TheHeap,
                               LAU,
                               SymToNat (Dictionary.GetSubject (MemToSym (Mem))));
         Mem := SeqAlgebra.NextMember (TheHeap, Mem);
      end loop;
      AU := LAU;
   end AbstractUnchanged;

   ----------------------------------------------

   procedure StepOne (RT         : in     RelationAlgebra.Relation;
                      RTDom      :    out SeqAlgebra.Seq;
                      AUnchanged :    out SeqAlgebra.Seq)
   --# global in     Dictionary.Dict;
   --#        in     SubProgSym;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives AUnchanged            from Dictionary.Dict,
   --#                                    SubProgSym,
   --#                                    TheHeap &
   --#         RTDom                 from Dictionary.Dict,
   --#                                    RT,
   --#                                    SubProgSym,
   --#                                    TheHeap &
   --#         Statistics.TableUsage,
   --#         TheHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    RT,
   --#                                    SubProgSym,
   --#                                    TheHeap;
   is
      CAExp,
      CExp,
      Unchanged,
      LAUnchanged : SeqAlgebra.Seq;
      EmptySeq    : SeqAlgebra.Seq;
   begin
      BuildCAExp (CAExp); -- set of all constituents of subject of any exported constituent
      BuildCExp (CExp);   -- set of actual exported constituents
      SeqAlgebra.Complement (TheHeap, CAExp, CExp, Unchanged); -- unwritten constituents
      AbstractUnchanged (Unchanged, LAUnchanged); -- abstract subjects of unwritten constituents
      RelationAlgebra.AddIdentity (TheHeap, RT, LAUnchanged); --depend on themselves

      SeqAlgebra.CreateSeq (TheHeap, EmptySeq);
      SeqAlgebra.Union (TheHeap, LAUnchanged, EmptySeq, RTDom);
      SeqAlgebra.DisposeOfSeq (TheHeap, EmptySeq);

      AUnchanged := LAUnchanged;

      SeqAlgebra.DisposeOfSeq (TheHeap, CAExp);
      SeqAlgebra.DisposeOfSeq (TheHeap, CExp);
      SeqAlgebra.DisposeOfSeq (TheHeap, Unchanged);
   end StepOne;

   ----------------------------------------------

   procedure StepsTwoAndThree (RT    : in RelationAlgebra.Relation;
                               RTDom : in SeqAlgebra.Seq)
   --# global in     Dictionary.Dict;
   --#        in     SubProgSym;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives Statistics.TableUsage,
   --#         TheHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    RT,
   --#                                    RTDom,
   --#                                    SubProgSym,
   --#                                    TheHeap;
   is
      ExportVar : Dictionary.Symbol;
      ImportVar : Dictionary.Symbol;
      ExportNum : Natural;
      ImportNum : Natural;

      ExportIt,
      ImportIt  : Dictionary.Iterator;

   begin
      ExportIt := Dictionary.FirstExport (Dictionary.IsRefined,
                                          SubprogSym);
      while not Dictionary.IsNullIterator (ExportIt) loop
         ExportVar := Dictionary.CurrentSymbol (ExportIt);
         if IsExpandableConstituent (ExportVar, SubprogSym) then
            ExportNum := SymToNat (Dictionary.GetSubject (ExportVar));
         else
            ExportNum := SymToNat (ExportVar);
         end if;
         SeqAlgebra.AddMember (TheHeap, RTDom, ExportNum);
         --if it's mode out stream add ExportNum, ExportNum to RT
         if Dictionary.GetOwnVariableOrConstituentMode (ExportVar) =
            Dictionary.OutMode then
            RelationAlgebra.InsertPair (TheHeap, RT, ExportNum, ExportNum);
         end if;

         ImportIt := Dictionary.FirstDependency (Dictionary.IsRefined,
                                                 SubprogSym,
                                                 ExportVar);
         while not Dictionary.IsNullIterator (ImportIt) loop
            ImportVar := Dictionary.CurrentSymbol (ImportIt);
            if IsExpandableConstituent (ImportVar, SubprogSym) then
               ImportNum := SymToNat (Dictionary.GetSubject (ImportVar));
            else
               ImportNum := SymToNat (ImportVar);
            end if;
            RelationAlgebra.InsertPair (TheHeap, RT, ExportNum, ImportNum);
            --if it's a mode in stream then add ImportNum, ImportNum to RT
            --and add ImportNum to RTDom
            if Dictionary.GetOwnVariableOrConstituentMode (ImportVar) =
               Dictionary.InMode then
               RelationAlgebra.InsertPair (TheHeap, RT, ImportNum, ImportNum);
               SeqAlgebra.AddMember (TheHeap, RTDom, ImportNum);
            end if;

            ImportIt := Dictionary.NextSymbol (ImportIt);
         end loop;
         ExportIt := Dictionary.NextSymbol (ExportIt);
      end loop;
   end StepsTwoAndThree;

   ----------------------------------------------

   procedure BuildRT (RT         : out RelationAlgebra.Relation;
                      RTDom      : out SeqAlgebra.Seq;
                      AUnchanged : out SeqAlgebra.Seq)
   --# global in     Dictionary.Dict;
   --#        in     SubProgSym;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives AUnchanged,
   --#         RTDom,
   --#         TheHeap               from Dictionary.Dict,
   --#                                    SubProgSym,
   --#                                    TheHeap &
   --#         RT                    from TheHeap &
   --#         Statistics.TableUsage from *,
   --#                                    Dictionary.Dict,
   --#                                    SubProgSym,
   --#                                    TheHeap;
   is
      LRT    : RelationAlgebra.Relation;
      LRTDom : SeqAlgebra.Seq;
   begin
      RelationAlgebra.CreateRelation (TheHeap, LRT);

      StepOne (LRT, LRTDom, AUnchanged);
      StepsTwoAndThree (LRT, LRTDom);

      RT := LRT;
      RTDom := LRTDom;
   end BuildRT;

   ----------------------------------------------

   procedure BuildR (R    : out RelationAlgebra.Relation;
                     RDom : out SeqAlgebra.Seq)
   --# global in     Dictionary.Dict;
   --#        in     SubProgSym;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives R,
   --#         RDom                  from TheHeap &
   --#         Statistics.TableUsage,
   --#         TheHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    SubProgSym,
   --#                                    TheHeap;
   is
      LR        : RelationAlgebra.Relation;
      LRDom     : SeqAlgebra.Seq;
      ExportVar : Dictionary.Symbol;
      ImportVar : Dictionary.Symbol;
      ExportNum : Natural;
      ImportNum : Natural;
      ExportIt,
      ImportIt  : Dictionary.Iterator;

   begin
      RelationAlgebra.CreateRelation (TheHeap, LR);
      SeqAlgebra.CreateSeq (TheHeap, LRDom);

      ExportIt := Dictionary.FirstExport (Dictionary.IsAbstract,
                                          SubprogSym);
      while not Dictionary.IsNullIterator (ExportIt) loop
         ExportVar := Dictionary.CurrentSymbol (ExportIt);
         ExportNum := SymToNat (ExportVar);
         SeqAlgebra.AddMember (TheHeap, LRDom, ExportNum);
         --if it's mode out stream add ExportNum, ExportNum to LR
         if Dictionary.GetOwnVariableOrConstituentMode (ExportVar) =
            Dictionary.OutMode then
            RelationAlgebra.InsertPair (TheHeap, LR, ExportNum, ExportNum);
         end if;

         ImportIt := Dictionary.FirstDependency (Dictionary.IsAbstract,
                                                 SubprogSym,
                                                 ExportVar);
         while not Dictionary.IsNullIterator (ImportIt) loop
            ImportVar := Dictionary.CurrentSymbol (ImportIt);
            ImportNum := SymToNat (ImportVar);
            RelationAlgebra.InsertPair (TheHeap, LR,
                                        ExportNum,
                                        ImportNum);
            --if it's a mode in stream then add ImportIt, ImportIt to LR
            --and add ImportIt to LRDom
            if Dictionary.GetOwnVariableOrConstituentMode (ImportVar) =
               Dictionary.InMode then
               RelationAlgebra.InsertPair (TheHeap, LR, ImportNum, ImportNum);
               SeqAlgebra.AddMember (TheHeap, LRDom, ImportNum);
            end if;

            ImportIt := Dictionary.NextSymbol (ImportIt);
         end loop;
         ExportIt := Dictionary.NextSymbol (ExportIt);
      end loop;
      R := LR;
      RDom := LRDom;
   end BuildR;

   ----------------------------------------------

   procedure TypeOneErrors (R          : in RelationAlgebra.Relation;
                            RDom       : in SeqAlgebra.Seq;
                            RT         : in RelationAlgebra.Relation;
                            RTDom      : in SeqAlgebra.Seq;
                            AUnchanged : in SeqAlgebra.Seq)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     Position;
   --#        in     SubProgSym;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from AUnchanged,
   --#                                        CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Position,
   --#                                        R,
   --#                                        RDom,
   --#                                        RT,
   --#                                        RTDom,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        SubProgSym,
   --#                                        TheHeap &
   --#         Statistics.TableUsage,
   --#         TheHeap                   from *,
   --#                                        R,
   --#                                        RDom,
   --#                                        RT,
   --#                                        RTDom,
   --#                                        TheHeap;
   is
      RRow      : SeqAlgebra.Seq;
      RTRow     : SeqAlgebra.Seq;
      Diff      : SeqAlgebra.Seq;
      RExp      : SeqAlgebra.MemberOfSeq;
      ErrImp    : SeqAlgebra.MemberOfSeq;
      SubprogScope : Dictionary.Scopes;
   begin
      SubprogScope := Dictionary.GetScope (SubprogSym);
      RExp := SeqAlgebra.FirstMember (TheHeap, RDom);
      loop
         exit when SeqAlgebra.IsNullMember (RExp);
         if not SeqAlgebra.IsMember (TheHeap,
                                     RTDom,
                                     SeqAlgebra.ValueOfMember (TheHeap, RExp))
         then
            -- special case: we do not want to issue Error 1 for an abstract own
            -- variable of mode in since a message saying we have failed to update
            -- such a variable is bound to misleading (the updating is only implicit anyway)
            if Dictionary.GetOwnVariableOrConstituentMode (MemToSym (RExp)) /=
               Dictionary.InMode then
               ErrorHandler.DepSemanticErrorSym (1,
                                                 Position,
                                                 MemToSym (RExp),
                                                 Dictionary.NullSymbol,
                                                 SubprogScope);
            end if;
         else
            RelationAlgebra.RowExtraction (TheHeap,
                                           R,
                                           SeqAlgebra.ValueOfMember (TheHeap, RExp),
                                           RRow);
            RelationAlgebra.RowExtraction (TheHeap,
                                           RT,
                                           SeqAlgebra.ValueOfMember (TheHeap, RExp),
                                           RTRow);
            SeqAlgebra.Complement (TheHeap, RTRow, RRow, Diff);
            ErrImp := SeqAlgebra.FirstMember (TheHeap, Diff);
            loop
               exit when SeqAlgebra.IsNullMember (ErrImp);
               if MemToSym (ErrImp) = MemToSym (RExp) and then
                  SeqAlgebra.IsMember (TheHeap,
                                       AUnchanged,
                                       SeqAlgebra.ValueOfMember (TheHeap, ErrImp))
               then
                  ErrorHandler.DepSemanticErrorSym (5,
                                                    Position,
                                                    MemToSym (RExp),
                                                    Dictionary.NullSymbol,
                                                    SubprogScope);
               else
                  ErrorHandler.DepSemanticErrorSym (4,
                                                    Position,
                                                    MemToSym (RExp),
                                                    MemToSym (ErrImp),
                                                    SubprogScope);
               end if;
               ErrImp := SeqAlgebra.NextMember (TheHeap, ErrImp);
            end loop;
         end if;
         RExp := SeqAlgebra.NextMember (TheHeap, RExp);
      end loop;
   end TypeOneErrors;

   ----------------------------------------------

   procedure TypeTwoErrors (R          : in RelationAlgebra.Relation;
                            RDom       : in SeqAlgebra.Seq;
                            RT         : in RelationAlgebra.Relation;
                            RTDom      : in SeqAlgebra.Seq)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     Position;
   --#        in     SubProgSym;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives ErrorHandler.ErrorContext,
   --#         SPARK_IO.FILE_SYS         from CommandLineData.Content,
   --#                                        Dictionary.Dict,
   --#                                        ErrorHandler.ErrorContext,
   --#                                        LexTokenManager.StringTable,
   --#                                        Position,
   --#                                        R,
   --#                                        RDom,
   --#                                        RT,
   --#                                        RTDom,
   --#                                        SPARK_IO.FILE_SYS,
   --#                                        SubProgSym,
   --#                                        TheHeap &
   --#         Statistics.TableUsage,
   --#         TheHeap                   from *,
   --#                                        Dictionary.Dict,
   --#                                        R,
   --#                                        RDom,
   --#                                        RT,
   --#                                        RTDom,
   --#                                        TheHeap;
   is
      RRow      : SeqAlgebra.Seq;
      RTRow     : SeqAlgebra.Seq;
      Diff      : SeqAlgebra.Seq;
      RTExp     : SeqAlgebra.MemberOfSeq;
      ErrImp    : SeqAlgebra.MemberOfSeq;
      SubprogScope : Dictionary.Scopes;
   begin
      SubprogScope := Dictionary.GetScope (SubprogSym);
      RTExp := SeqAlgebra.FirstMember (TheHeap, RTDom);
      loop
         exit when SeqAlgebra.IsNullMember (RTExp);
         if not SeqAlgebra.IsMember (TheHeap,
                                     RDom,
                                     SeqAlgebra.ValueOfMember (TheHeap, RTExp)) and then
           MemToSym (RTExp) /= Dictionary.GetNullVariable -- don't report refinement errors on Null
         then
            ErrorHandler.DepSemanticErrorSym (2,
                                              Position,
                                              MemToSym (RTExp),
                                              Dictionary.NullSymbol,
                                              SubprogScope);
         else
            RelationAlgebra.RowExtraction (TheHeap,
                                           R,
                                           SeqAlgebra.ValueOfMember (TheHeap, RTExp),
                                           RRow);
            RelationAlgebra.RowExtraction (TheHeap,
                                           RT,
                                           SeqAlgebra.ValueOfMember (TheHeap, RTExp),
                                           RTRow);
            SeqAlgebra.Complement (TheHeap, RRow, RTRow, Diff);
            ErrImp := SeqAlgebra.FirstMember (TheHeap, Diff);
            loop
               exit when SeqAlgebra.IsNullMember (ErrImp);

               -- guard to avoid spurious refinement errors in DFA mode
               if CommandLineData.Content.DoInformationFlow or else
                  MemToSym (RTExp) = MemToSym (ErrImp)
               then
                  ErrorHandler.DepSemanticErrorSym (3,
                                                    Position,
                                                    MemToSym (RTExp),
                                                    MemToSym (ErrImp),
                                                    SubprogScope);
               end if;
               ErrImp := SeqAlgebra.NextMember (TheHeap, ErrImp);
            end loop;
         end if;
         RTExp := SeqAlgebra.NextMember (TheHeap, RTExp);
      end loop;
   end TypeTwoErrors;

begin --CheckDerivesConsistency
   BuildRT (RT, RTDom, AUnchanged);
   BuildR (R, RDom);
   TypeOneErrors (R, RDom, RT, RTDom, AUnchanged);
   TypeTwoErrors (R, RDom, RT, RTDom);

   RelationAlgebra.DisposeOfRelation (TheHeap, RT);
   RelationAlgebra.DisposeOfRelation (TheHeap, R);
   SeqAlgebra.DisposeOfSeq (TheHeap, RTDom);
   SeqAlgebra.DisposeOfSeq (TheHeap, RDom);
   SeqAlgebra.DisposeOfSeq (TheHeap, AUnchanged);

   Heap.ReportUsage (TheHeap);
end CheckDerivesConsistency;
