-- $Id: sem-compunit-calcbinaryoperator.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

-- This    procedure    is    to   expression    evaluation    what       --
-- CheckBinaryOperator  is to  type  checking.  It  is called  from       --
-- wf_term,  wf_simple_expression,  wf_relation, wf_expression  and       --
-- wf_factor  to  calculate   effect  of  binary  operators.   This       --
-- procedure    is    called    immediately    after    calls    to       --
-- CheckBinaryOperator  so  that   if  the  sub-expression  is  not       --
-- wellformed  then Result  =  UnknownTypeRecord on  entry to  this       --
-- procedure.                                                             --
----------------------------------------------------------------------------

separate (Sem.CompUnit)
procedure CalcBinaryOperator (Node           : in     STree.SyntaxNode;
                              Operator       : in     SPSymbols.SPSymbol;
                              LeftVal,
                                 RightVal            : in     Maths.Value;
                              IsAnnotation   : in     Boolean;
                              Result         : in out ExpRecord)
is
   type ErrLookUp is array (Boolean) of Integer;
   WhichErr : constant ErrLookUp := ErrLookUp'(False => 402, True => 399);

   Err       : Maths.ErrorCode;
   Ans       : Maths.Value;

   procedure ApplyModulusIfNecessary
   --# global in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     Result;
   --#        in out Ans;
   --#        in out Err;
   --# derives Ans,
   --#         Err from Ans,
   --#                  Dictionary.Dict,
   --#                  Err,
   --#                  LexTokenManager.StringTable,
   --#                  Result;
   is
      TheModulusString : LexTokenManager.LexString;
      TempArg          : Maths.Value;
   begin
      if Err = Maths.NoError then
         if Dictionary.TypeIsModular (Result.TypeSymbol) then

            TheModulusString := Dictionary.GetScalarAttributeValue
               (Base     => False,
                Name     => LexTokenManager.ModulusToken,
                TypeMark => Result.TypeSymbol);

            TempArg := Ans;
            Maths.Modulus (FirstNum  => TempArg,
                           SecondNum => Maths.ValueRep (TheModulusString),
                           -- to get
                           Result    => Ans,
                           Ok        => Err);
         end if;
      end if;
   end ApplyModulusIfNecessary;

begin
   Err := Maths.NoError;
   if Result /= UnknownTypeRecord then
      -- expression was wellformed so we must try and calculate result
      case Operator is
         when SPSymbols.multiply =>
            Maths.Multiply (LeftVal,
                            RightVal,
                              -- to get
                            Ans,
                            Err);

            -- multiply might need a modulo reduction (See LRM 4.5.5(10))
            ApplyModulusIfNecessary;

         when SPSymbols.divide =>
            Maths.Divide (LeftVal,
                          RightVal,
                           -- to get
                          Ans,
                          Err);

            -- Divide never needs a modulo reduction (See LRM 4.5.5(10))

         when SPSymbols.RWmod =>
            Maths.Modulus (LeftVal,
                           RightVal,
                           -- to get
                           Ans,
                           Err);

            -- mod never needs a modulo reduction (See LRM 4.5.5(10))

         when SPSymbols.RWrem =>
            Maths.Remainder (LeftVal,
                             RightVal,
                              -- to get
                             Ans,
                             Err);

            -- rem never needs a modulo reduction (See LRM 4.5.5(10))

         when SPSymbols.plus =>
            Maths.Add (LeftVal,
                       RightVal,
                        -- to get
                       Ans,
                       Err);

            -- plus might need a modulo reduction (See LRM 4.5.3(11))
            ApplyModulusIfNecessary;

         when SPSymbols.minus =>
            Maths.Subtract (LeftVal,
                            RightVal,
                              -- to get
                            Ans,
                            Err);

            -- minus might need a modulo reduction (See LRM 4.5.3(11))
            ApplyModulusIfNecessary;

         when SPSymbols.RWand |
            SPSymbols.RWandthen =>
            Ans := Maths.AndOp (LeftVal, RightVal);

         when SPSymbols.RWor |
            SPSymbols.RWorelse =>
            Ans := Maths.OrOp (LeftVal, RightVal);

         when SPSymbols.RWxor =>
            Ans := Maths.XorOp (LeftVal, RightVal);

         when SPSymbols.double_star =>
            Maths.RaiseByPower (LeftVal,
                                RightVal,
                                 -- to get
                                Ans,
                                Err);

            -- ** might need a modulo reduction (See LRM 4.5.6(11))
            ApplyModulusIfNecessary;

         when SPSymbols.equals =>
            if LeftVal = Maths.NoValue or else RightVal = Maths.NoValue then
               Ans := Maths.NoValue;
            else
               Ans := Maths.BoolToValue (LeftVal = RightVal);
            end if;
            Err := Maths.NoError;

         when SPSymbols.not_equal =>
            if LeftVal = Maths.NoValue or else RightVal = Maths.NoValue then
               Ans := Maths.NoValue;
            else
               Ans := Maths.BoolToValue (LeftVal /= RightVal);
            end if;
            Err := Maths.NoError;

         when SPSymbols.less_than =>
            Maths.Lesser (LeftVal,
                          RightVal,
                           -- to get
                          Ans,
                          Err);

         when SPSymbols.less_or_equal =>
            Maths.LesserOrEqual (LeftVal,
                                 RightVal,
                                 -- to get
                                 Ans,
                                 Err);

         when SPSymbols.greater_or_equal =>
            Maths.GreaterOrEqual (LeftVal,
                                  RightVal,
                                    -- to get
                                  Ans,
                                  Err);

         when SPSymbols.greater_than =>
            Maths.Greater (LeftVal,
                           RightVal,
                           -- to get
                           Ans,
                           Err);

         when others =>
            SystemErrors.FatalError (SystemErrors.MathError, "in CalcBinaryOperator");
            Ans := Maths.NoValue; -- define Ans here to avoid subsequent flow errors
      end case;

      Result.Value := Ans;

      case Err is
         when Maths.NoError =>
            null;

         when Maths.DivideByZero =>
            ErrorHandler.SemanticError (400, ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        LexTokenManager.NullString);

         when Maths.ConstraintError =>
            ErrorHandler.SemanticError (WhichErr (IsAnnotation),
                                        ErrorHandler.NoReference,
                                        NodePosition (Node),
                                        LexTokenManager.NullString);

         when Maths.OverFlow =>
            Result.Value := Maths.NoValue;
            ErrorHandler.SemanticWarning (200,
                                          NodePosition (Node),
                                          LexTokenManager.NullString);


         when others => -- indicates internal error in maths package
            SystemErrors.FatalError (SystemErrors.MathError, "in CalcBinaryOperator (2nd case)");
      end case;
   end if;
end CalcBinaryOperator;
