-- $Id: sem-compunit-calcattribute.adb 11354 2008-10-06 17:02:56Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (Sem.CompUnit)
procedure CalcAttribute (Node         : in     STree.SyntaxNode;
                         AttribName   : in     LexTokenManager.LexString;
                         PrefixKind   : in     Dictionary.PrefixSort;
                         Prefix       : in     Dictionary.Symbol;
                         BaseFound    : in     Boolean;
                         IsAnnotation : in     Boolean;
                         Argument     : in out Maths.Value;
                         RHSofRange   :    out Maths.Value)
is
   pragma Unreferenced (PrefixKind);

   type ErrLookUp is array (Boolean) of Integer;
   WhichErr : constant ErrLookUp := ErrLookUp'(False => 402, True => 399);

   Err           : Maths.ErrorCode;
   LocalBaseType : Dictionary.Symbol;
   ArgumentLocal : Maths.Value;

   procedure CalcArrayAttribute
   --# global in     AttribName;
   --#        in     BaseFound;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.StringTable;
   --#        in     Prefix;
   --#        in out Argument;
   --#        in out Err;
   --#        in out RHSofRange;
   --# derives Argument,
   --#         Err,
   --#         RHSofRange from *,
   --#                         Argument,
   --#                         AttribName,
   --#                         BaseFound,
   --#                         Dictionary.Dict,
   --#                         LexTokenManager.StringTable,
   --#                         Prefix;
   is
      Dimension : Positive;
      Vlow,
      Vhigh     : Maths.Value;

      procedure GetDimension (Argument : Maths.Value)
      --# global out Dimension;
      --# derives Dimension from Argument;
      is
         UnusedErr : Maths.ErrorCode;
      begin
         if Maths.HasNoValue (Argument) then
            Dimension := 1;
         else
            --# accept Flow, 10, UnusedErr, "Expected ineffective assignment";
            Maths.ValueToInteger (Argument, -- expect ineffective assign to UnusedErr
                                    -- to get
                                  Dimension,
                                  UnusedErr);
            --# end accept;
         end if;
         --# accept Flow, 33, UnusedErr, "Expected to be neither referenced nor exported";
      end GetDimension;

   begin -- CalcArrayAttribute
      if BaseFound then
         -- the only valid attribute would be size and we never know what
         -- the size of things is so we can only return the null value
         Argument := Maths.NoValue;
      elsif Dictionary.IsUnconstrainedArrayType (Prefix) then
         Argument := Maths.NoValue;
      else -- a constrained array type or subtype
         GetDimension (Argument);
         if AttribName = LexTokenManager.RangeToken then
            Argument := Maths.ValueRep
               (Dictionary.GetArrayAttributeValue (LexTokenManager.FirstToken,
                                                   Prefix,
                                                   Dimension));
            RHSofRange := Maths.ValueRep
               (Dictionary.GetArrayAttributeValue (LexTokenManager.LastToken,
                                                   Prefix,
                                                   Dimension));
         elsif AttribName = LexTokenManager.LengthToken then
            Vlow := Maths.ValueRep
               (Dictionary.GetArrayAttributeValue (LexTokenManager.FirstToken,
                                                   Prefix,
                                                   Dimension));
            Vhigh := Maths.ValueRep
               (Dictionary.GetArrayAttributeValue (LexTokenManager.LastToken,
                                                   Prefix,
                                                   Dimension));
            --# accept Flow, 10, Err, "Expected ineffective assignment";
            Maths.Subtract (Vhigh,  -- flow error expected
                            Vlow,
                            Argument,
                            Err);
            --# end accept;
            Vhigh := Argument;
            Maths.Add (Vhigh,
                       Maths.OneInteger,
                       Argument,
                       Err);

         else -- first/last
            Argument := Maths.ValueRep
               (Dictionary.GetArrayAttributeValue (AttribName,
                                                   Prefix,
                                                   Dimension));
         end if;
      end if;
   end CalcArrayAttribute;

begin -- CalcAttribute
      -- this procedure is only called wf_attribute_designator if the attribute
      -- is well-formed.

   RHSofRange := Maths.NoValue;  -- default value unless 'RANGE processed

   if AttribName = LexTokenManager.SuccToken then
      Maths.SuccOp (Argument,
                    Err);

      LocalBaseType := Dictionary.GetRootType (Prefix);

      if Dictionary.TypeIsModular (LocalBaseType) then

         Maths.Modulus (FirstNum  => Argument,
                        SecondNum => Maths.ValueRep
                          (Dictionary.GetScalarAttributeValue
                           (Base     => False,
                            Name     => LexTokenManager.ModulusToken,
                            TypeMark => LocalBaseType)),
                        -- to get
                        Result    => ArgumentLocal,
                        Ok        => Err);

      else
         ConstraintCheck (Argument,
                          ArgumentLocal,
                          IsAnnotation,
                          LocalBaseType,
                          NodePosition (Node));
      end if;
      Argument := ArgumentLocal;

   elsif AttribName =  LexTokenManager.PredToken then
      Maths.PredOp (Argument,
                    Err);

      LocalBaseType := Dictionary.GetRootType (Prefix);

      if Dictionary.TypeIsModular (LocalBaseType) then

         Maths.Modulus (FirstNum  => Argument,
                        SecondNum => Maths.ValueRep
                          (Dictionary.GetScalarAttributeValue
                           (Base     => False,
                            Name     => LexTokenManager.ModulusToken,
                            TypeMark => LocalBaseType)),
                        -- to get
                        Result    => ArgumentLocal,
                        Ok        => Err);

      else
         ConstraintCheck (Argument,
                          ArgumentLocal,
                          IsAnnotation,
                          LocalBaseType,
                          NodePosition (Node));
      end if;
      Argument := ArgumentLocal;

   elsif AttribName = LexTokenManager.FloorToken then
      Maths.Floor (Argument, ArgumentLocal, Err);
      Argument := ArgumentLocal;
      LocalBaseType := Dictionary.GetRootType (Prefix);
      ConstraintCheck (Argument,
                       ArgumentLocal,
                       IsAnnotation,
                       LocalBaseType,
                       NodePosition (Node));
      Argument := ArgumentLocal;

   elsif AttribName = LexTokenManager.CeilingToken then
      Maths.Ceiling (Argument, ArgumentLocal, Err);
      Argument := ArgumentLocal;
      LocalBaseType := Dictionary.GetRootType (Prefix);
      ConstraintCheck (Argument,
                       ArgumentLocal,
                       IsAnnotation,
                       LocalBaseType,
                       NodePosition (Node));
      Argument := ArgumentLocal;

   elsif AttribName =  LexTokenManager.ValToken then
      Err := Maths.NoError;
      -- upper and lower bounds check required

   elsif AttribName =  LexTokenManager.PosToken then
      Err := Maths.NoError; -- no action required, no error can occur

   elsif AttribName =  LexTokenManager.ValidToken then
      Err := Maths.NoError; -- no action required, no error can occur

   elsif AttribName =  LexTokenManager.SizeToken then
      Err := Maths.NoError;
      Argument := Maths.ValueRep (Dictionary.TypeSizeAttribute (Prefix));

   elsif Dictionary.TypeIsScalar (Prefix) then
      Err := Maths.NoError;
      Argument := Maths.ValueRep (Dictionary.GetScalarAttributeValue (BaseFound,
                                                                      AttribName,
                                                                      Prefix));

   elsif Dictionary.TypeIsArray (Prefix) then
      Err := Maths.NoError;
      CalcArrayAttribute;

   else -- non-implemented attribute - should never occur
      Argument := Maths.NoValue;
      Err := Maths.NoError;
   end if;

   case Err is
      when Maths.NoError =>
         null;
      when Maths.DivideByZero =>
         Argument := Maths.NoValue;
         ErrorHandler.SemanticError (400,
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     LexTokenManager.NullString);
      when Maths.ConstraintError =>
         Argument := Maths.NoValue;
         ErrorHandler.SemanticError (WhichErr (IsAnnotation),
                                     ErrorHandler.NoReference,
                                     NodePosition (Node),
                                     LexTokenManager.NullString);
      when Maths.OverFlow =>
         Argument := Maths.NoValue;
         ErrorHandler.SemanticWarning (200,
                                       NodePosition (Node),
                                       LexTokenManager.NullString);
      when others => -- indicates internal error in maths package
         Argument := Maths.NoValue;
         SystemErrors.FatalError (SystemErrors.MathError, "in CalcAttribute");
   end case;
   --# accept Flow, 30, PrefixKind, "Expected to be unused";
end CalcAttribute; -- expect PrefixKind unused
