-- $Id: metafile.adb 11889 2008-12-12 15:49:12Z rod chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

-- Provides routines to support analysis of "files of files"                  --
--------------------------------------------------------------------------------

with StringUtilities,
     SystemErrors,
     FileSystem,
     CommandLineData,
     Ada.Characters.Handling,
     Ada.Characters.Latin_1,
     ScreenEcho,
     XMLReport,
     ErrorHandler;

package body MetaFile
is

   EmptyMetaFileElement : constant MetaFileElement :=
      MetaFileElement'(SPARK_IO.Null_File, EStrings.EmptyString);

   NullMetaFile : constant MetaFiles :=
      MetaFiles'(Vector => Vectors'(Indexes => EmptyMetaFileElement),
                 Ptr => 0);

   function IsEmpty (TheMetaFile : MetaFiles) return Boolean
   is
   begin
      return TheMetaFile.Ptr = 0;
   end IsEmpty;

   function IsWhiteSpace (SpaceChar : Character) return Boolean
   is
   begin
      return (SpaceChar = ' ') or
         (SpaceChar = Ada.Characters.Latin_1.HT) or
         (SpaceChar = Ada.Characters.Latin_1.CR);
   end IsWhiteSpace;


   procedure Push (TheFile     : in     SPARK_IO.File_Type;
                   ThePath     : in     EStrings.T;
                   TheMetaFile : in out MetaFiles)
   --# derives TheMetaFile from *,
   --#                          TheFile,
   --#                          ThePath;
   is
   begin
      if TheMetaFile.Ptr = Ptrs'Last then
         SystemErrors.FatalError (SystemErrors.MetaFileStackOverflow, "");
      end if;
      TheMetaFile.Ptr := TheMetaFile.Ptr + 1;
      TheMetaFile.Vector (TheMetaFile.Ptr) := MetaFileElement'(TheFile, ThePath);
   end Push;

   procedure Pop (TheMetaFile : in out MetaFiles;
                  TheFile     :    out SPARK_IO.File_Type;
                  ThePath     :    out EStrings.T)
   --# derives TheFile,
   --#         TheMetaFile,
   --#         ThePath     from TheMetaFile;
   --# pre not IsEmpty (TheMetaFile);
   is
   begin
      TheFile := TheMetaFile.Vector (TheMetaFile.Ptr).File_Handle;
      ThePath := TheMetaFile.Vector (TheMetaFile.Ptr).PathName;
      TheMetaFile.Ptr := TheMetaFile.Ptr - 1;
   end Pop;

   function StripAt (S : EStrings.T)
                    return EStrings.T
   is
   begin
      return StringUtilities.Section (S, 2, S.Length - 1);
   end StripAt;

   function StripLeadingSpaces (S : EStrings.T)
                               return EStrings.T
   is
      Ptr : EStrings.Positions := 1;
   begin
      loop
         exit when Ptr > S.Length;
         exit when not IsWhiteSpace (S.Content (Ptr));
         Ptr := Ptr + 1;
      end loop;
      return StringUtilities.Section (S, Ptr, (S.Length - Ptr) + 1);
   end StripLeadingSpaces;

   function StripComments (S : EStrings.T)
                          return EStrings.T
   is
      Res : EStrings.T;
   begin
      if S.Content (1) = '-' and then S.Content (2) = '-' then
         Res := EStrings.EmptyString;
      else
         Res := S;
      end if;
      return Res;
   end StripComments;

   function LineIsEmpty (S : EStrings.T) return Boolean
   is
   begin
      return S.Length = 0;
   end LineIsEmpty;

   function IsVaxOrNt return Boolean
   is
   begin
      return not FileSystem.UseUnixCommandLine;
   end IsVaxOrNt;

   function GetMetafileName (S : EStrings.T)
                            return EStrings.T
   is
      Res            : EStrings.T := EStrings.EmptyString;
      unused         : Boolean;
      InQuotedString : Boolean;
   begin
      InQuotedString := False;

      for I in EStrings.Positions range 1 .. S.Length loop
         exit when IsWhiteSpace (S.Content (I)) and not InQuotedString;

         -- allow for quoted strings containing spaces
         if S.Content (I) = Ada.Characters.Latin_1.Quotation then
            InQuotedString := not InQuotedString;
         else
            if S.Content (I) /= '@' then
               --# accept Flow, 10, unused, "Expected ineffective assignment to unused";
               EStrings.AppendChar (Res, S.Content (I), unused);
               --# end accept;
            end if;
         end if;
      end loop;
      FileSystem.CheckExtension (Res, CommandLineData.MetaFileExtension);
      --# accept Flow, 33, unused, "Expected ineffective assignment to unused";
      return Res;
   end GetMetafileName;

   -- Exported Meta File Operations

   procedure Create (FileName    : in     EStrings.T;
                     TheMetaFile :    out MetaFiles)
   is
      TheFile : SPARK_IO.File_Type := SPARK_IO.Null_File;
      Status  : SPARK_IO.File_Status;
      MetaFileLocal : MetaFiles := NullMetaFile;
      FileNameLocal : EStrings.T;
      FileNameFull  : EStrings.T;
      FindStatus : FileSystem.TypFileSpecStatus;
   begin
      FileNameLocal := StripAt (FileName);
      SPARK_IO.Open (TheFile,
                     SPARK_IO.In_File,
                     FileNameLocal.Length,
                     FileNameLocal.Content,
                     "",
                     -- to get
                     Status);
      if Status = SPARK_IO.Ok then
         --# accept F, 10, FindStatus, "Known to be ineffective, must be true at this point";
         FileSystem.FindFullFileName (FileNameLocal, FindStatus, FileNameFull);
         --# end accept;
         Push (TheFile, FileNameFull, MetaFileLocal);
      else
         ScreenEcho.New_Line (1);
         ScreenEcho.Put_String ("Cannot open file ");
         if CommandLineData.Content.PlainOutput then
            ScreenEcho.Put_ExaminerLine (EStrings.LowerCase (
               FileSystem.JustFile (FileNameLocal, True)));
         else
            ScreenEcho.Put_ExaminerLine (FileNameLocal);
         end if;
         ScreenEcho.New_Line (1);
      end if;
      TheMetaFile := MetaFileLocal;
      --# accept F, 33, FindStatus, "Known to be ineffective, must be true at this point";
   end Create;

   procedure NextName (TheMetaFile    : in out MetaFiles;
                       TheFileName    :    out EStrings.T;
                       DoListing      :    out Boolean;
                       TheListingName :    out EStrings.T;
                       FileFound      :    out Boolean)
   is
      TheFile        : SPARK_IO.File_Type;
      TmpFileName    : EStrings.T;
      ThePath        : EStrings.T;
      CurrentLine    : EStrings.T;
      DataLineFound  : Boolean;
      FileFoundLocal : Boolean;
      Unused,
      Status         : SPARK_IO.File_Status;

      procedure Parse (CurrentLine    : in     EStrings.T;
                       TheFileName    :    out EStrings.T;
                       DoListing      :    out Boolean;
                       TheListingName :    out EStrings.T)
         --# global in CommandLineData.Content;
         --# derives DoListing      from CurrentLine &
         --#         TheFileName,
         --#         TheListingName from CommandLineData.Content,
         --#                             CurrentLine;
      is
         Fname,
         Lname,
         Switch         : EStrings.T := EStrings.EmptyString;
         Ptr            : EStrings.Positions := 1;
         OptionMatch,
         unused         : Boolean;
         DoListingLocal : Boolean := True;
         InQuotedString : Boolean;

         procedure CheckOptionName (OptName : in     EStrings.T;
                                    Str     : in     String;
                                    Ok      :    out Boolean)
            --# derives OK from OptName,
            --#                 Str;
         is
            LOk : Boolean := False;
         begin
            if OptName.Length <= Str'Length then
               for I in EStrings.Lengths range 1 .. OptName.Length loop
                  LOk := Ada.Characters.Handling.To_Lower (OptName.Content (I)) =
                     Ada.Characters.Handling.To_Lower (Str (I));
                  exit when not LOk;
               end loop;
            end if;
            Ok := LOk;
         end CheckOptionName;

      begin -- Parse
         InQuotedString := False;
         loop
            exit when IsVaxOrNt and then CurrentLine.Content (Ptr) = FileSystem.SwitchCharacter;
            exit when IsWhiteSpace (CurrentLine.Content (Ptr)) and not InQuotedString;

            -- allow for quoted strings containing spaces
            if CurrentLine.Content (Ptr) = Ada.Characters.Latin_1.Quotation then
               InQuotedString := not InQuotedString;
            else
               --# accept Flow, 10, unused, "Expected ineffective assignment to unused";
               EStrings.AppendChar (Fname, CurrentLine.Content (Ptr), unused);
               --# end accept;
            end if;

            exit when Ptr = CurrentLine.Length;
            Ptr := Ptr + 1;
         end loop;
         FileSystem.CheckExtension (Fname, CommandLineData.Content.SourceExtension);

         -- at this point Fname has the whole of any file name and Ptr either points
         -- at a switch character or at the end of CurrentLine
         -- or at a space

         -- skip leading spaces before possible switch character
         if Ptr < CurrentLine.Length and then
            IsWhiteSpace (CurrentLine.Content (Ptr))
         then
            Ptr := Ptr + 1;
            loop
               -- To make metafiles compatible across platforms, we allow '-'
               -- as a switch character when it follows whitespace on all platforms.
               -- On platforms where '-' is already the switch character, this makes
               -- on difference.
               exit when (CurrentLine.Content (Ptr) = FileSystem.SwitchCharacter or
                          CurrentLine.Content (Ptr) = '-');

               exit when Ptr = CurrentLine.Length;
               Ptr := Ptr + 1;
            end loop;
         end if;

         -- At this point Ptr either points at a switch character or at the end of CurrentLine
         if (CurrentLine.Content (Ptr) = FileSystem.SwitchCharacter) or
            (CurrentLine.Content (Ptr) = '-') then

            Ptr := Ptr + 1;
            loop
               exit when IsWhiteSpace (CurrentLine.Content (Ptr));
               exit when CurrentLine.Content (Ptr) = '=';
               --# accept Flow, 10, unused, "Expected ineffective assignment to unused";
               EStrings.AppendChar (Switch, CurrentLine.Content (Ptr), unused);
               --# end accept;
               exit when Ptr = CurrentLine.Length;
               Ptr := Ptr + 1;
            end loop;

            -- at this point we have any command line argument in variable 'switch'
            CheckOptionName (Switch, "nolisting_file", OptionMatch);
            if OptionMatch then
               DoListingLocal := False;
            else
               CheckOptionName (Switch, "listing_file", OptionMatch);
               if OptionMatch then
                  -- the user has given a specific name to the listing file
                  -- first skip the '=' and any leading spaces
                  loop
                     exit when (not IsWhiteSpace (CurrentLine.Content (Ptr))) and
                        CurrentLine.Content (Ptr) /= '=';
                     exit when Ptr = CurrentLine.Length;
                     Ptr := Ptr + 1;
                  end loop;
                  -- we are either at the end of the line (error no file name provided)
                  -- at the start of a comment (error no file name provided)
                  -- at the start of the listing file name
                  if Ptr = CurrentLine.Length or else CurrentLine.Content (Ptr) = '-' then
                     -- error case, use default listing file name
                     Lname := FileSystem.JustFile (Fname, False);

                     FileSystem.CheckListingExtension (Fname,
                                                       Lname,
                                                       CommandLineData.Content.ListingExtension);
                  else -- ok case
                     InQuotedString := False;
                     loop
                        exit when IsWhiteSpace (CurrentLine.Content (Ptr)) and
                           not InQuotedString;

                        -- allow for quoted strings containing spaces
                        if CurrentLine.Content (Ptr) = Ada.Characters.Latin_1.Quotation then
                           InQuotedString := not InQuotedString;
                        else
                           --# accept Flow, 10, unused, "Expected ineffective assignment to unused";
                           EStrings.AppendChar (Lname, CurrentLine.Content (Ptr), unused);
                           --# end accept;
                        end if;

                        exit when Ptr = CurrentLine.Length;
                        Ptr := Ptr + 1;
                     end loop;


                     FileSystem.CheckListingExtension (Fname,
                                                       Lname,
                                                       CommandLineData.Content.ListingExtension);
                  end if;
               else -- invalid switch, treat as if no switch given
                  Lname := FileSystem.JustFile (Fname, False);


                  FileSystem.CheckListingExtension (Fname,
                                                    Lname,
                                                    CommandLineData.Content.ListingExtension);
               end if;
            end if;

         else  -- no switch found so listing is on and listing name is default listing name
            Lname := FileSystem.JustFile (Fname, False);


            FileSystem.CheckListingExtension (Fname,
                                              Lname,
                                              CommandLineData.Content.ListingExtension);
         end if;


         TheFileName := Fname;
         DoListing := DoListingLocal;
         TheListingName := Lname;
         --# accept Flow, 33, unused, "Expected unused to be neither referenced or exported";
      end Parse;


   begin -- NextName
      TheFileName := EStrings.EmptyString;
      TheListingName := EStrings.EmptyString;
      DoListing := False;
      FileFoundLocal := False;
      loop
         exit when IsEmpty (TheMetaFile); -- fail exit

         Pop (TheMetaFile,
               -- to get
              TheFile,
              ThePath);

         loop -- look for non-empty line in current file
            DataLineFound := False;

            if SPARK_IO.End_Of_File (TheFile) then
               --# accept Flow, 10, unused, "Expected ineffective assignment to unused" &
               --#        Flow, 10, TheFile, "Expected ineffective assignment to TheFile";
               SPARK_IO.Close (TheFile, Unused);
               --# end accept;
               exit;
            end if;

            EStrings.GetLine (TheFile,
                                       -- to get
                                     CurrentLine);
            CurrentLine := StripComments (StripLeadingSpaces (CurrentLine));

            if not LineIsEmpty (CurrentLine) then
               DataLineFound := True;
               Push (TheFile, ThePath, TheMetaFile); -- put file back ready for next call

               if CurrentLine.Content (1) = '@' then
                  CurrentLine := GetMetafileName (CurrentLine);
                  -- Interpret this FileSpec relative to the current
                  -- metafile's location
                  CurrentLine := FileSystem.InterpretRelative (CurrentLine,
                                                               ThePath);
                  SPARK_IO.Open (TheFile,
                                 SPARK_IO.In_File,
                                 CurrentLine.Length,
                                 CurrentLine.Content,
                                 "",
                                 -- to get
                                 Status);
                  if Status = SPARK_IO.Ok then
                     Push (TheFile, CurrentLine, TheMetaFile);
                  elsif Status = SPARK_IO.Use_Error then
                     -- for GNAT we get Use_Error if the file is already open; this means recursive
                     -- meta files have been detected.
                     ScreenEcho.New_Line (1);
                     ScreenEcho.Put_String ("Circular reference found to file ");
                     --# accept Flow, 41, "Expect stable expression";
                     if CommandLineData.Content.PlainOutput then
                     --# end accept;
                        ScreenEcho.Put_ExaminerLine (EStrings.LowerCase (
                           FileSystem.JustFile (CurrentLine, True)));
                     else
                        ScreenEcho.Put_ExaminerLine (CurrentLine);
                     end if;
                     ScreenEcho.New_Line (1);
                  else
                     ScreenEcho.New_Line (1);
                     ScreenEcho.Put_String ("Cannot open file ");
                     ErrorHandler.SetFileOpenError;
                     --# accept Flow, 41, "Expect stable expression";
                     if CommandLineData.Content.PlainOutput then
                     --# end accept;
                        ScreenEcho.Put_ExaminerLine (EStrings.LowerCase (
                           FileSystem.JustFile (CurrentLine, True)));
                     else
                        ScreenEcho.Put_ExaminerLine (CurrentLine);
                     end if;
                     ScreenEcho.New_Line (1);
                  end if;

               else -- ordinary file found
                  FileFoundLocal := True;
                  Parse (CurrentLine,
                           -- to get
                         TmpFileName,
                         DoListing,
                         TheListingName);
                  -- Interpret this FileSpec relative to the current
                  -- metafile's location
                  TheFileName := FileSystem.InterpretRelative (TmpFileName,
                                                               ThePath);
               end if;
            end if;
            exit when DataLineFound;
         end loop; -- looping through file lines for a non-blank and valid one
         exit when FileFoundLocal;  -- success exit
      end loop;  -- processing stacked metafile entries
      FileFound := FileFoundLocal;
      --# accept Flow, 33, unused, "Expected unused to be neither referenced or exported";
   end NextName;


   procedure ReportFileContent (ToFile       : in     SPARK_IO.File_Type;
                                FileName     : in     EStrings.T;
                                MetaFileUsed : in out Boolean)
   is
      TheFile        : SPARK_IO.File_Type := SPARK_IO.Null_File;
      Status,
      Unused         : SPARK_IO.File_Status;
      unused_bool,
      DataLineFound  : Boolean;
      TheMetaFile    : MetaFiles := NullMetaFile;
      CurrentLine,
      FileNameLocal,
      ThePath        : EStrings.T;
      FileNameFull   : EStrings.T;
      Ptr            : EStrings.Positions;
      InQuotedString : Boolean;
      FindStatus : FileSystem.TypFileSpecStatus;

      Margin         : Natural := 3;
      Offset         : constant Natural := 3;

      procedure PrintFileName (ToFile   : in SPARK_IO.File_Type;
                               FileName : in EStrings.T;
                               Margin   : in Natural)
         --# global in     CommandLineData.Content;
         --#        in out SPARK_IO.File_Sys;
         --# derives SPARK_IO.File_Sys from *,
         --#                                CommandLineData.Content,
         --#                                FileName,
         --#                                Margin,
         --#                                ToFile;
      is
      begin
         for I in Natural range 1 .. Margin loop
            SPARK_IO.Put_Char (ToFile, ' ');
         end loop;
         if CommandLineData.Content.PlainOutput then
            EStrings.PutLine (ToFile, EStrings.LowerCase (FileName));
         else
            EStrings.PutLine (ToFile, FileName);
         end if;
      end PrintFileName;

      procedure IncMargin
         --# global in out Margin;
         --# derives Margin from *;
      is
      begin
         Margin := Margin + Offset;
      end IncMargin;

      procedure DecMargin
         --# global in out Margin;
         --# derives Margin from *;
      is
      begin
         Margin := Margin - Offset;
      end DecMargin;

   begin -- ReportFileContent

      FileNameLocal := StripAt (FileName);

      SPARK_IO.Open (TheFile,
                     SPARK_IO.In_File,
                     FileNameLocal.Length,
                     FileNameLocal.Content,
                     "",
                     -- to get
                     Status);
      if Status = SPARK_IO.Ok then
         if not MetaFileUsed then
            MetaFileUsed := True;
            if not CommandLineData.Content.XML then -- Expect stable expression
               SPARK_IO.New_Line (ToFile, 1);
               SPARK_IO.Put_Line (ToFile, "Meta File(s) used were:", 0);
            end if;
         end if;
         if CommandLineData.Content.XML then -- Expect stable expression
            XMLReport.StartMetaFile (FileNameLocal,
                                     ToFile);
         else
            PrintFileName (ToFile, FileNameLocal, Margin);
            IncMargin;
         end if;

         --# accept F, 10, FindStatus, "Known to be ineffective, must be true at this point";
         FileSystem.FindFullFileName (FileNameLocal, FindStatus, FileNameFull);
         --# end accept;
         Push (TheFile, FileNameFull, TheMetaFile);

      end if;

      loop
         exit when IsEmpty (TheMetaFile);
         Pop (TheMetaFile,
               -- to get
              TheFile,
              ThePath);
         loop -- look for non-empty line in current file
            DataLineFound := False;

            if SPARK_IO.End_Of_File (TheFile) then
               --# accept Flow, 10, TheFile, "Expected ineffective assignment to TheFile" &
               --#        Flow, 10, unused, "Expected ineffective assignment to unused";
               SPARK_IO.Close (TheFile, Unused);
               --# end accept;
               --# accept Flow, 41, "Expect stable expression";
               if CommandLineData.Content.XML then
               --# end accept;
                  XMLReport.EndMetaFile (ToFile);
               else
                  DecMargin;
               end if;
               exit;
            end if;

            EStrings.GetLine (TheFile,
                                       -- to get
                                     CurrentLine);
            CurrentLine := StripComments (StripLeadingSpaces (CurrentLine));

            if not LineIsEmpty (CurrentLine) then
               DataLineFound := True;
               Push (TheFile, ThePath, TheMetaFile); -- put file back ready for next call

               if CurrentLine.Content (1) = '@' then
                  CurrentLine := GetMetafileName (CurrentLine);
                  CurrentLine := FileSystem.InterpretRelative (CurrentLine,
                                                               ThePath);
                  SPARK_IO.Open (TheFile,
                                 SPARK_IO.In_File,
                                 CurrentLine.Length,
                                 CurrentLine.Content,
                                 "",
                                 -- to get
                                 Status);
                  if Status = SPARK_IO.Ok then
                     Push (TheFile, CurrentLine, TheMetaFile);

                     --# accept Flow, 41, "Expect stable expression";
                     if CommandLineData.Content.PlainOutput then
                     --# end accept;
                        CurrentLine := FileSystem.JustFile (CurrentLine, True);
                     end if;

                     --# accept Flow, 41, "Expect stable expression";
                     if CommandLineData.Content.XML then
                     --# end accept;
                        XMLReport.StartMetaFile (CurrentLine,
                                                 ToFile);
                     else
                        PrintFileName (ToFile, CurrentLine, Margin);
                        IncMargin;
                     end if;
                  end if;

               else -- ordinary file found
                  FileNameLocal := EStrings.EmptyString;
                  Ptr := 1;
                  InQuotedString := False;
                  loop
                     exit when IsVaxOrNt and then CurrentLine.Content (Ptr) = FileSystem.SwitchCharacter;
                     exit when IsWhiteSpace (CurrentLine.Content (Ptr)) and not InQuotedString;

                     -- allow for quoted strings containing spaces
                     if CurrentLine.Content (Ptr) = Ada.Characters.Latin_1.Quotation then
                        InQuotedString := not InQuotedString;
                     else
                        --# accept Flow, 10, unused_bool, "Expected ineffective assignment to unused_bool";
                        EStrings.AppendChar (FileNameLocal, CurrentLine.Content (Ptr), unused_bool);
                        --# end accept;
                     end if;
                     exit when Ptr = CurrentLine.Length;
                     Ptr := Ptr + 1;
                  end loop;

                  --# accept Flow, 41, "Expect stable expression";
                  if CommandLineData.Content.PlainOutput then
                  --# end accept;
                     FileNameLocal := FileSystem.JustFile (FileNameLocal, True);
                  else
                     FileNameLocal := FileSystem.InterpretRelative (FileNameLocal,
                                                                    ThePath);
                  end if;

                  --# accept Flow, 41, "Expect stable expression";
                  if CommandLineData.Content.XML then
                  --# end accept;
                     XMLReport.Filename (FileNameLocal,
                                         ToFile);
                  else
                     PrintFileName (ToFile, FileNameLocal, Margin);
                  end if;
               end if;
            end if;
            exit when DataLineFound;
         end loop; -- looping through file lines for a non-blank and valid one
      end loop;
      --# accept Flow, 33, unused, "Expected unused to be neither referenced or exported" &
      --#        Flow, 33, unused_bool, "Expected unused_bool to be neither referenced or exported" &
      --#        Flow, 33, FindStatus, "Expected FindStatus to be neither referenced or exported";
   end ReportFileContent;
end MetaFile;
